"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MdExtractLinkDefinitionCodeActionProvider = void 0;
exports.createAddDefinitionEdit = createAddDefinitionEdit;
const l10n = require("@vscode/l10n");
const lsp = require("vscode-languageserver-protocol");
const documentLink_1 = require("../../types/documentLink");
const position_1 = require("../../types/position");
const range_1 = require("../../types/range");
const textDocument_1 = require("../../types/textDocument");
const editBuilder_1 = require("../../util/editBuilder");
const organizeLinkDefs_1 = require("../organizeLinkDefs");
const util_1 = require("./util");
const path_1 = require("../../util/path");
class MdExtractLinkDefinitionCodeActionProvider {
    static genericTitle = l10n.t('Extract to link definition');
    static #kind = lsp.CodeActionKind.RefactorExtract + '.linkDefinition';
    static notOnLinkAction = {
        title: this.genericTitle,
        kind: this.#kind,
        disabled: {
            reason: l10n.t('Not on link'),
        }
    };
    static alreadyRefLinkAction = {
        title: this.genericTitle,
        kind: this.#kind,
        disabled: {
            reason: l10n.t('Link is already a reference'),
        }
    };
    #linkProvider;
    constructor(linkProvider) {
        this.#linkProvider = linkProvider;
    }
    async getActions(doc, range, context, token) {
        if (!this.#isEnabled(context)) {
            return [];
        }
        const linkInfo = await this.#linkProvider.getLinks(doc);
        if (token.isCancellationRequested) {
            return [];
        }
        const linksInRange = linkInfo.links.filter(link => link.kind !== documentLink_1.MdLinkKind.Definition && (0, range_1.rangeIntersects)(range, link.source.range));
        if (!linksInRange.length) {
            return [_a.notOnLinkAction];
        }
        // Sort by range start to get most specific link
        linksInRange.sort((a, b) => (0, position_1.comparePosition)(b.source.range.start, a.source.range.start));
        // Even though multiple links may be in the selection, we only generate an action for the first link we find.
        // Creating actions for every link is overwhelming when users select all in a file
        const targetLink = linksInRange.find(link => link.href.kind === documentLink_1.HrefKind.External || link.href.kind === documentLink_1.HrefKind.Internal);
        if (!targetLink) {
            return [_a.alreadyRefLinkAction];
        }
        return [this.#getExtractLinkAction(doc, linkInfo, targetLink)];
    }
    #isEnabled(context) {
        if (typeof context.only === 'undefined') {
            return true;
        }
        return context.only.some(kind => (0, util_1.codeActionKindContains)(lsp.CodeActionKind.Refactor, kind));
    }
    #getExtractLinkAction(doc, linkInfo, targetLink) {
        const builder = new editBuilder_1.WorkspaceEditBuilder();
        const resource = (0, textDocument_1.getDocUri)(doc);
        const placeholder = this.#getPlaceholder(linkInfo.definitions);
        // Rewrite all inline occurrences of the link
        for (const link of linkInfo.links) {
            if (link.kind === documentLink_1.MdLinkKind.Link || link.kind === documentLink_1.MdLinkKind.AutoLink) {
                if (this.#matchesHref(targetLink.href, link)) {
                    const targetRange = link.kind === documentLink_1.MdLinkKind.AutoLink ? link.source.range : link.source.targetRange;
                    builder.replace(resource, targetRange, `[${placeholder}]`);
                }
            }
        }
        const definitionText = getLinkTargetText(doc, targetLink).trim();
        const definitions = linkInfo.links.filter(link => link.kind === documentLink_1.MdLinkKind.Definition);
        const defEdit = createAddDefinitionEdit(doc, definitions, [{ definitionText, placeholder }]);
        builder.insert(resource, defEdit.range.start, defEdit.newText);
        const renamePosition = (0, position_1.translatePosition)(targetLink.source.targetRange.start, { characterDelta: 1 });
        return {
            title: _a.genericTitle,
            kind: _a.#kind,
            edit: builder.getEdit(),
            command: {
                command: 'vscodeMarkdownLanguageservice.rename',
                title: 'Rename',
                arguments: [(0, textDocument_1.getDocUri)(doc), renamePosition],
            }
        };
    }
    #getPlaceholder(definitions) {
        const base = 'def';
        for (let i = 1;; ++i) {
            const name = i === 1 ? base : `${base}${i}`;
            if (typeof definitions.lookup(name) === 'undefined') {
                return name;
            }
        }
    }
    #matchesHref(href, link) {
        if (link.href.kind === documentLink_1.HrefKind.External && href.kind === documentLink_1.HrefKind.External) {
            return (0, path_1.isSameResource)(link.href.uri, href.uri);
        }
        if (link.href.kind === documentLink_1.HrefKind.Internal && href.kind === documentLink_1.HrefKind.Internal) {
            return (0, path_1.isSameResource)(link.href.path, href.path) && link.href.fragment === href.fragment;
        }
        return false;
    }
}
exports.MdExtractLinkDefinitionCodeActionProvider = MdExtractLinkDefinitionCodeActionProvider;
_a = MdExtractLinkDefinitionCodeActionProvider;
function createAddDefinitionEdit(doc, existingDefinitions, newDefs) {
    const defBlock = (0, organizeLinkDefs_1.getExistingDefinitionBlock)(doc, existingDefinitions);
    const newDefText = newDefs.map(({ definitionText, placeholder }) => `[${placeholder}]: ${definitionText}`).join('\n');
    if (!defBlock) {
        return lsp.TextEdit.insert({ line: doc.lineCount, character: 0 }, '\n\n' + newDefText);
    }
    else {
        const line = (0, textDocument_1.getLine)(doc, defBlock.endLine);
        return lsp.TextEdit.insert({ line: defBlock.endLine, character: line.length }, '\n' + newDefText);
    }
}
function getLinkTargetText(doc, link) {
    const afterHrefRange = link.kind === documentLink_1.MdLinkKind.AutoLink
        ? link.source.targetRange
        : lsp.Range.create((0, position_1.translatePosition)(link.source.targetRange.start, { characterDelta: 1 }), (0, position_1.translatePosition)(link.source.targetRange.end, { characterDelta: -1 }));
    return doc.getText(afterHrefRange);
}
//# sourceMappingURL=extractLinkDef.js.map