"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createHybridModeProjectProviderFactory = void 0;
const simpleProject_1 = require("@volar/language-server/lib/project/simpleProject");
const simpleProjectProvider_1 = require("@volar/language-server/lib/project/simpleProjectProvider");
const uriMap_1 = require("@volar/language-server/lib/utils/uriMap");
const language_core_1 = require("@vue/language-core");
const language_service_1 = require("@vue/language-service");
const utils_1 = require("@vue/typescript-plugin/lib/utils");
const vscode_uri_1 = require("vscode-uri");
function createHybridModeProjectProviderFactory(sys, getLanguagePlugins) {
    let initialized = false;
    const serviceEnvs = (0, uriMap_1.createUriMap)(sys.useCaseSensitiveFileNames);
    const tsconfigProjects = (0, uriMap_1.createUriMap)(sys.useCaseSensitiveFileNames);
    const simpleProjects = (0, uriMap_1.createUriMap)(sys.useCaseSensitiveFileNames);
    return {
        async get(uri) {
            if (!initialized) {
                initialized = true;
                initialize(this);
            }
            const parsedUri = vscode_uri_1.URI.parse(uri);
            const workspaceFolder = (0, simpleProjectProvider_1.getWorkspaceFolder)(parsedUri, this.workspaceFolders);
            let serviceEnv = serviceEnvs.get(workspaceFolder);
            if (!serviceEnv) {
                serviceEnv = (0, simpleProjectProvider_1.createServiceEnvironment)(this, workspaceFolder);
                serviceEnvs.set(workspaceFolder, serviceEnv);
            }
            const fileName = serviceEnv.typescript.uriToFileName(uri);
            const projectInfo = (await (0, utils_1.searchNamedPipeServerForFile)(fileName))?.projectInfo;
            if (projectInfo?.kind === 1) {
                const tsconfig = projectInfo.name;
                const tsconfigUri = vscode_uri_1.URI.parse(serviceEnv.typescript.fileNameToUri(tsconfig));
                if (!tsconfigProjects.has(tsconfigUri)) {
                    tsconfigProjects.set(tsconfigUri, (async () => {
                        const languagePlugins = await getLanguagePlugins(serviceEnv, tsconfig, undefined, {
                            ...sys,
                            version: 0,
                            async sync() {
                                return 0;
                            },
                            dispose() { },
                        });
                        return createTSConfigProject(this, serviceEnv, languagePlugins);
                    })());
                }
                return await tsconfigProjects.get(tsconfigUri);
            }
            else {
                if (!simpleProjects.has(workspaceFolder)) {
                    simpleProjects.set(workspaceFolder, (async () => {
                        const languagePlugins = await getLanguagePlugins(serviceEnv);
                        return (0, simpleProject_1.createSimpleServerProject)(this, serviceEnv, languagePlugins);
                    })());
                }
                return await simpleProjects.get(workspaceFolder);
            }
        },
        async all() {
            return Promise.all([
                ...tsconfigProjects.values(),
                ...simpleProjects.values(),
            ]);
        },
        reload() {
            for (const project of [
                ...tsconfigProjects.values(),
                ...simpleProjects.values(),
            ]) {
                project.then(p => p.dispose());
            }
            tsconfigProjects.clear();
            simpleProjects.clear();
        },
    };
    function initialize(server) {
        server.onDidChangeWatchedFiles(({ changes }) => {
            for (const change of changes) {
                const changeUri = vscode_uri_1.URI.parse(change.uri);
                if (tsconfigProjects.has(changeUri)) {
                    tsconfigProjects.get(changeUri)?.then(project => project.dispose());
                    tsconfigProjects.delete(changeUri);
                    server.clearPushDiagnostics();
                }
            }
        });
    }
    function createTSConfigProject(server, serviceEnv, languagePlugins) {
        let languageService;
        return {
            getLanguageService,
            getLanguageServiceDontCreate: () => languageService,
            dispose() {
                languageService?.dispose();
            },
        };
        function getLanguageService() {
            if (!languageService) {
                const language = (0, language_core_1.createLanguage)(languagePlugins, false, uri => {
                    const document = server.documents.get(uri);
                    if (document) {
                        language.scripts.set(uri, document.getSnapshot(), document.languageId);
                    }
                    else {
                        language.scripts.delete(uri);
                    }
                });
                languageService = (0, language_service_1.createLanguageService)(language, server.languageServicePlugins, serviceEnv);
            }
            return languageService;
        }
    }
}
exports.createHybridModeProjectProviderFactory = createHybridModeProjectProviderFactory;
//# sourceMappingURL=hybridModeProject.js.map