"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.
//
// Original code forked from https://github.com/Quramy/ts-graphql-plugin
const api_1 = require("typescript-styled-plugin/lib/api");
const typescript_template_language_service_decorator_1 = require("typescript-template-language-service-decorator");
const vscode_html_languageservice_1 = require("vscode-html-languageservice");
const config_1 = require("./config");
const configuration_1 = require("./configuration");
const html_template_language_service_1 = require("./html-template-language-service");
const substitutions_1 = require("./substitutions");
const virtual_document_provider_1 = require("./virtual-document-provider");
const litHtmlPluginMarker = Symbol('__litHtmlPluginMarker__');
class LanguageServiceLogger {
    constructor(info) {
        this.info = info;
    }
    log(msg) {
        this.info.project.projectService.logger.info(`[${config_1.pluginName}] ${msg}`);
    }
}
class HtmlPlugin {
    constructor(_typescript) {
        this._typescript = _typescript;
        this._virtualDocumentProvider = new virtual_document_provider_1.VirtualDocumentProvider();
        this._config = new configuration_1.Configuration();
    }
    create(info) {
        if (info.languageService[litHtmlPluginMarker]) {
            // Already decorated
            return info.languageService;
        }
        const logger = new LanguageServiceLogger(info);
        this._config.update(info.config);
        logger.log('config: ' + JSON.stringify(this._config));
        const styledLanguageService = new api_1.StyledTemplateLanguageService(this._typescript, {}, new virtual_document_provider_1.CssDocumentProvider(this.htmlLanguageService), logger);
        const htmlTemplateLanguageService = new html_template_language_service_1.default(this._typescript, this._config, this._virtualDocumentProvider, this.htmlLanguageService, styledLanguageService, logger);
        const languageService = typescript_template_language_service_decorator_1.decorateWithTemplateLanguageService(this._typescript, info.languageService, info.project, htmlTemplateLanguageService, this.getTemplateSettings(this._config, this._virtualDocumentProvider), { logger });
        languageService[litHtmlPluginMarker] = true;
        return languageService;
    }
    onConfigurationChanged(config) {
        this._config.update(config);
    }
    get htmlLanguageService() {
        if (!this._htmlLanguageService) {
            this._htmlLanguageService = vscode_html_languageservice_1.getLanguageService();
        }
        return this._htmlLanguageService;
    }
    getTemplateSettings(config, provider) {
        return {
            get tags() { return config.tags; },
            enableForStringWithSubstitutions: true,
            getSubstitutions: (templateString, spans) => {
                return substitutions_1.getSubstitutions(this._typescript, this.htmlLanguageService, provider, templateString, spans);
            },
        };
    }
}
module.exports = (mod) => new HtmlPlugin(mod.typescript);
//# sourceMappingURL=index.js.map