/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "vscode-uri"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.joinPath = exports.normalizePath = exports.resolvePath = exports.extname = exports.basename = exports.dirname = exports.isAbsolutePath = void 0;
    var vscode_uri_1 = require("vscode-uri");
    var Slash = '/'.charCodeAt(0);
    var Dot = '.'.charCodeAt(0);
    function isAbsolutePath(path) {
        return path.charCodeAt(0) === Slash;
    }
    exports.isAbsolutePath = isAbsolutePath;
    function dirname(uri) {
        var lastIndexOfSlash = uri.lastIndexOf('/');
        return lastIndexOfSlash !== -1 ? uri.substr(0, lastIndexOfSlash) : '';
    }
    exports.dirname = dirname;
    function basename(uri) {
        var lastIndexOfSlash = uri.lastIndexOf('/');
        return uri.substr(lastIndexOfSlash + 1);
    }
    exports.basename = basename;
    function extname(uri) {
        for (var i = uri.length - 1; i >= 0; i--) {
            var ch = uri.charCodeAt(i);
            if (ch === Dot) {
                if (i > 0 && uri.charCodeAt(i - 1) !== Slash) {
                    return uri.substr(i);
                }
                else {
                    break;
                }
            }
            else if (ch === Slash) {
                break;
            }
        }
        return '';
    }
    exports.extname = extname;
    function resolvePath(uriString, path) {
        if (isAbsolutePath(path)) {
            var uri = vscode_uri_1.URI.parse(uriString);
            var parts = path.split('/');
            return uri.with({ path: normalizePath(parts) }).toString();
        }
        return joinPath(uriString, path);
    }
    exports.resolvePath = resolvePath;
    function normalizePath(parts) {
        var newParts = [];
        for (var _i = 0, parts_1 = parts; _i < parts_1.length; _i++) {
            var part = parts_1[_i];
            if (part.length === 0 || part.length === 1 && part.charCodeAt(0) === Dot) {
                // ignore
            }
            else if (part.length === 2 && part.charCodeAt(0) === Dot && part.charCodeAt(1) === Dot) {
                newParts.pop();
            }
            else {
                newParts.push(part);
            }
        }
        if (parts.length > 1 && parts[parts.length - 1].length === 0) {
            newParts.push('');
        }
        var res = newParts.join('/');
        if (parts[0].length === 0) {
            res = '/' + res;
        }
        return res;
    }
    exports.normalizePath = normalizePath;
    function joinPath(uriString) {
        var paths = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            paths[_i - 1] = arguments[_i];
        }
        var uri = vscode_uri_1.URI.parse(uriString);
        var parts = uri.path.split('/');
        for (var _a = 0, paths_1 = paths; _a < paths_1.length; _a++) {
            var path = paths_1[_a];
            parts.push.apply(parts, path.split('/'));
        }
        return uri.with({ path: normalizePath(parts) }).toString();
    }
    exports.joinPath = joinPath;
});
