"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExperimentManager = void 0;
const query_1 = require("tsp-typescript-client/lib/models/query/query");
const signal_manager_1 = require("./signals/signal-manager");
class ExperimentManager {
    constructor(tspClient, traceManager) {
        this.fOpenExperiments = new Map();
        this.fTspClient = tspClient;
        this.fTraceManager = traceManager;
        (0, signal_manager_1.signalManager)().on(signal_manager_1.Signals.EXPERIMENT_DELETED, (experiment) => this.onExperimentDeleted(experiment));
    }
    /**
     * Get an array of opened experiments
     * @returns Array of experiment
     */
    async getOpenedExperiments() {
        const openedExperiments = [];
        // Look on the server for opened experiments
        const experimentsResponse = await this.fTspClient.fetchExperiments();
        const experiments = experimentsResponse.getModel();
        if (experimentsResponse.isOk() && experiments) {
            openedExperiments.push(...experiments);
        }
        return openedExperiments;
    }
    /**
     * Get a specific experiment information
     * @param experimentUUID experiment UUID
     */
    async getExperiment(experimentUUID) {
        // Check if the experiment is in "cache"
        let experiment = this.fOpenExperiments.get(experimentUUID);
        // If the experiment is undefined, check on the server
        if (!experiment) {
            const experimentResponse = await this.fTspClient.fetchExperiment(experimentUUID);
            if (experimentResponse.isOk()) {
                experiment = experimentResponse.getModel();
            }
        }
        return experiment;
    }
    /**
     * Get an array of OutputDescriptor for a given experiment
     * @param experimentUUID experiment UUID
     */
    async getAvailableOutputs(experimentUUID) {
        const outputsResponse = await this.fTspClient.experimentOutputs(experimentUUID);
        if (outputsResponse && outputsResponse.isOk()) {
            return outputsResponse.getModel();
        }
        return undefined;
    }
    /**
     * Open a given experiment on the server
     * @param experimentURI experiment URI to open
     * @param experimentName Optional name for the experiment. If not specified the URI name is used
     * @returns The opened experiment
     */
    async openExperiment(experimentName, traces) {
        const name = experimentName;
        const traceURIs = new Array();
        for (let i = 0; i < traces.length; i++) {
            traceURIs.push(traces[i].UUID);
        }
        const tryCreate = async function (tspClient, retry) {
            return tspClient.createExperiment(new query_1.Query({
                name: retry === 0 ? name : name + '(' + retry + ')',
                traces: traceURIs
            }));
        };
        let tryNb = 0;
        let experimentResponse;
        while (experimentResponse === undefined || experimentResponse.getStatusCode() === 409) {
            experimentResponse = await tryCreate(this.fTspClient, tryNb);
            tryNb++;
        }
        const experiment = experimentResponse.getModel();
        if (experimentResponse.isOk() && experiment) {
            this.addExperiment(experiment);
            (0, signal_manager_1.signalManager)().fireExperimentOpenedSignal(experiment);
            return experiment;
        }
        // TODO Handle any other experiment open errors
        return undefined;
    }
    /**
     * Update the experiment with the latest info from the server.
     * @param experimentUUID experiment UUID
     * @returns The updated experiment or undefined if the experiment failed to update
     */
    async updateExperiment(experimentUUID) {
        const experimentResponse = await this.fTspClient.fetchExperiment(experimentUUID);
        const experiment = experimentResponse.getModel();
        if (experiment && experimentResponse.isOk()) {
            this.fOpenExperiments.set(experimentUUID, experiment);
            return experiment;
        }
        return undefined;
    }
    /**
     * Delete the given experiment from the server
     * @param experimentUUID experiment UUID
     */
    async deleteExperiment(experimentUUID) {
        const experimentToDelete = this.fOpenExperiments.get(experimentUUID);
        if (experimentToDelete) {
            await this.fTspClient.deleteExperiment(experimentUUID);
            const deletedExperiment = this.removeExperiment(experimentUUID);
            if (deletedExperiment) {
                (0, signal_manager_1.signalManager)().fireExperimentDeletedSignal(deletedExperiment);
            }
        }
    }
    onExperimentDeleted(experiment) {
        /*
         * TODO: Do not close traces used by another experiment
         */
        // Close each trace
        const traces = experiment.traces;
        for (let i = 0; i < traces.length; i++) {
            this.fTraceManager.deleteTrace(traces[i].UUID);
        }
    }
    addExperiment(experiment) {
        this.fOpenExperiments.set(experiment.UUID, experiment);
        experiment.traces.forEach(trace => {
            this.fTraceManager.addTrace(trace);
        });
    }
    removeExperiment(experimentUUID) {
        const deletedExperiment = this.fOpenExperiments.get(experimentUUID);
        this.fOpenExperiments.delete(experimentUUID);
        return deletedExperiment;
    }
}
exports.ExperimentManager = ExperimentManager;
//# sourceMappingURL=experiment-manager.js.map