"use strict";
// *****************************************************************************
// Copyright (C) 2017 TypeFox and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var ThemeService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuiltinThemeProvider = exports.ThemeService = void 0;
const event_1 = require("../common/event");
const disposable_1 = require("../common/disposable");
const frontend_application_config_provider_1 = require("./frontend-application-config-provider");
const application_props_1 = require("@theia/application-package/lib/application-props");
const inversify_1 = require("inversify");
const promise_util_1 = require("../common/promise-util");
const preferences_1 = require("./preferences");
const debounce = require("lodash.debounce");
const COLOR_THEME_PREFERENCE_KEY = 'workbench.colorTheme';
const NO_THEME = { id: 'no-theme', label: 'Not a real theme.', type: 'dark' };
let ThemeService = ThemeService_1 = class ThemeService {
    constructor() {
        this.themes = {};
        this.activeTheme = NO_THEME;
        this.themeChange = new event_1.Emitter();
        this.deferredInitializer = new promise_util_1.Deferred();
        this.onDidColorThemeChange = this.themeChange.event;
        this.updateColorThemePreference = debounce(() => this.doUpdateColorThemePreference(), 500);
    }
    get initialized() {
        return this.deferredInitializer.promise;
    }
    init() {
        this.register(...BuiltinThemeProvider.themes);
        this.loadUserTheme();
        this.preferences.ready.then(() => {
            this.validateActiveTheme();
            this.updateColorThemePreference();
            this.preferences.onPreferencesChanged(changes => {
                if (COLOR_THEME_PREFERENCE_KEY in changes) {
                    this.validateActiveTheme();
                }
            });
        });
    }
    register(...themes) {
        for (const theme of themes) {
            this.themes[theme.id] = theme;
        }
        this.validateActiveTheme();
        this.updateColorThemePreference();
        return disposable_1.Disposable.create(() => {
            for (const theme of themes) {
                delete this.themes[theme.id];
                if (this.activeTheme === theme) {
                    this.setCurrentTheme(this.defaultTheme.id, false);
                }
            }
            this.updateColorThemePreference();
        });
    }
    validateActiveTheme() {
        if (this.preferences.isReady) {
            const configuredTheme = this.getConfiguredTheme();
            if (configuredTheme && configuredTheme !== this.activeTheme) {
                this.setCurrentTheme(configuredTheme.id, false);
            }
        }
    }
    doUpdateColorThemePreference() {
        const preference = this.schemaProvider.getSchemaProperty(COLOR_THEME_PREFERENCE_KEY);
        if (preference) {
            const sortedThemes = this.getThemes().sort((a, b) => a.label.localeCompare(b.label));
            preference.enum = sortedThemes.map(e => e.id);
            preference.enumItemLabels = sortedThemes.map(e => e.label);
            this.schemaProvider.updateSchemaProperty(COLOR_THEME_PREFERENCE_KEY, preference);
        }
    }
    getThemes() {
        const result = [];
        for (const o in this.themes) {
            if (this.themes.hasOwnProperty(o)) {
                result.push(this.themes[o]);
            }
        }
        return result;
    }
    getTheme(themeId) {
        return this.themes[themeId] || this.defaultTheme;
    }
    tryGetTheme(themeId) {
        return this.themes[themeId];
    }
    /** Should only be called at startup. */
    loadUserTheme() {
        var _a;
        const storedThemeId = (_a = window.localStorage.getItem(ThemeService_1.STORAGE_KEY)) !== null && _a !== void 0 ? _a : this.defaultTheme.id;
        const theme = this.getTheme(storedThemeId);
        this.setCurrentTheme(theme.id, false);
        this.deferredInitializer.resolve();
    }
    /**
     * @param persist If `true`, the value of the `workbench.colorTheme` preference will be set to the provided ID.
     */
    setCurrentTheme(themeId, persist = true) {
        var _a, _b;
        const newTheme = this.tryGetTheme(themeId);
        const oldTheme = this.activeTheme;
        if (newTheme && newTheme !== oldTheme) {
            (_a = oldTheme === null || oldTheme === void 0 ? void 0 : oldTheme.deactivate) === null || _a === void 0 ? void 0 : _a.call(oldTheme);
            (_b = newTheme.activate) === null || _b === void 0 ? void 0 : _b.call(newTheme);
            this.activeTheme = newTheme;
            this.themeChange.fire({ newTheme, oldTheme });
        }
        if (persist) {
            this.preferences.updateValue(COLOR_THEME_PREFERENCE_KEY, themeId);
        }
    }
    getCurrentTheme() {
        return this.activeTheme;
    }
    getConfiguredTheme() {
        const configuredId = this.preferences.get(COLOR_THEME_PREFERENCE_KEY);
        return configuredId ? this.themes[configuredId.toString()] : undefined;
    }
    /**
     * The default theme. If that is not applicable, returns with the fallback theme.
     */
    get defaultTheme() {
        var _a;
        return (_a = this.tryGetTheme(application_props_1.DefaultTheme.defaultForOSTheme(frontend_application_config_provider_1.FrontendApplicationConfigProvider.get().defaultTheme))) !== null && _a !== void 0 ? _a : this.getTheme(application_props_1.DefaultTheme.defaultForOSTheme(application_props_1.ApplicationProps.DEFAULT.frontend.config.defaultTheme));
    }
    /**
     * Resets the state to the user's default, or to the fallback theme. Also discards any persisted state in the local storage.
     */
    reset() {
        this.setCurrentTheme(this.defaultTheme.id);
    }
};
ThemeService.STORAGE_KEY = 'theme';
__decorate([
    (0, inversify_1.inject)(preferences_1.PreferenceService),
    __metadata("design:type", Object)
], ThemeService.prototype, "preferences", void 0);
__decorate([
    (0, inversify_1.inject)(preferences_1.PreferenceSchemaProvider),
    __metadata("design:type", preferences_1.PreferenceSchemaProvider)
], ThemeService.prototype, "schemaProvider", void 0);
__decorate([
    (0, inversify_1.postConstruct)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], ThemeService.prototype, "init", null);
ThemeService = ThemeService_1 = __decorate([
    (0, inversify_1.injectable)()
], ThemeService);
exports.ThemeService = ThemeService;
class BuiltinThemeProvider {
}
exports.BuiltinThemeProvider = BuiltinThemeProvider;
BuiltinThemeProvider.darkTheme = {
    id: 'dark',
    type: 'dark',
    label: 'Dark (Theia)',
    editorTheme: 'dark-theia' // loaded in /packages/monaco/src/browser/textmate/monaco-theme-registry.ts
};
BuiltinThemeProvider.lightTheme = {
    id: 'light',
    type: 'light',
    label: 'Light (Theia)',
    editorTheme: 'light-theia' // loaded in /packages/monaco/src/browser/textmate/monaco-theme-registry.ts
};
BuiltinThemeProvider.hcTheme = {
    id: 'hc-theia',
    type: 'hc',
    label: 'High Contrast (Theia)',
    editorTheme: 'hc-theia' // loaded in /packages/monaco/src/browser/textmate/monaco-theme-registry.ts
};
BuiltinThemeProvider.hcLightTheme = {
    id: 'hc-theia-light',
    type: 'hcLight',
    label: 'High Contrast Light (Theia)',
    editorTheme: 'hc-theia-light' // loaded in /packages/monaco/src/browser/textmate/monaco-theme-registry.ts
};
BuiltinThemeProvider.themes = [
    BuiltinThemeProvider.darkTheme,
    BuiltinThemeProvider.lightTheme,
    BuiltinThemeProvider.hcTheme,
    BuiltinThemeProvider.hcLightTheme
];
//# sourceMappingURL=theming.js.map