"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const utils_1 = require("./utils");
const chai_1 = require("chai");
const path = require("path");
const assert_1 = require("assert");
const os = require("os");
describe('multithread', function () {
    return __awaiter(this, void 0, void 0, function* () {
        let dc;
        const program = path.join(utils_1.testProgramsDir, 'MultiThread');
        const source = path.join(utils_1.testProgramsDir, 'MultiThread.cc');
        const threadNames = {
            monday: 0,
            tuesday: 1,
            wednesday: 2,
            thursday: 3,
            friday: 4,
        };
        const lineTags = {
            LINE_MAIN_ALL_THREADS_STARTED: 0,
        };
        before(function () {
            (0, utils_1.resolveLineTagLocations)(source, lineTags);
        });
        beforeEach(() => __awaiter(this, void 0, void 0, function* () {
            dc = yield (0, utils_1.standardBeforeEach)();
        }));
        afterEach(() => __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        }));
        it('sees all threads', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (!utils_1.gdbNonStop && os.platform() === 'win32' && utils_1.isRemoteTest) {
                    // The way thread names are set in remote tests on windows is unsupported
                    this.skip();
                }
                yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.test, {
                    program: program,
                }), {
                    path: source,
                    line: lineTags['LINE_MAIN_ALL_THREADS_STARTED'],
                });
                const threads = yield dc.threadsRequest();
                const nameToId = new Map(threads.body.threads.map((thread) => [thread.name, thread.id]));
                // Make sure all 5 threads are there
                (0, chai_1.expect)(nameToId).to.include.keys(Object.keys(threadNames));
                // and make sure that there is at least 6 threads.
                // We don't care about the name of the "main" thread
                (0, chai_1.expect)(threads.body.threads).length.greaterThanOrEqual(6);
                // check that each thread can be communicated with individually
                for (const [name, idInProgram] of Object.entries(threadNames)) {
                    // There are multiple ids/indexes.
                    // idInProgram cooresponds to the variable thread_id in the C++ source
                    // threadId is the id of the thread in DAP
                    const threadId = nameToId.get(name);
                    if (threadId === undefined) {
                        // unreachable because of expect above
                        (0, assert_1.fail)('unreachable');
                    }
                    if (utils_1.gdbNonStop) {
                        const waitForStopped = dc.waitForEvent('stopped');
                        const pr = dc.pauseRequest({ threadId });
                        yield Promise.all([pr, waitForStopped]);
                    }
                    const stack = yield dc.stackTraceRequest({ threadId });
                    let frameId = undefined;
                    for (const frame of stack.body.stackFrames) {
                        if (frame.name === 'PrintHello') {
                            frameId = frame.id;
                            break;
                        }
                    }
                    if (frameId === undefined) {
                        (0, assert_1.fail)("Failed to find frame with name 'PrintHello'");
                    }
                    const scopes = yield dc.scopesRequest({ frameId });
                    const vr = scopes.body.scopes[0].variablesReference;
                    const vars = yield dc.variablesRequest({ variablesReference: vr });
                    const varnameToValue = new Map(vars.body.variables.map((variable) => [
                        variable.name,
                        variable.value,
                    ]));
                    (0, chai_1.expect)(varnameToValue.get('thread_id')).to.equal(idInProgram.toString());
                    // The "name" variable is a pointer, so is displayed as an address + the
                    // extracted nul terminated string
                    (0, chai_1.expect)(varnameToValue.get('name')).to.contain(name);
                }
            });
        });
    });
});
//# sourceMappingURL=multithread.spec.js.map