"use strict";
/**
 * @file The whole purpose of this file is to encode the special logic
 * required to handle BigInt values in the various datatypes defined
 * by the TSP. Node.js doesn't handle BigInt and numbers the same way.
 *
 * Here are the cases we need to keep in mind:
 * 1. If what's supposed to be a BigInt value is too low, it will get
 *    deserialized as a number. It needs to be converted back into BigInt.
 * 2. If what's supposed to be a number value is too high, it will get
 *    deserialized as a BigInt. We need to throw in this case.
 *
 * To address (1) we define functions that will force the type of
 * fields to be converted into a BigInt.
 *
 * To address (2) we define functions that will throw an exception
 * if we get a BigInt instead of a number when deserializing.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.toBigInt = exports.assertNumber = exports.record = exports.array = exports.normalize = exports.createNormalizer = void 0;
/**
 * Create a normalizer function based on `descriptor` for `T` .
 *
 * General rules for a descriptor:
 * - If a field is `any`-like then you should either define a generic normalizer
 *   for the value or explicitly use `undefined`.
 * - Record objects (`{ [key: string]: any }`) are considered `any`-like.
 * - Any field that directly or indirectly contains `bigint` or `number` must
 *   have a normalizer function attached to it.
 */
function createNormalizer(descriptor) {
    return input => normalize(input, descriptor);
}
exports.createNormalizer = createNormalizer;
/**
 * Create a deep-copy of `input` while applying normalizers from `descriptor`.
 */
function normalize(input, descriptor) {
    if (input === undefined) {
        // Undefined
        return input;
    }
    if (typeof input === 'object') {
        if (input === null) {
            // Null
            return input;
        }
        else if (Array.isArray(input)) {
            // Array
            return typeof descriptor === 'function'
                ? descriptor(input) // Normalize
                : input.map(element => normalize(element)); // Deep-copy
        }
        else {
            // Object
            if (typeof descriptor === 'function') {
                return descriptor(input); // Normalize
            }
            const output = {};
            for (const [key, value] of Object.entries(input)) {
                output[key] = normalize(value, descriptor === null || descriptor === void 0 ? void 0 : descriptor[key]);
            }
            return output;
        }
    }
    // Primitive
    return typeof descriptor === 'function'
        ? descriptor(input) // Normalize (bigint or number or any)
        : input;
}
exports.normalize = normalize;
/**
 * Create a normalizer that operates on JS Array objects.
 */
function array(normalizer) {
    return input => input.map(element => normalizer(element));
}
exports.array = array;
/**
 * Create a normalizer that converts the values of a JS Object Map (aka Record).
 */
function record(normalizer) {
    return input => {
        const output = {};
        for (const [key, value] of Object.entries(input)) {
            output[key] = normalizer(value);
        }
        return output;
    };
}
exports.record = record;
/**
 * Throw if `input` is not a `number`.
 */
const assertNumber = 
// tslint:disable-next-line: no-shadowed-variable
function assertNumber(input) {
    if (typeof input !== 'number') {
        throw new TypeError(`"${input}" is not a number!`);
    }
    return input;
};
exports.assertNumber = assertNumber;
const toBigInt = function toBigInt(value) {
    return BigInt(value);
};
exports.toBigInt = toBigInt;
//# sourceMappingURL=serialization.js.map