"use strict";
// *****************************************************************************
// Copyright (C) 2018 Red Hat, Inc. and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeViewsExtImpl = void 0;
// TODO: extract `@theia/util` for event, disposable, cancellation and common types
// don't use @theia/core directly from plugin host
const event_1 = require("@theia/core/lib/common/event");
const disposable_1 = require("@theia/core/lib/common/disposable");
const types_impl_1 = require("../types-impl");
const plugin_api_rpc_1 = require("../../common/plugin-api-rpc");
const common_1 = require("../../common");
const plugin_icon_path_1 = require("../plugin-icon-path");
class TreeViewsExtImpl {
    constructor(rpc, commandRegistry) {
        this.commandRegistry = commandRegistry;
        this.treeViews = new Map();
        this.proxy = rpc.getProxy(plugin_api_rpc_1.PLUGIN_RPC_CONTEXT.TREE_VIEWS_MAIN);
        commandRegistry.registerArgumentProcessor({
            processArgument: arg => {
                if (!common_1.TreeViewSelection.is(arg)) {
                    return arg;
                }
                const { treeViewId, treeItemId } = arg;
                const treeView = this.treeViews.get(treeViewId);
                return treeView && treeView.getTreeItem(treeItemId);
            }
        });
    }
    registerTreeDataProvider(plugin, treeViewId, treeDataProvider) {
        const treeView = this.createTreeView(plugin, treeViewId, { treeDataProvider });
        return types_impl_1.Disposable.create(() => {
            this.treeViews.delete(treeViewId);
            treeView.dispose();
        });
    }
    createTreeView(plugin, treeViewId, options) {
        if (!options || !options.treeDataProvider) {
            throw new Error('Options with treeDataProvider is mandatory');
        }
        const treeView = new TreeViewExtImpl(plugin, treeViewId, options.treeDataProvider, this.proxy, this.commandRegistry.converter);
        this.treeViews.set(treeViewId, treeView);
        return {
            // tslint:disable:typedef
            get onDidExpandElement() {
                return treeView.onDidExpandElement;
            },
            get onDidCollapseElement() {
                return treeView.onDidCollapseElement;
            },
            get selection() {
                return treeView.selectedElements;
            },
            get onDidChangeSelection() {
                return treeView.onDidChangeSelection;
            },
            get visible() {
                return treeView.visible;
            },
            get onDidChangeVisibility() {
                return treeView.onDidChangeVisibility;
            },
            get message() {
                return treeView.message;
            },
            set message(message) {
                treeView.message = message;
            },
            get title() {
                return treeView.title;
            },
            set title(title) {
                treeView.title = title;
            },
            get description() {
                return treeView.description;
            },
            set description(description) {
                treeView.description = description;
            },
            reveal: (element, revealOptions) => treeView.reveal(element, revealOptions),
            dispose: () => {
                this.treeViews.delete(treeViewId);
                treeView.dispose();
            }
        };
    }
    async $getChildren(treeViewId, treeItemId) {
        const treeView = this.getTreeView(treeViewId);
        return treeView.getChildren(treeItemId);
    }
    async $setExpanded(treeViewId, treeItemId, expanded) {
        const treeView = this.getTreeView(treeViewId);
        if (expanded) {
            return treeView.onExpanded(treeItemId);
        }
        else {
            return treeView.onCollapsed(treeItemId);
        }
    }
    async $setSelection(treeViewId, treeItemIds) {
        this.getTreeView(treeViewId).setSelection(treeItemIds);
    }
    async $setVisible(treeViewId, isVisible) {
        this.getTreeView(treeViewId).setVisible(isVisible);
    }
    getTreeView(treeViewId) {
        const treeView = this.treeViews.get(treeViewId);
        if (!treeView) {
            throw new Error(`No tree view with id '${treeViewId}' registered.`);
        }
        return treeView;
    }
}
exports.TreeViewsExtImpl = TreeViewsExtImpl;
class TreeViewExtImpl {
    constructor(plugin, treeViewId, treeDataProvider, proxy, commandsConverter) {
        this.plugin = plugin;
        this.treeViewId = treeViewId;
        this.treeDataProvider = treeDataProvider;
        this.proxy = proxy;
        this.commandsConverter = commandsConverter;
        this.onDidExpandElementEmitter = new event_1.Emitter();
        this.onDidExpandElement = this.onDidExpandElementEmitter.event;
        this.onDidCollapseElementEmitter = new event_1.Emitter();
        this.onDidCollapseElement = this.onDidCollapseElementEmitter.event;
        this.onDidChangeSelectionEmitter = new event_1.Emitter();
        this.onDidChangeSelection = this.onDidChangeSelectionEmitter.event;
        this.onDidChangeVisibilityEmitter = new event_1.Emitter();
        this.onDidChangeVisibility = this.onDidChangeVisibilityEmitter.event;
        this.nodes = new Map();
        this.pendingRefresh = Promise.resolve();
        this.toDispose = new disposable_1.DisposableCollection(disposable_1.Disposable.create(() => this.clearAll()), this.onDidExpandElementEmitter, this.onDidCollapseElementEmitter, this.onDidChangeSelectionEmitter, this.onDidChangeVisibilityEmitter);
        this._message = '';
        this._title = '';
        this._description = '';
        this.selectedItemIds = new Set();
        this._visible = false;
        proxy.$registerTreeDataProvider(treeViewId);
        this.toDispose.push(disposable_1.Disposable.create(() => this.proxy.$unregisterTreeDataProvider(treeViewId)));
        if (treeDataProvider.onDidChangeTreeData) {
            treeDataProvider.onDidChangeTreeData((e) => {
                this.pendingRefresh = proxy.$refresh(treeViewId);
            });
        }
    }
    dispose() {
        this.toDispose.dispose();
    }
    async reveal(element, options) {
        await this.pendingRefresh;
        const elementParentChain = await this.calculateRevealParentChain(element);
        if (elementParentChain) {
            return this.proxy.$reveal(this.treeViewId, elementParentChain, Object.assign({ select: true, focus: false, expand: false }, options));
        }
    }
    get message() {
        return this._message;
    }
    set message(message) {
        this._message = message;
        this.proxy.$setMessage(this.treeViewId, this._message);
    }
    get title() {
        return this._title;
    }
    set title(title) {
        this._title = title;
        this.proxy.$setTitle(this.treeViewId, title);
    }
    get description() {
        return this._description;
    }
    set description(description) {
        this._description = description;
        this.proxy.$setDescription(this.treeViewId, this._description);
    }
    getTreeItem(treeItemId) {
        const element = this.nodes.get(treeItemId);
        return element && element.value;
    }
    /**
     * calculate the chain of node ids from root to element so that the frontend can expand all of them and reveal element.
     * this is needed as the frontend may not have the full tree nodes.
     * throughout the parent chain this.getChildren is called in order to fill this.nodes cache.
     *
     * returns undefined if wasn't able to calculate the path due to inconsistencies.
     *
     * @param element element to reveal
     */
    async calculateRevealParentChain(element) {
        if (!element) {
            // root
            return [];
        }
        const parent = this.treeDataProvider.getParent && await this.treeDataProvider.getParent(element);
        const chain = await this.calculateRevealParentChain(parent);
        if (!chain) {
            // parents are inconsistent
            return undefined;
        }
        const parentId = chain.length ? chain[chain.length - 1] : '';
        const treeItem = await this.treeDataProvider.getTreeItem(element);
        if (treeItem.id) {
            return chain.concat(treeItem.id);
        }
        const cachedParentNode = this.nodes.get(parentId);
        // first try to get children length from cache since getChildren disposes old nodes, which can cause a race
        // condition if command is executed together with reveal.
        // If not in cache, getChildren fills this.nodes and generate ids for them which are needed later
        const children = (cachedParentNode === null || cachedParentNode === void 0 ? void 0 : cachedParentNode.children) || await this.getChildren(parentId);
        if (!children) {
            return undefined; // parent is inconsistent
        }
        const idLabel = this.getTreeItemIdLabel(treeItem);
        let possibleIndex = children.length;
        // find the right element id by searching all possible id names in the cache
        while (possibleIndex-- > 0) {
            const candidateId = this.buildTreeItemId(parentId, possibleIndex, idLabel);
            if (this.nodes.has(candidateId)) {
                return chain.concat(candidateId);
            }
        }
        // couldn't calculate consistent parent chain and id
        return undefined;
    }
    getTreeItemLabel(treeItem) {
        const treeItemLabel = treeItem.label;
        if (typeof treeItemLabel === 'object' && typeof treeItemLabel.label === 'string') {
            return treeItemLabel.label;
        }
        else {
            return treeItem.label;
        }
    }
    getTreeItemIdLabel(treeItem) {
        let idLabel = this.getTreeItemLabel(treeItem);
        // Use resource URI if label is not set
        if (idLabel === undefined && treeItem.resourceUri) {
            idLabel = treeItem.resourceUri.path.toString();
            idLabel = decodeURIComponent(idLabel);
            if (idLabel.indexOf('/') >= 0) {
                idLabel = idLabel.substring(idLabel.lastIndexOf('/') + 1);
            }
        }
        return idLabel;
    }
    buildTreeItemId(parentId, index, idLabel) {
        return `${parentId}/${index}:${idLabel}`;
    }
    async getChildren(parentId) {
        const parentNode = this.nodes.get(parentId);
        const parent = parentNode === null || parentNode === void 0 ? void 0 : parentNode.value;
        if (parentId && !parent) {
            console.error(`No tree item with id '${parentId}' found.`);
            return [];
        }
        this.clearChildren(parentNode);
        // place root in the cache
        if (parentId === '') {
            this.nodes.set(parentId, { id: '', dispose: () => { } });
        }
        // ask data provider for children for cached element
        const result = await this.treeDataProvider.getChildren(parent);
        if (result) {
            const treeItemPromises = result.map(async (value, index) => {
                // Ask data provider for a tree item for the value
                // Data provider must return theia.TreeItem
                const treeItem = await this.treeDataProvider.getTreeItem(value);
                // Convert theia.TreeItem to the TreeViewItem
                const label = this.getTreeItemLabel(treeItem);
                const idLabel = this.getTreeItemIdLabel(treeItem);
                // Generate the ID
                // ID is used for caching the element
                const id = treeItem.id || this.buildTreeItemId(parentId, index, idLabel);
                const toDisposeElement = new disposable_1.DisposableCollection();
                const node = {
                    id,
                    value,
                    dispose: () => toDisposeElement.dispose()
                };
                if (parentNode) {
                    const children = parentNode.children || [];
                    children.push(node);
                    parentNode.children = children;
                }
                this.nodes.set(id, node);
                let icon;
                let iconUrl;
                let themeIconId;
                const { iconPath } = treeItem;
                if (typeof iconPath === 'string' && iconPath.indexOf('fa-') !== -1) {
                    icon = iconPath;
                }
                else if (types_impl_1.ThemeIcon.is(iconPath)) {
                    themeIconId = iconPath.id;
                }
                else {
                    iconUrl = plugin_icon_path_1.PluginIconPath.toUrl(iconPath, this.plugin);
                }
                const treeViewItem = {
                    id,
                    label,
                    icon,
                    iconUrl,
                    themeIconId,
                    description: treeItem.description,
                    resourceUri: treeItem.resourceUri,
                    tooltip: treeItem.tooltip,
                    collapsibleState: treeItem.collapsibleState,
                    contextValue: treeItem.contextValue,
                    command: this.commandsConverter.toSafeCommand(treeItem.command, toDisposeElement)
                };
                return treeViewItem;
            });
            return Promise.all(treeItemPromises);
        }
        else {
            return undefined;
        }
    }
    clearChildren(parentNode) {
        if (parentNode) {
            if (parentNode.children) {
                for (const child of parentNode.children) {
                    this.clear(child);
                }
            }
            delete parentNode['children'];
        }
        else {
            this.clearAll();
        }
    }
    clear(node) {
        if (node.children) {
            for (const child of node.children) {
                this.clear(child);
            }
        }
        this.nodes.delete(node.id);
        node.dispose();
    }
    clearAll() {
        this.nodes.forEach(node => node.dispose());
        this.nodes.clear();
    }
    async onExpanded(treeItemId) {
        // get element from a cache
        const cachedElement = this.getTreeItem(treeItemId);
        // fire an event
        if (cachedElement) {
            this.onDidExpandElementEmitter.fire({
                element: cachedElement
            });
        }
    }
    async onCollapsed(treeItemId) {
        // get element from a cache
        const cachedElement = this.getTreeItem(treeItemId);
        // fire an event
        if (cachedElement) {
            this.onDidCollapseElementEmitter.fire({
                element: cachedElement
            });
        }
    }
    get selectedElements() {
        const items = [];
        for (const id of this.selectedItemIds) {
            const item = this.getTreeItem(id);
            if (item) {
                items.push(item);
            }
        }
        return items;
    }
    setSelection(selectedItemIds) {
        const toDelete = new Set(this.selectedItemIds);
        for (const id of selectedItemIds) {
            toDelete.delete(id);
            if (!this.selectedItemIds.has(id)) {
                this.doSetSelection(selectedItemIds);
                return;
            }
        }
        if (toDelete.size) {
            this.doSetSelection(selectedItemIds);
        }
    }
    doSetSelection(selectedItemIts) {
        this.selectedItemIds = new Set(selectedItemIts);
        this.onDidChangeSelectionEmitter.fire(Object.freeze({ selection: this.selectedElements }));
    }
    get visible() {
        return this._visible;
    }
    setVisible(visible) {
        if (visible !== this._visible) {
            this._visible = visible;
            this.onDidChangeVisibilityEmitter.fire(Object.freeze({ visible: this._visible }));
        }
    }
}
//# sourceMappingURL=tree-views.js.map