"use strict";
// *****************************************************************************
// Copyright (C) 2017-2018 TypeFox and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
Object.defineProperty(exports, "__esModule", { value: true });
exports.NsfwFileSystemWatcherServer = void 0;
const fs = require("fs");
const nsfw = require("@theia/core/shared/nsfw");
const paths = require("path");
const minimatch_1 = require("minimatch");
const disposable_1 = require("@theia/core/lib/common/disposable");
const file_uri_1 = require("@theia/core/lib/node/file-uri");
const file_change_collection_1 = require("../file-change-collection");
const timers_1 = require("timers");
const debounce = require("@theia/core/shared/lodash.debounce");
/**
 * @deprecated since 1.6.0 use `NsfwFileSystemWatcherService` instead.
 */
class NsfwFileSystemWatcherServer {
    constructor(options) {
        this.watcherSequence = 1;
        this.watchers = new Map();
        this.watcherOptions = new Map();
        this.toDispose = new disposable_1.DisposableCollection(disposable_1.Disposable.create(() => this.setClient(undefined)));
        this.changes = new file_change_collection_1.FileChangeCollection();
        /**
         * Fires file changes to clients.
         * It is debounced in the case if the filesystem is spamming to avoid overwhelming clients with events.
         */
        this.fireDidFilesChanged = debounce(() => this.doFireDidFilesChanged(), 50);
        this.options = Object.assign({ nsfwOptions: {}, verbose: false, info: (message, ...args) => console.info(message, ...args), error: (message, ...args) => console.error(message, ...args) }, options);
    }
    dispose() {
        this.toDispose.dispose();
    }
    async watchFileChanges(uri, options) {
        const watcherId = this.watcherSequence++;
        const basePath = file_uri_1.FileUri.fsPath(uri);
        this.debug('Starting watching:', basePath);
        const toDisposeWatcher = new disposable_1.DisposableCollection();
        this.watchers.set(watcherId, toDisposeWatcher);
        toDisposeWatcher.push(disposable_1.Disposable.create(() => this.watchers.delete(watcherId)));
        if (fs.existsSync(basePath)) {
            this.start(watcherId, basePath, options, toDisposeWatcher);
        }
        else {
            const toClearTimer = new disposable_1.DisposableCollection();
            const timer = (0, timers_1.setInterval)(() => {
                if (fs.existsSync(basePath)) {
                    toClearTimer.dispose();
                    this.pushAdded(watcherId, basePath);
                    this.start(watcherId, basePath, options, toDisposeWatcher);
                }
            }, 500);
            toClearTimer.push(disposable_1.Disposable.create(() => (0, timers_1.clearInterval)(timer)));
            toDisposeWatcher.push(toClearTimer);
        }
        this.toDispose.push(toDisposeWatcher);
        return watcherId;
    }
    async start(watcherId, basePath, rawOptions, toDisposeWatcher) {
        const options = Object.assign({ ignored: [] }, rawOptions);
        if (options.ignored.length > 0) {
            this.debug('Files ignored for watching', options.ignored);
        }
        let watcher = await nsfw(fs.realpathSync(basePath), (events) => {
            for (const event of events) {
                if (event.action === 0 /* CREATED */) {
                    this.pushAdded(watcherId, this.resolvePath(event.directory, event.file));
                }
                if (event.action === 1 /* DELETED */) {
                    this.pushDeleted(watcherId, this.resolvePath(event.directory, event.file));
                }
                if (event.action === 2 /* MODIFIED */) {
                    this.pushUpdated(watcherId, this.resolvePath(event.directory, event.file));
                }
                if (event.action === 3 /* RENAMED */) {
                    this.pushDeleted(watcherId, this.resolvePath(event.directory, event.oldFile));
                    this.pushAdded(watcherId, this.resolvePath(event.newDirectory || event.directory, event.newFile));
                }
            }
        }, Object.assign({ errorCallback: error => {
                // see https://github.com/atom/github/issues/342
                console.warn(`Failed to watch "${basePath}":`, error);
                if (error === 'Inotify limit reached') {
                    if (this.client) {
                        this.client.onError();
                    }
                }
                this.unwatchFileChanges(watcherId);
            } }, this.options.nsfwOptions));
        await watcher.start();
        this.options.info('Started watching:', basePath);
        if (toDisposeWatcher.disposed) {
            this.debug('Stopping watching:', basePath);
            await watcher.stop();
            // remove a reference to nsfw otherwise GC cannot collect it
            watcher = undefined;
            this.options.info('Stopped watching:', basePath);
            return;
        }
        toDisposeWatcher.push(disposable_1.Disposable.create(async () => {
            this.watcherOptions.delete(watcherId);
            if (watcher) {
                this.debug('Stopping watching:', basePath);
                await watcher.stop();
                // remove a reference to nsfw otherwise GC cannot collect it
                watcher = undefined;
                this.options.info('Stopped watching:', basePath);
            }
        }));
        this.watcherOptions.set(watcherId, {
            ignored: options.ignored.map(pattern => new minimatch_1.Minimatch(pattern, { dot: true }))
        });
    }
    unwatchFileChanges(watcherId) {
        const disposable = this.watchers.get(watcherId);
        if (disposable) {
            this.watchers.delete(watcherId);
            disposable.dispose();
        }
        return Promise.resolve();
    }
    setClient(client) {
        if (client && this.toDispose.disposed) {
            return;
        }
        this.client = client;
    }
    pushAdded(watcherId, path) {
        this.debug('Added:', path);
        this.pushFileChange(watcherId, path, 1 /* ADDED */);
    }
    pushUpdated(watcherId, path) {
        this.debug('Updated:', path);
        this.pushFileChange(watcherId, path, 0 /* UPDATED */);
    }
    pushDeleted(watcherId, path) {
        this.debug('Deleted:', path);
        this.pushFileChange(watcherId, path, 2 /* DELETED */);
    }
    pushFileChange(watcherId, path, type) {
        if (this.isIgnored(watcherId, path)) {
            return;
        }
        const uri = file_uri_1.FileUri.create(path).toString();
        this.changes.push({ uri, type });
        this.fireDidFilesChanged();
    }
    resolvePath(directory, file) {
        const path = paths.join(directory, file);
        try {
            return fs.realpathSync(path);
        }
        catch (_a) {
            try {
                // file does not exist try to resolve directory
                return paths.join(fs.realpathSync(directory), file);
            }
            catch (_b) {
                // directory does not exist fall back to symlink
                return path;
            }
        }
    }
    doFireDidFilesChanged() {
        const changes = this.changes.values();
        this.changes = new file_change_collection_1.FileChangeCollection();
        const event = { changes };
        if (this.client) {
            this.client.onDidFilesChanged(event);
        }
    }
    isIgnored(watcherId, path) {
        const options = this.watcherOptions.get(watcherId);
        return !!options && options.ignored.length > 0 && options.ignored.some(m => m.match(path));
    }
    debug(message, ...params) {
        if (this.options.verbose) {
            this.options.info(message, ...params);
        }
    }
}
exports.NsfwFileSystemWatcherServer = NsfwFileSystemWatcherServer;
//# sourceMappingURL=nsfw-filesystem-watcher.js.map