/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.persistence.CascadeType;
import javax.persistence.FetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalDiscriminatorClass;
import org.eclipse.persistence.tools.mapping.orm.ExternalDiscriminatorColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.mapping.orm.ExternalVariableOneToOneMapping;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of a variable one to one, which is a child of an entity.
 *
 * @see Entity
 *
 * @version 2.6
 */
final class VariableOneToOneMapping extends NonTransientMapping
                                    implements ExternalVariableOneToOneMapping {

	/**
	 * The list of ordered element names required for insertion of children at the right location.
	 */
	private List<String> cascadeTypeIndices;

	/**
	 * Creates a new <code>VariableOneToOneMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	VariableOneToOneMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addCascadeType(CascadeType type) {

		Element element = getChild(CASCADE);

		if (element == null) {
			element = addChild(CASCADE);
		}

		addChild(element, cascadeType(type), cascadeTypeIndices);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalDiscriminatorClass addDiscriminatorClass(String discrimator, String value) {
		DiscriminatorClass discriminatorClass = buildDiscriminatorClass(-1);
		discriminatorClass.addSelf();
		discriminatorClass.setDiscriminator(discrimator);
		discriminatorClass.setValue(value);
		return discriminatorClass;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalDiscriminatorColumn addDiscriminatorColumn() {
		DiscriminatorColumn discriminatorColumn = buildDiscriminatorColumn();
		discriminatorColumn.addSelf();
		return discriminatorColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn addJoinColumn(String name) {
		JoinColumn joinColumn = buildJoinColumn(-1);
		joinColumn.addSelf();
		joinColumn.setName(name);
		return joinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_INTERFACE);
		names.add(FETCH);
		names.add(OPTIONAL);
		names.add(ORPHAN_REMOVAL);
		return names;
	}

	private List<String> buildCascadeTypeIndices() {
		List<String> names = new ArrayList<String>();
		names.add(CASCADE_ALL);
		names.add(CASCADE_PERSIST);
		names.add(CASCADE_MERGE);
		names.add(CASCADE_REMOVE);
		names.add(CASCADE_REFRESH);
		names.add(CASCADE_DETACH);
		return names;
	}

	private DiscriminatorClass buildDiscriminatorClass(int index) {
		return new DiscriminatorClass(this, index);
	}

	private DiscriminatorColumn buildDiscriminatorColumn() {
		return new DiscriminatorColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(CASCADE);
		names.add(DiscriminatorColumn.DISCRIMINATOR_COLUMN);
		names.add(DiscriminatorClass.DISCRIMINATOR_CLASS);
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(PRIVATE_OWNED);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, JoinColumn.JOIN_COLUMN, index);
	}

	private String cascadeType(CascadeType type) {
		switch (type) {
			case ALL:     return CASCADE_ALL;
			case DETACH:  return CASCADE_DETACH;
			case MERGE:   return CASCADE_MERGE;
			case PERSIST: return CASCADE_PERSIST;
			case REFRESH: return CASCADE_REFRESH;
			case REMOVE:  return CASCADE_REMOVE;
			default:      return null;
		}
	}

	private CascadeType cascadeType(Element element) {

		String elementName = getNodeName(element);

		if (ObjectTools.equals(elementName, CASCADE_ALL)) {
			return CascadeType.ALL;
		}

		if (ObjectTools.equals(elementName, CASCADE_MERGE)) {
			return CascadeType.MERGE;
		}

		if (ObjectTools.equals(elementName, CASCADE_DETACH)) {
			return CascadeType.DETACH;
		}

		if (ObjectTools.equals(elementName, CASCADE_PERSIST)) {
			return CascadeType.PERSIST;
		}

		if (ObjectTools.equals(elementName, CASCADE_REFRESH)) {
			return CascadeType.REFRESH;
		}

		if (ObjectTools.equals(elementName, CASCADE_REMOVE)) {
			return CascadeType.REMOVE;
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<CascadeType> cascadeTypes() {

		Element element = getChild(CASCADE);

		if (element == null) {
			return Collections.emptyList();
		}

		List<Element> children = getChildren(element);
		List<CascadeType> cascadeTypes = new ArrayList<CascadeType>(children.size());

		for (Element childElement : children) {
			CascadeType cascadeType = cascadeType(childElement);

			if (cascadeType != null) {
				cascadeTypes.add(cascadeType);
			}
		}

		return cascadeTypes;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalDiscriminatorClass> discriminatorClasses() {

		int count = discriminatorClassesSize();
		List<ExternalDiscriminatorClass> discriminatorClasses = new ArrayList<ExternalDiscriminatorClass>(count);

		for (int index = 0; index < count; index++) {
			discriminatorClasses.add(buildDiscriminatorClass(index));
		}

		return discriminatorClasses;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int discriminatorClassesSize() {
		return getChildrenSize(DiscriminatorClass.DISCRIMINATOR_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalDiscriminatorClass getDiscriminatorClass(String value, String discriminator) {

		int count = discriminatorClassesSize();

		for (int index = 0; index < count; index++) {
			ExternalDiscriminatorClass discriminatorClass = buildDiscriminatorClass(index);

			if (ObjectTools.equals(discriminator, discriminatorClass.getDiscriminator()) &&
			    ObjectTools.equals(value, discriminatorClass.getValue())) {

				return discriminatorClass;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalDiscriminatorColumn getDiscriminatorColumn() {

		if (hasChild(DiscriminatorColumn.DISCRIMINATOR_COLUMN)) {
			return buildDiscriminatorColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return VARIABLE_ONE_TO_ONE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getFetchTypeTextRange() {
		return getAttributeTextRange(FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		if (hasChild(JoinColumn.JOIN_COLUMN, index)) {
			return buildJoinColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getOptionalTextRange() {
		return getAttributeTextRange(OPTIONAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getOrphanRemovalTextRange() {
		return getAttributeTextRange(ORPHAN_REMOVAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPrivateOwnedTextRange() {
		return getChildTextRange(PRIVATE_OWNED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTargetInterfaceName() {
		return getAttribute(TARGET_INTERFACE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTargetInterfaceNameTextRange() {
		return getAttributeTextRange(TARGET_INTERFACE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void initialize() {
		super.initialize();
		cascadeTypeIndices = buildCascadeTypeIndices();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isOptional() {
		return getBooleanAttribute(OPTIONAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isOrphanRemoval() {
		return getBooleanAttribute(ORPHAN_REMOVAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isPrivateOwned() {
		return hasChild(PRIVATE_OWNED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			joinColumns.add(buildJoinColumn(index));
		}

		return joinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeCascadeType(CascadeType type) {

		Element element = getChild(CASCADE);

		if (element != null) {
			removeChild(element, cascadeType(type));

			if (!hasAnyChildren(element)) {
				remove(element);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeDiscriminatorClass(int index) {
		DiscriminatorClass discClass = buildDiscriminatorClass(index);
		discClass.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeDiscriminatorColumn() {
		DiscriminatorColumn column = buildDiscriminatorColumn();
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOptional(Boolean optional) {
		setAttribute(OPTIONAL, optional);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOrphanRemoval(Boolean value) {
		setAttribute(ORPHAN_REMOVAL, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrivateOwned(Boolean privateOwned) {
		if (privateOwned == Boolean.TRUE) {
			addChild(PRIVATE_OWNED);
		}
		else {
			removeChild(PRIVATE_OWNED);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTargetInterfaceName(String entityName) {
		setAttribute(TARGET_INTERFACE, entityName);
	}
}