/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.OptimisticLockingType;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalOptimisticLocking;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form of a optimistic locking, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.6
 */
final class OptimisticLocking extends AbstractExternalForm
                              implements ExternalOptimisticLocking {

	/**
	 * Creates a new <code>OptimisticLocking</code>.
	 *
	 * @param parent The parent of this external form
	 */
	OptimisticLocking(MappedSuperclass parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addColumn(String columnName) {
		EntityColumn column = buildColumn(-1);
		column.addSelf();
		column.setName(columnName);
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(TYPE);
		names.add(CASCADE);
		return names;
	}

	private EntityColumn buildColumn(int index) {
		return new EntityColumn(this, SELECTED_COLUMN, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(SELECTED_COLUMN);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalEntityColumn> columns() {

		int count = columnsSize();
		List<ExternalEntityColumn> columns = new ArrayList<ExternalEntityColumn>(count);

		for (int index = 0; index < count; index++) {
			columns.add(buildColumn(index));
		}

		return columns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int columnsSize() {
		return getChildrenSize(SELECTED_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getCascadeTextRange() {
		return getAttributeTextRange(CASCADE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn(int index) {

		if (hasChild(SELECTED_COLUMN, index)) {
			return buildColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getColumnTextRange(String columnName) {

		for (ExternalEntityColumn column : columns()) {
			if (columnName.equals(column.getName())) {
				return column.getTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return OPTIMISTIC_LOCKING;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public OptimisticLockingType getOptimisticLockingType() {
		return getEnumAttribute(TYPE, OptimisticLockingType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getOptimisticLockingTypeTextRange() {
		return getAttributeTextRange(TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAllColumns() {
		removeChildren(SELECTED_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn(int index) {
		EntityColumn column = buildColumn(index);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOptimisticLockingType(OptimisticLockingType type) {

		if (type == null) {
			if (!hasAnyChildren()) {
				removeSelf();
			}
			else {
				setAttribute(TYPE, type);
			}
		}
		else {
			setAttribute(TYPE, type);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setShouldCascade(Boolean value) {
		setAttribute(CASCADE, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean shouldCascade() {
		return getBooleanAttribute(CASCADE);
	}
}