/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.utility.StringTools;

import static org.eclipse.persistence.tools.mapping.orm.XmlConstants.*;
import static org.eclipse.persistence.tools.mapping.persistence.PersistenceXmlConstants.*;

/**
 * Represents the document version of orm XML files.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.5
 */
public enum PersistenceDocumentType {

	/**
	 * Indicates the JPA doc version is 1.0.
	 */
	JPA_1_0(PERSISTENCE_NAMESPACE_URI, VERSION_1_0, PERSISTENCE_XSD_URI_1_0),

	/**
	 * Indicates the JPA doc version is 2.0.
	 */
	JPA_2_0(PERSISTENCE_NAMESPACE_URI, VERSION_2_0, PERSISTENCE_XSD_URI_2_0),

	/**
	 * Indicates the JPA doc version is 2.1.
	 */
	JPA_2_1(PERSISTENCE_NAMESPACE_URI, VERSION_2_1, PERSISTENCE_XSD_URI_2_1),

	/**
	 * Indicates the JPA doc version unknown or unsupported.
	 */
	UNKNOWN(PERSISTENCE_NAMESPACE_URI, VERSION_2_0, PERSISTENCE_XSD_URI_2_0);

	/**
	 * The location of the schema for this document type;
	 */
	private String schemaURI;

	/**
	 * The document version number.
	 */
	private String version;

	/**
	 * The document, XML namespace;
	 */
	private String xmlNamespace;

	/**
	 * Creates a new <code>PersistenceDocumentType</code>.
	 *
	 * @param xmlNamespace the namespace of the document.
	 * @param version The actual version number
	 * @param schemaURI the schema URI.
	 */
	private PersistenceDocumentType(String xmlNamespace, String version, String schemaURI) {
		this.xmlNamespace = xmlNamespace;
		this.version = version;
		this.schemaURI = schemaURI;
	}

	/**
	 * Returns the list of the types.
	 *
	 * @return The list of unique constants
	 */
	public static Iterable<PersistenceDocumentType> types() {
		List<PersistenceDocumentType> types = new ArrayList<PersistenceDocumentType>(2);
		types.add(JPA_1_0);
		types.add(JPA_2_0);
//		types.add(JPA_2_1); // Not supported yet
		return types;
	}

	/**
	 * Retrieves the enumeration constant for the document parameters. If the type is
	 * not known, then {@link #UNKNOWN} will be returned.  If the <code>schemaURI</code>
	 * is identified, then the namespace and version are ignored.
	 *
	 * @param xmlNamespace the namespace of the document.
	 * @param version the version of the orm document.
	 * @param schemaURI the schema uri of the document.
	 *
	 * @return The constant version of the given value
	 */
	public static PersistenceDocumentType value(String xmlNamespace, String version, String schemaURI) {

		for (PersistenceDocumentType type : types()) {

			if (StringTools.equalsIgnoreCase(schemaURI, type.schemaURI) ||
			   (StringTools.equalsIgnoreCase(type.xmlNamespace, xmlNamespace) &&
			    StringTools.equalsIgnoreCase(type.version, version))) {

				return type;
			}
		}

		return UNKNOWN;
	}

	/**
	 * Returns the schema {@link URI} of this type.
	 *
	 * @return the schema URI.
	 */
	public String getSchemaURI() {
		return this.schemaURI;
	}

	/**
	 * Returns the real version this constant represents.
	 *
	 * @return The string value of the version
	 */
	public String getVersion() {
		return this.version;
	}

	/**
	 * Returns the namespace of this type.
	 *
	 * @return the namespace.
	 */
	public String getXmlNamespace() {
		return this.xmlNamespace;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return this.version;
	}
}