/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import javax.persistence.DiscriminatorType;
import org.eclipse.persistence.tools.mapping.orm.ExternalTenantDiscriminatorColumn;
import org.w3c.dom.Element;

/**
 * The external form of the tenant discriminator column.
 *
 * @see MultitenancyPolicy
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class TenantDiscriminatorColumn extends AbstractExternalForm
                                      implements ExternalTenantDiscriminatorColumn {

	/**
	 * The index of this column
	 */
	private int index;

	/**
	 * Determines whether this external form is owned by an entity, the ORM's persistence unit,
	 * or the ORM configuration since the structure is different.
	 */
	private boolean pUnit;

	/**
	 * The attribute name used to store and retrieve the column-definition property.
	 */
	static final String COLUMN_DEFINITION = "column-definition";

	/**
	 * The attribute name used to store and retrieve the context-property property.
	 */
	static final String CONTEXT_PROPERTY = "context-property";

	/**
	 * The attribute name used to store and retrieve the discriminator-type property.
	 */
	static final String DISCRIMINATOR_TYPE = "discriminator-type";

	/**
	 * The attribute name used to store and retrieve the length property.
	 */
	static final String LENGTH = "length";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the primary-key property.
	 */
	static final String PRIMARY_KEY = "primary-key";

	/**
	 * The attribute name used to store and retrieve the table property.
	 */
	static final String TABLE = "table";

	/**
	 * The node name used to store and retrieve the <code>Element</code>
	 * encapsulated by this external form.
	 */
	static final String TENANT_DISCRIMINATOR_COLUMN = "tenant-discriminator-column";

	/**
	 * Creates a new <code>TenantDiscriminatorColumn</code>.
	 *
	 * @param parent The parent of this external form
	 */
	public TenantDiscriminatorColumn(MultitenancyPolicy parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>TenantDiscriminatorColumn</code>.
	 *
	 * @param parent The parent of this external form
	 */
	public TenantDiscriminatorColumn(ORMConfiguration parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>TenantDiscriminatorColumn</code>.
	 *
	 * @param parent The parent of this external form
	 */
	public TenantDiscriminatorColumn(PersistenceUnit parent, int index) {
		super(parent);
		this.index = index;
		this.pUnit = true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName) {
		Element parentElement;

		if (pUnit) {
			parentElement = getChild(getParent(), PersistenceUnit.PERSISTENCE_UNIT_DEFAULTS);
			if (parentElement == null) {
				parentElement = addChild(
					getParent(),
					PersistenceUnit.PERSISTENCE_UNIT_DEFAULTS
				);
			}
		} else {
			parentElement = getParentElement();
		}

		return addChild(parentElement, elementName, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getColumnDefinition() {
		return getAttribute(COLUMN_DEFINITION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getContextProperty() {
		return getAttribute(CONTEXT_PROPERTY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DiscriminatorType getDiscriminatorType() {
		return getEnumAttribute(DISCRIMINATOR_TYPE, DiscriminatorType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element getElement() {

		Element parentElement;

		if (pUnit) {
			parentElement = getChild(getParent(), PersistenceUnit.PERSISTENCE_UNIT_DEFAULTS);
		}
		else {
			parentElement = getParentElement();
		}

		if (parentElement == null) {
			return null;
		}

		return getChild(parentElement, TENANT_DISCRIMINATOR_COLUMN, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return TENANT_DISCRIMINATOR_COLUMN;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getLength() {
		return getIntegerAttribute(LENGTH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTableName() {
		return getAttribute(TABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isPrimaryKey() {
		return getBooleanAttribute(PRIMARY_KEY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {

		Element parentElement;

		if (pUnit) {
			parentElement = getChild(getParent(), PersistenceUnit.PERSISTENCE_UNIT_DEFAULTS);
		}
		else {
			parentElement = getParentElement();
		}

		if (parentElement == null) {
			return;
		}

		removeChild(parentElement, getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setColumnDefinition(String definition) {

		setAttribute(COLUMN_DEFINITION, definition);

		if (!hasAttributes() && !hasAnyChildren()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setContextProperty(String contextProperty) {

		setAttribute(CONTEXT_PROPERTY, contextProperty);

		if (!hasAttributes() && !hasAnyChildren()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDiscriminatorType(DiscriminatorType type) {

		setAttribute(DISCRIMINATOR_TYPE, type);

		if (!hasAttributes() && !hasAnyChildren()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIndex(int index) {
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIsPrimaryKey(Boolean primaryKey) {

		setAttribute(PRIMARY_KEY, primaryKey);

		if (!hasAttributes() && !hasAnyChildren()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLength(Integer length) {

		setAttribute(LENGTH, length);

		if (!hasAttributes() && !hasAnyChildren()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {

		setAttribute(NAME, name);

		if (!hasAttributes() && !hasAnyChildren()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTableName(String tableName) {

		setAttribute(TABLE, tableName);

		if (!hasAttributes() && !hasAnyChildren()) {
			removeSelf();
		}
	}
}