/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.persistence.AccessType;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddableEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappedSuperClassEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalNativeQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalORMConfiguration;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalPersistenceUnit;
import org.eclipse.persistence.tools.mapping.orm.ExternalSQLResultSetMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalSequenceGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalStoredProcedureQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalStructConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalTableGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTenantDiscriminatorColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ORMDocumentType;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import static org.eclipse.persistence.tools.mapping.orm.ORMXmlConstants.*;
import static org.eclipse.persistence.tools.mapping.orm.XmlConstants.*;

/**
 * The external form interacting with the XML document for the ORM Configuration file.
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings({"nls", "unused"}) // unused used for the import statement: see bug 330740
final class ORMConfiguration extends AbstractExternalForm
                             implements ExternalORMConfiguration {

	/**
	 * The root document that represents this orm config's xml.
	 */
	private Document document;

	/**
	 * The element name of the child text node for the access type.
	 */
	static final String ACCESS = "access";

	/**
	 * The element name of the child text node for the default catalog name.
	 */
	static final String CATALOG = "catalog";

	/**
	 * The element name of the child text node for the description.
	 */
	static final String DESCRIPTION = "description";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String ENTITY_MAPPINGS = "entity-mappings";

	/**
	 * The element name of the child text node for the default package name.
	 */
	static final String PACKAGE = "package";

	/**
	 * The element name of the child text node for the default schema name.
	 */
	static final String SCHEMA = "schema";

	/**
	 * Creates a new <code>ORMConfiguration</code>.
	 *
	 * @param xmlModel The XML model containing the actual information
	 * @param sourceRoot The root location of the ORM configuration file
	 */
	ORMConfiguration(Document document) {
		super(null);
		this.document = document;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalConverter addConverter(int index) {
		Converter converter = buildConverter(index);
		converter.addSelf();
		return converter;
	}

	@Override
	public ExternalTenantDiscriminatorColumn addDiscriminatorColumn(int index) {
		TenantDiscriminatorColumn column = buildDiscriminatorColumn(index);
		column.addSelf();
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addEmbeddableEntity(int index, String className) {
		EmbeddableEntity entity = buildEmbeddableEntity(index);
		entity.addSelf();
		entity.setClassName(className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEmbeddableEntity addEmbeddableEntity(String className) {
		EmbeddableEntity entity = buildEmbeddableEntity(embeddableEntitiesSize());
		entity.addSelf();
		entity.setClassName(className);

		return entity;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addEntity(int index, String className) {
		Entity entity = buildEntity(index);
		entity.addSelf();
		entity.setClassName(className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntity addEntity(String entityClassName) {

		Entity entity = buildEntity(entitiesSize());
		entity.addSelf();
		entity.setClassName(entityClassName);

		return entity;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addMappedSuperClass(int index, String className) {
		MappedSuperClassEntity entity = buildMappedSuperclass(index);
		entity.addSelf();
		entity.setClassName(className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addNamedNativeQuery(int index, String name) {
		NamedNativeQuery namedNativeQuery = buildNamedNativeQuery(index);
		namedNativeQuery.addSelf();
		namedNativeQuery.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalNamedQuery addNamedQuery(int index, String name) {
		NamedQuery namedQuery = buildNamedQuery(index);
		namedQuery.addSelf();
		namedQuery.setName(name);
		return namedQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalNamedQuery addNamedQuery(String name) {
		NamedQuery namedQuery = buildNamedQuery(namedQueriesSize());
		namedQuery.addSelf();
		namedQuery.setName(name);
		return namedQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalObjectTypeConverter addObjectTypeConverter(int index) {
		ObjectTypeConverter converter = buildObjectTypeConverter(index);
		converter.addSelf();
		return converter;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName) {
		return addSelf(elementName, Collections.<String>emptyList());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {

		Document document = getDocument();

		Element element = document.createElementNS(ECLIPSELINK_ORM_NAMESPACE_URI, elementName);
		element.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns", ECLIPSELINK_ORM_NAMESPACE_URI);
		addXmlns(element, "xsi", XSI_URI);
		element.setAttributeNS(XSI_URI, XSD_URI_ATTRIBUTE, buildSchemaLocation(ORMDocumentType.ECLIPELINK_2_5));
		element.setAttribute(VERSION, getBuildVersion());

		document.appendChild(element);

		return element;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addSequenceGenerator(int index, String name) {
		SequenceGenerator sequenceGenerator = buildSequenceGenerator(index);
		sequenceGenerator.addSelf();
		sequenceGenerator.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addSqlResultSetMapping(int index, String name) {
		SQLResultSetMapping sqlResultSetMapping = buildSqlResultSetMapping(index);
		sqlResultSetMapping.addSelf();
		sqlResultSetMapping.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addStoredProcedureQuery(int index, String name, String procedureName) {
		NamedStoredProcedureQuery storedProcedureQuery = buildStoredProcedureQuery(index);
		storedProcedureQuery.addSelf();
		storedProcedureQuery.setName(name);
		storedProcedureQuery.setProcedureName(procedureName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStructConverter addStructConverter(int index) {
		StructConverter converter = buildStructConverter(index);
		converter.addSelf();
		return converter;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addTableGenerator(int index, String name) {
		TableGenerator sequenceGenerator = buildTableGenerator(index);
		sequenceGenerator.addSelf();
		sequenceGenerator.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTypeConverter addTypeConverter(int index) {
		TypeConverter converter = buildTypeConverter(index);
		converter.addSelf();
		return converter;
	}

	private Converter buildConverter(int index) {
		return new Converter(this, index);
	}

	private TenantDiscriminatorColumn buildDiscriminatorColumn(int index) {
		return new TenantDiscriminatorColumn(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {

		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(PersistenceUnit.PERSISTENCE_UNIT_METADATA);
		names.add(PACKAGE);
		names.add(SCHEMA);
		names.add(CATALOG);
		names.add(ACCESS);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(SequenceGenerator.SEQUENCE_GENERATOR);
		names.add(TableGenerator.TABLE_GENERATOR);
		names.add(NamedQuery.NAMED_QUERY);
		names.add(NamedNativeQuery.NAMED_NATIVE_QUERY);
		names.add(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY);
		names.add(SQLResultSetMapping.SQL_RESULT_SET_MAPPING);
		names.add(MappedSuperClassEntity.MAPPED_SUPERCLASS);
		names.add(Entity.ENTITY);
		names.add(EmbeddableEntity.EMBEDDABLE);
		return names;
	}

	private EmbeddableEntity buildEmbeddableEntity(int index) {
		return new EmbeddableEntity(this, index);
	}

	private Entity buildEntity(int index) {
		return new Entity(this, index);
	}

	private MappedSuperClassEntity buildMappedSuperclass(int index) {
		return new MappedSuperClassEntity(this, index);
	}

	private NamedNativeQuery buildNamedNativeQuery(int index) {
		return new NamedNativeQuery(this, index);
	}

	private NamedQuery buildNamedQuery(int index) {
		return new NamedQuery(this, index);
	}

	private ObjectTypeConverter buildObjectTypeConverter(int index) {
		return new ObjectTypeConverter(this, index);
	}

	private PersistenceUnit buildPersistenceUnit() {
		return new PersistenceUnit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPersistenceUnit buildPersistenceUnitMetaData() {
		PersistenceUnit persistenceUnit = buildPersistenceUnit();
		persistenceUnit.addSelf();
		return persistenceUnit;
	}

	private String buildSchemaLocation(ORMDocumentType version) {

		StringBuilder sb = new StringBuilder();
		sb.append(version.getXmlNamespace());
		sb.append(" ");
		sb.append(version.getSchemaURI());

		return sb.toString();
	}

	private SequenceGenerator buildSequenceGenerator(int index) {
		return new SequenceGenerator(this, index);
	}

	private SQLResultSetMapping buildSqlResultSetMapping(int index) {
		return new SQLResultSetMapping(this, index);
	}

	private NamedStoredProcedureQuery buildStoredProcedureQuery(int index) {
		return new NamedStoredProcedureQuery(this, index);
	}

	private StructConverter buildStructConverter(int index) {
		return new StructConverter(this, index);
	}

	private TableGenerator buildTableGenerator(int index) {
		return new TableGenerator(this, index);
	}

	private TypeConverter buildTypeConverter(int index) {
		return new TypeConverter(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalConverter> converters() {

		int count = convertersSize();
		List<ExternalConverter> converters = new ArrayList<ExternalConverter>(count);

		for (int index = count; --index >= 0;) {
			ExternalConverter converter = buildConverter(index);
			converters.add(0, converter);
		}

		return new ListListIterable<ExternalConverter>(converters);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int convertersSize() {
		return getChildrenSize(Converter.CONVERTER);
	}

	@Override
	public ListIterable<ExternalTenantDiscriminatorColumn> discriminatorColumns() {
		int count = discriminatorColumnSize();
		List<ExternalTenantDiscriminatorColumn> columns = new ArrayList<ExternalTenantDiscriminatorColumn>(count);

		for (int index = count; --index >= 0; )
		{
			ExternalTenantDiscriminatorColumn column = buildDiscriminatorColumn(index);
			columns.add(0, column);
		}

		return new ListListIterable<ExternalTenantDiscriminatorColumn>(columns);
	}

	@Override
	public int discriminatorColumnSize() {
		return getChildrenSize(TenantDiscriminatorColumn.TENANT_DISCRIMINATOR_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalEmbeddableEntity> embeddableEntities() {

		int count = embeddableEntitiesSize();
		List<ExternalEmbeddableEntity> entities = new ArrayList<ExternalEmbeddableEntity>(count);

		for (int index = count; --index >= 0;) {
			ExternalEmbeddableEntity entity = buildEmbeddableEntity(index);
			entities.add(0, entity);
		}

		return new ListListIterable<ExternalEmbeddableEntity>(entities);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int embeddableEntitiesSize() {
		return getChildrenSize(EmbeddableEntity.EMBEDDABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalEntity> entities() {

		int count = entitiesSize();
		List<ExternalEntity> entities = new ArrayList<ExternalEntity>(count);

		for (int index = count; --index >= 0;) {
			ExternalEntity entity = buildEntity(index);
			entities.add(0, entity);
		}

		return new ListListIterable<ExternalEntity>(entities);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int entitiesSize() {
		return getChildrenSize(Entity.ENTITY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public AccessType getAccessType() {
		return getChildEnumNode(ACCESS, AccessType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getBuildVersion() {
		return VERSION_2_0;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getCatalogName() {
		return getChildTextNode(CATALOG);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalConverter getConverter(int index) {

		Element element = getChild(Converter.CONVERTER, index);

		if (element == null) {
			return null;
		}

		return buildConverter(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	@Override
	public ExternalTenantDiscriminatorColumn getDiscriminatorColumn(int index) {
		Element element = getChild(TenantDiscriminatorColumn.TENANT_DISCRIMINATOR_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildDiscriminatorColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Document getDocument() {
		return this.document;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ORMDocumentType getDocumentType() {
		return ORMDocumentType.value(getNamespace(), getVersion(), getSchemaLocation());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getRootElement();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ENTITY_MAPPINGS;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEmbeddableEntity getEmbeddableEntity(int index) {

		Element element = getChild(EmbeddableEntity.EMBEDDABLE, index);

		if (element == null) {
			return null;
		}

		return buildEmbeddableEntity(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntity getEntity(int index) {

		Element element = getChild(Entity.ENTITY, index);

		if (element == null) {
			return null;
		}

		return buildEntity(index);
	}

	@Override
	public ExternalEntity getEntity(String className) {
		for (ExternalEntity entity : entities()) {
			if (ObjectTools.equals(entity.getClassName(), className)) {
				return entity;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalMappedSuperClassEntity getMappedSuperClass(int index) {

		Element element = getChild(MappedSuperClassEntity.MAPPED_SUPERCLASS, index);

		if (element == null) {
			return null;
		}

		return buildMappedSuperclass(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalNativeQuery getNamedNativeQuery(int index) {

		Element element = getChild(NamedNativeQuery.NAMED_NATIVE_QUERY, index);

		if (element == null) {
			return null;
		}

		return buildNamedNativeQuery(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBasicNamedQuery getNamedQuery(int index) {

		Element element = getChild(NamedQuery.NAMED_QUERY, index);

		if (element == null) {
			return null;
		}

		return buildNamedQuery(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBasicNamedQuery getNamedQuery(String queryName) {
		for (ExternalNamedQuery namedQuery : namedQueries()) {
			if (namedQuery.getName().equals(queryName)) {
				return (ExternalBasicNamedQuery)namedQuery;
			}
		}
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalObjectTypeConverter getObjectTypeConverter(int index) {

		Element element = getChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER, index);

		if (element == null) {
			return null;
		}

		return buildObjectTypeConverter(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPackageName() {
		return getChildTextNode(PACKAGE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PersistenceUnit getPersistenceUnitMetaData() {

		Element element = getChild(PersistenceUnit.PERSISTENCE_UNIT_METADATA);

		if (element == null) {
			return null;
		}

		return buildPersistenceUnit();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getSchemaName() {
		return getChildTextNode(SCHEMA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSequenceGenerator getSequenceGenerator(int index) {

		Element element = getChild(SequenceGenerator.SEQUENCE_GENERATOR, index);

		if (element == null) {
			return null;
		}

		return buildSequenceGenerator(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSQLResultSetMapping getSqlResultSetMapping(int index) {

		Element element = getChild(SQLResultSetMapping.SQL_RESULT_SET_MAPPING, index);

		if (element == null) {
			return null;
		}

		return buildSqlResultSetMapping(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStoredProcedureQuery getStoredProcedureQuery(int index) {

		Element element = getChild(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY, index);

		if (element == null) {
			return null;
		}

		return buildStoredProcedureQuery(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStoredProcedureQuery getStoredProcedureQuery(int index, String name) {

		ExternalStoredProcedureQuery storedProcedureQuery = getStoredProcedureQuery(index);

		if (storedProcedureQuery == null) {
			return null;
		}

		if (ObjectTools.notEquals(name, storedProcedureQuery.getName())) {
			storedProcedureQuery = null;
		}

		return storedProcedureQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStructConverter getStructConverter(int index) {

		Element element = getChild(StructConverter.STRUCT_CONVERTER, index);

		if (element == null) {
			return null;
		}

		return buildStructConverter(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTableGenerator getTableGenerator(int index) {

		Element element = getChild(TableGenerator.TABLE_GENERATOR, index);

		if (element == null) {
			return null;
		}

		return buildTableGenerator(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTypeConverter getTypeConverter(int index) {

		Element element = getChild(TypeConverter.TYPE_CONVERTER, index);

		if (element == null) {
			return null;
		}

		return buildTypeConverter(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getXML() {
		try {

			// write the content into xml file
			TransformerFactory transformerFactory = TransformerFactory.newInstance();
			javax.xml.transform.Transformer transformer = transformerFactory.newTransformer();
			DOMSource source = new DOMSource(getDocument());
			ByteArrayOutputStream ouput = new ByteArrayOutputStream();
			StreamResult result = new StreamResult(ouput);

			transformer.transform(source, result);

			return ouput.toString();

		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalMappedSuperClassEntity> mappedSuperClasses() {

		int count = mappedSuperClassesSize();
		List<ExternalMappedSuperClassEntity> entities = new ArrayList<ExternalMappedSuperClassEntity>(count);

		for (int index = count; --index >= 0;) {
			ExternalMappedSuperClassEntity entity = buildMappedSuperclass(index);
			entities.add(0, entity);
		}

		return new ListListIterable<ExternalMappedSuperClassEntity>(entities);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mappedSuperClassesSize() {
		return getChildrenSize(MappedSuperClassEntity.MAPPED_SUPERCLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalNativeQuery> namedNativeQueries() {

		int count = namedNativeQueriesSize();
		List<ExternalNativeQuery> nativeQueries = new ArrayList<ExternalNativeQuery>(count);

		for (int index = count; --index >= 0;) {
			ExternalNativeQuery nativeQuery = buildNamedNativeQuery(index);
			nativeQueries.add(0, nativeQuery);
		}

		return new ListListIterable<ExternalNativeQuery>(nativeQueries);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int namedNativeQueriesSize() {
		return getChildrenSize(NamedNativeQuery.NAMED_NATIVE_QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalNamedQuery> namedQueries() {

		int count = namedQueriesSize();
		List<ExternalNamedQuery> namedQueries = new ArrayList<ExternalNamedQuery>(count);

		for (int index = count; --index >= 0;) {
			ExternalNamedQuery namedQuery = buildNamedQuery(index);
			namedQueries.add(0, namedQuery);
		}

		return new ListListIterable<ExternalNamedQuery>(namedQueries);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int namedQueriesSize() {
		return getChildrenSize(NamedQuery.NAMED_QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalObjectTypeConverter> objectTypeConverters() {

		int count = objectTypeConvertersSize();
		List<ExternalObjectTypeConverter> converters = new ArrayList<ExternalObjectTypeConverter>(count);

		for (int index = count; --index >= 0;) {
			ExternalObjectTypeConverter converter = buildObjectTypeConverter(index);
			converters.add(0, converter);
		}

		return new ListListIterable<ExternalObjectTypeConverter>(converters);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int objectTypeConvertersSize() {
		return getChildrenSize(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeConverter(int index) {
		Converter converter = buildConverter(index);
		converter.removeSelf();
	}

	@Override
	public void removeDiscriminatorColumn(int index) {
		TenantDiscriminatorColumn column = buildDiscriminatorColumn(index);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeEmbeddableEntity(int index) {
		EmbeddableEntity entity = buildEmbeddableEntity(index);
		entity.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeEntity(int index) {
		Entity entity = buildEntity(index);
		entity.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeEntity(String entityClassName) {
		for (ExternalEntity entity : entities()) {
			if (entity.getClassName().equals(entityClassName)) {
				((Entity)entity).removeSelf();

				return;
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMappedSuperClass(int index) {
		MappedSuperClassEntity entity = buildMappedSuperclass(index);
		entity.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeNamedQuery(int index) {
		NamedQuery namedQuery = buildNamedQuery(index);
		namedQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeNamedQuery(String queryName) {
		for (ExternalNamedQuery query : namedQueries()) {
			if (query.getName().equals(queryName)) {
				((NamedQuery)query).removeSelf();
				return;
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeNativeQuery(int index) {
		NamedNativeQuery namedNativeQuery = buildNamedNativeQuery(index);
		namedNativeQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeObjectTypeConverter(int index) {
		ObjectTypeConverter converter = buildObjectTypeConverter(index);
		converter.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePersistenceUnitMetaData() {
		PersistenceUnit persistenceUnit = buildPersistenceUnit();
		persistenceUnit.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSequenceGenerator(int index) {
		SequenceGenerator sequenceGenerator = buildSequenceGenerator(index);
		sequenceGenerator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSqlResultSetMapping(int index) {
		SQLResultSetMapping sqlResultSetMapping = buildSqlResultSetMapping(index);
		sqlResultSetMapping.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeStoredProcedureQuery(int index) {
		NamedStoredProcedureQuery storedProcedureQuery = buildStoredProcedureQuery(index);
		storedProcedureQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeStructConverter(int index) {
		StructConverter converter = buildStructConverter(index);
		converter.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeTableGenerator(int index) {
		TableGenerator tableGenerator = buildTableGenerator(index);
		tableGenerator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeTypeConverter(int index) {
		TypeConverter converter = buildTypeConverter(index);
		converter.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalSequenceGenerator> sequenceGenerators() {

		int count = sequenceGeneratorsSize();
		List<ExternalSequenceGenerator> sequenceGenerators = new ArrayList<ExternalSequenceGenerator>(count);

		for (int index = count; --index >= 0;) {
			ExternalSequenceGenerator sequenceGenerator = buildSequenceGenerator(index);
			sequenceGenerators.add(0, sequenceGenerator);
		}

		return new ListListIterable<ExternalSequenceGenerator>(sequenceGenerators);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int sequenceGeneratorsSize() {
		return getChildrenSize(SequenceGenerator.SEQUENCE_GENERATOR);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAccessType(AccessType type) {
		updateChildTextNode(ACCESS, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCatalogName(String catalog) {
		updateChildTextNode(CATALOG, catalog);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDocumentType(ORMDocumentType version) {
		setVersion(version.getVersion());
		setSchemaLocation(buildSchemaLocation(version));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPackageName(String packageName) {
		updateChildTextNode(PACKAGE, packageName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSchemaName(String schema) {
		updateChildTextNode(SCHEMA, schema);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalSQLResultSetMapping> sqlResultSetMappings() {

		int count = sqlResultSetMappingsSize();
		List<ExternalSQLResultSetMapping> sqlResultSetMappings = new ArrayList<ExternalSQLResultSetMapping>(count);

		for (int index = count; --index >= 0;) {
			ExternalSQLResultSetMapping sqlResultSetMapping = buildSqlResultSetMapping(index);
			sqlResultSetMappings.add(0, sqlResultSetMapping);
		}

		return new ListListIterable<ExternalSQLResultSetMapping>(sqlResultSetMappings);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int sqlResultSetMappingsSize() {
		return getChildrenSize(SQLResultSetMapping.SQL_RESULT_SET_MAPPING);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalStoredProcedureQuery> storedProcedureQueries() {

		int count = storedProcedureQueriesSize();
		List<ExternalStoredProcedureQuery> storedProcedureQueries = new ArrayList<ExternalStoredProcedureQuery>(count);

		for (int index = count; --index >= 0;) {
			ExternalStoredProcedureQuery nativeQuery = buildStoredProcedureQuery(index);
			storedProcedureQueries.add(0, nativeQuery);
		}

		return new ListListIterable<ExternalStoredProcedureQuery>(storedProcedureQueries);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int storedProcedureQueriesSize() {
		return getChildrenSize(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalStructConverter> structConverters() {
		int count = structConvertersSize();
		List<ExternalStructConverter> converters = new ArrayList<ExternalStructConverter>(count);

		for (int index = count; --index >= 0;) {
			ExternalStructConverter converter = buildStructConverter(index);
			converters.add(0, converter);
		}

		return new ListListIterable<ExternalStructConverter>(converters);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int structConvertersSize() {
		return getChildrenSize(StructConverter.STRUCT_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalTableGenerator> tableGenerators() {

		int count = tableGeneratorsSize();
		List<ExternalTableGenerator> tableGenerators = new ArrayList<ExternalTableGenerator>(count);

		for (int index = count; --index >= 0;) {
			ExternalTableGenerator tableGenerator = buildTableGenerator(index);
			tableGenerators.add(0, tableGenerator);
		}

		return new ListListIterable<ExternalTableGenerator>(tableGenerators);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int tableGeneratorsSize() {
		return getChildrenSize(TableGenerator.TABLE_GENERATOR);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalTypeConverter> typeConverters() {

		int count = typeConvertersSize();
		List<ExternalTypeConverter> converters = new ArrayList<ExternalTypeConverter>(count);

		for (int index = count; --index >= 0;) {
			ExternalTypeConverter converter = buildTypeConverter(index);
			converters.add(0, converter);
		}

		return new ListListIterable<ExternalTypeConverter>(converters);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int typeConvertersSize() {
		return getChildrenSize(TypeConverter.TYPE_CONVERTER);
	}

	/**
	 * Simple buffer implementation of an {@link OutputStream} that allows for capturing the stream as a String.
	 */
	public class StringOutputStream extends OutputStream {

		private StringBuffer buffer = new StringBuffer();

		public StringOutputStream() {
			super();
		}

		public void clear() {
			this.buffer.delete(0, this.buffer.length());
		}

		@Override
		public String toString() {
			return this.buffer.toString();
		}

		@Override
		public void write(int bt) throws IOException {
			this.buffer.append((char) bt);
		}
	}
}