/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalStoredProcedureParameter;
import org.eclipse.persistence.tools.mapping.orm.ExternalStoredProcedureQuery;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of a stored procedure query, which is a child of an entity or an ORM configuration.
 *
 * @see Entity
 * @see ORMConfiguration
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class NamedStoredProcedureQuery extends Query
                                      implements ExternalStoredProcedureQuery {

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String NAMED_STORED_PROCEDURE_QUERY = "named-stored-procedure-query";

	/**
	 * The attribute name used to store and retrieve the procedure-name property.
	 */
	static final String PROCEDURE_NAME = "procedure-name";

	/**
	 * The attribute name used to store and retrieve the result-class property.
	 */
	static final String RESULT_CLASS = "result-class";

	/**
	 * The attribute name used to store and retrieve the result-set-mapping property.
	 */
	static final String RESULT_SET_MAPPING = "result-set-mapping";

	/**
	 * The attribute name used to store and retrieve the returns-result-set property.
	 */
	static final String RETURNS_RESULT_SET = "returns-result-set";

	/**
	 * Creates a new <code>NamedStoredProcedureQuery</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	NamedStoredProcedureQuery(Entity parent, int index) {
		super(parent, index);
	}

	/**
	 * Creates a new <code>NamedStoredProcedureQuery</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	NamedStoredProcedureQuery(ORMConfiguration parent, int index) {
		super(parent, index);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(RESULT_CLASS);
		names.add(RESULT_SET_MAPPING);
		names.add(PROCEDURE_NAME);
		names.add(RETURNS_RESULT_SET);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStoredProcedureParameter addParameter(int index) {
		StoredProcedureParameter parameter = buildParameter(index);
		parameter.addSelf();
		return parameter;
	}

	private StoredProcedureParameter buildParameter(int index) {
		return new StoredProcedureParameter(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean doesReturnResultSet() {
		return getBooleanAttribute(RETURNS_RESULT_SET);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return NAMED_STORED_PROCEDURE_QUERY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStoredProcedureParameter getParameter(int index) {

		Element element = getChild(StoredProcedureParameter.STORED_PROCEDURE_PARAMETER, index);

		if (element == null) {
			return null;
		}

		return buildParameter(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getProcedureName() {
		return getAttribute(PROCEDURE_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getResultClassName() {
		return getAttribute(RESULT_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getResultSetMapping() {
		return getAttribute(RESULT_SET_MAPPING);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalStoredProcedureParameter> parameters() {

		int count = parametersSize();
		List<ExternalStoredProcedureParameter> parameters = new ArrayList<ExternalStoredProcedureParameter>(count);

		for (int index = count; --index >= 0;) {
			ExternalStoredProcedureParameter queryHint = buildParameter(index);
			parameters.add(0, queryHint);
		}

		return new ListListIterable<ExternalStoredProcedureParameter>(parameters);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int parametersSize() {
		return getChildrenSize(StoredProcedureParameter.STORED_PROCEDURE_PARAMETER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeParameter(int index) {
		StoredProcedureParameter parameter = buildParameter(index);
		parameter.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDoesReturnResultSet(Boolean returnResultSet) {
		setAttribute(RETURNS_RESULT_SET, returnResultSet);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setProcedureName(String name) {
		setAttribute(PROCEDURE_NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setResultClassName(String name) {
		setAttribute(RESULT_CLASS, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setResultSetMapping(String mappingName) {
		setAttribute(RESULT_SET_MAPPING, mappingName);
	}
}