/*******************************************************************************
 * Copyright (c) 1998, 2012 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation from Oracle TopLink
******************************************************************************/
package org.eclipse.persistence.tools.db.relational.handles;

import org.eclipse.persistence.tools.db.relational.ELColumnPair;
import org.eclipse.persistence.tools.db.relational.ELModel;
import org.eclipse.persistence.tools.utility.node.Node;

/**
 * MWColumnPairHandle is used to isolate the painful bits of code
 * necessary to correctly handle references to MWColumnPairs.
 * Since a MWColumnPair is nested within the XML file
 * for a MWTable, we need to store a reference to a particular
 * column pair as a list of instance variables:
 *   - the name of the containing table
 *   - the name of the containing reference
 *   - the name of the column pair
 *
 * This causes no end of pain when dealing with TopLink, property
 * change listeners, backward-compatibility, etc.
 *
 * @version 2.5
 */
@SuppressWarnings("nls")
public final class MWColumnPairHandle extends MWHandle {

	/**
	 * This is the actual column pair.
	 * It is built from the table, reference, and column pair names, below.
	 */
	private volatile ELColumnPair columnPair;

	/**
	 * The table, reference, and column pair names are transient. They
	 * are used only to hold their values until #postProjectBuild()
	 * is called and we can resolve the actual column pair.
	 * We do not keep these in synch with the column pair itself because
	 * we cannot know when the column pair has been renamed etc.
	 */
	private volatile String tableName;
	private volatile String referenceName;
	private volatile String columnPairName;

	// ********** constructors **********

	/**
	 * default constructor - for TopLink use only
	 */
	private MWColumnPairHandle() {
		super();
	}

	public MWColumnPairHandle(ELModel parent, NodeReferenceScrubber scrubber) {
		super(parent, scrubber);
	}

	public MWColumnPairHandle(ELModel parent, ELColumnPair columnPair, NodeReferenceScrubber scrubber) {
		super(parent, scrubber);
		this.columnPair = columnPair;
	}


	// ********** instance methods **********

	public ELColumnPair getColumnPair() {
		return this.columnPair;
	}

	public void setColumnPair(ELColumnPair columnPair) {
		this.columnPair = columnPair;
	}

	@Override
	protected Node node() {
		return this.getColumnPair();
	}

	@Override
	public void toString(StringBuffer sb) {
		if (this.columnPair == null) {
			sb.append("null");
		} else {
			this.columnPair.toString(sb);
		}
	}

	public MWColumnPairHandle setScrubber(NodeReferenceScrubber scrubber) {
		this.setScrubberInternal(scrubber);
		return this;
	}
}