/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityListener;
import org.w3c.dom.Element;

/**
 * The external form of an entity listener.
 *
 * @see MappedSuperClassEntity
 * @see PersistenceUnit
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class EntityListener extends AbstractExternalForm
                           implements ExternalEntityListener {

	/**
	 * Determines whether this external form is owned by an entity or the ORM's persistence unit
	 * since the structure is different.
	 */
	private boolean entity;

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The attribute name used to store and retrieve the class property.
	 */
	static String CLASS = "class";

	/**
	 * The element name used to store and retrieve the description child node.
	 */
	static String DESCRIPTION = "description";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static String ENTITY_LISTENER = "entity-listener";

	/**
	 * The element name used to store and retrieve the entity-listeners parent node encapsulating the
	 * {@link Element} represented by this external form.
	 */
	static String ENTITY_LISTENERS = "entity-listeners";

	/**
	 * The attribute name used to store and retrieve the method-name property of the callback child nodes.
	 */
	static String METHOD_NAME = "method-name";

	/**
	 * The element name used to store and retrieve the post-load child node.
	 */
	static String POST_LOAD = "post-load";

	/**
	 * The element name used to store and retrieve the post-persist child node.
	 */
	static String POST_PERSIST = "post-persist";

	/**
	 * The element name used to store and retrieve the post-remove child node.
	 */
	static String POST_REMOVE = "post-remove";

	/**
	 * The element name used to store and retrieve the post-update child node.
	 */
	static String POST_UPDATE = "post-update";

	/**
	 * The element name used to store and retrieve the pre-persist child node.
	 */
	static String PRE_PERSIST = "pre-persist";

	/**
	 * The element name used to store and retrieve the pre-remove child node.
	 */
	static String PRE_REMOVE = "pre-remove";

	/**
	 * The element name used to store and retrieve the pre-update child node.
	 */
	static String PRE_UPDATE = "pre-update";

	/**
	 * Creates a new <code>EntityListener</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	EntityListener(MappedSuperClassEntity parent, int index) {
		super(parent);
		this.index  = index;
		this.entity = true;
	}

	/**
	 * Creates a new <code>EntityListener</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	EntityListener(PersistenceUnit parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {

		Element parentElement;

		if (entity) {
			parentElement = getParentElement();
		}
		else {
			parentElement = getChild(getParent(), PersistenceUnit.PERSISTENCE_UNIT_DEFAULTS);

			if (parentElement == null) {
				parentElement = addChild(
					getParent(),
					PersistenceUnit.PERSISTENCE_UNIT_DEFAULTS,
					elementNamesOrder
				);
			}
		}

		Element element = getChild(parentElement, ENTITY_LISTENERS);

		if (element == null) {
			element = addChild(parentElement, ENTITY_LISTENERS, index);
		}

		return addChild(element, ENTITY_LISTENER);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {

		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(PRE_PERSIST);
		names.add(POST_PERSIST);
		names.add(PRE_REMOVE);
		names.add(POST_REMOVE);
		names.add(PRE_UPDATE);
		names.add(POST_UPDATE);
		names.add(POST_LOAD);
		return names;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getClassName() {
		return getAttribute(CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		Element parentElement;

		if (entity) {
			parentElement = getParentElement();
		}
		else {
			parentElement = getChild(getParent(), PersistenceUnit.PERSISTENCE_UNIT_DEFAULTS);
		}

		if (parentElement == null) {
			return null;
		}

		Element element = getChild(parentElement, ENTITY_LISTENERS);

		if (element == null) {
			return null;
		}

		return getChild(element, ENTITY_LISTENER, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ENTITY_LISTENER;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostLoadMethod() {
		return getChildAttribute(POST_LOAD, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostLoadMethodDescription() {
		return getChildAttribute(POST_LOAD, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostPersistMethod() {
		return getChildAttribute(POST_PERSIST, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostPersistMethodDescription() {
		return getChildAttribute(POST_PERSIST, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostRemoveMethod() {
		return getChildAttribute(POST_REMOVE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostRemoveMethodDescription() {
		return getChildAttribute(POST_REMOVE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostUpdateMethod() {
		return getChildAttribute(POST_UPDATE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostUpdateMethodDescription() {
		return getChildAttribute(POST_UPDATE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPrePersistMethod() {
		return getChildAttribute(PRE_PERSIST, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPrePersistMethodDescription() {
		return getChildAttribute(PRE_PERSIST, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPreRemoveMethod() {
		return getChildAttribute(PRE_REMOVE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPreRemoveMethodDescription() {
		return getChildAttribute(PRE_REMOVE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPreUpdateMethod() {
		return getChildAttribute(PRE_UPDATE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPreUpdateMethodDescription() {
		return getChildAttribute(PRE_UPDATE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {

		Element parentElement = entity ? getParentElement() : getChild(getParent(), PersistenceUnit.PERSISTENCE_UNIT_DEFAULTS);

		if (parentElement == null) {
			return;
		}

		Element element = getChild(parentElement, ENTITY_LISTENERS);

		if (element != null) {
			removeChild(element, ENTITY_LISTENER, index);

			// Remove "entity-listeners" if it has no children
			if (!hasAnyChildren(element)) {
				remove(parentElement, element);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setClassName(String name) {
		setAttribute(CLASS, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostLoadMethod(String value) {
		updateChildAttribute(POST_LOAD, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostLoadMethodDescription(String name) {
		updateChildChildTextNode(POST_LOAD, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostPersistMethod(String value) {
		updateChildAttribute(POST_PERSIST, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostPersistMethodDescription(String name) {
		updateChildChildTextNode(POST_PERSIST, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostRemoveMethod(String value) {
		updateChildAttribute(POST_REMOVE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostRemoveMethodDescription(String name) {
		updateChildChildTextNode(POST_REMOVE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostUpdateMethod(String value) {
		updateChildAttribute(POST_UPDATE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostUpdateMethodDescription(String name) {
		updateChildChildTextNode(POST_UPDATE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrePersistMethod(String value) {
		updateChildAttribute(PRE_PERSIST, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrePersistMethodDescription(String name) {
		updateChildChildTextNode(PRE_PERSIST, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPreRemoveMethod(String value) {
		updateChildAttribute(PRE_REMOVE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPreRemoveMethodDescription(String name) {
		updateChildChildTextNode(PRE_REMOVE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPreUpdateMethod(String value) {
		updateChildAttribute(PRE_UPDATE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPreUpdateMethodDescription(String name) {
		updateChildChildTextNode(PRE_UPDATE, DESCRIPTION, name);
	}
}