/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalConversionValue;
import org.w3c.dom.Element;

/**
 * The external form of a conversion value, which is a child of an object type converter.
 *
 * @see ObjectTypeConverter
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class ConversionValue extends AbstractExternalForm implements ExternalConversionValue {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String CONVERSION_VALUE = "conversion-value";

	/**
	 * The attribute name used to store and retrieve the data-value property.
	 */
	static final String DATA_VALUE = "data-value";

	/**
	 * The attribute name used to store and retrieve the object-value property.
	 */
	static final String OBJECT_VALUE = "object-value";

	/**
	 * Creates a new <code>ConversionValue</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ConversionValue(ObjectTypeConverter parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {

		if (index == -1) {
			return super.addSelf(elementName, elementNamesOrder);
		}

		return addChild(getParent(), elementName, index, elementNamesOrder);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DATA_VALUE);
		names.add(OBJECT_VALUE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDataValue() {
		return getAttribute(DATA_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return CONVERSION_VALUE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getObjectValue() {
		return getAttribute(OBJECT_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {

		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDataValue(String value) {
		setAttribute(DATA_VALUE, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIndex(int value) {
		this.index = value;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setObjectValue(String value) {
		setAttribute(OBJECT_VALUE, value);
	}
}