/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalManyToOneMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlJoinField;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a M:1 mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class ManyToOneMapping extends RelationshipMapping
                             implements ExternalManyToOneMapping {

	/**
	 * The attribute name used to store and retrieve the id property.
	 */
	static final String ID = "id";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String MANY_TO_ONE = "many-to-one";

	/**
	 * The attribute name used to store and retrieve the mapped-by property.
	 */
	static final String MAPPED_BY = "mapped-by-id";

	/**
	 * The attribute name used to store and retrieve the maps-id property.
	 */
	static final String MAPS_ID = "maps-id";

	/**
	 * Creates a new <code>ManyToOneMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ManyToOneMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinColumn(int index, String name) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinField(int index, String name) {
		ExternalNoSqlJoinField joinField = buildJoinField(index);
		joinField.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_ENTITY);
		names.add(FETCH);
		names.add(OPTIONAL);
		names.add(ACCESS);
		names.add(MAPPED_BY);
		names.add(MAPS_ID);
		names.add(ID);
		return names;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(JoinTable.JOIN_TABLE);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, index);
	}

	private ExternalNoSqlJoinField buildJoinField(int index) {
		return new NoSqlJoinField(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return MANY_TO_ONE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean getId() {
		return getBooleanAttribute(ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		Element element = getChild(JoinColumn.JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildJoinColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMappedByMappingName() {
		return getAttribute(MAPPED_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapsId() {
		return getAttribute(MAPS_ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildJoinColumn(index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalNoSqlJoinField> joinFields() {

		int count = joinFieldSize();
		List<ExternalNoSqlJoinField> joinFields = new ArrayList<ExternalNoSqlJoinField>(count);

		for (int index = count; --index >= 0;) {
			ExternalNoSqlJoinField joinField = buildJoinField(index);
			joinFields.add(0, joinField);
		}

		return new ListListIterable<ExternalNoSqlJoinField>(joinFields);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinFieldSize() {
		return getChildrenSize(NoSqlJoinField.JOIN_FIELD);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinField(int index) {
		removeChild(NoSqlJoinField.JOIN_FIELD, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setId(Boolean id) {
		setAttribute(ID, id);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappedByMappingName(String name) {
		setAttribute(MAPPED_BY, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapsId(String mapsId) {
		setAttribute(MAPS_ID, mapsId);
	}
}