/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.w3c.dom.Element;

/**
 * The external form of an entity column.
 *
 * @see Mapping
 * @see AttributeOverride
 * @see OptimisticLocking
 * @see WriteTransfomer
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class EntityColumn extends AbstractColumn
                         implements ExternalEntityColumn {

	/**
	 * The name of the element represented by this external form.
	 */
	private String elementName;

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The attribute name used to store and retrieve the column property.
	 */
	static final String COLUMN = "column";

	/**
	 * The attribute name used to store and retrieve the length property.
	 */
	static final String LENGTH = "length";

	/**
	 * The attribute name used to store and retrieve the precision property.
	 */
	static final String PRECISION = "precision";

	/**
	 * The attribute name used to store and retrieve the scale property.
	 */
	static final String SCALE = "scale";

	/**
	 * Creates a new <code>EntityColumn</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	EntityColumn(AttributeOverride parent, int index) {
		super(parent);
		this.index = index;
		this.elementName = COLUMN;
	}

	/**
	 * Creates a new <code>EntityColumn</code>.
	 *
	 * @param parent The parent of this external form
	 */
	EntityColumn(Mapping parent) {
		this(parent, COLUMN);
	}

	/**
	 * Creates a new <code>EntityColumn</code>.
	 *
	 * @param parent The parent of this external form
	 * @param elementName The name of the element represented by this external form
	 */
	EntityColumn(Mapping parent, String elementName) {
		super(parent);
		this.index = -1;
		this.elementName = elementName;
	}

	/**
	 * Creates a new <code>EntityColumn</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 * @param elementName The name of the element represented by this external form
	 */
	EntityColumn(OptimisticLocking parent, String elementName, int index) {
		super(parent);
		this.index = index;
		this.elementName = elementName;
	}

	/**
	 * Creates a new <code>EntityColumn</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	EntityColumn(PrimaryKey parent, int index) {
		super(parent);
		this.index = index;
		this.elementName = COLUMN;
	}

	/**
	 * Creates a new <code>EntityColumn</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	EntityColumn(WriteTransfomer parent, int index) {
		super(parent);
		this.index = index;
		this.elementName = COLUMN;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {

		if (index == -1) {
			return super.addSelf(elementName, elementNamesOrder);
		}

		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return elementName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getLength() {
		return getIntegerAttribute(LENGTH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getPrecision() {
		return getIntegerAttribute(PRECISION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getScale() {
		return getIntegerAttribute(SCALE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLength(Integer length) {

		setAttribute(LENGTH, length);

		if (shouldRemoveEmptyElement() && !hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrecision(Integer precision) {

		setAttribute(PRECISION, precision);

		if (shouldRemoveEmptyElement() && !hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setScale(Integer scale) {

		setAttribute(SCALE, scale);

		if (shouldRemoveEmptyElement() && !hasAnyChildren() && !hasAttributes()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	boolean shouldRemoveEmptyElement() {
		return true;
	}
}