/*******************************************************************************
 * Copyright (c) 2009, 2018 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Lars Vogel <Lars.Vogel@vogella.com> - Bug 472654
 ******************************************************************************/
package org.eclipse.e4.ui.workbench;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.model.application.MApplicationElement;
import org.eclipse.e4.ui.model.application.ui.MUIElement;
import org.eclipse.e4.ui.model.internal.ModelUtils;
import org.osgi.service.event.Event;

/**
 * E4 UI events and event topic definitions.
 *
 * This file contains generated and hand crafted event topic constants. There are also hand crafted
 * utility methods for constructing topic strings and publishing events.
 *
 * When the UI model changes org.eclipse.e4.ui.internal.workbench.swt.GenTopic should be run as an
 * Eclipse application and the console results should be pasted into this file replacing the code
 * below the "Place Generated Code Here" comment
 *
 * @noextend This class is not intended to be subclassed by clients.
 * @noinstantiate This class is not intended to be instantiated by clients.
 * @since 1.0
 */
public class UIEvents {

	/**
	 * Topic separator character
	 */
	public static final String TOPIC_SEP = "/"; //$NON-NLS-1$

	/**
	 * Wild card character for matching all sub topics
	 */
	public static final String ALL_SUB_TOPICS = "*"; //$NON-NLS-1$

	/**
	 * Base name of all E4 UI events
	 */
	public static final String UITopicBase = "org/eclipse/e4/ui"; //$NON-NLS-1$

	/**
	 * Name element for all E4 UI model events (these are generated by GenTopic)
	 */
	public static final String UIModelTopicBase = UITopicBase + "/model"; //$NON-NLS-1$

	/**
	 * Name element for E4 Renderer events
	 *
	 * @since 1.1
	 */
	public static final String UIRendererTopicBase = UITopicBase + "/renderer"; //$NON-NLS-1$

	/**
	 * Event sent when a enabled update is requested for tool-items.
	 * <p>
	 * As a value you may pass in to the {@link IEventBroker#DATA}:
	 * </p>
	 * <ul>
	 * <li>the special value {@link #ALL_ELEMENT_ID} to request changes of all
	 * applicable elements (the default)</li>
	 * <li>the {@link MApplicationElement#getElementId()} of the element to
	 * check</li>
	 * <li>instance of {@link Selector}</li>
	 * </ul>
	 *
	 * @since 1.1
	 */
	public static final String REQUEST_ENABLEMENT_UPDATE_TOPIC = UIRendererTopicBase
			+ "/requestEnablementUpdate"; //$NON-NLS-1$

	/**
	 * Special id passed to force all elements to be checked
	 *
	 * @since 1.1
	 */
	public static final String ALL_ELEMENT_ID = "ALL"; //$NON-NLS-1$

	/**
	 * E4 UI Event Types. Add appropriate utility is &lt;Test&gt; method below if
	 * new types added
	 *
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface EventTypes {
		/**
		 * Creation event
		 */
		String CREATE = "CREATE"; //$NON-NLS-1$
		/**
		 * Set event
		 */
		String SET = "SET"; //$NON-NLS-1$
		/**
		 * Add event: value added is {@link EventTags#NEW_VALUE}.
		 *
		 * @see UIEvents#isADD(Event)
		 */
		String ADD = "ADD"; //$NON-NLS-1$
		/**
		 * Add many items: values added are {@link EventTags#NEW_VALUE}
		 *
		 * @see UIEvents#isADD(Event)
		 */
		String ADD_MANY = "ADD_MANY";//$NON-NLS-1$
		/**
		 * Remove event: value removed is {@link EventTags#OLD_VALUE}
		 *
		 * @see UIEvents#isREMOVE(Event)
		 */
		String REMOVE = "REMOVE"; //$NON-NLS-1$
		/**
		 * Remove many event: the values removed are the {@link EventTags#OLD_VALUE} (a collection).
		 * The former positions of the removed values are provided as an integer array in
		 * {@link EventTags#POSITION}.
		 *
		 * @see UIEvents#isREMOVE(Event)
		 */
		String REMOVE_MANY = "REMOVE_MANY"; //$NON-NLS-1$
		/**
		 * Value moved: the value moved is the {@link EventTags#NEW_VALUE}, the old position is
		 * {@link EventTags#OLD_VALUE}, and the new position in {@link EventTags#POSITION}.
		 */
		String MOVE = "MOVE"; //$NON-NLS-1$
	}

	/**
	 * @param event
	 *            An OSGI event representing a UIEvent
	 * @return true if it is an add event (i.e., {@link EventTypes#ADD} or
	 *         {@link EventTypes#ADD_MANY}), or false otherwise.
	 * @see UIEvents.EventTags#NEW_VALUE
	 * @see #asIterable(Event, String)
	 */
	public static boolean isADD(Event event) {
		Object type = event.getProperty(UIEvents.EventTags.TYPE);
		return UIEvents.EventTypes.ADD.equals(type) || UIEvents.EventTypes.ADD_MANY.equals(type);
	}

	/**
	 * @param event
	 *            An OSGI event representing a UIEvent
	 * @return true if it is a remove event (i.e., {@link EventTypes#REMOVE} or
	 *         {@link EventTypes#REMOVE_MANY}), or false otherwise.
	 * @see UIEvents.EventTags#OLD_VALUE
	 * @see #asIterable(Event, String)
	 */
	public static boolean isREMOVE(Event event) {
		Object type = event.getProperty(UIEvents.EventTags.TYPE);
		return UIEvents.EventTypes.REMOVE.equals(type)
				|| UIEvents.EventTypes.REMOVE_MANY.equals(type);
	}

	/**
	 * @param event An OSGI event representing a UIEvent
	 * @return true if it is a move event, false otherwise.
	 * @since 1.10
	 */
	public static boolean isMOVE(Event event) {
		return UIEvents.EventTypes.MOVE.equals(event.getProperty(UIEvents.EventTags.TYPE));
	}

	/**
	 * @param event An OSGI event representing a UIEvent
	 * @return true if it is a set event, false otherwise.
	 */
	public static boolean isSET(Event event) {
		return UIEvents.EventTypes.SET.equals(event.getProperty(UIEvents.EventTags.TYPE));
	}

	/**
	 * @param event
	 *            An OSGI event representing a UIEvent
	 * @return true if it is a create event, false otherwise.
	 */
	public static boolean isCREATE(Event event) {
		return UIEvents.EventTypes.CREATE.equals(event.getProperty(UIEvents.EventTags.TYPE));
	}

	/**
	 * Return true if the specified property contains {@code o}. Intended as a helper function for
	 * {@link EventTypes#ADD}, {@link EventTypes#ADD_MANY}, {@link EventTypes#REMOVE}, and
	 * {@link EventTypes#REMOVE_MANY}. If the property is not a container (e.g., a collection or
	 * array), then return true then if {@code container} is equal to {@code o}.
	 *
	 * @param event
	 *            the event
	 * @param propertyName
	 *            the property name
	 * @param o
	 *            the object to check for containment
	 * @return true if the property value contains {@code o} or is equal to {@code o}
	 */
	public static boolean contains(Event event, String propertyName, Object o) {
		Object container = event.getProperty(propertyName);
		if (container == null) {
			return false;
		} else if (container instanceof Collection<?> && ((Collection<?>) container).contains(o)) {
			return true;
		} else if (container instanceof Object[]) {
			for (Object element : (Object[]) container) {
				if (o.equals(element)) {
					return true;
				}
			}
		}
		return o.equals(container);
	}

	/**
	 * Return the provided event property as an iterable. If already a collection, return the
	 * collection.
	 *
	 * @param event
	 *            the event object
	 * @param propertyName
	 *            the name of the property
	 * @return an iterable collection over the property elements
	 */
	public static Iterable<?> asIterable(Event event, String propertyName) {
		Object o = event.getProperty(propertyName);
		return o instanceof Collection<?> ? (Collection<?>) o : Collections.singleton(o);
	}

	/**
	 * E4 UI Event argument attribute keys. These are used as keys for the event
	 * arguments map. Each event may have none, some, or all arguments set.
	 *
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface EventTags {
		/**
		 * The element that caused the event to be published
		 */
		String ELEMENT = "ChangedElement"; //$NON-NLS-1$
		/**
		 * The widget that generated the event
		 */
		String WIDGET = "Widget"; //$NON-NLS-1$
		/**
		 * The event type @see UIEvents.EventTypes
		 */
		String TYPE = "EventType"; //$NON-NLS-1$
		/**
		 * The attribute name
		 */
		String ATTNAME = "AttName"; //$NON-NLS-1$
		/**
		 * The old value
		 */
		String OLD_VALUE = "OldValue"; //$NON-NLS-1$
		/**
		 * The new value
		 */
		String NEW_VALUE = "NewValue"; //$NON-NLS-1$
		/**
		 * The position (if applicable) of the change within the list.
		 */
		String POSITION = "Position"; //$NON-NLS-1$
	}

	/**
	 * E4 UI life cycle events. These events are explicitly published by specific
	 * operations. They are not directly generated by UI model changes.
	 *
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface UILifeCycle {
		/**
		 * Base name for all UI life cycle events
		 */
		String TOPIC = UITopicBase + "/LifeCycle"; //$NON-NLS-1$

		/**
		 * Sent when a UIElement is brought to top
		 */
		String BRINGTOTOP = TOPIC + TOPIC_SEP + "bringToTop"; //$NON-NLS-1$

		/**
		 * Sent when an MPart is activated
		 */
		String ACTIVATE = TOPIC + TOPIC_SEP + "activate"; //$NON-NLS-1$

		/**
		 * Sent when a perspective is saved
		 */
		String PERSPECTIVE_SAVED = TOPIC + TOPIC_SEP + "perpSaved"; //$NON-NLS-1$

		/**
		 * Sent when a perspective is opened
		 */
		String PERSPECTIVE_OPENED = TOPIC + TOPIC_SEP + "perspOpened"; //$NON-NLS-1$

		/**
		 * Sent when a perspective is reset
		 *
		 * @since 1.2
		 */
		String PERSPECTIVE_RESET = TOPIC + TOPIC_SEP + "perspReset"; //$NON-NLS-1$

		/**
		 * Sent when a perspective is switched
		 *
		 * @since 1.14
		 */
		String PERSPECTIVE_SWITCHED = TOPIC + TOPIC_SEP + "perspSwitched"; //$NON-NLS-1$

		/**
		 * Sent when application startup is complete
		 */
		String APP_STARTUP_COMPLETE = TOPIC + TOPIC_SEP + "appStartupComplete"; //$NON-NLS-1$
		/**
		 * Sent when application shutdown is starting
		 *
		 * @since 1.1
		 */
		String APP_SHUTDOWN_STARTED = TOPIC + TOPIC_SEP + "appShutdownStarted"; //$NON-NLS-1$

		/**
		 * Sent when the theme is changed
		 *
		 * @since 1.1
		 */
		String THEME_CHANGED = TOPIC + TOPIC_SEP + "themeChanged"; //$NON-NLS-1$

		/**
		 * Sent when the theme definition is changed
		 *
		 * @since 1.1
		 */
		String THEME_DEFINITION_CHANGED = TOPIC + TOPIC_SEP
				+ "themeDefinitionChanged"; //$NON-NLS-1$
	}

	/**
	 * Publish the topic to the changedElements global event bus. The changedElement is added the
	 * the EventTags.ELEMENT tag.
	 *
	 * @param topic
	 *            to broadcast
	 * @param changedElement
	 *            the element that changed
	 * @return true if the event is published correctly, false otherwise
	 */
	public static boolean publishEvent(String topic, MUIElement changedElement) {
		if (topic == null || topic.isEmpty() || changedElement == null)
			return false;

		Map<String, Object> argMap = new HashMap<>(1);
		argMap.put(EventTags.ELEMENT, changedElement);
		return publishEvent(topic, argMap);
	}

	/**
	 * Publish the topic with the provided arguments to the global event bus. argMap MUST contain an
	 * EventTags.ELEMENT argument that is an MUIElement. the contained MUIElement will be used to
	 * determine the event bus to publish to.
	 *
	 * @param topic
	 *            to broadcast
	 * @param argMap
	 *            arguments map with a minimum of a changedElement
	 * @return true if the event is published correctly, false otherwise
	 */
	public static boolean publishEvent(String topic, Map<String, Object> argMap) {
		if (topic == null || topic.isEmpty() || argMap == null)
			return false;

		Object uiElement = argMap.get(EventTags.ELEMENT);
		if (!(uiElement instanceof MUIElement)) {
			return false;
		}

		IEclipseContext context = uiElement instanceof MApplication ? ((MApplication) uiElement)
				.getContext() : ModelUtils.getContainingContext((MUIElement) uiElement);
		if (context == null)
			return false;

		IEventBroker eventBroker = context.get(IEventBroker.class);
		if (eventBroker == null)
			return false;

		return eventBroker.send(topic, argMap);
	}

	@Deprecated
	/**
	 * @deprecated Subscribe to an all attribute events on a topic using the TOPIC_ALL constant directly
	 */
	public static String buildTopic(String topic) {
		return topic + TOPIC_SEP + ALL_SUB_TOPICS;
	}

	@Deprecated
	/**
	 * @deprecated Subscribe to an attribute event by using the TOPIC_<attribute> constant directly
	 */
	public static String buildTopic(String topic, String attrName) {
		return topic + TOPIC_SEP + attrName + TOPIC_SEP + ALL_SUB_TOPICS;
	}

	@Deprecated
	/**
	 * @deprecated Subscribing to a particular event type on a topic attribute is not longer supported
	 */
	public static String buildTopic(String topic, String attrName, String eventType) {
		return topic + TOPIC_SEP + attrName + TOPIC_SEP + eventType;
	}

	/*************************************************************************************
	 * GENERATED CODE!!
	 *
	 * NOTE: *All* non-generated code must be above this comment.
	 *
	 * Replace the generated code below this comment with the output of GenTopic.
	 *
	 *************************************************************************************/

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface BindingContext {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/BindingContext"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/BindingContext/*"; //$NON-NLS-1$
		String TOPIC_CHILDREN = "org/eclipse/e4/ui/model/commands/BindingContext/children/*"; //$NON-NLS-1$
		String TOPIC_DESCRIPTION = "org/eclipse/e4/ui/model/commands/BindingContext/description/*"; //$NON-NLS-1$
		String TOPIC_NAME = "org/eclipse/e4/ui/model/commands/BindingContext/name/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CHILDREN = "children"; //$NON-NLS-1$
		String DESCRIPTION = "description"; //$NON-NLS-1$
		String NAME = "name"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface BindingTable {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/BindingTable"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/BindingTable/*"; //$NON-NLS-1$
		String TOPIC_BINDINGCONTEXT = "org/eclipse/e4/ui/model/commands/BindingTable/bindingContext/*"; //$NON-NLS-1$
		String TOPIC_BINDINGS = "org/eclipse/e4/ui/model/commands/BindingTable/bindings/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String BINDINGCONTEXT = "bindingContext"; //$NON-NLS-1$
		String BINDINGS = "bindings"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface BindingTableContainer {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/BindingTableContainer"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/BindingTableContainer/*"; //$NON-NLS-1$
		String TOPIC_BINDINGTABLES = "org/eclipse/e4/ui/model/commands/BindingTableContainer/bindingTables/*"; //$NON-NLS-1$
		String TOPIC_ROOTCONTEXT = "org/eclipse/e4/ui/model/commands/BindingTableContainer/rootContext/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String BINDINGTABLES = "bindingTables"; //$NON-NLS-1$
		String ROOTCONTEXT = "rootContext"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Bindings {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/Bindings"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/Bindings/*"; //$NON-NLS-1$
		String TOPIC_BINDINGCONTEXTS = "org/eclipse/e4/ui/model/commands/Bindings/bindingContexts/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String BINDINGCONTEXTS = "bindingContexts"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Category {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/Category"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/Category/*"; //$NON-NLS-1$
		String TOPIC_DESCRIPTION = "org/eclipse/e4/ui/model/commands/Category/description/*"; //$NON-NLS-1$
		String TOPIC_NAME = "org/eclipse/e4/ui/model/commands/Category/name/*"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String TOPIC_LOCALIZED_NAME = "org/eclipse/e4/ui/model/commands/Category/localizedName/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String DESCRIPTION = "description"; //$NON-NLS-1$
		String NAME = "name"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String LOCALIZED_NAME = "localizedName"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Command {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/Command"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/Command/*"; //$NON-NLS-1$
		String TOPIC_CATEGORY = "org/eclipse/e4/ui/model/commands/Command/category/*"; //$NON-NLS-1$
		String TOPIC_COMMANDNAME = "org/eclipse/e4/ui/model/commands/Command/commandName/*"; //$NON-NLS-1$
		String TOPIC_DESCRIPTION = "org/eclipse/e4/ui/model/commands/Command/description/*"; //$NON-NLS-1$
		String TOPIC_PARAMETERS = "org/eclipse/e4/ui/model/commands/Command/parameters/*"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String TOPIC_LOCALIZED_COMMANDNAME = "org/eclipse/e4/ui/model/commands/Command/localizedCommandName/*"; //$NON-NLS-1$
		/**
		 * @since 1.1
		 */
		String TOPIC_LOCALIZED_DESCRIPTION = "org/eclipse/e4/ui/model/commands/Command/localizedDescription/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CATEGORY = "category"; //$NON-NLS-1$
		String COMMANDNAME = "commandName"; //$NON-NLS-1$
		String DESCRIPTION = "description"; //$NON-NLS-1$
		String PARAMETERS = "parameters"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String LOCALIZED_COMMANDNAME = "localizedCommandName"; //$NON-NLS-1$
		/**
		 * @since 1.1
		 */
		String LOCALIZED_DESCRIPTION = "localizedDescription"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface CommandParameter {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/CommandParameter"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/CommandParameter/*"; //$NON-NLS-1$
		String TOPIC_NAME = "org/eclipse/e4/ui/model/commands/CommandParameter/name/*"; //$NON-NLS-1$
		String TOPIC_OPTIONAL = "org/eclipse/e4/ui/model/commands/CommandParameter/optional/*"; //$NON-NLS-1$
		String TOPIC_TYPEID = "org/eclipse/e4/ui/model/commands/CommandParameter/typeId/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String NAME = "name"; //$NON-NLS-1$
		String OPTIONAL = "optional"; //$NON-NLS-1$
		String TYPEID = "typeId"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Handler {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/Handler"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/Handler/*"; //$NON-NLS-1$
		String TOPIC_COMMAND = "org/eclipse/e4/ui/model/commands/Handler/command/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String COMMAND = "command"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface HandlerContainer {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/HandlerContainer"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/HandlerContainer/*"; //$NON-NLS-1$
		String TOPIC_HANDLERS = "org/eclipse/e4/ui/model/commands/HandlerContainer/handlers/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String HANDLERS = "handlers"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface KeyBinding {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/KeyBinding"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/KeyBinding/*"; //$NON-NLS-1$
		String TOPIC_COMMAND = "org/eclipse/e4/ui/model/commands/KeyBinding/command/*"; //$NON-NLS-1$
		String TOPIC_PARAMETERS = "org/eclipse/e4/ui/model/commands/KeyBinding/parameters/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String COMMAND = "command"; //$NON-NLS-1$
		String PARAMETERS = "parameters"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface KeySequence {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/KeySequence"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/KeySequence/*"; //$NON-NLS-1$
		String TOPIC_KEYSEQUENCE = "org/eclipse/e4/ui/model/commands/KeySequence/keySequence/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String KEYSEQUENCE = "keySequence"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Parameter {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/commands/Parameter"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/commands/Parameter/*"; //$NON-NLS-1$
		String TOPIC_NAME = "org/eclipse/e4/ui/model/commands/Parameter/name/*"; //$NON-NLS-1$
		String TOPIC_VALUE = "org/eclipse/e4/ui/model/commands/Parameter/value/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String NAME = "name"; //$NON-NLS-1$
		String VALUE = "value"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface PartDescriptor {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/basic/PartDescriptor"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/basic/PartDescriptor/*"; //$NON-NLS-1$
		String TOPIC_ALLOWMULTIPLE = "org/eclipse/e4/ui/model/basic/PartDescriptor/allowMultiple/*"; //$NON-NLS-1$
		String TOPIC_CATEGORY = "org/eclipse/e4/ui/model/basic/PartDescriptor/category/*"; //$NON-NLS-1$
		String TOPIC_CLOSEABLE = "org/eclipse/e4/ui/model/basic/PartDescriptor/closeable/*"; //$NON-NLS-1$
		String TOPIC_CONTRIBUTIONURI = "org/eclipse/e4/ui/model/basic/PartDescriptor/contributionURI/*"; //$NON-NLS-1$
		String TOPIC_DESCRIPTION = "org/eclipse/e4/ui/model/basic/PartDescriptor/description/*"; //$NON-NLS-1$
		/**
		 * @deprecated
		 * @noreference This field is not intended to be referenced by clients.
		 * @see <a href="https://bugs.eclipse.org/bugs/show_bug.cgi?id=530887">Bug
		 *      530887</a>
		 */
		@Deprecated
		String TOPIC_DIRTYABLE = "org/eclipse/e4/ui/model/basic/PartDescriptor/dirtyable/*"; //$NON-NLS-1$
		String TOPIC_MENUS = "org/eclipse/e4/ui/model/basic/PartDescriptor/menus/*"; //$NON-NLS-1$
		String TOPIC_TOOLBAR = "org/eclipse/e4/ui/model/basic/PartDescriptor/toolbar/*"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String TOPIC_LOCALIZED_DESCRIPTION = "org/eclipse/e4/ui/model/basic/PartDescriptor/localizedDescription/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String ALLOWMULTIPLE = "allowMultiple"; //$NON-NLS-1$
		String CATEGORY = "category"; //$NON-NLS-1$
		String CLOSEABLE = "closeable"; //$NON-NLS-1$
		String CONTRIBUTIONURI = "contributionURI"; //$NON-NLS-1$
		String DESCRIPTION = "description"; //$NON-NLS-1$
		/**
		 * @deprecated
		 * @noreference This field is not intended to be referenced by clients.
		 * @see <a href="https://bugs.eclipse.org/bugs/show_bug.cgi?id=530887">Bug
		 *      530887</a>
		 */
		@Deprecated
		String DIRTYABLE = "dirtyable"; //$NON-NLS-1$
		String MENUS = "menus"; //$NON-NLS-1$
		String TOOLBAR = "toolbar"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String LOCALIZED_DESCRIPTION = "localizedDescription"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface PartDescriptorContainer {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/basic/PartDescriptorContainer"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/basic/PartDescriptorContainer/*"; //$NON-NLS-1$
		String TOPIC_DESCRIPTORS = "org/eclipse/e4/ui/model/basic/PartDescriptorContainer/descriptors/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String DESCRIPTORS = "descriptors"; //$NON-NLS-1$
	}

	/**
	 * @since 1.1
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Application {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/application/Application"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/application/Application/*"; //$NON-NLS-1$
		String TOPIC_ADDONS = "org/eclipse/e4/ui/model/application/Application/addons/*"; //$NON-NLS-1$
		String TOPIC_CATEGORIES = "org/eclipse/e4/ui/model/application/Application/categories/*"; //$NON-NLS-1$
		String TOPIC_COMMANDS = "org/eclipse/e4/ui/model/application/Application/commands/*"; //$NON-NLS-1$
		/**
		 * @since 1.1
		 * @deprecated
		 * @noreference This field is not intended to be referenced by clients.
		 * @see <a href="https://bugs.eclipse.org/bugs/show_bug.cgi?id=531054">Bug
		 *      531054 </a>
		 */
		@Deprecated
		String TOPIC_DIALOGS = "org/eclipse/e4/ui/model/application/Application/dialogs/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String ADDONS = "addons"; //$NON-NLS-1$
		String CATEGORIES = "categories"; //$NON-NLS-1$
		String COMMANDS = "commands"; //$NON-NLS-1$
		/**
		 * @since 1.1
		 * @deprecated
		 * @noreference This field is not intended to be referenced by clients.
		 * @see <a href="https://bugs.eclipse.org/bugs/show_bug.cgi?id=531054">Bug
		 *      531054 </a>
		 */
		@Deprecated
		String DIALOGS = "dialogs"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface ApplicationElement {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/application/ApplicationElement"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/application/ApplicationElement/*"; //$NON-NLS-1$
		String TOPIC_CONTRIBUTORURI = "org/eclipse/e4/ui/model/application/ApplicationElement/contributorURI/*"; //$NON-NLS-1$
		String TOPIC_ELEMENTID = "org/eclipse/e4/ui/model/application/ApplicationElement/elementId/*"; //$NON-NLS-1$
		String TOPIC_PERSISTEDSTATE = "org/eclipse/e4/ui/model/application/ApplicationElement/persistedState/*"; //$NON-NLS-1$
		String TOPIC_TAGS = "org/eclipse/e4/ui/model/application/ApplicationElement/tags/*"; //$NON-NLS-1$
		String TOPIC_TRANSIENTDATA = "org/eclipse/e4/ui/model/application/ApplicationElement/transientData/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CONTRIBUTORURI = "contributorURI"; //$NON-NLS-1$
		String ELEMENTID = "elementId"; //$NON-NLS-1$
		String PERSISTEDSTATE = "persistedState"; //$NON-NLS-1$
		String TAGS = "tags"; //$NON-NLS-1$
		String TRANSIENTDATA = "transientData"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Contribution {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/application/Contribution"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/application/Contribution/*"; //$NON-NLS-1$
		String TOPIC_CONTRIBUTIONURI = "org/eclipse/e4/ui/model/application/Contribution/contributionURI/*"; //$NON-NLS-1$
		String TOPIC_OBJECT = "org/eclipse/e4/ui/model/application/Contribution/object/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CONTRIBUTIONURI = "contributionURI"; //$NON-NLS-1$
		String OBJECT = "object"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface StringToObjectMap {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/application/StringToObjectMap"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/application/StringToObjectMap/*"; //$NON-NLS-1$
		String TOPIC_KEY = "org/eclipse/e4/ui/model/application/StringToObjectMap/key/*"; //$NON-NLS-1$
		String TOPIC_VALUE = "org/eclipse/e4/ui/model/application/StringToObjectMap/value/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String KEY = "key"; //$NON-NLS-1$
		String VALUE = "value"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface StringToStringMap {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/application/StringToStringMap"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/application/StringToStringMap/*"; //$NON-NLS-1$
		String TOPIC_KEY = "org/eclipse/e4/ui/model/application/StringToStringMap/key/*"; //$NON-NLS-1$
		String TOPIC_VALUE = "org/eclipse/e4/ui/model/application/StringToStringMap/value/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String KEY = "key"; //$NON-NLS-1$
		String VALUE = "value"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Perspective {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/advanced/Perspective"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/advanced/Perspective/*"; //$NON-NLS-1$
		String TOPIC_WINDOWS = "org/eclipse/e4/ui/model/advanced/Perspective/windows/*"; //$NON-NLS-1$
		/**
		 * @since 1.5
		 */
		String TOPIC_TRIMBARS = "org/eclipse/e4/ui/model/advanced/Perspective/trimBars/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String WINDOWS = "windows"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Placeholder {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/advanced/Placeholder"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/advanced/Placeholder/*"; //$NON-NLS-1$
		String TOPIC_CLOSEABLE = "org/eclipse/e4/ui/model/advanced/Placeholder/closeable/*"; //$NON-NLS-1$
		String TOPIC_REF = "org/eclipse/e4/ui/model/advanced/Placeholder/ref/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CLOSEABLE = "closeable"; //$NON-NLS-1$
		String REF = "ref"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Part {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/basic/Part"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/basic/Part/*"; //$NON-NLS-1$
		String TOPIC_CLOSEABLE = "org/eclipse/e4/ui/model/basic/Part/closeable/*"; //$NON-NLS-1$
		String TOPIC_DESCRIPTION = "org/eclipse/e4/ui/model/basic/Part/description/*"; //$NON-NLS-1$
		String TOPIC_MENUS = "org/eclipse/e4/ui/model/basic/Part/menus/*"; //$NON-NLS-1$
		String TOPIC_TOOLBAR = "org/eclipse/e4/ui/model/basic/Part/toolbar/*"; //$NON-NLS-1$
		/**
		 * @since 1.5
		 */
		String TOPIC_TRIMBARS = "org/eclipse/e4/ui/model/basic/Part/trimBars/*"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String TOPIC_LOCALIZED_DESCRIPTION = "org/eclipse/e4/ui/model/basic/Part/localizedDescription/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CLOSEABLE = "closeable"; //$NON-NLS-1$
		String DESCRIPTION = "description"; //$NON-NLS-1$
		String MENUS = "menus"; //$NON-NLS-1$
		String TOOLBAR = "toolbar"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String LOCALIZED_DESCRIPTION = "localizedDescription"; //$NON-NLS-1$
	}

	/**
	 * @since 1.1
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface TrimBar {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/basic/TrimBar"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/basic/TrimBar/*"; //$NON-NLS-1$
		String TOPIC_PENDINGCLEANUP = "org/eclipse/e4/ui/model/basic/TrimBar/pendingCleanup/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String PENDINGCLEANUP = "pendingCleanup"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface TrimmedWindow {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/basic/TrimmedWindow"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/basic/TrimmedWindow/*"; //$NON-NLS-1$
		String TOPIC_TRIMBARS = "org/eclipse/e4/ui/model/basic/TrimmedWindow/trimBars/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String TRIMBARS = "trimBars"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Window {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/basic/Window"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/basic/Window/*"; //$NON-NLS-1$
		String TOPIC_HEIGHT = "org/eclipse/e4/ui/model/basic/Window/height/*"; //$NON-NLS-1$
		String TOPIC_MAINMENU = "org/eclipse/e4/ui/model/basic/Window/mainMenu/*"; //$NON-NLS-1$
		String TOPIC_SHAREDELEMENTS = "org/eclipse/e4/ui/model/basic/Window/sharedElements/*"; //$NON-NLS-1$
		String TOPIC_WIDTH = "org/eclipse/e4/ui/model/basic/Window/width/*"; //$NON-NLS-1$
		String TOPIC_WINDOWS = "org/eclipse/e4/ui/model/basic/Window/windows/*"; //$NON-NLS-1$
		String TOPIC_X = "org/eclipse/e4/ui/model/basic/Window/x/*"; //$NON-NLS-1$
		String TOPIC_Y = "org/eclipse/e4/ui/model/basic/Window/y/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String HEIGHT = "height"; //$NON-NLS-1$
		String MAINMENU = "mainMenu"; //$NON-NLS-1$
		String SHAREDELEMENTS = "sharedElements"; //$NON-NLS-1$
		String WIDTH = "width"; //$NON-NLS-1$
		String WINDOWS = "windows"; //$NON-NLS-1$
		String X = "x"; //$NON-NLS-1$
		String Y = "y"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Context {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/Context"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/Context/*"; //$NON-NLS-1$
		String TOPIC_CONTEXT = "org/eclipse/e4/ui/model/ui/Context/context/*"; //$NON-NLS-1$
		String TOPIC_PROPERTIES = "org/eclipse/e4/ui/model/ui/Context/properties/*"; //$NON-NLS-1$
		String TOPIC_VARIABLES = "org/eclipse/e4/ui/model/ui/Context/variables/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CONTEXT = "context"; //$NON-NLS-1$
		String PROPERTIES = "properties"; //$NON-NLS-1$
		String VARIABLES = "variables"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface CoreExpression {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/CoreExpression"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/CoreExpression/*"; //$NON-NLS-1$
		String TOPIC_COREEXPRESSION = "org/eclipse/e4/ui/model/ui/CoreExpression/coreExpression/*"; //$NON-NLS-1$
		String TOPIC_COREEXPRESSIONID = "org/eclipse/e4/ui/model/ui/CoreExpression/coreExpressionId/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String COREEXPRESSION = "coreExpression"; //$NON-NLS-1$
		String COREEXPRESSIONID = "coreExpressionId"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Dirtyable {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/Dirtyable"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/Dirtyable/*"; //$NON-NLS-1$
		String TOPIC_DIRTY = "org/eclipse/e4/ui/model/ui/Dirtyable/dirty/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String DIRTY = "dirty"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface ElementContainer {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/ElementContainer"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/ElementContainer/*"; //$NON-NLS-1$
		String TOPIC_CHILDREN = "org/eclipse/e4/ui/model/ui/ElementContainer/children/*"; //$NON-NLS-1$
		String TOPIC_SELECTEDELEMENT = "org/eclipse/e4/ui/model/ui/ElementContainer/selectedElement/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CHILDREN = "children"; //$NON-NLS-1$
		String SELECTEDELEMENT = "selectedElement"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface GenericTile {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/GenericTile"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/GenericTile/*"; //$NON-NLS-1$
		String TOPIC_HORIZONTAL = "org/eclipse/e4/ui/model/ui/GenericTile/horizontal/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String HORIZONTAL = "horizontal"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface GenericTrimContainer {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/GenericTrimContainer"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/GenericTrimContainer/*"; //$NON-NLS-1$
		String TOPIC_SIDE = "org/eclipse/e4/ui/model/ui/GenericTrimContainer/side/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String SIDE = "side"; //$NON-NLS-1$
	}

	/**
	 * @since 1.5
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface ImperativeExpression {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/ImperativeExpression"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/ImperativeExpression/*"; //$NON-NLS-1$
		String TOPIC_TRACKING = "org/eclipse/e4/ui/model/ui/ImperativeExpression/tracking/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String TRACKING = "tracking"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface SnippetContainer {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/SnippetContainer"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/SnippetContainer/*"; //$NON-NLS-1$
		String TOPIC_SNIPPETS = "org/eclipse/e4/ui/model/ui/SnippetContainer/snippets/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String SNIPPETS = "snippets"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface UIElement {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/UIElement"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/UIElement/*"; //$NON-NLS-1$
		String TOPIC_ACCESSIBILITYPHRASE = "org/eclipse/e4/ui/model/ui/UIElement/accessibilityPhrase/*"; //$NON-NLS-1$
		String TOPIC_CONTAINERDATA = "org/eclipse/e4/ui/model/ui/UIElement/containerData/*"; //$NON-NLS-1$
		String TOPIC_CURSHAREDREF = "org/eclipse/e4/ui/model/ui/UIElement/curSharedRef/*"; //$NON-NLS-1$
		String TOPIC_ONTOP = "org/eclipse/e4/ui/model/ui/UIElement/onTop/*"; //$NON-NLS-1$
		String TOPIC_PARENT = "org/eclipse/e4/ui/model/ui/UIElement/parent/*"; //$NON-NLS-1$
		String TOPIC_RENDERER = "org/eclipse/e4/ui/model/ui/UIElement/renderer/*"; //$NON-NLS-1$
		String TOPIC_TOBERENDERED = "org/eclipse/e4/ui/model/ui/UIElement/toBeRendered/*"; //$NON-NLS-1$
		String TOPIC_VISIBLE = "org/eclipse/e4/ui/model/ui/UIElement/visible/*"; //$NON-NLS-1$
		String TOPIC_VISIBLEWHEN = "org/eclipse/e4/ui/model/ui/UIElement/visibleWhen/*"; //$NON-NLS-1$
		String TOPIC_WIDGET = "org/eclipse/e4/ui/model/ui/UIElement/widget/*"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String TOPIC_LOCALIZED_ACCESSIBILITYPHRASE = "org/eclipse/e4/ui/model/ui/UIElement/localizedAccessibilityPhrase/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String ACCESSIBILITYPHRASE = "accessibilityPhrase"; //$NON-NLS-1$
		String CONTAINERDATA = "containerData"; //$NON-NLS-1$
		String CURSHAREDREF = "curSharedRef"; //$NON-NLS-1$
		String ONTOP = "onTop"; //$NON-NLS-1$
		String PARENT = "parent"; //$NON-NLS-1$
		String RENDERER = "renderer"; //$NON-NLS-1$
		String TOBERENDERED = "toBeRendered"; //$NON-NLS-1$
		String VISIBLE = "visible"; //$NON-NLS-1$
		String VISIBLEWHEN = "visibleWhen"; //$NON-NLS-1$
		String WIDGET = "widget"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String LOCALIZED_ACCESSIBILITYPHRASE = "localizedAccessibilityPhrase"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface UILabel {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/ui/UILabel"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/ui/UILabel/*"; //$NON-NLS-1$
		String TOPIC_ICONURI = "org/eclipse/e4/ui/model/ui/UILabel/iconURI/*"; //$NON-NLS-1$
		String TOPIC_LABEL = "org/eclipse/e4/ui/model/ui/UILabel/label/*"; //$NON-NLS-1$
		String TOPIC_TOOLTIP = "org/eclipse/e4/ui/model/ui/UILabel/tooltip/*"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String TOPIC_LOCALIZED_LABEL = "org/eclipse/e4/ui/model/ui/UILabel/localizedLabel/*"; //$NON-NLS-1$
		/**
		 * @since 1.1
		 */
		String TOPIC_LOCALIZED_TOOLTIP = "org/eclipse/e4/ui/model/ui/UILabel/localizedTooltip/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String ICONURI = "iconURI"; //$NON-NLS-1$
		String LABEL = "label"; //$NON-NLS-1$
		String TOOLTIP = "tooltip"; //$NON-NLS-1$

		/**
		 * @since 1.1
		 */
		String LOCALIZED_LABEL = "localizedLabel"; //$NON-NLS-1$
		/**
		 * @since 1.1
		 */
		String LOCALIZED_TOOLTIP = "localizedTooltip"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface HandledItem {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/HandledItem"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/HandledItem/*"; //$NON-NLS-1$
		String TOPIC_COMMAND = "org/eclipse/e4/ui/model/menu/HandledItem/command/*"; //$NON-NLS-1$
		String TOPIC_PARAMETERS = "org/eclipse/e4/ui/model/menu/HandledItem/parameters/*"; //$NON-NLS-1$
		String TOPIC_WBCOMMAND = "org/eclipse/e4/ui/model/menu/HandledItem/wbCommand/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String COMMAND = "command"; //$NON-NLS-1$
		String PARAMETERS = "parameters"; //$NON-NLS-1$
		String WBCOMMAND = "wbCommand"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Item {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/Item"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/Item/*"; //$NON-NLS-1$
		String TOPIC_ENABLED = "org/eclipse/e4/ui/model/menu/Item/enabled/*"; //$NON-NLS-1$
		String TOPIC_SELECTED = "org/eclipse/e4/ui/model/menu/Item/selected/*"; //$NON-NLS-1$
		String TOPIC_TYPE = "org/eclipse/e4/ui/model/menu/Item/type/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String ENABLED = "enabled"; //$NON-NLS-1$
		String SELECTED = "selected"; //$NON-NLS-1$
		String TYPE = "type"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface Menu {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/Menu"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/Menu/*"; //$NON-NLS-1$
		String TOPIC_ENABLED = "org/eclipse/e4/ui/model/menu/Menu/enabled/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String ENABLED = "enabled"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface MenuContribution {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/MenuContribution"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/MenuContribution/*"; //$NON-NLS-1$
		String TOPIC_PARENTID = "org/eclipse/e4/ui/model/menu/MenuContribution/parentId/*"; //$NON-NLS-1$
		String TOPIC_POSITIONINPARENT = "org/eclipse/e4/ui/model/menu/MenuContribution/positionInParent/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String PARENTID = "parentId"; //$NON-NLS-1$
		String POSITIONINPARENT = "positionInParent"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface MenuContributions {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/MenuContributions"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/MenuContributions/*"; //$NON-NLS-1$
		String TOPIC_MENUCONTRIBUTIONS = "org/eclipse/e4/ui/model/menu/MenuContributions/menuContributions/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String MENUCONTRIBUTIONS = "menuContributions"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface MenuElement {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/MenuElement"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/MenuElement/*"; //$NON-NLS-1$
		String TOPIC_MNEMONICS = "org/eclipse/e4/ui/model/menu/MenuElement/mnemonics/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String MNEMONICS = "mnemonics"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface OpaqueMenuItem {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/OpaqueMenuItem"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/OpaqueMenuItem/*"; //$NON-NLS-1$
		String TOPIC_OPAQUEITEM = "org/eclipse/e4/ui/model/menu/OpaqueMenuItem/opaqueItem/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String OPAQUEITEM = "opaqueItem"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface OpaqueMenuSeparator {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/OpaqueMenuSeparator"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/OpaqueMenuSeparator/*"; //$NON-NLS-1$
		String TOPIC_OPAQUEITEM = "org/eclipse/e4/ui/model/menu/OpaqueMenuSeparator/opaqueItem/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String OPAQUEITEM = "opaqueItem"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface OpaqueToolItem {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/OpaqueToolItem"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/OpaqueToolItem/*"; //$NON-NLS-1$
		String TOPIC_OPAQUEITEM = "org/eclipse/e4/ui/model/menu/OpaqueToolItem/opaqueItem/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String OPAQUEITEM = "opaqueItem"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface RenderedMenu {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/RenderedMenu"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/RenderedMenu/*"; //$NON-NLS-1$
		String TOPIC_CONTRIBUTIONMANAGER = "org/eclipse/e4/ui/model/menu/RenderedMenu/contributionManager/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CONTRIBUTIONMANAGER = "contributionManager"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface RenderedMenuItem {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/RenderedMenuItem"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/RenderedMenuItem/*"; //$NON-NLS-1$
		String TOPIC_CONTRIBUTIONITEM = "org/eclipse/e4/ui/model/menu/RenderedMenuItem/contributionItem/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CONTRIBUTIONITEM = "contributionItem"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface RenderedToolBar {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/RenderedToolBar"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/RenderedToolBar/*"; //$NON-NLS-1$
		String TOPIC_CONTRIBUTIONMANAGER = "org/eclipse/e4/ui/model/menu/RenderedToolBar/contributionManager/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String CONTRIBUTIONMANAGER = "contributionManager"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface ToolBarContribution {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/ToolBarContribution"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/ToolBarContribution/*"; //$NON-NLS-1$
		String TOPIC_PARENTID = "org/eclipse/e4/ui/model/menu/ToolBarContribution/parentId/*"; //$NON-NLS-1$
		String TOPIC_POSITIONINPARENT = "org/eclipse/e4/ui/model/menu/ToolBarContribution/positionInParent/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String PARENTID = "parentId"; //$NON-NLS-1$
		String POSITIONINPARENT = "positionInParent"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface ToolBarContributions {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/ToolBarContributions"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/ToolBarContributions/*"; //$NON-NLS-1$
		String TOPIC_TOOLBARCONTRIBUTIONS = "org/eclipse/e4/ui/model/menu/ToolBarContributions/toolBarContributions/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String TOOLBARCONTRIBUTIONS = "toolBarContributions"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface ToolItem {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/ToolItem"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/ToolItem/*"; //$NON-NLS-1$
		String TOPIC_MENU = "org/eclipse/e4/ui/model/menu/ToolItem/menu/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String MENU = "menu"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface TrimContribution {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/TrimContribution"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/TrimContribution/*"; //$NON-NLS-1$
		String TOPIC_PARENTID = "org/eclipse/e4/ui/model/menu/TrimContribution/parentId/*"; //$NON-NLS-1$
		String TOPIC_POSITIONINPARENT = "org/eclipse/e4/ui/model/menu/TrimContribution/positionInParent/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String PARENTID = "parentId"; //$NON-NLS-1$
		String POSITIONINPARENT = "positionInParent"; //$NON-NLS-1$
	}

	/**
	 * @noimplement This interface is not intended to be implemented by clients.
	 */
	public interface TrimContributions {

		// Topics that can be subscribed to

		@Deprecated
		String TOPIC = "org/eclipse/e4/ui/model/menu/TrimContributions"; //$NON-NLS-1$

		String TOPIC_ALL = "org/eclipse/e4/ui/model/menu/TrimContributions/*"; //$NON-NLS-1$
		String TOPIC_TRIMCONTRIBUTIONS = "org/eclipse/e4/ui/model/menu/TrimContributions/trimContributions/*"; //$NON-NLS-1$

		// Attributes that can be tested in event handlers
		String TRIMCONTRIBUTIONS = "trimContributions"; //$NON-NLS-1$
	}
}
