package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link Swarm}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableSwarm.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableSwarm implements Swarm {
  private final String id;
  private final Version version;
  private final Date createdAt;
  private final Date updatedAt;
  private final SwarmSpec swarmSpec;
  private final JoinTokens joinTokens;

  private ImmutableSwarm(
      String id,
      Version version,
      Date createdAt,
      Date updatedAt,
      SwarmSpec swarmSpec,
      JoinTokens joinTokens) {
    this.id = id;
    this.version = version;
    this.createdAt = createdAt;
    this.updatedAt = updatedAt;
    this.swarmSpec = swarmSpec;
    this.joinTokens = joinTokens;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("ID")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code version} attribute
   */
  @JsonProperty("Version")
  @Override
  public Version version() {
    return version;
  }

  /**
   * @return The value of the {@code createdAt} attribute
   */
  @JsonProperty("CreatedAt")
  @Override
  public Date createdAt() {
    return createdAt;
  }

  /**
   * @return The value of the {@code updatedAt} attribute
   */
  @JsonProperty("UpdatedAt")
  @Override
  public Date updatedAt() {
    return updatedAt;
  }

  /**
   * @return The value of the {@code swarmSpec} attribute
   */
  @JsonProperty("Spec")
  @Override
  public SwarmSpec swarmSpec() {
    return swarmSpec;
  }

  /**
   * @return The value of the {@code joinTokens} attribute
   */
  @JsonProperty("JoinTokens")
  @Override
  public JoinTokens joinTokens() {
    return joinTokens;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Swarm#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSwarm withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableSwarm(newValue, this.version, this.createdAt, this.updatedAt, this.swarmSpec, this.joinTokens);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Swarm#version() version} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for version
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSwarm withVersion(Version value) {
    if (this.version == value) return this;
    Version newValue = Objects.requireNonNull(value, "version");
    return new ImmutableSwarm(this.id, newValue, this.createdAt, this.updatedAt, this.swarmSpec, this.joinTokens);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Swarm#createdAt() createdAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for createdAt
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSwarm withCreatedAt(Date value) {
    if (this.createdAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "createdAt");
    return new ImmutableSwarm(this.id, this.version, newValue, this.updatedAt, this.swarmSpec, this.joinTokens);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Swarm#updatedAt() updatedAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for updatedAt
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSwarm withUpdatedAt(Date value) {
    if (this.updatedAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "updatedAt");
    return new ImmutableSwarm(this.id, this.version, this.createdAt, newValue, this.swarmSpec, this.joinTokens);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Swarm#swarmSpec() swarmSpec} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for swarmSpec
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSwarm withSwarmSpec(SwarmSpec value) {
    if (this.swarmSpec == value) return this;
    SwarmSpec newValue = Objects.requireNonNull(value, "swarmSpec");
    return new ImmutableSwarm(this.id, this.version, this.createdAt, this.updatedAt, newValue, this.joinTokens);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Swarm#joinTokens() joinTokens} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for joinTokens
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSwarm withJoinTokens(JoinTokens value) {
    if (this.joinTokens == value) return this;
    JoinTokens newValue = Objects.requireNonNull(value, "joinTokens");
    return new ImmutableSwarm(this.id, this.version, this.createdAt, this.updatedAt, this.swarmSpec, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableSwarm} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableSwarm
        && equalTo((ImmutableSwarm) another);
  }

  private boolean equalTo(ImmutableSwarm another) {
    return id.equals(another.id)
        && version.equals(another.version)
        && createdAt.equals(another.createdAt)
        && updatedAt.equals(another.updatedAt)
        && swarmSpec.equals(another.swarmSpec)
        && joinTokens.equals(another.joinTokens);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code version}, {@code createdAt}, {@code updatedAt}, {@code swarmSpec}, {@code joinTokens}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + version.hashCode();
    h += (h << 5) + createdAt.hashCode();
    h += (h << 5) + updatedAt.hashCode();
    h += (h << 5) + swarmSpec.hashCode();
    h += (h << 5) + joinTokens.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Swarm} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Swarm{"
        + "id=" + id
        + ", version=" + version
        + ", createdAt=" + createdAt
        + ", updatedAt=" + updatedAt
        + ", swarmSpec=" + swarmSpec
        + ", joinTokens=" + joinTokens
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Swarm} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Swarm instance
   */
  public static ImmutableSwarm copyOf(Swarm instance) {
    if (instance instanceof ImmutableSwarm) {
      return (ImmutableSwarm) instance;
    }
    return ImmutableSwarm.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableSwarm ImmutableSwarm}.
   * <pre>
   * ImmutableSwarm.builder()
   *    .id(String) // required {@link Swarm#id() id}
   *    .version(org.mandas.docker.client.messages.swarm.Version) // required {@link Swarm#version() version}
   *    .createdAt(Date) // required {@link Swarm#createdAt() createdAt}
   *    .updatedAt(Date) // required {@link Swarm#updatedAt() updatedAt}
   *    .swarmSpec(org.mandas.docker.client.messages.swarm.SwarmSpec) // required {@link Swarm#swarmSpec() swarmSpec}
   *    .joinTokens(org.mandas.docker.client.messages.swarm.JoinTokens) // required {@link Swarm#joinTokens() joinTokens}
   *    .build();
   * </pre>
   * @return A new ImmutableSwarm builder
   */
  public static ImmutableSwarm.Builder builder() {
    return new ImmutableSwarm.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableSwarm ImmutableSwarm}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private static final long INIT_BIT_VERSION = 0x2L;
    private static final long INIT_BIT_CREATED_AT = 0x4L;
    private static final long INIT_BIT_UPDATED_AT = 0x8L;
    private static final long INIT_BIT_SWARM_SPEC = 0x10L;
    private static final long INIT_BIT_JOIN_TOKENS = 0x20L;
    private long initBits = 0x3fL;

    private String id;
    private Version version;
    private Date createdAt;
    private Date updatedAt;
    private SwarmSpec swarmSpec;
    private JoinTokens joinTokens;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Swarm} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Swarm instance) {
      Objects.requireNonNull(instance, "instance");
      id(instance.id());
      version(instance.version());
      createdAt(instance.createdAt());
      updatedAt(instance.updatedAt());
      swarmSpec(instance.swarmSpec());
      joinTokens(instance.joinTokens());
      return this;
    }

    /**
     * Initializes the value for the {@link Swarm#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ID")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link Swarm#version() version} attribute.
     * @param version The value for version 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Version")
    public final Builder version(Version version) {
      this.version = Objects.requireNonNull(version, "version");
      initBits &= ~INIT_BIT_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link Swarm#createdAt() createdAt} attribute.
     * @param createdAt The value for createdAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("CreatedAt")
    public final Builder createdAt(Date createdAt) {
      this.createdAt = Objects.requireNonNull(createdAt, "createdAt");
      initBits &= ~INIT_BIT_CREATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Swarm#updatedAt() updatedAt} attribute.
     * @param updatedAt The value for updatedAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("UpdatedAt")
    public final Builder updatedAt(Date updatedAt) {
      this.updatedAt = Objects.requireNonNull(updatedAt, "updatedAt");
      initBits &= ~INIT_BIT_UPDATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Swarm#swarmSpec() swarmSpec} attribute.
     * @param swarmSpec The value for swarmSpec 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Spec")
    public final Builder swarmSpec(SwarmSpec swarmSpec) {
      this.swarmSpec = Objects.requireNonNull(swarmSpec, "swarmSpec");
      initBits &= ~INIT_BIT_SWARM_SPEC;
      return this;
    }

    /**
     * Initializes the value for the {@link Swarm#joinTokens() joinTokens} attribute.
     * @param joinTokens The value for joinTokens 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("JoinTokens")
    public final Builder joinTokens(JoinTokens joinTokens) {
      this.joinTokens = Objects.requireNonNull(joinTokens, "joinTokens");
      initBits &= ~INIT_BIT_JOIN_TOKENS;
      return this;
    }

    /**
     * Builds a new {@link ImmutableSwarm ImmutableSwarm}.
     * @return An immutable instance of Swarm
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableSwarm build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableSwarm(id, version, createdAt, updatedAt, swarmSpec, joinTokens);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_VERSION) != 0) attributes.add("version");
      if ((initBits & INIT_BIT_CREATED_AT) != 0) attributes.add("createdAt");
      if ((initBits & INIT_BIT_UPDATED_AT) != 0) attributes.add("updatedAt");
      if ((initBits & INIT_BIT_SWARM_SPEC) != 0) attributes.add("swarmSpec");
      if ((initBits & INIT_BIT_JOIN_TOKENS) != 0) attributes.add("joinTokens");
      return "Cannot build Swarm, some of required attributes are not set " + attributes;
    }
  }
}
