"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ESLintAdapter = exports.translateToCodeFixesFromESLintResult = exports.isIntersect = exports.translateToDiagnosticsFromESLintResult = void 0;
const readPkgUp = require("read-pkg-up");
const typescript_1 = __importDefault(require("typescript"));
const eslint_1 = require("eslint");
const errors_1 = require("./errors");
const consts_1 = require("./consts");
function translateToDiagnosticsFromESLintResult(result, sourceFile) {
    return result.map(({ message, severity, ruleId, line, column, endLine, endColumn }) => {
        const messageText = `[${ruleId}] ${message}`;
        const category = severity === 2
            ? typescript_1.default.DiagnosticCategory.Error
            : severity === 1
                ? typescript_1.default.DiagnosticCategory.Warning
                : typescript_1.default.DiagnosticCategory.Suggestion;
        /**
         * ESLint uses 1-started index. On the other hand, TypeScript 0-started index.
         * So we should minus from ESLint result's row/column to create TypeScript position.
         */
        const start = typescript_1.default.getPositionOfLineAndCharacter(sourceFile, line - 1, column - 1);
        const end = endLine && endColumn ? typescript_1.default.getPositionOfLineAndCharacter(sourceFile, endLine - 1, endColumn - 1) : start;
        const length = Math.max(0, end - start);
        const diagnostic = {
            file: sourceFile,
            category,
            code: consts_1.TS_LANGSERVICE_ESLINT_DIAGNOSTIC_ERROR_CODE,
            messageText,
            start,
            length,
        };
        return diagnostic;
    });
}
exports.translateToDiagnosticsFromESLintResult = translateToDiagnosticsFromESLintResult;
function isIntersect(message, range, sourceFile) {
    const { line, column, endLine, endColumn } = message;
    const mStart = typescript_1.default.getPositionOfLineAndCharacter(sourceFile, line - 1, column - 1);
    const mEnd = endLine && endColumn ? typescript_1.default.getPositionOfLineAndCharacter(sourceFile, endLine - 1, endColumn - 1) : mStart;
    return !(mEnd < range.start || mStart > range.end);
}
exports.isIntersect = isIntersect;
function translateToCodeFixesFromESLintResult(result, fileName) {
    return result.reduce((acc, { message, ruleId, fix }) => {
        if (!fix) {
            return acc;
        }
        const rid = ruleId || "eslint";
        const rangeStart = fix.range[0];
        const rangeLength = fix.range[1] ? fix.range[1] - fix.range[0] : 0;
        const codeFixAction = {
            description: `Fix: ${message}`,
            fixId: rid,
            fixName: rid,
            changes: [
                {
                    fileName,
                    isNewFile: false,
                    textChanges: [
                        {
                            span: {
                                start: rangeStart,
                                length: rangeLength,
                            },
                            newText: fix.text,
                        },
                    ],
                },
            ],
        };
        return [...acc, codeFixAction];
    }, []);
}
exports.translateToCodeFixesFromESLintResult = translateToCodeFixesFromESLintResult;
class ESLintAdapter {
    constructor({ logger, converter, configProvider, getSourceFile }) {
        this.linter = new eslint_1.Linter();
        this.logger = logger;
        this.converter = converter;
        this.configProvider = configProvider;
        this.getSourceFile = getSourceFile;
    }
    getESLintResult(fileName, sourceFile) {
        var _a, _b;
        if (new eslint_1.CLIEngine({}).isPathIgnored(fileName)) {
            return [];
        }
        const configArray = this.configProvider.getConfigArrayForFile(fileName);
        const configFileContent = configArray.extractConfig(fileName).toCompatibleObjectAsConfigFileContent();
        if (!configFileContent.parser ||
            ((_b = (_a = readPkgUp.sync({ cwd: configFileContent.parser })) === null || _a === void 0 ? void 0 : _a.packageJson.name) !== null && _b !== void 0 ? _b : "") !== "@typescript-eslint/parser") {
            throw new errors_1.InvalidParserError();
        }
        const parserOptions = (configFileContent.parserOptions ? configFileContent.parserOptions : {});
        const sourceCode = this.converter.convertToESLintSourceCode(sourceFile, parserOptions);
        // See https://github.com/eslint/eslint/blob/v6.1.0/lib/linter/linter.js#L1130
        return this.linter.verify(sourceCode, configArray, { filename: fileName });
    }
    getSemanticDiagnostics(delegate, fileName) {
        const original = delegate(fileName);
        try {
            const sourceFile = this.getSourceFile(fileName);
            if (!sourceFile) {
                return original;
            }
            const eslintResult = this.getESLintResult(fileName, sourceFile);
            return [...original, ...translateToDiagnosticsFromESLintResult(eslintResult, sourceFile)];
        }
        catch (error) {
            this.logger(error.message ? error.message : "unknow error");
            return original;
        }
    }
    getCodeFixesAtPosition(delegate, fileName, start, end, errorCodes, formatOptions, preferences) {
        const original = delegate(fileName, start, end, errorCodes, formatOptions, preferences);
        try {
            if (!errorCodes.includes(consts_1.TS_LANGSERVICE_ESLINT_DIAGNOSTIC_ERROR_CODE)) {
                return original;
            }
            const sourceFile = this.getSourceFile(fileName);
            if (!sourceFile) {
                return original;
            }
            const eslintResult = this.getESLintResult(fileName, sourceFile);
            return [
                ...original,
                ...translateToCodeFixesFromESLintResult(eslintResult.filter(r => isIntersect(r, { start, end }, sourceFile)), fileName),
            ];
        }
        catch (error) {
            this.logger(error.message ? error.message : "unknow error");
            return original;
        }
        return original;
    }
}
exports.ESLintAdapter = ESLintAdapter;
//# sourceMappingURL=eslint-adapter.js.map