/**
 * Copyright (c) 2010-2017, Grill Balázs, IncQueryLabs
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.testing.core.coverage;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.eclipse.viatra.query.runtime.matchers.psystem.PBody;
import org.eclipse.viatra.query.runtime.matchers.psystem.PTraceable;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.testing.core.coverage.CoverageContext;
import org.eclipse.viatra.query.testing.core.coverage.CoverageState;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Associates {@link CoverageState}s to objects.
 * @param <T> type of objects whose coverages are stored
 * @since 1.6
 */
@SuppressWarnings("all")
public class CoverageInfo<T extends Object> extends HashMap<CoverageContext<T>, CoverageState> {
  protected static final long serialVersionUID = (-8699692647123679741L);
  
  /**
   * Merge coverage. The keys of this and other should be disjunct.
   */
  public CoverageInfo<T> mergeWith(final CoverageInfo<T> other) {
    final CoverageInfo<T> result = new CoverageInfo<T>();
    result.putAll(this);
    result.putAll(other);
    return result;
  }
  
  /**
   * Returns the coverage for elements aggregated per scope.
   */
  public Map<T, CoverageState> getElementCoverage() {
    Map<T, CoverageState> _xblockexpression = null;
    {
      final Map<T, CoverageState> result = CollectionLiterals.<T, CoverageState>newHashMap();
      Set<Map.Entry<CoverageContext<T>, CoverageState>> _entrySet = this.entrySet();
      for (final Map.Entry<CoverageContext<T>, CoverageState> it : _entrySet) {
        {
          final T element = it.getKey().getElement();
          final CoverageState state = it.getValue();
          final CoverageState otherState = result.get(element);
          if (((state != null) && (otherState != null))) {
            result.put(element, state.best(otherState));
          } else {
            CoverageState _elvis = null;
            if (state != null) {
              _elvis = state;
            } else {
              _elvis = otherState;
            }
            result.put(element, _elvis);
          }
        }
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the ratio of the covered and the represented {@link PBody}s contained in this info, in percents.
   */
  public double getAggregatedCoveragePercent() {
    return this.getCoveragePercent(Iterables.<PBody>filter(this.getElementCoverage().keySet(), PBody.class));
  }
  
  /**
   * Returns the ratio of the given covered and the represented {@link PTraceable}s, in percents.
   */
  public double getCoveragePercent(final Iterable<? extends PTraceable> traceables) {
    double _xblockexpression = (double) 0;
    {
      final Map<T, CoverageState> elementCoverage = this.getElementCoverage();
      final Function1<PTraceable, Boolean> _function = (PTraceable it) -> {
        CoverageState _get = elementCoverage.get(it);
        return Boolean.valueOf(Objects.equal(_get, CoverageState.COVERED));
      };
      int _size = IterableExtensions.size(IterableExtensions.filter(traceables, _function));
      final double coveredCount = ((double) _size);
      final Function1<PTraceable, Boolean> _function_1 = (PTraceable it) -> {
        return Boolean.valueOf(Collections.<CoverageState>unmodifiableSet(CollectionLiterals.<CoverageState>newHashSet(CoverageState.COVERED, CoverageState.NOT_COVERED)).contains(elementCoverage.get(it)));
      };
      final int representedCount = IterableExtensions.size(IterableExtensions.filter(traceables, _function_1));
      double _xifexpression = (double) 0;
      if ((representedCount == 0)) {
        _xifexpression = 0;
      } else {
        _xifexpression = ((coveredCount / representedCount) * 100);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the ratio of the covered and the represented {@link PBody}s contained in the given {@link PQuery}, in percents.
   */
  public double getCoveragePercent(final PQuery query) {
    return this.getCoveragePercent(query.getDisjunctBodies().getBodies());
  }
}
