/**
 * Copyright (c) 2010-2016, Gabor Bergmann, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Gabor Bergmann - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.eclipse.viatra.query.runtime.api.GenericQueryGroup;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.testing.core.QueryPerformanceTest;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This abstract test class can be used to measure the steady-state memory requirements
 * of the Rete networks of individual queries on a given QueryScope,
 * relative to the network built to evaluate its dependencies.
 * In other words, the "local cost" of a query is measured; this is the memory footprint of the query
 * imposed on top of the memory footprint of all other queries invoked by it.
 * 
 * <p>
 * This test case prepares an ViatraEngine on the given scope and with the provided query group.
 * After the initial preparation is done, the engine is wiped (deletes the Rete network but keeps the base index).
 * Next, the following is performed for each query in the group:
 * <p/>
 * <ol>
 *   <li> Wipe the engine </li>
 *   <li> Prepare all queries referenced by the query under consideration, then measure memory </li>
 *   <li> Create the matcher for the query and count matches, then measure memory again to see the difference </li>
 *   <li> Wipe the engine </li>
 * </ol>
 * 
 * After each step, the used, total and free heap space is logged in MBytes after 5 GC calls and 1 second of waiting.
 * Note that even this does not always provide an absolute steady state or a precise result, but can be useful for
 * finding problematic queries.
 * 
 * @since 1.3
 */
@SuppressWarnings("all")
public abstract class RelativeQueryPerformanceTest extends QueryPerformanceTest {
  private Map<String, Long> absoluteHeapResults = Maps.<String, Long>newTreeMap();
  
  private Map<String, Long> relativeHeapResults = Maps.<String, Long>newTreeMap();
  
  @Override
  public <MATCH extends IPatternMatch, MATCHER extends ViatraQueryMatcher<MATCH>> QueryPerformanceTest.QueryPerformanceData performMeasurements(final IQuerySpecification<MATCHER> specification, final int current, final long usedHeapBefore) {
    final IQueryGroup prerequisites = RelativeQueryPerformanceTest.getDirectPrerequisites(specification);
    long prerequisitesHeap = 0L;
    {
      QueryPerformanceTest.logger.debug("Building Prerequisites");
      final Stopwatch watch = Stopwatch.createStarted();
      this.queryEngine.prepareGroup(prerequisites, null);
      watch.stop();
      final long usedHeapAfter = QueryPerformanceTest.logMemoryProperties("Prerequisites built");
      prerequisitesHeap = (usedHeapAfter - usedHeapBefore);
      String _fullyQualifiedName = specification.getFullyQualifiedName();
      String _plus = ("Prerequisites of query " + _fullyQualifiedName);
      String _plus_1 = (_plus + "(used ");
      String _plus_2 = (_plus_1 + Long.valueOf(prerequisitesHeap));
      String _plus_3 = (_plus_2 + 
        " kByte heap, took ");
      long _elapsed = watch.elapsed(TimeUnit.MILLISECONDS);
      String _plus_4 = (_plus_3 + Long.valueOf(_elapsed));
      String _plus_5 = (_plus_4 + " ms)");
      QueryPerformanceTest.logger.info(_plus_5);
    }
    final QueryPerformanceTest.QueryPerformanceData result = super.<MATCH, MATCHER>performMeasurements(specification, current, usedHeapBefore);
    this.absoluteHeapResults.put(specification.getFullyQualifiedName(), Long.valueOf(result.getUsedHeap()));
    String _fullyQualifiedName = specification.getFullyQualifiedName();
    long _usedHeap = result.getUsedHeap();
    long _minus = (_usedHeap - prerequisitesHeap);
    this.relativeHeapResults.put(_fullyQualifiedName, Long.valueOf(_minus));
    return result;
  }
  
  @Override
  protected void printResults() {
    super.printResults();
    final StringBuilder resultSB = new StringBuilder("\nAbsoluteHeap[kB]\trelativeHeap[kB]\tquery\n");
    final Consumer<Map.Entry<String, Long>> _function = (Map.Entry<String, Long> entry) -> {
      final String query = entry.getKey();
      resultSB.append(String.format("%12d\t%12d\t%s\n", entry.getValue(), this.relativeHeapResults.get(query), query));
    };
    this.absoluteHeapResults.entrySet().forEach(_function);
    QueryPerformanceTest.logger.info(resultSB);
  }
  
  protected static IQueryGroup getDirectPrerequisites(final IQuerySpecification<?> query) {
    IQueryGroup _xblockexpression = null;
    {
      final Set<PQuery> referredQueries = query.getInternalQueryRepresentation().getDisjunctBodies().getDirectReferredQueries();
      final Function1<PQuery, Iterable<IQuerySpecification>> _function = (PQuery it) -> {
        return Iterables.<IQuerySpecification>filter(it.publishedAs(), IQuerySpecification.class);
      };
      final Iterable<IQuerySpecification> referredSpecifications = Iterables.<IQuerySpecification>concat(IterableExtensions.<PQuery, Iterable<IQuerySpecification>>map(referredQueries, _function));
      _xblockexpression = GenericQueryGroup.of(((IQuerySpecification<?>[])Conversions.unwrapArray(referredSpecifications, IQuerySpecification.class)));
    }
    return _xblockexpression;
  }
}
