/*******************************************************************************
 * Copyright (c) 2010 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.ui.views;

import java.util.Arrays;

import org.eclipse.emf.mint.IItemJavaElementDescriptor2;
import org.eclipse.emf.mint.IJavaTypeReference;
import org.eclipse.jdt.ui.JavaElementLabelProvider;
import org.eclipse.jface.preference.JFacePreferences;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.StyledCellLabelProvider;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.IWorkbenchSiteProgressService;

public class ItemJavaElementLabelProvider extends StyledCellLabelProvider {
	
	private static final Object DEFERRED = new Object();
	
	private final JavaElementLabelProvider delegate;
	
	private final IWorkbenchSiteProgressService progressService;
	
	private final UpdateViewerJob updateViewerJob;
	
	public ItemJavaElementLabelProvider(IWorkbenchSiteProgressService progressService, UpdateViewerJob updateViewerJob) {
		this.progressService = progressService;
		this.updateViewerJob = updateViewerJob;
		delegate = new JavaElementLabelProvider(JavaElementLabelProvider.SHOW_DEFAULT | JavaElementLabelProvider.SHOW_POST_QUALIFIED);
	}

	@Override
	public void update(ViewerCell cell) {
		if (cell.getElement() instanceof ItemJavaElementDescriptor) {
			ItemJavaElementDescriptor desc = (ItemJavaElementDescriptor) cell.getElement();
			
			if (desc.getDescriptor() instanceof IItemJavaElementDescriptor2) {
				IItemJavaElementDescriptor2 desc2 = ((IItemJavaElementDescriptor2) desc.getDescriptor());
				if (!desc2.isElementOwner(desc.getTarget())) 
					cell.setForeground(JFaceResources.getColorRegistry().get(JFacePreferences.QUALIFIER_COLOR));
			}
			
			switch (cell.getColumnIndex()) {
			case 0:
				Object element = getJavaElement(desc);
				if (element == null) {
					// TODO improve text
					cell.setText(Messages.ItemJavaElementLabelProvider_UnknownElementLabel);
					// TODO check icon
					cell.setImage(PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJS_WARN_TSK));
				} else if (element == DEFERRED) {
					cell.setText(desc.getDescriptor().getJavaTypeReference(desc.getTarget()).getTypeName());
					// TODO what about image?
				} else if (element == ItemJavaElementDescriptor.ERROR_ELEMENT) {
					cell.setText(desc.getDescriptor().getJavaTypeReference(desc.getTarget()).getTypeName());
					// TODO check icon
					cell.setImage(PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJS_ERROR_TSK));
				} else {
					StyledString styledString = delegate.getStyledText(element);
					String newText= styledString.toString();
					
					StyleRange[] oldStyleRanges= cell.getStyleRanges();
					StyleRange[] newStyleRanges= isOwnerDrawEnabled() ? styledString.getStyleRanges() : null;
					
					if (!Arrays.equals(oldStyleRanges, newStyleRanges)) {
						cell.setStyleRanges(newStyleRanges);
						if (cell.getText().equals(newText)) {
							// make sure there will be a refresh from a change
							cell.setText(""); //$NON-NLS-1$
						}
					}

					cell.setText(newText);
					cell.setImage(delegate.getImage(element));
				}
				
				break;
			case 1:
				cell.setText(desc.getDescriptor().getDisplayName(desc.getTarget()));
				break;
			case 2:
				cell.setText(desc.getDescriptor().getCategory(desc.getTarget()));
				break;
			}
		}
	}
	
	private Object getJavaElement(ItemJavaElementDescriptor desc) {
		switch (desc.getDescriptor().getKind(desc.getTarget())) {
		case JAVA_ELEMENT:
			return desc.getDescriptor().getJavaElement(desc.getTarget());
		case JAVA_TYPE_REFERENCE:
			if (desc.getElement() != null)
				return desc.getElement();
			
			IJavaTypeReference ref = desc.getDescriptor().getJavaTypeReference(desc.getTarget());
			FindTypeJob job = new FindTypeJob(desc, ref, updateViewerJob);
			if (progressService == null)
				job.schedule();
			else
				progressService.schedule(job, 0L, true);
			
			return DEFERRED;
		case NON_JAVA_RESOURCE:
			return desc.getDescriptor().getNonJavaElement(desc.getTarget());
		}
		
		return null;
	}
	
	@Override
	public void addListener(ILabelProviderListener listener) {
		super.addListener(listener);
		delegate.addListener(listener);
	}
	
	@Override
	public void removeListener(ILabelProviderListener listener) {
		super.removeListener(listener);
		delegate.removeListener(listener);
	}
	
	@Override
	public boolean isLabelProperty(Object element, String property) {
		return delegate.isLabelProperty(element, property);
	}

	@Override
	public void dispose() {
		super.dispose();
		delegate.dispose();
	}
}
