/*
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 *
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2009 by
 * + Christian-Albrechts-University of Kiel
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This code is provided under the terms of the Eclipse Public License (EPL).
 * See the file epl-v10.html for the license text.
 */
package de.cau.cs.kieler.kiml.gmf;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.operations.IOperationHistory;
import org.eclipse.core.commands.operations.OperationHistoryFactory;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.draw2d.Connection;
import org.eclipse.draw2d.ConnectionLocator;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.Label;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Insets;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.impl.InternalTransactionalEditingDomain;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.RootEditPart;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CommandStack;
import org.eclipse.gmf.runtime.diagram.ui.editparts.AbstractBorderItemEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.CompartmentEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.ConnectionEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.DiagramEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.DiagramRootEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.LabelEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.ResizableCompartmentEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.ShapeNodeEditPart;
import org.eclipse.gmf.runtime.diagram.ui.figures.BorderedNodeFigure;
import org.eclipse.gmf.runtime.diagram.ui.figures.ResizableCompartmentFigure;
import org.eclipse.gmf.runtime.diagram.ui.parts.DiagramCommandStack;
import org.eclipse.gmf.runtime.diagram.ui.parts.DiagramEditor;
import org.eclipse.gmf.runtime.draw2d.ui.figures.WrappingLabel;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.graphics.Font;
import org.eclipse.ui.IWorkbenchPart;

import com.google.common.collect.BiMap;

import de.cau.cs.kieler.core.WrappedException;
import de.cau.cs.kieler.core.kgraph.KEdge;
import de.cau.cs.kieler.core.kgraph.KGraphElement;
import de.cau.cs.kieler.core.kgraph.KGraphFactory;
import de.cau.cs.kieler.core.kgraph.KLabel;
import de.cau.cs.kieler.core.kgraph.KNode;
import de.cau.cs.kieler.core.kgraph.KPort;
import de.cau.cs.kieler.core.math.KVector;
import de.cau.cs.kieler.core.properties.IProperty;
import de.cau.cs.kieler.core.properties.Property;
import de.cau.cs.kieler.core.util.Maybe;
import de.cau.cs.kieler.kiml.LayoutContext;
import de.cau.cs.kieler.kiml.config.VolatileLayoutConfig;
import de.cau.cs.kieler.kiml.klayoutdata.KEdgeLayout;
import de.cau.cs.kieler.kiml.klayoutdata.KInsets;
import de.cau.cs.kieler.kiml.klayoutdata.KLayoutDataFactory;
import de.cau.cs.kieler.kiml.klayoutdata.KPoint;
import de.cau.cs.kieler.kiml.klayoutdata.KShapeLayout;
import de.cau.cs.kieler.kiml.klayoutdata.impl.KEdgeLayoutImpl;
import de.cau.cs.kieler.kiml.klayoutdata.impl.KShapeLayoutImpl;
import de.cau.cs.kieler.kiml.options.EdgeLabelPlacement;
import de.cau.cs.kieler.kiml.options.LayoutOptions;
import de.cau.cs.kieler.kiml.ui.diagram.LayoutMapping;
import de.cau.cs.kieler.kiml.util.KimlUtil;

/**
 * Diagram layout manager that is able to generically layout diagrams generated by GMF. The internal
 * KGraph graph structure is built from the structure of edit parts in the diagram. The new layout
 * is applied to the diagram using {@link GmfLayoutEditPolicy}, which creates a
 * {@link GmfLayoutCommand} to directly manipulate data in the GMF notation model, where layout
 * information is stored persistently.
 * 
 * @kieler.rating yellow 2012-07-19 review KI-20 by cds, jjc
 * @author ars
 * @author msp
 */
public class GmfDiagramLayoutManager extends GefDiagramLayoutManager<IGraphicalEditPart> {

    /** list of connection edit parts that were found in the diagram. */
    public static final IProperty<List<ConnectionEditPart>> CONNECTIONS = 
            new Property<List<ConnectionEditPart>>("gmf.connections");

    /** editor part of the currently layouted diagram. */
    public static final IProperty<DiagramEditor> DIAGRAM_EDITOR = new Property<DiagramEditor>(
            "gmf.diagramEditor");

    /** diagram edit part of the currently layouted diagram. */
    public static final IProperty<DiagramEditPart> DIAGRAM_EDIT_PART = new Property<DiagramEditPart>(
            "gmf.diagramEditPart");

    /** the command that applies the transferred layout to the diagram. */
    public static final IProperty<Command> LAYOUT_COMMAND = new Property<Command>(
            "gmf.applyLayoutCommand");

    /** the command stack that executes the command. */
    public static final IProperty<CommandStack> COMMAND_STACK = new Property<CommandStack>(
            "gmf.applyLayoutCommandStack");
    
    /** the volatile layout config for static properties such as minimal node sizes. */
    public static final IProperty<VolatileLayoutConfig> STATIC_CONFIG
            = new Property<VolatileLayoutConfig>("gmf.staticLayoutConfig");
    
    /**
     * Determines the insets for a parent figure, relative to the given child.
     * 
     * @param parent the figure of a parent edit part
     * @param child the figure of a child edit part
     * @return the insets to add to the relative coordinates of the child
     */
    public static Insets calcInsets(final IFigure parent, final IFigure child) {
        Insets result = new Insets(0);
        IFigure currentChild = child;
        IFigure currentParent = child.getParent();
        Point coordsToAdd = null;
        boolean isRelative = false;
        // follow the chain of parents in the figure hierarchy up to the given parent figure
        while (currentChild != parent && currentParent != null) {
            if (currentParent.isCoordinateSystem()) {
                // the content of the current parent is relative to that figure's position
                isRelative = true;
                result.add(currentParent.getInsets());
                if (coordsToAdd != null) {
                    // add the position of the previous parent with local coordinate system
                    result.left += coordsToAdd.x;
                    result.top += coordsToAdd.y;
                }
                coordsToAdd = currentParent.getBounds().getLocation();
            } else if (currentParent == parent && coordsToAdd != null) {
                // we found the top parent, and it does not have local coordinate system,
                // so subtract the parent's coordinates from the previous parent's position
                Point parentCoords = parent.getBounds().getLocation();
                result.left += coordsToAdd.x - parentCoords.x;
                result.top += coordsToAdd.y - parentCoords.y;
            }
            currentChild = currentParent;
            currentParent = currentChild.getParent();
        }
        if (!isRelative) {
            // there is no local coordinate system, so just subtract the coordinates
            Rectangle parentBounds = parent.getBounds();
            currentParent = child.getParent();
            Rectangle containerBounds = currentParent.getBounds();
            result.left = containerBounds.x - parentBounds.x;
            result.top = containerBounds.y - parentBounds.y;
        }
        // In theory it would be better to get the bottom and right insets from the size.
        // However, due to the inpredictability of Draw2D layout managers, this leads to
        // bad results in many cases, so a fixed insets value is more stable.
        result.right = result.left;
        result.bottom = result.left;
        return result;
    }
    
    /**
     * Calculates the absolute bounds of the given figure.
     * 
     * @param figure a figure
     * @return the absolute bounds
     */
    public static Rectangle getAbsoluteBounds(final IFigure figure) {
        Rectangle bounds = new Rectangle(figure.getBounds()) {
            static final long serialVersionUID = 1;
            @Override
            public void performScale(final double factor) {
                // don't perform any scaling to avoid distortion by the zoom level
            }
        };
        figure.translateToAbsolute(bounds);
        return bounds;
    }
    
    /**
     * Calculates an absolute position for one of the bend points of the given connection.
     * 
     * @param connection a connection figure
     * @param index the index in the point list
     * @return the absolute point
     * @deprecated this method does not correctly compensate panning of the diagram
     *          (deprecated since July 2012)
     */
    public static Point getAbsolutePoint(final Connection connection, final int index) {
        Point point = new Point(connection.getPoints().getPoint(index)) {
            static final long serialVersionUID = 1;
            @Override
            public void performScale(final double factor) {
                // don't perform any scaling to avoid distortion by the zoom level
            }
        };
        connection.translateToAbsolute(point);
        return point;
    }

    /**
     * Finds the diagram edit part of an edit part.
     * 
     * @param editPart an edit part
     * @return the diagram edit part, or {@code null} if there is no containing diagram
     *     edit part
     */
    public static DiagramEditPart getDiagramEditPart(final EditPart editPart) {
        EditPart ep = editPart;
        while (ep != null && !(ep instanceof DiagramEditPart) && !(ep instanceof RootEditPart)) {
            ep = ep.getParent();
        }
        if (ep instanceof RootEditPart) {
            // the diagram edit part is a direct child of the root edit part
            RootEditPart root = (RootEditPart) ep;
            ep = null;
            for (Object child : root.getChildren()) {
                if (child instanceof DiagramEditPart) {
                    ep = (EditPart) child;
                }
            }
        }
        return (DiagramEditPart) ep;
    }

    /**
     * {@inheritDoc}
     */
    public boolean supports(final Object object) {
        return object instanceof DiagramEditor || object instanceof IGraphicalEditPart;
    }

    /** the cached layout configuration for GMF. */
    private GmfLayoutConfig layoutConfig = new GmfLayoutConfig();

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings({ "rawtypes", "unchecked" }) // the signature of IAdapterFactory is unchecked
    public Object getAdapter(final Object object, final Class adapterType) {
        try {
            if (adapterType.isAssignableFrom(GmfLayoutConfig.class)) {
                return layoutConfig;
            } else if (adapterType.isAssignableFrom(IGraphicalEditPart.class)) {
                if (object instanceof CompartmentEditPart) {
                    return ((CompartmentEditPart) object).getParent();
                } else if (object instanceof IGraphicalEditPart) {
                    return object;
                } else if (object instanceof DiagramEditor) {
                    return ((DiagramEditor) object).getDiagramEditPart();
                } else if (object instanceof DiagramRootEditPart) {
                    return ((DiagramRootEditPart) object).getContents();
                }
            } else if (adapterType.isAssignableFrom(EObject.class)) {
                if (object instanceof IGraphicalEditPart) {
                    IGraphicalEditPart editPart = (IGraphicalEditPart) object;
                    EObject element = editPart.getNotationView().getElement();
                    if (editPart.getParent() != null) {
                        // return the EObject only if the edit part has its own model element
                        Object model = editPart.getParent().getModel();
                        if (model instanceof View) {
                            EObject parentElement = ((View) model).getElement();
                            if (element == parentElement) {
                                return null;
                            }
                        }
                    }
                    return element;
                } else if (object instanceof View) {
                    return ((View) object).getElement();
                }
            } else if (adapterType.isAssignableFrom(TransactionalEditingDomain.class)) {
                if (object instanceof DiagramEditor) {
                    return ((DiagramEditor) object).getEditingDomain();
                } else if (object instanceof IGraphicalEditPart) {
                    return ((IGraphicalEditPart) object).getEditingDomain();
                }
            }
            if (object instanceof IAdaptable) {
                return ((IAdaptable) object).getAdapter(adapterType);
            }
        } catch (RuntimeException exception) {
            // when the editor part has been closed NPEs can occur
        }
        return null;
    }

    /**
     * {@inheritDoc}
     */
    public Class<?>[] getAdapterList() {
        return new Class<?>[] { IGraphicalEditPart.class };
    }

    /**
     * {@inheritDoc}
     */
    public LayoutMapping<IGraphicalEditPart> buildLayoutGraph(final IWorkbenchPart workbenchPart,
            final Object diagramPart) {
        DiagramEditor diagramEditor = null;

        // get the diagram editor part
        if (workbenchPart instanceof DiagramEditor) {
            diagramEditor = (DiagramEditor) workbenchPart;
        }

        // choose the layout root edit part
        IGraphicalEditPart layoutRootPart = null;
        if (diagramPart instanceof ShapeNodeEditPart || diagramPart instanceof DiagramEditPart) {
            layoutRootPart = (IGraphicalEditPart) diagramPart;
        } else if (diagramPart instanceof IGraphicalEditPart) {
            EditPart tgEditPart = ((IGraphicalEditPart) diagramPart).getTopGraphicEditPart();
            if (tgEditPart instanceof ShapeNodeEditPart) {
                layoutRootPart = (IGraphicalEditPart) tgEditPart;
            }
        }
        if (layoutRootPart == null && diagramEditor != null) {
            layoutRootPart = diagramEditor.getDiagramEditPart();
        }
        if (layoutRootPart == null) {
            throw new UnsupportedOperationException(
                    "Not supported by this layout manager: Workbench part " + workbenchPart
                            + ", Edit part " + diagramPart);
        }

        // create the mapping
        LayoutMapping<IGraphicalEditPart> mapping = buildLayoutGraph(layoutRootPart);

        // set optional diagram editor
        if (diagramEditor != null) {
            mapping.setProperty(DIAGRAM_EDITOR, diagramEditor);
        }
        
        // create a layout configuration
        mapping.getLayoutConfigs().add(mapping.getProperty(STATIC_CONFIG));
        mapping.getLayoutConfigs().add(layoutConfig);

        return mapping;
    }

    /**
     * Creates the actual mapping given an edit part which functions as the root for the layout.
     * 
     * @param layoutRootPart
     *            the layout root edit part
     * @return a layout graph mapping
     */
    protected LayoutMapping<IGraphicalEditPart> buildLayoutGraph(
            final IGraphicalEditPart layoutRootPart) {
        LayoutMapping<IGraphicalEditPart> mapping = new LayoutMapping<IGraphicalEditPart>(this);
        mapping.setProperty(CONNECTIONS, new LinkedList<ConnectionEditPart>());
        mapping.setProperty(STATIC_CONFIG, new VolatileLayoutConfig(GmfLayoutConfig.PRIORITY - 1));

        // set the parent element
        mapping.setParentElement(layoutRootPart);

        // find the diagram edit part
        mapping.setProperty(DIAGRAM_EDIT_PART, getDiagramEditPart(layoutRootPart));

        KNode topNode = KimlUtil.createInitializedNode();
        KShapeLayout shapeLayout = topNode.getData(KShapeLayout.class);
        Rectangle rootBounds = layoutRootPart.getFigure().getBounds();
        if (layoutRootPart instanceof DiagramEditPart) {
            // start with the whole diagram as root for layout
            String labelText = ((DiagramEditPart) layoutRootPart).getDiagramView().getName();
            if (labelText.length() > 0) {
                KLabel label = KimlUtil.createInitializedLabel(topNode);
                label.setText(labelText);
            }
        } else {
            // start with a specific node as root for layout
            shapeLayout.setPos(rootBounds.x, rootBounds.y);
        }
        shapeLayout.setSize(rootBounds.width, rootBounds.height);
        mapping.getGraphMap().put(topNode, layoutRootPart);
        mapping.setLayoutGraph(topNode);

        // traverse the children of the layout root part
        buildLayoutGraphRecursively(mapping, layoutRootPart, topNode, layoutRootPart);
        // transform all connections in the selected area
        processConnections(mapping);

        return mapping;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void transferLayout(final LayoutMapping<IGraphicalEditPart> mapping) {
        // create a new request to change the layout
        ApplyLayoutRequest applyLayoutRequest = new ApplyLayoutRequest();
        for (Entry<KGraphElement, IGraphicalEditPart> entry : mapping.getGraphMap().entrySet()) {
            if (!(entry.getValue() instanceof DiagramEditPart)) {
                applyLayoutRequest.addElement(entry.getKey(), entry.getValue());
            }
        }
        KShapeLayout graphLayout = mapping.getLayoutGraph().getData(KShapeLayout.class);
        applyLayoutRequest.setUpperBound(graphLayout.getWidth(), graphLayout.getHeight());

        // check the validity of the editing domain to catch cases where it is disposed
        DiagramEditPart diagramEditPart = mapping.getProperty(DIAGRAM_EDIT_PART);
        if (((InternalTransactionalEditingDomain) diagramEditPart.getEditingDomain())
                .getChangeRecorder() != null) {
            // retrieve a command for the request; the command is created by GmfLayoutEditPolicy
            Command applyLayoutCommand = diagramEditPart.getCommand(applyLayoutRequest);
            mapping.setProperty(LAYOUT_COMMAND, applyLayoutCommand);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void applyLayout(final LayoutMapping<IGraphicalEditPart> mapping) {
        Command applyLayoutCommand = mapping.getProperty(LAYOUT_COMMAND);
        
        if (applyLayoutCommand != null) {
            // get a command stack to execute the command
            CommandStack commandStack = mapping.getProperty(COMMAND_STACK);
            DiagramEditor diagramEditor = mapping.getProperty(DIAGRAM_EDITOR);
            if (commandStack == null) {
                if (diagramEditor != null) {
                    Object adapter = diagramEditor.getAdapter(CommandStack.class);
                    if (adapter instanceof CommandStack) {
                        commandStack = (CommandStack) adapter;
                    }
                }
                if (commandStack == null) {
                    commandStack = mapping.getParentElement().getDiagramEditDomain()
                            .getDiagramCommandStack();
                }
            }
    
            // execute the command
            commandStack.execute(applyLayoutCommand);
            
            // refresh the border items in the diagram
            if (diagramEditor != null || mapping.getParentElement() != null) {
                refreshDiagram(diagramEditor, mapping.getParentElement());
            }
        }
    }
    
    /**
     * Perform undo in the original diagram.
     *
     * @param mapping a layout mapping that was created by this layout manager
     */
    @Override
    protected void performUndo(final LayoutMapping<IGraphicalEditPart> mapping) {
        try {
            IOperationHistory history = OperationHistoryFactory.getOperationHistory();
            history.undoOperation(DiagramCommandStack.getICommand(mapping.getProperty(LAYOUT_COMMAND)),
                    new NullProgressMonitor(), null);
        } catch (ExecutionException e) {
            throw new WrappedException(e);
        }
    }

    /**
     * Recursively builds a layout graph by analyzing the children of the given edit part.
     * 
     * @param mapping
     *            the layout mapping
     * @param parentEditPart
     *            the parent edit part of the current elements
     * @param parentLayoutNode
     *            the corresponding KNode
     * @param currentEditPart
     *            the currently analyzed edit part
     */
    private void buildLayoutGraphRecursively(final LayoutMapping<IGraphicalEditPart> mapping,
            final IGraphicalEditPart parentEditPart, final KNode parentLayoutNode,
            final IGraphicalEditPart currentEditPart) {
        Maybe<KInsets> kinsets = new Maybe<KInsets>();

        // iterate through the children of the element
        for (Object obj : currentEditPart.getChildren()) {

            // check visibility of the child
            if (obj instanceof IGraphicalEditPart) {
                IFigure figure = ((IGraphicalEditPart) obj).getFigure();
                if (!figure.isVisible()) {
                    continue;
                }
            }

            // process a port (border item)
            if (obj instanceof AbstractBorderItemEditPart) {
                createPort(mapping, (AbstractBorderItemEditPart) obj, parentEditPart,
                        parentLayoutNode);

            // process a compartment, which may contain other elements
            } else if (obj instanceof ResizableCompartmentEditPart
                    && ((CompartmentEditPart) obj).getChildren().size() > 0) {
                CompartmentEditPart compartment = (CompartmentEditPart) obj;
                if (!GmfLayoutConfig.isNoLayout(compartment)) {
                    boolean compExp = true;
                    IFigure compartmentFigure = compartment.getFigure();
                    if (compartmentFigure instanceof ResizableCompartmentFigure) {
                        ResizableCompartmentFigure resizCompFigure = 
                                (ResizableCompartmentFigure) compartmentFigure;
                        // check whether the compartment is collapsed
                        compExp = resizCompFigure.isExpanded();
                    }

                    if (compExp) {
                        buildLayoutGraphRecursively(mapping, parentEditPart, parentLayoutNode,
                                compartment);
                    }
                }

            // process a node, which may be a parent of ports, compartments, or other nodes
            } else if (obj instanceof ShapeNodeEditPart) {
                ShapeNodeEditPart childNodeEditPart = (ShapeNodeEditPart) obj;
                if (!GmfLayoutConfig.isNoLayout(childNodeEditPart)) {
                    createNode(mapping, childNodeEditPart, parentEditPart, parentLayoutNode,
                            kinsets);
                }

            // process a label of the current node
            } else if (obj instanceof IGraphicalEditPart) {
                createNodeLabel(mapping, (IGraphicalEditPart) obj, parentEditPart, parentLayoutNode);
            }
        }
    }

    /**
     * Create a node while building the layout graph.
     * 
     * @param mapping
     *            the layout mapping
     * @param nodeEditPart
     *            the node edit part
     * @param parentEditPart
     *            the parent node edit part that contains the current node
     * @param parentKNode
     *            the corresponding parent layout node
     * @param kinsets
     *            reference parameter for insets; the insets are calculated if this has not been
     *            done before
     */
    private void createNode(final LayoutMapping<IGraphicalEditPart> mapping,
            final ShapeNodeEditPart nodeEditPart, final IGraphicalEditPart parentEditPart,
            final KNode parentKNode, final Maybe<KInsets> kinsets) {
        IFigure nodeFigure = nodeEditPart.getFigure();
        KNode childLayoutNode = KimlUtil.createInitializedNode();

        // set location and size
        Rectangle childBounds = getAbsoluteBounds(nodeFigure);
        Rectangle containerBounds = getAbsoluteBounds(nodeFigure.getParent());
        KShapeLayout nodeLayout = childLayoutNode.getData(KShapeLayout.class);
        nodeLayout.setXpos(childBounds.x - containerBounds.x);
        nodeLayout.setYpos(childBounds.y - containerBounds.y);
        nodeLayout.setSize(childBounds.width, childBounds.height);
        // the modification flag must initially be false
        ((KShapeLayoutImpl) nodeLayout).resetModificationFlag();
        
        // determine minimal size of the node
        try {
            Dimension minSize = nodeFigure.getMinimumSize();
            VolatileLayoutConfig staticConfig = mapping.getProperty(STATIC_CONFIG);
            staticConfig.setValue(LayoutOptions.MIN_WIDTH, childLayoutNode, LayoutContext.GRAPH_ELEM,
                    (float) minSize.width);
            staticConfig.setValue(LayoutOptions.MIN_HEIGHT, childLayoutNode, LayoutContext.GRAPH_ELEM,
                    (float) minSize.height);
        } catch (SWTException exception) {
            // getMinimumSize() can cause this exception when fonts are disposed for some reason;
            // ignore exception and leave the default minimal size
        }

        // set insets if not yet defined
        if (kinsets.get() == null) {
            KInsets ki = parentKNode.getData(KShapeLayout.class).getInsets();
            Insets insets = calcInsets(parentEditPart.getFigure(), nodeFigure);
            ki.setLeft(insets.left);
            ki.setTop(insets.top);
            ki.setRight(insets.right);
            ki.setBottom(insets.bottom);
            kinsets.set(ki);
        }

        parentKNode.getChildren().add(childLayoutNode);
        mapping.getGraphMap().put(childLayoutNode, nodeEditPart);
        // process the child as new current edit part
        buildLayoutGraphRecursively(mapping, nodeEditPart, childLayoutNode, nodeEditPart);

        // store all the connections to process them later
        addConnections(mapping, nodeEditPart);
    }

    /**
     * Create a port while building the layout graph.
     * 
     * @param mapping
     *            the layout mapping
     * @param portEditPart
     *            the port edit part
     * @param nodeEditPart
     *            the parent node edit part
     * @param knode
     *            the corresponding layout node
     * @param layoutConfig
     *            a layout configuration
     */
    private void createPort(final LayoutMapping<IGraphicalEditPart> mapping,
            final AbstractBorderItemEditPart portEditPart, final IGraphicalEditPart nodeEditPart,
            final KNode knode) {
        KPort port = KimlUtil.createInitializedPort();
        port.setNode(knode);

        // set the port's layout, relative to the node position
        KShapeLayout portLayout = port.getData(KShapeLayout.class);
        Rectangle portBounds = getAbsoluteBounds(portEditPart.getFigure());
        Rectangle nodeBounds = getAbsoluteBounds(nodeEditPart.getFigure());
        float xpos = portBounds.x - nodeBounds.x;
        float ypos = portBounds.y - nodeBounds.y;
        portLayout.setPos(xpos, ypos);
        portLayout.setSize(portBounds.width, portBounds.height);
        // the modification flag must initially be false
        ((KShapeLayoutImpl) portLayout).resetModificationFlag();

        mapping.getGraphMap().put(port, portEditPart);

        // store all the connections to process them later
        addConnections(mapping, portEditPart);

        // set the port label
        for (Object portChildObj : portEditPart.getChildren()) {
            if (portChildObj instanceof IGraphicalEditPart) {
                IFigure labelFigure = ((IGraphicalEditPart) portChildObj).getFigure();
                String text = null;
                if (labelFigure instanceof WrappingLabel) {
                    text = ((WrappingLabel) labelFigure).getText();
                } else if (labelFigure instanceof Label) {
                    text = ((Label) labelFigure).getText();
                }
                if (text != null) {
                    KLabel portLabel = KimlUtil.createInitializedLabel(port);
                    portLabel.setText(text);
                    mapping.getGraphMap().put(portLabel, (IGraphicalEditPart) portChildObj);
                    // set the port label's layout
                    KShapeLayout labelLayout = portLabel.getData(KShapeLayout.class);
                    Rectangle labelBounds = getAbsoluteBounds(labelFigure);
                    labelLayout.setXpos(labelBounds.x - portBounds.x);
                    labelLayout.setYpos(labelBounds.y - portBounds.y);
                    try {
                        Dimension size = labelFigure.getPreferredSize();
                        labelLayout.setWidth(size.width);
                        labelLayout.setHeight(size.height);
                    } catch (SWTException exception) {
                        // ignore exception and leave the label size to (0, 0)
                    }
                    // the modification flag must initially be false
                    ((KShapeLayoutImpl) labelLayout).resetModificationFlag();
                }
            }
        }
    }

    /**
     * Create a node label while building the layout graph.
     * 
     * @param mapping
     *            the layout mapping
     * @param labelEditPart
     *            the label edit part
     * @param nodeEditPart
     *            the parent node edit part
     * @param knode
     *            the layout node for which the label is set
     */
    private void createNodeLabel(final LayoutMapping<IGraphicalEditPart> mapping,
            final IGraphicalEditPart labelEditPart, final IGraphicalEditPart nodeEditPart,
            final KNode knode) {
        IFigure labelFigure = labelEditPart.getFigure();
        String text = null;
        Font font = null;
        if (labelFigure instanceof WrappingLabel) {
            WrappingLabel wrappingLabel = (WrappingLabel) labelFigure;
            text = wrappingLabel.getText();
            font = wrappingLabel.getFont();
        } else if (labelFigure instanceof Label) {
            Label label = (Label) labelFigure;
            text = label.getText();
            font = label.getFont();
        }
        if (text != null) {
            KLabel label = KimlUtil.createInitializedLabel(knode);
            label.setText(text);
            mapping.getGraphMap().put(label, labelEditPart);
            KShapeLayout labelLayout = label.getData(KShapeLayout.class);
            Rectangle labelBounds = getAbsoluteBounds(labelFigure);
            Rectangle nodeBounds = getAbsoluteBounds(nodeEditPart.getFigure());
            labelLayout.setXpos(labelBounds.x - nodeBounds.x);
            labelLayout.setYpos(labelBounds.y - nodeBounds.y);
            try {
                Dimension size = labelFigure.getPreferredSize();
                labelLayout.setSize(size.width, size.height);
                if (font != null && !font.isDisposed()) {
                    VolatileLayoutConfig staticConfig = mapping.getProperty(STATIC_CONFIG);
                    staticConfig.setValue(LayoutOptions.FONT_NAME, label, LayoutContext.GRAPH_ELEM,
                            font.getFontData()[0].getName());
                    staticConfig.setValue(LayoutOptions.FONT_SIZE, label, LayoutContext.GRAPH_ELEM,
                            font.getFontData()[0].getHeight());
                }
            } catch (SWTException exception) {
                // ignore exception and leave the label size to (0, 0)
            }
            // the modification flag must initially be false
            ((KShapeLayoutImpl) labelLayout).resetModificationFlag();
        }
    }

    /**
     * Adds all target connections and connected connections to the list of connections that must be
     * processed later.
     * 
     * @param mapping
     *            the layout mapping
     * @param editPart
     *            an edit part
     */
    protected void addConnections(final LayoutMapping<IGraphicalEditPart> mapping,
            final IGraphicalEditPart editPart) {
        for (Object targetConn : editPart.getTargetConnections()) {
            if (targetConn instanceof ConnectionEditPart) {
                ConnectionEditPart connectionEditPart = (ConnectionEditPart) targetConn;
                mapping.getProperty(CONNECTIONS).add(connectionEditPart);
                addConnections(mapping, connectionEditPart);
            }
        }
    }

    /**
     * Creates new edges and takes care of the labels for each connection identified in the
     * {@code buildLayoutGraphRecursively} method.
     * 
     * @param mapping
     *            the layout mapping
     */
    protected void processConnections(final LayoutMapping<IGraphicalEditPart> mapping) {
        Map<EReference, KEdge> reference2EdgeMap = new HashMap<EReference, KEdge>();
        for (ConnectionEditPart connection : mapping.getProperty(CONNECTIONS)) {
            boolean isOppositeEdge = false;
            EdgeLabelPlacement edgeLabelPlacement = EdgeLabelPlacement.UNDEFINED;
            KEdge edge;

            // Check whether the edge belongs to an Ecore reference, which may have opposites.
            // This is required for the layout of Ecore diagrams, since the bend points of
            // opposite references are kept synchronized by the editor.
            EObject modelObject = connection.getNotationView().getElement();
            if (modelObject instanceof EReference) {
                EReference reference = (EReference) modelObject;
                edge = reference2EdgeMap.get(reference.getEOpposite());
                if (edge != null) {
                    edgeLabelPlacement = EdgeLabelPlacement.TAIL;
                    isOppositeEdge = true;
                } else {
                    edge = KimlUtil.createInitializedEdge();
                    reference2EdgeMap.put(reference, edge);
                }
            } else {
                edge = KimlUtil.createInitializedEdge();
            }

            BiMap<KGraphElement, IGraphicalEditPart> graphMap = mapping.getGraphMap();

            // find a proper source node and source port
            KGraphElement sourceElem;
            EditPart sourceObj = connection.getSource();
            if (sourceObj instanceof ConnectionEditPart) {
                sourceElem = graphMap.inverse().get(((ConnectionEditPart) sourceObj).getSource());
                if (sourceElem == null) {
                    sourceElem = graphMap.inverse().get(
                            ((ConnectionEditPart) sourceObj).getTarget());
                }
            } else {
                sourceElem = graphMap.inverse().get(sourceObj);
            }
            KNode sourceNode = null;
            KPort sourcePort = null;
            if (sourceElem instanceof KNode) {
                sourceNode = (KNode) sourceElem;
            } else if (sourceElem instanceof KPort) {
                sourcePort = (KPort) sourceElem;
                sourceNode = sourcePort.getNode();
            } else {
                continue;
            }

            // find a proper target node and target port
            KGraphElement targetElem;
            EditPart targetObj = connection.getTarget();
            if (targetObj instanceof ConnectionEditPart) {
                targetElem = graphMap.inverse().get(((ConnectionEditPart) targetObj).getTarget());
                if (targetElem == null) {
                    targetElem = graphMap.inverse().get(
                            ((ConnectionEditPart) targetObj).getSource());
                }
            } else {
                targetElem = graphMap.inverse().get(targetObj);
            }
            KNode targetNode = null;
            KPort targetPort = null;
            if (targetElem instanceof KNode) {
                targetNode = (KNode) targetElem;
            } else if (targetElem instanceof KPort) {
                targetPort = (KPort) targetElem;
                targetNode = targetPort.getNode();
            } else {
                continue;
            }

            // calculate offset for edge and label coordinates
            KVector offset = new KVector();
            if (KimlUtil.isDescendant(targetNode, sourceNode)) {
                KimlUtil.toAbsolute(offset, sourceNode);
            } else {
                KimlUtil.toAbsolute(offset, sourceNode.getParent());
            }

            if (!isOppositeEdge) {
                // set source and target
                edge.setSource(sourceNode);
                if (sourcePort != null) {
                    edge.setSourcePort(sourcePort);
                    sourcePort.getEdges().add(edge);
                }
                edge.setTarget(targetNode);
                if (targetPort != null) {
                    edge.setTargetPort(targetPort);
                    targetPort.getEdges().add(edge);
                }

                graphMap.put(edge, connection);

                // store the current coordinates of the edge
                KEdgeLayout edgeLayout = edge.getData(KEdgeLayout.class);
                setEdgeLayout(edgeLayout, connection, offset);
            }

            // process edge labels
            processEdgeLabels(mapping, connection, edge, edgeLabelPlacement, offset);
        }
    }

    /**
     * Stores the layout information of the given connection edit part into an edge layout.
     * 
     * @param edgeLayout
     *            an edge layout
     * @param connection
     *            a connection edit part
     * @param offset
     *            offset to be subtracted from coordinates
     */
    protected void setEdgeLayout(final KEdgeLayout edgeLayout, final ConnectionEditPart connection,
            final KVector offset) {
        Connection figure = connection.getConnectionFigure();
        PointList pointList = figure.getPoints();

        KPoint sourcePoint = edgeLayout.getSourcePoint();
        Point firstPoint = pointList.getPoint(0);
        sourcePoint.setX(firstPoint.x - (float) offset.x);
        sourcePoint.setY(firstPoint.y - (float) offset.y);

        for (int i = 1; i < pointList.size() - 1; i++) {
            Point point = pointList.getPoint(i);
            KPoint kpoint = KLayoutDataFactory.eINSTANCE.createKPoint();
            kpoint.setX(point.x - (float) offset.x);
            kpoint.setY(point.y - (float) offset.y);
            edgeLayout.getBendPoints().add(kpoint);
        }
        KPoint targetPoint = edgeLayout.getTargetPoint();
        Point lastPoint = pointList.getPoint(pointList.size() - 1);
        targetPoint.setX(lastPoint.x - (float) offset.x);
        targetPoint.setY(lastPoint.y - (float) offset.y);
        
        // the modification flag must initially be false
        ((KEdgeLayoutImpl) edgeLayout).resetModificationFlag();
    }

    /**
     * Process the labels of an edge.
     * 
     * @param mapping
     *            the layout mapping
     * @param connection
     *            the connection edit part
     * @param edge
     *            the layout edge
     * @param placement
     *            predefined placement for all labels, or {@code UNDEFINED} if the placement shall
     *            be derived from the edit part
     * @param offset
     *            the offset for coordinates
     */
    private void processEdgeLabels(final LayoutMapping<IGraphicalEditPart> mapping,
            final ConnectionEditPart connection, final KEdge edge,
            final EdgeLabelPlacement placement, final KVector offset) {
        VolatileLayoutConfig staticConfig = mapping.getProperty(STATIC_CONFIG);
        /*
         * ars: source and target is exchanged when defining it in the gmfgen file. So if Emma sets
         * a label to be placed as target on a connection, then the label will show up next to the
         * source node in the diagram editor. So correct it here, very ugly.
         */
        for (Object obj : connection.getChildren()) {
            if (obj instanceof LabelEditPart) {
                LabelEditPart labelEditPart = (LabelEditPart) obj;
                IFigure labelFigure = labelEditPart.getFigure();
                
                // Check if the label is visible in the first place
                if (labelFigure == null || !labelFigure.isVisible()) {
                    continue;
                }
                
                Rectangle labelBounds = getAbsoluteBounds(labelFigure);
                String labelText = null;
                Dimension iconBounds = null;
                
                if (labelFigure instanceof WrappingLabel) {
                    WrappingLabel wrappingLabel = (WrappingLabel) labelFigure;
                    labelText = wrappingLabel.getText();
                    if (wrappingLabel.getIcon() != null) {
                        iconBounds = new Dimension();
                        iconBounds.width = wrappingLabel.getIcon().getBounds().width
                                + wrappingLabel.getIconTextGap();
                        iconBounds.height = wrappingLabel.getIcon().getBounds().height;
                        // Add more characters to the text for layouters that need the text to
                        // determine the label size.
                        labelText = "O " + labelText;
                    }
                } else if (labelFigure instanceof Label) {
                    Label label = (Label) labelFigure;
                    labelText = label.getText();
                    if (label.getIcon() != null) {
                        iconBounds = label.getIconBounds().getSize();
                        iconBounds.width += label.getIconTextGap();
                        // Add more characters to the text for layouters that need the text to
                        // determine the label size.
                        labelText = "O " + labelText;
                    }
                }
                
                if (labelText != null && labelText.length() > 0) {
                    KLabel label = KimlUtil.createInitializedLabel(edge);
                    KShapeLayout labelLayout = label.getData(KShapeLayout.class);
                    if (placement == EdgeLabelPlacement.UNDEFINED) {
                        switch (labelEditPart.getKeyPoint()) {
                        case ConnectionLocator.SOURCE:
                            staticConfig.setValue(LayoutOptions.EDGE_LABEL_PLACEMENT, label,
                                    LayoutContext.GRAPH_ELEM, EdgeLabelPlacement.HEAD);
                            break;
                        case ConnectionLocator.MIDDLE:
                            staticConfig.setValue(LayoutOptions.EDGE_LABEL_PLACEMENT, label,
                                    LayoutContext.GRAPH_ELEM, EdgeLabelPlacement.CENTER);
                            break;
                        case ConnectionLocator.TARGET:
                            staticConfig.setValue(LayoutOptions.EDGE_LABEL_PLACEMENT, label,
                                    LayoutContext.GRAPH_ELEM, EdgeLabelPlacement.TAIL);
                            break;
                        }
                    } else {
                        staticConfig.setValue(LayoutOptions.EDGE_LABEL_PLACEMENT, label,
                                LayoutContext.GRAPH_ELEM, placement);
                    }
                    Font font = labelFigure.getFont();
                    if (font != null && !font.isDisposed()) {
                        staticConfig.setValue(LayoutOptions.FONT_NAME, label, LayoutContext.GRAPH_ELEM,
                                font.getFontData()[0].getName());
                        staticConfig.setValue(LayoutOptions.FONT_SIZE, label, LayoutContext.GRAPH_ELEM,
                                font.getFontData()[0].getHeight());
                    }
                    labelLayout.setXpos(labelBounds.x - (float) offset.x);
                    labelLayout.setYpos(labelBounds.y - (float) offset.y);
                    if (iconBounds != null) {
                        labelLayout.setWidth(labelBounds.width + iconBounds.width);
                    } else {
                        labelLayout.setWidth(labelBounds.width);
                    }
                    labelLayout.setHeight(labelBounds.height);
                    ((KShapeLayoutImpl) labelLayout).resetModificationFlag();
                    label.setText(labelText);
                    mapping.getGraphMap().put(label, labelEditPart);
                } else {
                    // add the label to the mapping anyway so it is reset to its reference location
                    KLabel label = KGraphFactory.eINSTANCE.createKLabel();
                    KShapeLayout labelLayout = KLayoutDataFactory.eINSTANCE.createKShapeLayout();
                    label.getData().add(labelLayout);
                    mapping.getGraphMap().put(label, labelEditPart);
                }
            }
        }
    }

    /**
     * Refreshes all ports in the diagram. This is necessary in order correctly move ports, which
     * does not work due to GMF bugs. See Eclipse bug #291484.
     * 
     * @param editor
     *            the diagram editor
     * @param rootPart
     *            the root edit part
     * @see https://bugs.eclipse.org/bugs/show_bug.cgi?id=291484
     */
    private static void refreshDiagram(final DiagramEditor editor, final IGraphicalEditPart rootPart) {
        EditPart editPart = rootPart;
        if (editPart == null) {
            editPart = editor.getDiagramEditPart();
        }
        for (Object obj : editPart.getViewer().getEditPartRegistry().values()) {
            if (obj instanceof ShapeNodeEditPart) {
                IFigure figure = ((ShapeNodeEditPart) obj).getFigure();
                if (figure instanceof BorderedNodeFigure) {
                    IFigure portContainer = ((BorderedNodeFigure) figure).getBorderItemContainer();
                    portContainer.invalidate();
                    portContainer.validate();
                }
            }
        }
    }

}
