/***********************************************************************
 * Copyright (c) 2004 Actuate Corporation.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0/.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 *
 * Contributors:
 * Actuate Corporation - initial API and implementation
 ***********************************************************************/

package org.eclipse.birt.report.engine.layout.pdf;

import org.eclipse.birt.core.exception.BirtException;
import org.eclipse.birt.report.engine.content.IContent;
import org.eclipse.birt.report.engine.content.IStyle;
import org.eclipse.birt.report.engine.extension.IReportItemExecutor;
import org.eclipse.birt.report.engine.layout.ILayoutContext;
import org.eclipse.birt.report.engine.layout.IStackingLayoutManager;
import org.eclipse.birt.report.engine.layout.area.impl.AbstractArea;
import org.eclipse.birt.report.engine.layout.area.impl.ContainerArea;
import org.eclipse.birt.report.engine.layout.pdf.util.PropertyUtil;

public abstract class PDFStackingLM extends PDFAbstractLM implements IStackingLayoutManager, ILayoutContext {

	protected int maxWidth = -1;

	protected int maxHeight = -1;

	protected int maxAvaHeight = 0;

	protected int maxAvaWidth = 0;

	protected int currentIP = 0;

	protected int currentBP = 0;

	protected int offsetX = 0;

	protected int offsetY = 0;

	protected ContainerArea root;

	protected PDFAbstractLM child;

	protected int minHeight = 0;

	protected int minWidth = 0;

	protected boolean pageBreakAvoid = false;

	public int getMaxAvaHeight() {
		return this.maxAvaHeight;
	}

	@Override
	public int getCurrentMaxContentWidth() {
		return maxAvaWidth - currentIP;// FIXME
	}

	@Override
	public int getCurrentMaxContentHeight() {
		return maxAvaHeight - currentBP;
	}

	@Override
	public int getCurrentIP() {
		return this.currentIP;
	}

	@Override
	public int getCurrentBP() {
		return this.currentBP;
	}

	@Override
	public void setCurrentBP(int bp) {
		this.currentBP = bp;
	}

	@Override
	public void setCurrentIP(int ip) {
		this.currentIP = ip;
	}

	@Override
	public int getOffsetX() {
		return offsetX;
	}

	@Override
	public void setOffsetX(int x) {
		this.offsetX = x;
	}

	@Override
	public int getOffsetY() {
		return offsetY;
	}

	@Override
	public void setOffsetY(int y) {
		this.offsetY = y;
	}

	public PDFStackingLM(PDFLayoutEngineContext context, PDFStackingLM parent, IContent content,
			IReportItemExecutor executor) {
		super(context, parent, content, executor);
		pageBreakAvoid = pageBreakInsideAvoid() || pageBreakAfterAvoid();
	}

	public boolean isPageBreakAvoidInside() {
		if (parent == null) {
			return false;
		} else if (parent.pageBreakAvoid) {
			return true;
		} else {
			return parent.isPageBreakAvoidInside();
		}
	}

	@Override
	protected boolean layoutChildren() throws BirtException {
		initialize();
		boolean hasNextPage;
		hasNextPage = traverseChildren();
		if (!hasNextPage) {
			isLast = true;
			clearCache();
		} else if (!context.isAutoPageBreak()) {
			clearCache();
		}
		// if ( !isRootEmpty( ) &&!(isPageBreakAvoidInside( )&&context.isAutoPageBreak(
		// )&&hasNextPage))
		if (!isRootEmpty()) {
			closeLayout();
			hasNextPage = !submitRoot() || hasNextPage;
		}
		return hasNextPage;
	}

	protected boolean clearCache() {
		return true;
	}

	protected boolean isRootEmpty() {
		return !(root != null && root.getChildrenCount() > 0);
	}

	protected abstract boolean traverseChildren() throws BirtException;

	/**
	 * submit the current layout result
	 *
	 * @return
	 */
	protected boolean submitRoot() {
		if (root == null) {
			return true;
		}
		boolean success = true;
		if (parent != null) {
			// FIXME to support keepWithPrevious
			success = parent.addArea(root, false, isKeepWithNext());
			// parent.submit( root );
			if (success) {
				isFirst = false;
				root = null;
			}
		} else if (content != null) {
			content.setExtension(IContent.LAYOUT_EXTENSION, root);
			root = null;
		}
		return success;
	}

	protected boolean isKeepWithNext() {
		return isLast && isFirst && pageBreakAfterAvoid();
	}

	/**
	 * initialize dynamic layout information
	 * <ul>
	 * <li>create root area</li>
	 * <li>set MaxAvaHeight and MaxAvaWidth</li>
	 * <li>set OffsetX and OffsetY</li>
	 * <li>set CurrentIP and CurrentBP</li>
	 * </ul>
	 *
	 * @throws BirtException
	 */
	protected abstract void initialize() throws BirtException;

	/**
	 * end current area if it is the last area of content, add bottom box property
	 *
	 */
	protected abstract void closeLayout();

	protected abstract void createRoot();

	protected int getIntrisicHeight() {
		if (root != null) {
			return root.getIntrisicHeight();
		}
		return 0;
	}

	@Override
	public boolean isPageEmpty() {
		if (!isRootEmpty()) {
			return false;
		} else if (parent != null) {
			return parent.isPageEmpty();
		}
		return true;
	}

	public int getLineHeight() {
		if (content != null) {
			IStyle contentStyle = content.getComputedStyle();
			return PropertyUtil.getLineHeight(contentStyle.getLineHeight());
		}
		return 0;
	}

	public abstract void submit(AbstractArea area);

}
