/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)mlib_s_SignalFFT_Tbl.c	9.3	07/11/05 SMI"

#include <mlib_signal.h>
#include <mlib_s_sse_int.h>

/* *********************************************************** */

#include <mlib_s_SignalFFTUtil.h>

/* *********************************************************** */

#define	_SIGNAL_FFT_TABLE

#include <mlib_s_SignalFFT_Tbl.h>
#include <mlib_s_SignalFFTDisp.h>

/* *********************************************************** */

const mlib_d64 mlib_s_fftK_SinCosTable[] = {
	0.0,
	1.0,
	0.707106781186547573,
	0.707106781186547462,
	0.923879532511286738,
	0.382683432365089782,
	0.980785280403230431,
	0.195090322016128248,
	0.995184726672196929,
	0.098017140329560604,
	0.998795456205172405,
	0.049067674327418015,
	0.999698818696204250,
	0.024541228522912288,
	0.999924701839144503,
	0.012271538285719925,
	0.999981175282601109,
	0.006135884649154475,
	0.999995293809576191,
	0.003067956762965976,
	0.999998823451701879,
	0.001533980186284766,
	0.999999705862882227,
	0.000766990318742704,
	0.999999926465717892,
	0.000383495187571396,
	0.999999981616429334,
	0.000191747597310703,
	0.999999995404107334,
	0.000095873799095977,
	0.999999998851026861,
	0.000047936899603067,
	0.999999999712756660,
	0.000023968449808418,
	0.999999999928189220,
	0.000011984224905070,
	0.999999999982047250,
	0.000005992112452642,
	0.999999999995511812,
	0.000002996056226335,
	0.999999999998878009,
	0.000001498028113169,
	0.999999999999719447,
	0.000000749014056585,
	0.999999999999929834,
	0.000000374507028292,
	0.999999999999982458,
	0.000000187253514146,
	0.999999999999995670,
	0.000000093626757073,
	0.999999999999998890,
	0.000000046813378537,
	0.999999999999999778,
	0.000000023406689268,
	0.999999999999999889,
	0.000000011703344634,
	1.000000000000000000,
	0.000000005851672317,
	1.000000000000000000,
	0.000000002925836159
};

/* *********************************************************** */

typedef union
{
	mlib_d64 value;
	struct
	{
		mlib_s32 int0, int1;
	} twoint;
} type_mlib_d64;

/* *********************************************************** */

#define	D64_TO_S32(val)	((mlib_s32)(val))

/* *********************************************************** */

void
mlib_s_fftK_CalcTbl(
    void *buff,
    mlib_s32 order,
    mlib_s32 isign)
{
	mlib_s32 layer = order;
	mlib_d64 *tbl = (mlib_d64 *)buff + (1 << (layer - 2));
	mlib_d64 wp_r, wp_i, w_r, w_i, dw0_r, dw1_r, dw2_r;
	mlib_d64 dw3_r, dw0_i, dw1_i, dw2_i, dw3_i;
	mlib_s32 j, p;

	p = (order - 2) << 1;

	dw1_r = mlib_s_fftK_SinCosTable[p] * 2147418112.0;
	dw1_i = isign * mlib_s_fftK_SinCosTable[p + 1] * 2147418112.0;

	dw2_r = mlib_s_fftK_SinCosTable[p - 2];
	dw2_i = isign * mlib_s_fftK_SinCosTable[p - 1];

	dw3_r = (dw2_r * dw1_r) - (dw2_i * dw1_i);
	dw3_i = (dw2_i * dw1_r) + (dw2_r * dw1_i);

	dw2_r *= 2147418112.0;
	dw2_i *= 2147418112.0;

	dw0_r = 2147418112.0;
	dw0_i = 0;

	wp_r = mlib_s_fftK_SinCosTable[p - 4];
	wp_i = isign * mlib_s_fftK_SinCosTable[p - 3];
	{
		type_mlib_d64 str0, str1;

		str0.twoint.int0 = D64_TO_S32(dw0_r);
		str0.twoint.int1 = D64_TO_S32(dw1_r);
		str1.twoint.int0 = D64_TO_S32(dw2_r);
		str1.twoint.int1 = D64_TO_S32(dw3_r);
		*((mlib_s16 *)&w_r + 0) = str0.twoint.int0 >> 16;
		*((mlib_s16 *)&w_r + 1) = str0.twoint.int1 >> 16;
		*((mlib_s16 *)&w_r + 2) = str1.twoint.int0 >> 16;
		*((mlib_s16 *)&w_r + 3) = str1.twoint.int1 >> 16;
		str0.twoint.int0 = D64_TO_S32(dw0_i);
		str0.twoint.int1 = D64_TO_S32(dw1_i);
		str1.twoint.int0 = D64_TO_S32(dw2_i);
		str1.twoint.int1 = D64_TO_S32(dw3_i);
		*((mlib_s16 *)&w_i + 0) = str0.twoint.int0 >> 16;
		*((mlib_s16 *)&w_i + 1) = str0.twoint.int1 >> 16;
		*((mlib_s16 *)&w_i + 2) = str1.twoint.int0 >> 16;
		*((mlib_s16 *)&w_i + 3) = str1.twoint.int1 >> 16;
	}

	for (j = 0; j < (1 << (order - 3)); j++) {
		mlib_d64 tmp0_r, tmp1_r, tmp2_r, tmp3_r;
		type_mlib_d64 str0, str1;

		tmp0_r = dw0_r;
		dw0_r = (dw0_r * wp_r) - (dw0_i * wp_i);
		tmp1_r = dw1_r;
		dw1_r = (dw1_r * wp_r) - (dw1_i * wp_i);
		tmp2_r = dw2_r;
		dw2_r = (dw2_r * wp_r) - (dw2_i * wp_i);
		tmp3_r = dw3_r;
		dw3_r = (dw3_r * wp_r) - (dw3_i * wp_i);

		tbl[2 * j] = w_r; /* 0, 2, 4, 6 ... */
		tbl[2 * j + 1] = w_i; /* 1, 3, 5, 7 ... */

		str0.twoint.int0 = D64_TO_S32(dw0_r);
		str0.twoint.int1 = D64_TO_S32(dw1_r);
		str1.twoint.int0 = D64_TO_S32(dw2_r);
		str1.twoint.int1 = D64_TO_S32(dw3_r);
		*((mlib_s16 *)&w_r + 0) = str0.twoint.int0 >> 16;
		*((mlib_s16 *)&w_r + 1) = str0.twoint.int1 >> 16;
		*((mlib_s16 *)&w_r + 2) = str1.twoint.int0 >> 16;
		*((mlib_s16 *)&w_r + 3) = str1.twoint.int1 >> 16;

		dw0_i = (dw0_i * wp_r) + (tmp0_r * wp_i);
		dw1_i = (dw1_i * wp_r) + (tmp1_r * wp_i);
		dw2_i = (dw2_i * wp_r) + (tmp2_r * wp_i);
		dw3_i = (dw3_i * wp_r) + (tmp3_r * wp_i);

		str0.twoint.int0 = D64_TO_S32(dw0_i);
		str0.twoint.int1 = D64_TO_S32(dw1_i);
		str1.twoint.int0 = D64_TO_S32(dw2_i);
		str1.twoint.int1 = D64_TO_S32(dw3_i);
		*((mlib_s16 *)&w_i + 0) = str0.twoint.int0 >> 16;
		*((mlib_s16 *)&w_i + 1) = str0.twoint.int1 >> 16;
		*((mlib_s16 *)&w_i + 2) = str1.twoint.int0 >> 16;
		*((mlib_s16 *)&w_i + 3) = str1.twoint.int1 >> 16;
	}

	for (layer = order - 1; layer > FFT_VIS_TBL_ORDER; layer--) {
		mlib_d64 *tbl1 = (mlib_d64 *)buff + (1 << (layer - 2));
		mlib_s32 tscale = 4 + (order - layer);
		mlib_s32 t_off = 4 << (order - layer);
		mlib_s32 j, off1, off2, off3;

		if ((order - layer) > 1) {
			off1 = t_off;
			off2 = 2 * t_off;
		} else {
			off1 = 4;
			off2 = 16;
		}

		off3 = off1 + off2;

#pragma pipeloop(0)
		for (j = 0; j < (1 << (layer - 3)); j++) {
			mlib_u8 *ptbl;
			mlib_s16 *tbl10, *tbl11;

			/* reminder: should be accelerated with SSE2 later */
			ptbl = (mlib_u8 *)tbl + (j << tscale);
			tbl10 = (mlib_s16 *)(&(tbl1[2 * j]));
			tbl11 = (mlib_s16 *)(&(tbl1[2 * j + 1]));

			tbl10[3] = *(mlib_s16 *)(ptbl + off3);
			tbl10[2] = *(mlib_s16 *)(ptbl + off2);
			tbl10[1] = *(mlib_s16 *)(ptbl + off1);
			tbl10[0] = *(mlib_s16 *)(ptbl + 0);

			tbl11[3] = *(mlib_s16 *)(ptbl + off3 + 8);
			tbl11[2] = *(mlib_s16 *)(ptbl + off3 + 8);
			tbl11[1] = *(mlib_s16 *)(ptbl + off3 + 8);
			tbl11[0] = *(mlib_s16 *)(ptbl + 8);

		}
	}



	/* reminder: following code is added for table format conversion */
	/* performance will be considered later */
	tbl = (mlib_d64 *)buff;
	mlib_d64 temp;
	for (j = 0; j < (1 << (order - 3)); j++) {
		temp = tbl[4 * j + 1];
		tbl[4 * j + 1] = tbl[4 * j + 2];
		tbl[4 * j + 2] = temp;
	}
}

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_1_M[] = {
	0x00000000, 0x00000000, 0x5a817fff, 0xa57e0000,
	0x00000000, 0x00000000, 0xa57e0000, 0xa57e8001,
	0x76407fff, 0x30fb5a81, 0xcf040000, 0x89bfa57e,
	0xcf040000, 0x89bfa57e, 0x89bf8001, 0xcf04a57e,
	0x7d897fff, 0x6a6c7640, 0x471c5a81, 0x18f830fb,
	0xe7070000, 0xb8e3cf04, 0x9593a57e, 0x827689bf,
	0xe7070000, 0xb8e3cf04, 0x9593a57e, 0x827689bf,
	0x82768001, 0x959389bf, 0xb8e3a57e, 0xe707cf04,
	0x7f617fff, 0x7a7c7d89, 0x70e17640, 0x62f16a6c,
	0xf3740000, 0xdad8e707, 0xc3a9cf04, 0xaeccb8e3,
	0x51335a81, 0x3c56471c, 0x252730fb, 0x0c8b18f8,
	0x9d0ea57e, 0x8f1e9593, 0x858389bf, 0x809e8276,
	0xf3740000, 0xdad8e707, 0xc3a9cf04, 0xaeccb8e3,
	0x809e8001, 0x85838276, 0x8f1e89bf, 0x9d0e9593,
	0x9d0ea57e, 0x8f1e9593, 0x858389bf, 0x809e8276,
	0xaecca57e, 0xc3a9b8e3, 0xdad8cf04, 0xf374e707,
	0x7fd77fff, 0x7e9c7f61, 0x7c297d89, 0x78837a7c,
	0xf9b80000, 0xed38f374, 0xe0e6e707, 0xd4e1dad8,
	0x73b57640, 0x6dc970e1, 0x66ce6a6c, 0x5ed662f1,
	0xc946cf04, 0xbe32c3a9, 0xb3c0b8e3, 0xaa0baecc,
	0x55f45a81, 0x4c3f5133, 0x41cd471c, 0x36b93c56,
	0xa129a57e, 0x99319d0e, 0x92369593, 0x8c4a8f1e,
	0x2b1e30fb, 0x1f192527, 0x12c718f8, 0x06470c8b,
	0x877c89bf, 0x83d68583, 0x81638276, 0x8028809e,
	0xf9b80000, 0xed38f374, 0xe0e6e707, 0xd4e1dad8,
	0x80288001, 0x8163809e, 0x83d68276, 0x877c8583,
	0xc946cf04, 0xbe32c3a9, 0xb3c0b8e3, 0xaa0baecc,
	0x8c4a89bf, 0x92368f1e, 0x99319593, 0xa1299d0e,
	0xa129a57e, 0x99319d0e, 0x92369593, 0x8c4a8f1e,
	0xaa0ba57e, 0xb3c0aecc, 0xbe32b8e3, 0xc946c3a9,
	0x877c89bf, 0x83d68583, 0x81638276, 0x8028809e,
	0xd4e1cf04, 0xe0e6dad8, 0xed38e707, 0xf9b8f374,
	0x7ff57fff, 0x7fa67fd7, 0x7f087f61, 0x7e1c7e9c,
	0xfcdb0000, 0xf695f9b8, 0xf054f374, 0xea1eed38,
	0x7ce27d89, 0x7b5c7c29, 0x79897a7c, 0x776b7883,
	0xe3f4e707, 0xdddce0e6, 0xd7d9dad8, 0xd1efd4e1,
	0x75037640, 0x725473b5, 0x6f5e70e1, 0x6c236dc9,
	0xcc21cf04, 0xc673c946, 0xc0e9c3a9, 0xbb85be32,
	0x68a56a6c, 0x64e766ce, 0x60eb62f1, 0x5cb35ed6,
	0xb64cb8e3, 0xb140b3c0, 0xac65aecc, 0xa7bdaa0b,
	0x58425a81, 0x539a55f4, 0x4ebf5133, 0x49b34c3f,
	0xa34ca57e, 0x9f14a129, 0x9b189d0e, 0x975a9931,
	0x447a471c, 0x3f1641cd, 0x398c3c56, 0x33de36b9,
	0x93dc9593, 0x90a19236, 0x8dab8f1e, 0x8afc8c4a,
	0x2e1030fb, 0x28262b1e, 0x22232527, 0x1c0b1f19,
	0x889489bf, 0x8676877c, 0x84a38583, 0x831d83d6,
	0x15e118f8, 0x0fab12c7, 0x096a0c8b, 0x03240647,
	0x81e38276, 0x80f78163, 0x8059809e, 0x800a8028,
	0xfcdb0000, 0xf695f9b8, 0xf054f374, 0xea1eed38,
	0x800a8001, 0x80598028, 0x80f7809e, 0x81e38163,
	0xe3f4e707, 0xdddce0e6, 0xd7d9dad8, 0xd1efd4e1,
	0x831d8276, 0x84a383d6, 0x86768583, 0x8894877c,
	0xcc21cf04, 0xc673c946, 0xc0e9c3a9, 0xbb85be32,
	0x8afc89bf, 0x8dab8c4a, 0x90a18f1e, 0x93dc9236,
	0xb64cb8e3, 0xb140b3c0, 0xac65aecc, 0xa7bdaa0b,
	0x975a9593, 0x9b189931, 0x9f149d0e, 0xa34ca129,
	0xa34ca57e, 0x9f14a129, 0x9b189d0e, 0x975a9931,
	0xa7bda57e, 0xac65aa0b, 0xb140aecc, 0xb64cb3c0,
	0x93dc9593, 0x90a19236, 0x8dab8f1e, 0x8afc8c4a,
	0xbb85b8e3, 0xc0e9be32, 0xc673c3a9, 0xcc21c946,
	0x889489bf, 0x8676877c, 0x84a38583, 0x831d83d6,
	0xd1efcf04, 0xd7d9d4e1, 0xdddcdad8, 0xe3f4e0e6,
	0x81e38276, 0x80f78163, 0x8059809e, 0x800a8028,
	0xea1ee707, 0xf054ed38, 0xf695f374, 0xfcdbf9b8,
	0x7ffc7fff, 0x7fe87ff5, 0x7fc17fd7, 0x7f867fa6,
	0xfe6d0000, 0xfb49fcdb, 0xf826f9b8, 0xf504f695,
	0x7f377f61, 0x7ed47f08, 0x7e5e7e9c, 0x7dd57e1c,
	0xf1e4f374, 0xeec6f054, 0xebaaed38, 0xe892ea1e,
	0x7d387d89, 0x7c887ce2, 0x7bc47c29, 0x7aee7b5c,
	0xe57de707, 0xe26ce3f4, 0xdf60e0e6, 0xdc59dddc,
	0x7a047a7c, 0x79087989, 0x77f97883, 0x76d8776b,
	0xd958dad8, 0xd65cd7d9, 0xd367d4e1, 0xd079d1ef,
	0x75a47640, 0x745e7503, 0x730673b5, 0x719d7254,
	0xcd92cf04, 0xcab2cc21, 0xc7dbc946, 0xc50dc673,
	0x702270e1, 0x6e956f5e, 0x6cf86dc9, 0x6b4a6c23,
	0xc248c3a9, 0xbf8cc0e9, 0xbcdabe32, 0xba33bb85,
	0x698b6a6c, 0x67bc68a5, 0x65dd66ce, 0x63ee64e7,
	0xb796b8e3, 0xb505b64c, 0xb27fb3c0, 0xb005b140,
	0x61f062f1, 0x5fe260eb, 0x5dc65ed6, 0x5b9c5cb3,
	0xad97aecc, 0xab36ac65, 0xa8e2aa0b, 0xa69ca7bd,
	0x59635a81, 0x571d5842, 0x54c955f4, 0x5268539a,
	0xa463a57e, 0xa239a34c, 0xa01da129, 0x9e0f9f14,
	0x4ffa5133, 0x4d804ebf, 0x4afa4c3f, 0x486949b3,
	0x9c119d0e, 0x9a229b18, 0x98439931, 0x9674975a,
	0x45cc471c, 0x4325447a, 0x407341cd, 0x3db73f16,
	0x94b59593, 0x930793dc, 0x916a9236, 0x8fdd90a1,
	0x3af23c56, 0x3824398c, 0x354d36b9, 0x326d33de,
	0x8e628f1e, 0x8cf98dab, 0x8ba18c4a, 0x8a5b8afc,
	0x2f8630fb, 0x2c982e10, 0x29a32b1e, 0x26a72826,
	0x892789bf, 0x88068894, 0x86f7877c, 0x85fb8676,
	0x23a62527, 0x209f2223, 0x1d931f19, 0x1a821c0b,
	0x85118583, 0x843b84a3, 0x837783d6, 0x82c7831d,
	0x176d18f8, 0x145515e1, 0x113912c7, 0x0e1b0fab,
	0x822a8276, 0x81a181e3, 0x812b8163, 0x80c880f7,
	0x0afb0c8b, 0x07d9096a, 0x04b60647, 0x01920324,
	0x8079809e, 0x803e8059, 0x80178028, 0x8003800a,
	0xfe6d0000, 0xfb49fcdb, 0xf826f9b8, 0xf504f695,
	0x80038001, 0x8017800a, 0x803e8028, 0x80798059,
	0xf1e4f374, 0xeec6f054, 0xebaaed38, 0xe892ea1e,
	0x80c8809e, 0x812b80f7, 0x81a18163, 0x822a81e3,
	0xe57de707, 0xe26ce3f4, 0xdf60e0e6, 0xdc59dddc,
	0x82c78276, 0x8377831d, 0x843b83d6, 0x851184a3,
	0xd958dad8, 0xd65cd7d9, 0xd367d4e1, 0xd079d1ef,
	0x85fb8583, 0x86f78676, 0x8806877c, 0x89278894,
	0xcd92cf04, 0xcab2cc21, 0xc7dbc946, 0xc50dc673,
	0x8a5b89bf, 0x8ba18afc, 0x8cf98c4a, 0x8e628dab,
	0xc248c3a9, 0xbf8cc0e9, 0xbcdabe32, 0xba33bb85,
	0x8fdd8f1e, 0x916a90a1, 0x93079236, 0x94b593dc,
	0xb796b8e3, 0xb505b64c, 0xb27fb3c0, 0xb005b140,
	0x96749593, 0x9843975a, 0x9a229931, 0x9c119b18,
	0xad97aecc, 0xab36ac65, 0xa8e2aa0b, 0xa69ca7bd,
	0x9e0f9d0e, 0xa01d9f14, 0xa239a129, 0xa463a34c,
	0xa463a57e, 0xa239a34c, 0xa01da129, 0x9e0f9f14,
	0xa69ca57e, 0xa8e2a7bd, 0xab36aa0b, 0xad97ac65,
	0x9c119d0e, 0x9a229b18, 0x98439931, 0x9674975a,
	0xb005aecc, 0xb27fb140, 0xb505b3c0, 0xb796b64c,
	0x94b59593, 0x930793dc, 0x916a9236, 0x8fdd90a1,
	0xba33b8e3, 0xbcdabb85, 0xbf8cbe32, 0xc248c0e9,
	0x8e628f1e, 0x8cf98dab, 0x8ba18c4a, 0x8a5b8afc,
	0xc50dc3a9, 0xc7dbc673, 0xcab2c946, 0xcd92cc21,
	0x892789bf, 0x88068894, 0x86f7877c, 0x85fb8676,
	0xd079cf04, 0xd367d1ef, 0xd65cd4e1, 0xd958d7d9,
	0x85118583, 0x843b84a3, 0x837783d6, 0x82c7831d,
	0xdc59dad8, 0xdf60dddc, 0xe26ce0e6, 0xe57de3f4,
	0x822a8276, 0x81a181e3, 0x812b8163, 0x80c880f7,
	0xe892e707, 0xebaaea1e, 0xeec6ed38, 0xf1e4f054,
	0x8079809e, 0x803e8059, 0x80178028, 0x8003800a,
	0xf504f374, 0xf826f695, 0xfb49f9b8, 0xfe6dfcdb,
	0x7ffe7fff, 0x7ff97ffc, 0x7fef7ff5, 0x7fe07fe8,
	0xff360000, 0xfda4fe6d, 0xfc12fcdb, 0xfa81fb49,
	0x7fcd7fd7, 0x7fb47fc1, 0x7f967fa6, 0x7f747f86,
	0xf8eff9b8, 0xf75ef826, 0xf5cdf695, 0xf43cf504,
	0x7f4c7f61, 0x7f207f37, 0x7eef7f08, 0x7eb97ed4,
	0xf2acf374, 0xf11cf1e4, 0xef8df054, 0xedffeec6,
	0x7e7e7e9c, 0x7e3e7e5e, 0x7df97e1c, 0x7db07dd5,
	0xec71ed38, 0xeae4ebaa, 0xe958ea1e, 0xe7cce892,
	0x7d617d89, 0x7d0e7d38, 0x7cb67ce2, 0x7c597c88,
	0xe642e707, 0xe4b9e57d, 0xe330e3f4, 0xe1a9e26c,
	0x7bf77c29, 0x7b917bc4, 0x7b257b5c, 0x7ab57aee,
	0xe023e0e6, 0xde9edf60, 0xdd1bdddc, 0xdb98dc59,
	0x7a417a7c, 0x79c77a04, 0x79497989, 0x78c67908,
	0xda18dad8, 0xd898d958, 0xd71ad7d9, 0xd59ed65c,
	0x783f7883, 0x77b377f9, 0x7722776b, 0x768d76d8,
	0xd424d4e1, 0xd2abd367, 0xd133d1ef, 0xcfbed079,
	0x75f37640, 0x755475a4, 0x74b17503, 0x740a745e,
	0xce4bcf04, 0xccd9cd92, 0xcb69cc21, 0xc9fccab2,
	0x735e73b5, 0x72ae7306, 0x71f97254, 0x7140719d,
	0xc890c946, 0xc727c7db, 0xc5c0c673, 0xc45bc50d,
	0x708270e1, 0x6fc07022, 0x6efa6f5e, 0x6e306e95,
	0xc2f8c3a9, 0xc198c248, 0xc03ac0e9, 0xbedfbf8c,
	0x6d616dc9, 0x6c8e6cf8, 0x6bb76c23, 0x6adb6b4a,
	0xbd86be32, 0xbc2fbcda, 0xbadcbb85, 0xb98bba33,
	0x69fc6a6c, 0x6919698b, 0x683168a5, 0x674567bc,
	0xb83cb8e3, 0xb6f1b796, 0xb5a8b64c, 0xb462b505,
	0x665666ce, 0x656265dd, 0x646b64e7, 0x637063ee,
	0xb31fb3c0, 0xb1dfb27f, 0xb0a2b140, 0xaf68b005,
	0x627162f1, 0x616e61f0, 0x606760eb, 0x5f5d5fe2,
	0xae31aecc, 0xacfead97, 0xabcdac65, 0xaaa0ab36,
	0x5e4f5ed6, 0x5d3d5dc6, 0x5c285cb3, 0x5b0f5b9c,
	0xa976aa0b, 0xa84fa8e2, 0xa72ca7bd, 0xa60ca69c,
	0x59f35a81, 0x58d35963, 0x57b05842, 0x5689571d,
	0xa4f0a57e, 0xa3d7a463, 0xa2c2a34c, 0xa1b0a239,
	0x555f55f4, 0x543254c9, 0x5301539a, 0x51ce5268,
	0xa0a2a129, 0x9f98a01d, 0x9e919f14, 0x9d8e9e0f,
	0x50975133, 0x4f5d4ffa, 0x4e204ebf, 0x4ce04d80,
	0x9c8f9d0e, 0x9b949c11, 0x9a9d9b18, 0x99a99a22,
	0x4b9d4c3f, 0x4a574afa, 0x490e49b3, 0x47c34869,
	0x98ba9931, 0x97ce9843, 0x96e6975a, 0x96039674,
	0x4674471c, 0x452345cc, 0x43d0447a, 0x42794325,
	0x95249593, 0x944894b5, 0x937193dc, 0x929e9307,
	0x412041cd, 0x3fc54073, 0x3e673f16, 0x3d073db7,
	0x91cf9236, 0x9105916a, 0x903f90a1, 0x8f7d8fdd,
	0x3ba43c56, 0x3a3f3af2, 0x38d8398c, 0x376f3824,
	0x8ebf8f1e, 0x8e068e62, 0x8d518dab, 0x8ca18cf9,
	0x360336b9, 0x3496354d, 0x332633de, 0x31b4326d,
	0x8bf58c4a, 0x8b4e8ba1, 0x8aab8afc, 0x8a0c8a5b,
	0x304130fb, 0x2ecc2f86, 0x2d542e10, 0x2bdb2c98,
	0x897289bf, 0x88dd8927, 0x884c8894, 0x87c08806,
	0x2a612b1e, 0x28e529a3, 0x27672826, 0x25e726a7,
	0x8739877c, 0x86b686f7, 0x86388676, 0x85be85fb,
	0x24672527, 0x22e423a6, 0x21612223, 0x1fdc209f,
	0x854a8583, 0x84da8511, 0x846e84a3, 0x8408843b,
	0x1e561f19, 0x1ccf1d93, 0x1b461c0b, 0x19bd1a82,
	0x83a683d6, 0x83498377, 0x82f1831d, 0x829e82c7,
	0x183318f8, 0x16a7176d, 0x151b15e1, 0x138e1455,
	0x824f8276, 0x8206822a, 0x81c181e3, 0x818181a1,
	0x120012c7, 0x10721139, 0x0ee30fab, 0x0d530e1b,
	0x81468163, 0x8110812b, 0x80df80f7, 0x80b380c8,
	0x0bc30c8b, 0x0a320afb, 0x08a1096a, 0x071007d9,
	0x808b809e, 0x80698079, 0x804b8059, 0x8032803e,
	0x057e0647, 0x03ed04b6, 0x025b0324, 0x00c90192,
	0x801f8028, 0x80108017, 0x8006800a, 0x80018003,
	0xff360000, 0xfda4fe6d, 0xfc12fcdb, 0xfa81fb49,
	0x80018001, 0x80068003, 0x8010800a, 0x801f8017,
	0xf8eff9b8, 0xf75ef826, 0xf5cdf695, 0xf43cf504,
	0x80328028, 0x804b803e, 0x80698059, 0x808b8079,
	0xf2acf374, 0xf11cf1e4, 0xef8df054, 0xedffeec6,
	0x80b3809e, 0x80df80c8, 0x811080f7, 0x8146812b,
	0xec71ed38, 0xeae4ebaa, 0xe958ea1e, 0xe7cce892,
	0x81818163, 0x81c181a1, 0x820681e3, 0x824f822a,
	0xe642e707, 0xe4b9e57d, 0xe330e3f4, 0xe1a9e26c,
	0x829e8276, 0x82f182c7, 0x8349831d, 0x83a68377,
	0xe023e0e6, 0xde9edf60, 0xdd1bdddc, 0xdb98dc59,
	0x840883d6, 0x846e843b, 0x84da84a3, 0x854a8511,
	0xda18dad8, 0xd898d958, 0xd71ad7d9, 0xd59ed65c,
	0x85be8583, 0x863885fb, 0x86b68676, 0x873986f7,
	0xd424d4e1, 0xd2abd367, 0xd133d1ef, 0xcfbed079,
	0x87c0877c, 0x884c8806, 0x88dd8894, 0x89728927,
	0xce4bcf04, 0xccd9cd92, 0xcb69cc21, 0xc9fccab2,
	0x8a0c89bf, 0x8aab8a5b, 0x8b4e8afc, 0x8bf58ba1,
	0xc890c946, 0xc727c7db, 0xc5c0c673, 0xc45bc50d,
	0x8ca18c4a, 0x8d518cf9, 0x8e068dab, 0x8ebf8e62,
	0xc2f8c3a9, 0xc198c248, 0xc03ac0e9, 0xbedfbf8c,
	0x8f7d8f1e, 0x903f8fdd, 0x910590a1, 0x91cf916a,
	0xbd86be32, 0xbc2fbcda, 0xbadcbb85, 0xb98bba33,
	0x929e9236, 0x93719307, 0x944893dc, 0x952494b5,
	0xb83cb8e3, 0xb6f1b796, 0xb5a8b64c, 0xb462b505,
	0x96039593, 0x96e69674, 0x97ce975a, 0x98ba9843,
	0xb31fb3c0, 0xb1dfb27f, 0xb0a2b140, 0xaf68b005,
	0x99a99931, 0x9a9d9a22, 0x9b949b18, 0x9c8f9c11,
	0xae31aecc, 0xacfead97, 0xabcdac65, 0xaaa0ab36,
	0x9d8e9d0e, 0x9e919e0f, 0x9f989f14, 0xa0a2a01d,
	0xa976aa0b, 0xa84fa8e2, 0xa72ca7bd, 0xa60ca69c,
	0xa1b0a129, 0xa2c2a239, 0xa3d7a34c, 0xa4f0a463,
	0xa4f0a57e, 0xa3d7a463, 0xa2c2a34c, 0xa1b0a239,
	0xa60ca57e, 0xa72ca69c, 0xa84fa7bd, 0xa976a8e2,
	0xa0a2a129, 0x9f98a01d, 0x9e919f14, 0x9d8e9e0f,
	0xaaa0aa0b, 0xabcdab36, 0xacfeac65, 0xae31ad97,
	0x9c8f9d0e, 0x9b949c11, 0x9a9d9b18, 0x99a99a22,
	0xaf68aecc, 0xb0a2b005, 0xb1dfb140, 0xb31fb27f,
	0x98ba9931, 0x97ce9843, 0x96e6975a, 0x96039674,
	0xb462b3c0, 0xb5a8b505, 0xb6f1b64c, 0xb83cb796,
	0x95249593, 0x944894b5, 0x937193dc, 0x929e9307,
	0xb98bb8e3, 0xbadcba33, 0xbc2fbb85, 0xbd86bcda,
	0x91cf9236, 0x9105916a, 0x903f90a1, 0x8f7d8fdd,
	0xbedfbe32, 0xc03abf8c, 0xc198c0e9, 0xc2f8c248,
	0x8ebf8f1e, 0x8e068e62, 0x8d518dab, 0x8ca18cf9,
	0xc45bc3a9, 0xc5c0c50d, 0xc727c673, 0xc890c7db,
	0x8bf58c4a, 0x8b4e8ba1, 0x8aab8afc, 0x8a0c8a5b,
	0xc9fcc946, 0xcb69cab2, 0xccd9cc21, 0xce4bcd92,
	0x897289bf, 0x88dd8927, 0x884c8894, 0x87c08806,
	0xcfbecf04, 0xd133d079, 0xd2abd1ef, 0xd424d367,
	0x8739877c, 0x86b686f7, 0x86388676, 0x85be85fb,
	0xd59ed4e1, 0xd71ad65c, 0xd898d7d9, 0xda18d958,
	0x854a8583, 0x84da8511, 0x846e84a3, 0x8408843b,
	0xdb98dad8, 0xdd1bdc59, 0xde9edddc, 0xe023df60,
	0x83a683d6, 0x83498377, 0x82f1831d, 0x829e82c7,
	0xe1a9e0e6, 0xe330e26c, 0xe4b9e3f4, 0xe642e57d,
	0x824f8276, 0x8206822a, 0x81c181e3, 0x818181a1,
	0xe7cce707, 0xe958e892, 0xeae4ea1e, 0xec71ebaa,
	0x81468163, 0x8110812b, 0x80df80f7, 0x80b380c8,
	0xedffed38, 0xef8deec6, 0xf11cf054, 0xf2acf1e4,
	0x808b809e, 0x80698079, 0x804b8059, 0x8032803e,
	0xf43cf374, 0xf5cdf504, 0xf75ef695, 0xf8eff826,
	0x801f8028, 0x80108017, 0x8006800a, 0x80018003,
	0xfa81f9b8, 0xfc12fb49, 0xfda4fcdb, 0xff36fe6d,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_1_P[] = {
	0x00000000, 0x00000000, 0x5a817fff, 0xa57e0000,
	0x00000000, 0x00000000, 0x5a810000, 0x5a817fff,
	0x76407fff, 0x30fb5a81, 0xcf040000, 0x89bfa57e,
	0x30fb0000, 0x76405a81, 0x76407fff, 0x30fb5a81,
	0x7d897fff, 0x6a6c7640, 0x471c5a81, 0x18f830fb,
	0x18f80000, 0x471c30fb, 0x6a6c5a81, 0x7d897640,
	0xe7070000, 0xb8e3cf04, 0x9593a57e, 0x827689bf,
	0x7d897fff, 0x6a6c7640, 0x471c5a81, 0x18f830fb,
	0x7f617fff, 0x7a7c7d89, 0x70e17640, 0x62f16a6c,
	0x0c8b0000, 0x252718f8, 0x3c5630fb, 0x5133471c,
	0x51335a81, 0x3c56471c, 0x252730fb, 0x0c8b18f8,
	0x62f15a81, 0x70e16a6c, 0x7a7c7640, 0x7f617d89,
	0xf3740000, 0xdad8e707, 0xc3a9cf04, 0xaeccb8e3,
	0x7f617fff, 0x7a7c7d89, 0x70e17640, 0x62f16a6c,
	0x9d0ea57e, 0x8f1e9593, 0x858389bf, 0x809e8276,
	0x51335a81, 0x3c56471c, 0x252730fb, 0x0c8b18f8,
	0x7fd77fff, 0x7e9c7f61, 0x7c297d89, 0x78837a7c,
	0x06470000, 0x12c70c8b, 0x1f1918f8, 0x2b1e2527,
	0x73b57640, 0x6dc970e1, 0x66ce6a6c, 0x5ed662f1,
	0x36b930fb, 0x41cd3c56, 0x4c3f471c, 0x55f45133,
	0x55f45a81, 0x4c3f5133, 0x41cd471c, 0x36b93c56,
	0x5ed65a81, 0x66ce62f1, 0x6dc96a6c, 0x73b570e1,
	0x2b1e30fb, 0x1f192527, 0x12c718f8, 0x06470c8b,
	0x78837640, 0x7c297a7c, 0x7e9c7d89, 0x7fd77f61,
	0xf9b80000, 0xed38f374, 0xe0e6e707, 0xd4e1dad8,
	0x7fd77ffe, 0x7e9c7f61, 0x7c297d89, 0x78837a7c,
	0xc946cf04, 0xbe32c3a9, 0xb3c0b8e3, 0xaa0baecc,
	0x73b57640, 0x6dc970e1, 0x66ce6a6c, 0x5ed662f1,
	0xa129a57e, 0x99319d0e, 0x92369593, 0x8c4a8f1e,
	0x55f45a81, 0x4c3f5133, 0x41cd471c, 0x36b93c56,
	0x877c89bf, 0x83d68583, 0x81638276, 0x8028809e,
	0x2b1e30fb, 0x1f192527, 0x12c718f8, 0x06470c8b,
	0x7ff57fff, 0x7fa67fd7, 0x7f087f61, 0x7e1c7e9c,
	0x03240000, 0x096a0647, 0x0fab0c8b, 0x15e112c7,
	0x7ce27d89, 0x7b5c7c29, 0x79897a7c, 0x776b7883,
	0x1c0b18f8, 0x22231f19, 0x28262527, 0x2e102b1e,
	0x75037640, 0x725473b5, 0x6f5e70e1, 0x6c236dc9,
	0x33de30fb, 0x398c36b9, 0x3f163c56, 0x447a41cd,
	0x68a56a6c, 0x64e766ce, 0x60eb62f1, 0x5cb35ed6,
	0x49b3471c, 0x4ebf4c3f, 0x539a5133, 0x584255f4,
	0x58425a81, 0x539a55f4, 0x4ebf5133, 0x49b34c3f,
	0x5cb35a81, 0x60eb5ed6, 0x64e762f1, 0x68a566ce,
	0x447a471c, 0x3f1641cd, 0x398c3c56, 0x33de36b9,
	0x6c236a6c, 0x6f5e6dc9, 0x725470e1, 0x750373b5,
	0x2e1030fb, 0x28262b1e, 0x22232527, 0x1c0b1f19,
	0x776b7640, 0x79897883, 0x7b5c7a7c, 0x7ce27c29,
	0x15e118f8, 0x0fab12c7, 0x096a0c8b, 0x03240647,
	0x7e1c7d89, 0x7f087e9c, 0x7fa67f61, 0x7ff57fd7,
	0xfcdb0000, 0xf695f9b8, 0xf054f374, 0xea1eed38,
	0x7ff57fff, 0x7fa67fd7, 0x7f087f61, 0x7e1c7e9c,
	0xe3f4e707, 0xdddce0e6, 0xd7d9dad8, 0xd1efd4e1,
	0x7ce27d89, 0x7b5c7c29, 0x79897a7c, 0x776b7883,
	0xcc21cf04, 0xc673c946, 0xc0e9c3a9, 0xbb85be32,
	0x75037640, 0x725473b5, 0x6f5e70e1, 0x6c236dc9,
	0xb64cb8e3, 0xb140b3c0, 0xac65aecc, 0xa7bdaa0b,
	0x68a56a6c, 0x64e766ce, 0x60eb62f1, 0x5cb35ed6,
	0xa34ca57e, 0x9f14a129, 0x9b189d0e, 0x975a9931,
	0x58425a81, 0x539a55f4, 0x4ebf5133, 0x49b34c3f,
	0x93dc9593, 0x90a19236, 0x8dab8f1e, 0x8afc8c4a,
	0x447a471c, 0x3f1641cd, 0x398c3c56, 0x33de36b9,
	0x889489bf, 0x8676877c, 0x84a38583, 0x831d83d6,
	0x2e1030fb, 0x28262b1e, 0x22232527, 0x1c0b1f19,
	0x81e38276, 0x80f78163, 0x8059809e, 0x800a8028,
	0x15e118f8, 0x0fab12c7, 0x096a0c8b, 0x03240647,
	0x7ffc7fff, 0x7fe87ff5, 0x7fc17fd7, 0x7f867fa6,
	0x01920000, 0x04b60324, 0x07d90647, 0x0afb096a,
	0x7f377f61, 0x7ed47f08, 0x7e5e7e9c, 0x7dd57e1c,
	0x0e1b0c8b, 0x11390fab, 0x145512c7, 0x176d15e1,
	0x7d387d89, 0x7c887ce2, 0x7bc47c29, 0x7aee7b5c,
	0x1a8218f8, 0x1d931c0b, 0x209f1f19, 0x23a62223,
	0x7a047a7c, 0x79087989, 0x77f97883, 0x76d8776b,
	0x26a72527, 0x29a32826, 0x2c982b1e, 0x2f862e10,
	0x75a47640, 0x745e7503, 0x730673b5, 0x719d7254,
	0x326d30fb, 0x354d33de, 0x382436b9, 0x3af2398c,
	0x702270e1, 0x6e956f5e, 0x6cf86dc9, 0x6b4a6c23,
	0x3db73c56, 0x40733f16, 0x432541cd, 0x45cc447a,
	0x698b6a6c, 0x67bc68a5, 0x65dd66ce, 0x63ee64e7,
	0x4869471c, 0x4afa49b3, 0x4d804c3f, 0x4ffa4ebf,
	0x61f062f1, 0x5fe260eb, 0x5dc65ed6, 0x5b9c5cb3,
	0x52685133, 0x54c9539a, 0x571d55f4, 0x59635842,
	0x59635a81, 0x571d5842, 0x54c955f4, 0x5268539a,
	0x5b9c5a81, 0x5dc65cb3, 0x5fe25ed6, 0x61f060eb,
	0x4ffa5133, 0x4d804ebf, 0x4afa4c3f, 0x486949b3,
	0x63ee62f1, 0x65dd64e7, 0x67bc66ce, 0x698b68a5,
	0x45cc471c, 0x4325447a, 0x407341cd, 0x3db73f16,
	0x6b4a6a6c, 0x6cf86c23, 0x6e956dc9, 0x70226f5e,
	0x3af23c56, 0x3824398c, 0x354d36b9, 0x326d33de,
	0x719d70e1, 0x73067254, 0x745e73b5, 0x75a47503,
	0x2f8630fb, 0x2c982e10, 0x29a32b1e, 0x26a72826,
	0x76d87640, 0x77f9776b, 0x79087883, 0x7a047989,
	0x23a62527, 0x209f2223, 0x1d931f19, 0x1a821c0b,
	0x7aee7a7c, 0x7bc47b5c, 0x7c887c29, 0x7d387ce2,
	0x176d18f8, 0x145515e1, 0x113912c7, 0x0e1b0fab,
	0x7dd57d89, 0x7e5e7e1c, 0x7ed47e9c, 0x7f377f08,
	0x0afb0c8b, 0x07d9096a, 0x04b60647, 0x01920324,
	0x7f867f61, 0x7fc17fa6, 0x7fe87fd7, 0x7ffc7ff5,
	0xfe6d0000, 0xfb49fcdb, 0xf826f9b8, 0xf504f695,
	0x7ffc7fff, 0x7fe87ff5, 0x7fc17fd7, 0x7f867fa6,
	0xf1e4f374, 0xeec6f054, 0xebaaed38, 0xe892ea1e,
	0x7f377f61, 0x7ed47f08, 0x7e5e7e9c, 0x7dd57e1c,
	0xe57de707, 0xe26ce3f4, 0xdf60e0e6, 0xdc59dddc,
	0x7d387d89, 0x7c887ce2, 0x7bc47c29, 0x7aee7b5c,
	0xd958dad8, 0xd65cd7d9, 0xd367d4e1, 0xd079d1ef,
	0x7a047a7c, 0x79087989, 0x77f97883, 0x76d8776b,
	0xcd92cf04, 0xcab2cc21, 0xc7dbc946, 0xc50dc673,
	0x75a47640, 0x745e7503, 0x730673b5, 0x719d7254,
	0xc248c3a9, 0xbf8cc0e9, 0xbcdabe32, 0xba33bb85,
	0x702270e1, 0x6e956f5e, 0x6cf86dc9, 0x6b4a6c23,
	0xb796b8e3, 0xb505b64c, 0xb27fb3c0, 0xb005b140,
	0x698b6a6c, 0x67bc68a5, 0x65dd66ce, 0x63ee64e7,
	0xad97aecc, 0xab36ac65, 0xa8e2aa0b, 0xa69ca7bd,
	0x61f062f1, 0x5fe260eb, 0x5dc65ed6, 0x5b9c5cb3,
	0xa463a57e, 0xa239a34c, 0xa01da129, 0x9e0f9f14,
	0x59635a81, 0x571d5842, 0x54c955f4, 0x5268539a,
	0x9c119d0e, 0x9a229b18, 0x98439931, 0x9674975a,
	0x4ffa5133, 0x4d804ebf, 0x4afa4c3f, 0x486949b3,
	0x94b59593, 0x930793dc, 0x916a9236, 0x8fdd90a1,
	0x45cc471c, 0x4325447a, 0x407341cd, 0x3db73f16,
	0x8e628f1e, 0x8cf98dab, 0x8ba18c4a, 0x8a5b8afc,
	0x3af23c56, 0x3824398c, 0x354d36b9, 0x326d33de,
	0x892789bf, 0x88068894, 0x86f7877c, 0x85fb8676,
	0x2f8630fb, 0x2c982e10, 0x29a32b1e, 0x26a72826,
	0x85118583, 0x843b84a3, 0x837783d6, 0x82c7831d,
	0x23a62527, 0x209f2223, 0x1d931f19, 0x1a821c0b,
	0x822a8276, 0x81a181e3, 0x812b8163, 0x80c880f7,
	0x176d18f8, 0x145515e1, 0x113912c7, 0x0e1b0fab,
	0x8079809e, 0x803e8059, 0x80178028, 0x8003800a,
	0x0afb0c8b, 0x07d9096a, 0x04b60647, 0x01920324,
	0x7ffe7fff, 0x7ff97ffc, 0x7fef7ff5, 0x7fe07fe8,
	0x00c90000, 0x025b0192, 0x03ed0324, 0x057e04b6,
	0x7fcd7fd7, 0x7fb47fc1, 0x7f967fa6, 0x7f747f86,
	0x07100647, 0x08a107d9, 0x0a32096a, 0x0bc30afb,
	0x7f4c7f61, 0x7f207f37, 0x7eef7f08, 0x7eb97ed4,
	0x0d530c8b, 0x0ee30e1b, 0x10720fab, 0x12001139,
	0x7e7e7e9c, 0x7e3e7e5e, 0x7df97e1c, 0x7db07dd5,
	0x138e12c7, 0x151b1455, 0x16a715e1, 0x1833176d,
	0x7d617d89, 0x7d0e7d38, 0x7cb67ce2, 0x7c597c88,
	0x19bd18f8, 0x1b461a82, 0x1ccf1c0b, 0x1e561d93,
	0x7bf77c29, 0x7b917bc4, 0x7b257b5c, 0x7ab57aee,
	0x1fdc1f19, 0x2161209f, 0x22e42223, 0x246723a6,
	0x7a417a7c, 0x79c77a04, 0x79497989, 0x78c67908,
	0x25e72527, 0x276726a7, 0x28e52826, 0x2a6129a3,
	0x783f7883, 0x77b377f9, 0x7722776b, 0x768d76d8,
	0x2bdb2b1e, 0x2d542c98, 0x2ecc2e10, 0x30412f86,
	0x75f37640, 0x755475a4, 0x74b17503, 0x740a745e,
	0x31b430fb, 0x3326326d, 0x349633de, 0x3603354d,
	0x735e73b5, 0x72ae7306, 0x71f97254, 0x7140719d,
	0x376f36b9, 0x38d83824, 0x3a3f398c, 0x3ba43af2,
	0x708270e1, 0x6fc07022, 0x6efa6f5e, 0x6e306e95,
	0x3d073c56, 0x3e673db7, 0x3fc53f16, 0x41204073,
	0x6d616dc9, 0x6c8e6cf8, 0x6bb76c23, 0x6adb6b4a,
	0x427941cd, 0x43d04325, 0x4523447a, 0x467445cc,
	0x69fc6a6c, 0x6919698b, 0x683168a5, 0x674567bc,
	0x47c3471c, 0x490e4869, 0x4a5749b3, 0x4b9d4afa,
	0x665666ce, 0x656265dd, 0x646b64e7, 0x637063ee,
	0x4ce04c3f, 0x4e204d80, 0x4f5d4ebf, 0x50974ffa,
	0x627162f1, 0x616e61f0, 0x606760eb, 0x5f5d5fe2,
	0x51ce5133, 0x53015268, 0x5432539a, 0x555f54c9,
	0x5e4f5ed6, 0x5d3d5dc6, 0x5c285cb3, 0x5b0f5b9c,
	0x568955f4, 0x57b0571d, 0x58d35842, 0x59f35963,
	0x59f35a81, 0x58d35963, 0x57b05842, 0x5689571d,
	0x5b0f5a81, 0x5c285b9c, 0x5d3d5cb3, 0x5e4f5dc6,
	0x555f55f4, 0x543254c9, 0x5301539a, 0x51ce5268,
	0x5f5d5ed6, 0x60675fe2, 0x616e60eb, 0x627161f0,
	0x50975133, 0x4f5d4ffa, 0x4e204ebf, 0x4ce04d80,
	0x637062f1, 0x646b63ee, 0x656264e7, 0x665665dd,
	0x4b9d4c3f, 0x4a574afa, 0x490e49b3, 0x47c34869,
	0x674566ce, 0x683167bc, 0x691968a5, 0x69fc698b,
	0x4674471c, 0x452345cc, 0x43d0447a, 0x42794325,
	0x6adb6a6c, 0x6bb76b4a, 0x6c8e6c23, 0x6d616cf8,
	0x412041cd, 0x3fc54073, 0x3e673f16, 0x3d073db7,
	0x6e306dc9, 0x6efa6e95, 0x6fc06f5e, 0x70827022,
	0x3ba43c56, 0x3a3f3af2, 0x38d8398c, 0x376f3824,
	0x714070e1, 0x71f9719d, 0x72ae7254, 0x735e7306,
	0x360336b9, 0x3496354d, 0x332633de, 0x31b4326d,
	0x740a73b5, 0x74b1745e, 0x75547503, 0x75f375a4,
	0x304130fb, 0x2ecc2f86, 0x2d542e10, 0x2bdb2c98,
	0x768d7640, 0x772276d8, 0x77b3776b, 0x783f77f9,
	0x2a612b1e, 0x28e529a3, 0x27672826, 0x25e726a7,
	0x78c67883, 0x79497908, 0x79c77989, 0x7a417a04,
	0x24672527, 0x22e423a6, 0x21612223, 0x1fdc209f,
	0x7ab57a7c, 0x7b257aee, 0x7b917b5c, 0x7bf77bc4,
	0x1e561f19, 0x1ccf1d93, 0x1b461c0b, 0x19bd1a82,
	0x7c597c29, 0x7cb67c88, 0x7d0e7ce2, 0x7d617d38,
	0x183318f8, 0x16a7176d, 0x151b15e1, 0x138e1455,
	0x7db07d89, 0x7df97dd5, 0x7e3e7e1c, 0x7e7e7e5e,
	0x120012c7, 0x10721139, 0x0ee30fab, 0x0d530e1b,
	0x7eb97e9c, 0x7eef7ed4, 0x7f207f08, 0x7f4c7f37,
	0x0bc30c8b, 0x0a320afb, 0x08a1096a, 0x071007d9,
	0x7f747f61, 0x7f967f86, 0x7fb47fa6, 0x7fcd7fc1,
	0x057e0647, 0x03ed04b6, 0x025b0324, 0x00c90192,
	0x7fe07fd7, 0x7fef7fe8, 0x7ff97ff5, 0x7ffe7ffc,
	0xff360000, 0xfda4fe6d, 0xfc12fcdb, 0xfa81fb49,
	0x7ffe7fff, 0x7ff97ffc, 0x7fef7ff5, 0x7fe07fe8,
	0xf8eff9b8, 0xf75ef826, 0xf5cdf695, 0xf43cf504,
	0x7fcd7fd7, 0x7fb47fc1, 0x7f967fa6, 0x7f747f86,
	0xf2acf374, 0xf11cf1e4, 0xef8df054, 0xedffeec6,
	0x7f4c7f61, 0x7f207f37, 0x7eef7f08, 0x7eb97ed4,
	0xec71ed38, 0xeae4ebaa, 0xe958ea1e, 0xe7cce892,
	0x7e7e7e9c, 0x7e3e7e5e, 0x7df97e1c, 0x7db07dd5,
	0xe642e707, 0xe4b9e57d, 0xe330e3f4, 0xe1a9e26c,
	0x7d617d89, 0x7d0e7d38, 0x7cb67ce2, 0x7c597c88,
	0xe023e0e6, 0xde9edf60, 0xdd1bdddc, 0xdb98dc59,
	0x7bf77c29, 0x7b917bc4, 0x7b257b5c, 0x7ab57aee,
	0xda18dad8, 0xd898d958, 0xd71ad7d9, 0xd59ed65c,
	0x7a417a7c, 0x79c77a04, 0x79497989, 0x78c67908,
	0xd424d4e1, 0xd2abd367, 0xd133d1ef, 0xcfbed079,
	0x783f7883, 0x77b377f9, 0x7722776b, 0x768d76d8,
	0xce4bcf04, 0xccd9cd92, 0xcb69cc21, 0xc9fccab2,
	0x75f37640, 0x755475a4, 0x74b17503, 0x740a745e,
	0xc890c946, 0xc727c7db, 0xc5c0c673, 0xc45bc50d,
	0x735e73b5, 0x72ae7306, 0x71f97254, 0x7140719d,
	0xc2f8c3a9, 0xc198c248, 0xc03ac0e9, 0xbedfbf8c,
	0x708270e1, 0x6fc07022, 0x6efa6f5e, 0x6e306e95,
	0xbd86be32, 0xbc2fbcda, 0xbadcbb85, 0xb98bba33,
	0x6d616dc9, 0x6c8e6cf8, 0x6bb76c23, 0x6adb6b4a,
	0xb83cb8e3, 0xb6f1b796, 0xb5a8b64c, 0xb462b505,
	0x69fc6a6c, 0x6919698b, 0x683168a5, 0x674567bc,
	0xb31fb3c0, 0xb1dfb27f, 0xb0a2b140, 0xaf68b005,
	0x665666ce, 0x656265dd, 0x646b64e7, 0x637063ee,
	0xae31aecc, 0xacfead97, 0xabcdac65, 0xaaa0ab36,
	0x627162f1, 0x616e61f0, 0x606760eb, 0x5f5d5fe2,
	0xa976aa0b, 0xa84fa8e2, 0xa72ca7bd, 0xa60ca69c,
	0x5e4f5ed6, 0x5d3d5dc6, 0x5c285cb3, 0x5b0f5b9c,
	0xa4f0a57e, 0xa3d7a463, 0xa2c2a34c, 0xa1b0a239,
	0x59f35a81, 0x58d35963, 0x57b05842, 0x5689571d,
	0xa0a2a129, 0x9f98a01d, 0x9e919f14, 0x9d8e9e0f,
	0x555f55f4, 0x543254c9, 0x5301539a, 0x51ce5268,
	0x9c8f9d0e, 0x9b949c11, 0x9a9d9b18, 0x99a99a22,
	0x50975133, 0x4f5d4ffa, 0x4e204ebf, 0x4ce04d80,
	0x98ba9931, 0x97ce9843, 0x96e6975a, 0x96039674,
	0x4b9d4c3f, 0x4a574afa, 0x490e49b3, 0x47c34869,
	0x95249593, 0x944894b5, 0x937193dc, 0x929e9307,
	0x4674471c, 0x452345cc, 0x43d0447a, 0x42794325,
	0x91cf9236, 0x9105916a, 0x903f90a1, 0x8f7d8fdd,
	0x412041cd, 0x3fc54073, 0x3e673f16, 0x3d073db7,
	0x8ebf8f1e, 0x8e068e62, 0x8d518dab, 0x8ca18cf9,
	0x3ba43c56, 0x3a3f3af2, 0x38d8398c, 0x376f3824,
	0x8bf58c4a, 0x8b4e8ba1, 0x8aab8afc, 0x8a0c8a5b,
	0x360336b9, 0x3496354d, 0x332633de, 0x31b4326d,
	0x897289bf, 0x88dd8927, 0x884c8894, 0x87c08806,
	0x304130fb, 0x2ecc2f86, 0x2d542e10, 0x2bdb2c98,
	0x8739877c, 0x86b686f7, 0x86388676, 0x85be85fb,
	0x2a612b1e, 0x28e529a3, 0x27672826, 0x25e726a7,
	0x854a8583, 0x84da8511, 0x846e84a3, 0x8408843b,
	0x24672527, 0x22e423a6, 0x21612223, 0x1fdc209f,
	0x83a683d6, 0x83498377, 0x82f1831d, 0x829e82c7,
	0x1e561f19, 0x1ccf1d93, 0x1b461c0b, 0x19bd1a82,
	0x824f8276, 0x8206822a, 0x81c181e3, 0x818181a1,
	0x183318f8, 0x16a7176d, 0x151b15e1, 0x138e1455,
	0x81468163, 0x8110812b, 0x80df80f7, 0x80b380c8,
	0x120012c7, 0x10721139, 0x0ee30fab, 0x0d530e1b,
	0x808b809e, 0x80698079, 0x804b8059, 0x8032803e,
	0x0bc30c8b, 0x0a320afb, 0x08a1096a, 0x071007d9,
	0x801f8028, 0x80108017, 0x8006800a, 0x80018003,
	0x057e0647, 0x03ed04b6, 0x025b0324, 0x00c90192,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_3_M[] = {
	0x00000000, 0x00000000, 0xa57e7fff, 0x5a810000,
	0x00000000, 0x00000000, 0xa57e0000, 0xa57e7fff,
	0x30fb7fff, 0x89bfa57e, 0x76400000, 0xcf045a81,
	0x89bf0000, 0x30fba57e, 0x30fb7fff, 0x89bfa57e,
	0x6a6c7fff, 0xe70730fb, 0x8276a57e, 0xb8e389bf,
	0xb8e30000, 0x827689bf, 0xe707a57e, 0x6a6c30fb,
	0x471c0000, 0x7d897640, 0x18f85a81, 0x9593cf04,
	0x6a6c7fff, 0xe70730fb, 0x8276a57e, 0xb8e389bf,
	0x7a7c7fff, 0x51336a6c, 0x0c8b30fb, 0xc3a9e707,
	0xdad80000, 0x9d0eb8e3, 0x809e89bf, 0x8f1e8276,
	0x8f1ea57e, 0x809e8276, 0x9d0e89bf, 0xdad8b8e3,
	0xc3a9a57e, 0x0c8be707, 0x513330fb, 0x7a7c6a6c,
	0x25270000, 0x62f1471c, 0x7f617640, 0x70e17d89,
	0x7a7c7ffe, 0x51336a6c, 0x0c8b30fb, 0xc3a9e707,
	0x3c565a81, 0xf37418f8, 0xaecccf04, 0x85839593,
	0x8f1ea57e, 0x809e8276, 0x9d0e89bf, 0xdad8b8e3,
	0x7e9c7fff, 0x73b57a7c, 0x5ed66a6c, 0x41cd5133,
	0xed380000, 0xc946dad8, 0xaa0bb8e3, 0x92369d0e,
	0x1f1930fb, 0xf9b80c8b, 0xd4e1e707, 0xb3c0c3a9,
	0x83d689bf, 0x8028809e, 0x877c8276, 0x99318f1e,
	0x9931a57e, 0x877c8f1e, 0x80288276, 0x83d6809e,
	0xb3c0a57e, 0xd4e1c3a9, 0xf9b8e707, 0x1f190c8b,
	0x923689bf, 0xaa0b9d0e, 0xc946b8e3, 0xed38dad8,
	0x41cd30fb, 0x5ed65133, 0x73b56a6c, 0x7e9c7a7c,
	0x12c70000, 0x36b92527, 0x55f4471c, 0x6dc962f1,
	0x7e9c7ffe, 0x73b57a7c, 0x5ed66a6c, 0x41cd5133,
	0x7c297640, 0x7fd77f61, 0x78837d89, 0x66ce70e1,
	0x1f1930fb, 0xf9b80c8b, 0xd4e1e707, 0xb3c0c3a9,
	0x4c3f5a81, 0x2b1e3c56, 0x064718f8, 0xe0e6f374,
	0x9931a57e, 0x877c8f1e, 0x80288276, 0x83d6809e,
	0xbe32cf04, 0xa129aecc, 0x8c4a9593, 0x81638583,
	0x923689bf, 0xaa0b9d0e, 0xc946b8e3, 0xed38dad8,
	0x7fa67fff, 0x7ce27e9c, 0x776b7a7c, 0x6f5e73b5,
	0xf6950000, 0xe3f4ed38, 0xd1efdad8, 0xc0e9c946,
	0x64e76a6c, 0x58425ed6, 0x49b35133, 0x398c41cd,
	0xb140b8e3, 0xa34caa0b, 0x975a9d0e, 0x8dab9236,
	0x282630fb, 0x15e11f19, 0x03240c8b, 0xf054f9b8,
	0x867689bf, 0x81e383d6, 0x800a809e, 0x80f78028,
	0xdddce707, 0xcc21d4e1, 0xbb85c3a9, 0xac65b3c0,
	0x84a38276, 0x8afc877c, 0x93dc8f1e, 0x9f149931,
	0x9f14a57e, 0x93dc9931, 0x8afc8f1e, 0x84a3877c,
	0xac65a57e, 0xbb85b3c0, 0xcc21c3a9, 0xdddcd4e1,
	0x80f78276, 0x800a8028, 0x81e3809e, 0x867683d6,
	0xf054e707, 0x0324f9b8, 0x15e10c8b, 0x28261f19,
	0x8dab89bf, 0x975a9236, 0xa34c9d0e, 0xb140aa0b,
	0x398c30fb, 0x49b341cd, 0x58425133, 0x64e75ed6,
	0xc0e9b8e3, 0xd1efc946, 0xe3f4dad8, 0xf695ed38,
	0x6f5e6a6c, 0x776b73b5, 0x7ce27a7c, 0x7fa67e9c,
	0x096a0000, 0x1c0b12c7, 0x2e102527, 0x3f1636b9,
	0x7fa67fff, 0x7ce27e9c, 0x776b7a7c, 0x6f5e73b5,
	0x4ebf471c, 0x5cb355f4, 0x68a562f1, 0x72546dc9,
	0x64e76a6c, 0x58425ed6, 0x49b35133, 0x398c41cd,
	0x79897640, 0x7e1c7c29, 0x7ff57f61, 0x7f087fd7,
	0x282630fb, 0x15e11f19, 0x03240c8b, 0xf054f9b8,
	0x7b5c7d89, 0x75037883, 0x6c2370e1, 0x60eb66ce,
	0xdddce707, 0xcc21d4e1, 0xbb85c3a9, 0xac65b3c0,
	0x539a5a81, 0x447a4c3f, 0x33de3c56, 0x22232b1e,
	0x9f14a57e, 0x93dc9931, 0x8afc8f1e, 0x84a3877c,
	0x0fab18f8, 0xfcdb0647, 0xea1ef374, 0xd7d9e0e6,
	0x80f78276, 0x800a8028, 0x81e3809e, 0x867683d6,
	0xc673cf04, 0xb64cbe32, 0xa7bdaecc, 0x9b18a129,
	0x8dab89bf, 0x975a9236, 0xa34c9d0e, 0xb140aa0b,
	0x90a19593, 0x88948c4a, 0x831d8583, 0x80598163,
	0xc0e9b8e3, 0xd1efc946, 0xe3f4dad8, 0xf695ed38,
	0x7fe87fff, 0x7f377fa6, 0x7dd57e9c, 0x7bc47ce2,
	0xfb490000, 0xf1e4f695, 0xe892ed38, 0xdf60e3f4,
	0x79087a7c, 0x75a4776b, 0x719d73b5, 0x6cf86f5e,
	0xd65cdad8, 0xcd92d1ef, 0xc50dc946, 0xbcdac0e9,
	0x67bc6a6c, 0x61f064e7, 0x5b9c5ed6, 0x54c95842,
	0xb505b8e3, 0xad97b140, 0xa69caa0b, 0xa01da34c,
	0x4d805133, 0x45cc49b3, 0x3db741cd, 0x354d398c,
	0x9a229d0e, 0x94b5975a, 0x8fdd9236, 0x8ba18dab,
	0x2c9830fb, 0x23a62826, 0x1a821f19, 0x113915e1,
	0x880689bf, 0x85118676, 0x82c783d6, 0x812b81e3,
	0x07d90c8b, 0xfe6d0324, 0xf504f9b8, 0xebaaf054,
	0x803e809e, 0x8003800a, 0x80798028, 0x81a180f7,
	0xe26ce707, 0xd958dddc, 0xd079d4e1, 0xc7dbcc21,
	0x83778276, 0x85fb84a3, 0x8927877c, 0x8cf98afc,
	0xbf8cc3a9, 0xb796bb85, 0xb005b3c0, 0xa8e2ac65,
	0x916a8f1e, 0x967493dc, 0x9c119931, 0xa2399f14,
	0xa239a57e, 0x9c119f14, 0x96749931, 0x916a93dc,
	0xa8e2a57e, 0xb005ac65, 0xb796b3c0, 0xbf8cbb85,
	0x8cf98f1e, 0x89278afc, 0x85fb877c, 0x837784a3,
	0xc7dbc3a9, 0xd079cc21, 0xd958d4e1, 0xe26cdddc,
	0x81a18276, 0x807980f7, 0x80038028, 0x803e800a,
	0xebaae707, 0xf504f054, 0xfe6df9b8, 0x07d90324,
	0x812b809e, 0x82c781e3, 0x851183d6, 0x88068676,
	0x11390c8b, 0x1a8215e1, 0x23a61f19, 0x2c982826,
	0x8ba189bf, 0x8fdd8dab, 0x94b59236, 0x9a22975a,
	0x354d30fb, 0x3db7398c, 0x45cc41cd, 0x4d8049b3,
	0xa01d9d0e, 0xa69ca34c, 0xad97aa0b, 0xb505b140,
	0x54c95133, 0x5b9c5842, 0x61f05ed6, 0x67bc64e7,
	0xbcdab8e3, 0xc50dc0e9, 0xcd92c946, 0xd65cd1ef,
	0x6cf86a6c, 0x719d6f5e, 0x75a473b5, 0x7908776b,
	0xdf60dad8, 0xe892e3f4, 0xf1e4ed38, 0xfb49f695,
	0x7bc47a7c, 0x7dd57ce2, 0x7f377e9c, 0x7fe87fa6,
	0x04b60000, 0x0e1b096a, 0x176d12c7, 0x209f1c0b,
	0x7fe87fff, 0x7f377fa6, 0x7dd57e9c, 0x7bc47ce2,
	0x29a32527, 0x326d2e10, 0x3af236b9, 0x43253f16,
	0x79087a7c, 0x75a4776b, 0x719d73b5, 0x6cf86f5e,
	0x4afa471c, 0x52684ebf, 0x596355f4, 0x5fe25cb3,
	0x67bc6a6c, 0x61f064e7, 0x5b9c5ed6, 0x54c95842,
	0x65dd62f1, 0x6b4a68a5, 0x70226dc9, 0x745e7254,
	0x4d805133, 0x45cc49b3, 0x3db741cd, 0x354d398c,
	0x77f97640, 0x7aee7989, 0x7d387c29, 0x7ed47e1c,
	0x2c9830fb, 0x23a62826, 0x1a821f19, 0x113915e1,
	0x7fc17f61, 0x7ffc7ff5, 0x7f867fd7, 0x7e5e7f08,
	0x07d90c8b, 0xfe6d0324, 0xf504f9b8, 0xebaaf054,
	0x7c887d89, 0x7a047b5c, 0x76d87883, 0x73067503,
	0xe26ce707, 0xd958dddc, 0xd079d4e1, 0xc7dbcc21,
	0x6e9570e1, 0x698b6c23, 0x63ee66ce, 0x5dc660eb,
	0xbf8cc3a9, 0xb796bb85, 0xb005b3c0, 0xa8e2ac65,
	0x571d5a81, 0x4ffa539a, 0x48694c3f, 0x4073447a,
	0xa239a57e, 0x9c119f14, 0x96749931, 0x916a93dc,
	0x38243c56, 0x2f8633de, 0x26a72b1e, 0x1d932223,
	0x8cf98f1e, 0x89278afc, 0x85fb877c, 0x837784a3,
	0x145518f8, 0x0afb0fab, 0x01920647, 0xf826fcdb,
	0x81a18276, 0x807980f7, 0x80038028, 0x803e800a,
	0xeec6f374, 0xe57dea1e, 0xdc59e0e6, 0xd367d7d9,
	0x812b809e, 0x82c781e3, 0x851183d6, 0x88068676,
	0xcab2cf04, 0xc248c673, 0xba33be32, 0xb27fb64c,
	0x8ba189bf, 0x8fdd8dab, 0x94b59236, 0x9a22975a,
	0xab36aecc, 0xa463a7bd, 0x9e0fa129, 0x98439b18,
	0xa01d9d0e, 0xa69ca34c, 0xad97aa0b, 0xb505b140,
	0x93079593, 0x8e6290a1, 0x8a5b8c4a, 0x86f78894,
	0xbcdab8e3, 0xc50dc0e9, 0xcd92c946, 0xd65cd1ef,
	0x843b8583, 0x822a831d, 0x80c88163, 0x80178059,
	0xdf60dad8, 0xe892e3f4, 0xf1e4ed38, 0xfb49f695,
	0x7ff97fff, 0x7fcd7fe8, 0x7f747fa6, 0x7eef7f37,
	0xfda40000, 0xf8effb49, 0xf43cf695, 0xef8df1e4,
	0x7e3e7e9c, 0x7d617dd5, 0x7c597ce2, 0x7b257bc4,
	0xeae4ed38, 0xe642e892, 0xe1a9e3f4, 0xdd1bdf60,
	0x79c77a7c, 0x783f7908, 0x768d776b, 0x74b175a4,
	0xd898dad8, 0xd424d65c, 0xcfbed1ef, 0xcb69cd92,
	0x72ae73b5, 0x7082719d, 0x6e306f5e, 0x6bb76cf8,
	0xc727c946, 0xc2f8c50d, 0xbedfc0e9, 0xbadcbcda,
	0x69196a6c, 0x665667bc, 0x637064e7, 0x606761f0,
	0xb6f1b8e3, 0xb31fb505, 0xaf68b140, 0xabcdad97,
	0x5d3d5ed6, 0x59f35b9c, 0x56895842, 0x530154c9,
	0xa84faa0b, 0xa4f0a69c, 0xa1b0a34c, 0x9e91a01d,
	0x4f5d5133, 0x4b9d4d80, 0x47c349b3, 0x43d045cc,
	0x9b949d0e, 0x98ba9a22, 0x9603975a, 0x937194b5,
	0x3fc541cd, 0x3ba43db7, 0x376f398c, 0x3326354d,
	0x91059236, 0x8ebf8fdd, 0x8ca18dab, 0x8aab8ba1,
	0x2ecc30fb, 0x2a612c98, 0x25e72826, 0x216123a6,
	0x88dd89bf, 0x87398806, 0x85be8676, 0x846e8511,
	0x1ccf1f19, 0x18331a82, 0x138e15e1, 0x0ee31139,
	0x834983d6, 0x824f82c7, 0x818181e3, 0x80df812b,
	0x0a320c8b, 0x057e07d9, 0x00c90324, 0xfc12fe6d,
	0x8069809e, 0x801f803e, 0x8001800a, 0x80108003,
	0xf75ef9b8, 0xf2acf504, 0xedfff054, 0xe958ebaa,
	0x804b8028, 0x80b38079, 0x814680f7, 0x820681a1,
	0xe4b9e707, 0xe023e26c, 0xdb98dddc, 0xd71ad958,
	0x82f18276, 0x84088377, 0x854a84a3, 0x86b685fb,
	0xd2abd4e1, 0xce4bd079, 0xc9fccc21, 0xc5c0c7db,
	0x884c877c, 0x8a0c8927, 0x8bf58afc, 0x8e068cf9,
	0xc198c3a9, 0xbd86bf8c, 0xb98bbb85, 0xb5a8b796,
	0x903f8f1e, 0x929e916a, 0x952493dc, 0x97ce9674,
	0xb1dfb3c0, 0xae31b005, 0xaaa0ac65, 0xa72ca8e2,
	0x9a9d9931, 0x9d8e9c11, 0xa0a29f14, 0xa3d7a239,
	0xa3d7a57e, 0xa0a2a239, 0x9d8e9f14, 0x9a9d9c11,
	0xa72ca57e, 0xaaa0a8e2, 0xae31ac65, 0xb1dfb005,
	0x97ce9931, 0x95249674, 0x929e93dc, 0x903f916a,
	0xb5a8b3c0, 0xb98bb796, 0xbd86bb85, 0xc198bf8c,
	0x8e068f1e, 0x8bf58cf9, 0x8a0c8afc, 0x884c8927,
	0xc5c0c3a9, 0xc9fcc7db, 0xce4bcc21, 0xd2abd079,
	0x86b6877c, 0x854a85fb, 0x840884a3, 0x82f18377,
	0xd71ad4e1, 0xdb98d958, 0xe023dddc, 0xe4b9e26c,
	0x82068276, 0x814681a1, 0x80b380f7, 0x804b8079,
	0xe958e707, 0xedffebaa, 0xf2acf054, 0xf75ef504,
	0x80108028, 0x80018003, 0x801f800a, 0x8069803e,
	0xfc12f9b8, 0x00c9fe6d, 0x057e0324, 0x0a3207d9,
	0x80df809e, 0x8181812b, 0x824f81e3, 0x834982c7,
	0x0ee30c8b, 0x138e1139, 0x183315e1, 0x1ccf1a82,
	0x846e83d6, 0x85be8511, 0x87398676, 0x88dd8806,
	0x21611f19, 0x25e723a6, 0x2a612826, 0x2ecc2c98,
	0x8aab89bf, 0x8ca18ba1, 0x8ebf8dab, 0x91058fdd,
	0x332630fb, 0x376f354d, 0x3ba4398c, 0x3fc53db7,
	0x93719236, 0x960394b5, 0x98ba975a, 0x9b949a22,
	0x43d041cd, 0x47c345cc, 0x4b9d49b3, 0x4f5d4d80,
	0x9e919d0e, 0xa1b0a01d, 0xa4f0a34c, 0xa84fa69c,
	0x53015133, 0x568954c9, 0x59f35842, 0x5d3d5b9c,
	0xabcdaa0b, 0xaf68ad97, 0xb31fb140, 0xb6f1b505,
	0x60675ed6, 0x637061f0, 0x665664e7, 0x691967bc,
	0xbadcb8e3, 0xbedfbcda, 0xc2f8c0e9, 0xc727c50d,
	0x6bb76a6c, 0x6e306cf8, 0x70826f5e, 0x72ae719d,
	0xcb69c946, 0xcfbecd92, 0xd424d1ef, 0xd898d65c,
	0x74b173b5, 0x768d75a4, 0x783f776b, 0x79c77908,
	0xdd1bdad8, 0xe1a9df60, 0xe642e3f4, 0xeae4e892,
	0x7b257a7c, 0x7c597bc4, 0x7d617ce2, 0x7e3e7dd5,
	0xef8ded38, 0xf43cf1e4, 0xf8eff695, 0xfda4fb49,
	0x7eef7e9c, 0x7f747f37, 0x7fcd7fa6, 0x7ff97fe8,
	0x025b0000, 0x071004b6, 0x0bc3096a, 0x10720e1b,
	0x7ff97fff, 0x7fcd7fe8, 0x7f747fa6, 0x7eef7f37,
	0x151b12c7, 0x19bd176d, 0x1e561c0b, 0x22e4209f,
	0x7e3e7e9c, 0x7d617dd5, 0x7c597ce2, 0x7b257bc4,
	0x27672527, 0x2bdb29a3, 0x30412e10, 0x3496326d,
	0x79c77a7c, 0x783f7908, 0x768d776b, 0x74b175a4,
	0x38d836b9, 0x3d073af2, 0x41203f16, 0x45234325,
	0x72ae73b5, 0x7082719d, 0x6e306f5e, 0x6bb76cf8,
	0x490e471c, 0x4ce04afa, 0x50974ebf, 0x54325268,
	0x69196a6c, 0x665667bc, 0x637064e7, 0x606761f0,
	0x57b055f4, 0x5b0f5963, 0x5e4f5cb3, 0x616e5fe2,
	0x5d3d5ed6, 0x59f35b9c, 0x56895842, 0x530154c9,
	0x646b62f1, 0x674565dd, 0x69fc68a5, 0x6c8e6b4a,
	0x4f5d5133, 0x4b9d4d80, 0x47c349b3, 0x43d045cc,
	0x6efa6dc9, 0x71407022, 0x735e7254, 0x7554745e,
	0x3fc541cd, 0x3ba43db7, 0x376f398c, 0x3326354d,
	0x77227640, 0x78c677f9, 0x7a417989, 0x7b917aee,
	0x2ecc30fb, 0x2a612c98, 0x25e72826, 0x216123a6,
	0x7cb67c29, 0x7db07d38, 0x7e7e7e1c, 0x7f207ed4,
	0x1ccf1f19, 0x18331a82, 0x138e15e1, 0x0ee31139,
	0x7f967f61, 0x7fe07fc1, 0x7ffe7ff5, 0x7fef7ffc,
	0x0a320c8b, 0x057e07d9, 0x00c90324, 0xfc12fe6d,
	0x7fb47fd7, 0x7f4c7f86, 0x7eb97f08, 0x7df97e5e,
	0xf75ef9b8, 0xf2acf504, 0xedfff054, 0xe958ebaa,
	0x7d0e7d89, 0x7bf77c88, 0x7ab57b5c, 0x79497a04,
	0xe4b9e707, 0xe023e26c, 0xdb98dddc, 0xd71ad958,
	0x77b37883, 0x75f376d8, 0x740a7503, 0x71f97306,
	0xd2abd4e1, 0xce4bd079, 0xc9fccc21, 0xc5c0c7db,
	0x6fc070e1, 0x6d616e95, 0x6adb6c23, 0x6831698b,
	0xc198c3a9, 0xbd86bf8c, 0xb98bbb85, 0xb5a8b796,
	0x656266ce, 0x627163ee, 0x5f5d60eb, 0x5c285dc6,
	0xb1dfb3c0, 0xae31b005, 0xaaa0ac65, 0xa72ca8e2,
	0x58d35a81, 0x555f571d, 0x51ce539a, 0x4e204ffa,
	0xa3d7a57e, 0xa0a2a239, 0x9d8e9f14, 0x9a9d9c11,
	0x4a574c3f, 0x46744869, 0x4279447a, 0x3e674073,
	0x97ce9931, 0x95249674, 0x929e93dc, 0x903f916a,
	0x3a3f3c56, 0x36033824, 0x31b433de, 0x2d542f86,
	0x8e068f1e, 0x8bf58cf9, 0x8a0c8afc, 0x884c8927,
	0x28e52b1e, 0x246726a7, 0x1fdc2223, 0x1b461d93,
	0x86b6877c, 0x854a85fb, 0x840884a3, 0x82f18377,
	0x16a718f8, 0x12001455, 0x0d530fab, 0x08a10afb,
	0x82068276, 0x814681a1, 0x80b380f7, 0x804b8079,
	0x03ed0647, 0xff360192, 0xfa81fcdb, 0xf5cdf826,
	0x80108028, 0x80018003, 0x801f800a, 0x8069803e,
	0xf11cf374, 0xec71eec6, 0xe7ccea1e, 0xe330e57d,
	0x80df809e, 0x8181812b, 0x824f81e3, 0x834982c7,
	0xde9ee0e6, 0xda18dc59, 0xd59ed7d9, 0xd133d367,
	0x846e83d6, 0x85be8511, 0x87398676, 0x88dd8806,
	0xccd9cf04, 0xc890cab2, 0xc45bc673, 0xc03ac248,
	0x8aab89bf, 0x8ca18ba1, 0x8ebf8dab, 0x91058fdd,
	0xbc2fbe32, 0xb83cba33, 0xb462b64c, 0xb0a2b27f,
	0x93719236, 0x960394b5, 0x98ba975a, 0x9b949a22,
	0xacfeaecc, 0xa976ab36, 0xa60ca7bd, 0xa2c2a463,
	0x9e919d0e, 0xa1b0a01d, 0xa4f0a34c, 0xa84fa69c,
	0x9f98a129, 0x9c8f9e0f, 0x99a99b18, 0x96e69843,
	0xabcdaa0b, 0xaf68ad97, 0xb31fb140, 0xb6f1b505,
	0x94489593, 0x91cf9307, 0x8f7d90a1, 0x8d518e62,
	0xbadcb8e3, 0xbedfbcda, 0xc2f8c0e9, 0xc727c50d,
	0x8b4e8c4a, 0x89728a5b, 0x87c08894, 0x863886f7,
	0xcb69c946, 0xcfbecd92, 0xd424d1ef, 0xd898d65c,
	0x84da8583, 0x83a6843b, 0x829e831d, 0x81c1822a,
	0xdd1bdad8, 0xe1a9df60, 0xe642e3f4, 0xeae4e892,
	0x81108163, 0x808b80c8, 0x80328059, 0x80068017,
	0xef8ded38, 0xf43cf1e4, 0xf8eff695, 0xfda4fb49,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_3_P[] = {
	0x00000000, 0x00000000, 0xa57e7fff, 0x5a810000,
	0x00000000, 0x00000000, 0x5a810000, 0x5a818001,
	0x30fb7fff, 0x89bfa57e, 0x76400000, 0xcf045a81,
	0x76400000, 0xcf045a81, 0xcf048001, 0x76405a81,
	0x6a6c7fff, 0xe70730fb, 0x8276a57e, 0xb8e389bf,
	0x471c0000, 0x7d897640, 0x18f85a81, 0x9593cf04,
	0x471c0000, 0x7d897640, 0x18f85a81, 0x9593cf04,
	0x95938001, 0x18f8cf04, 0x7d895a81, 0x471c7640,
	0x7a7c7fff, 0x51336a6c, 0x0c8b30fb, 0xc3a9e707,
	0x25270000, 0x62f1471c, 0x7f617640, 0x70e17d89,
	0x8f1ea57e, 0x809e8276, 0x9d0e89bf, 0xdad8b8e3,
	0x3c565a81, 0xf37418f8, 0xaecccf04, 0x85839593,
	0x25270000, 0x62f1471c, 0x7f617640, 0x70e17d89,
	0x85838001, 0xaecc9593, 0xf374cf04, 0x3c5618f8,
	0x3c565a81, 0xf37418f8, 0xaecccf04, 0x85839593,
	0x70e15a81, 0x7f617d89, 0x62f17640, 0x2527471c,
	0x7e9c7fff, 0x73b57a7c, 0x5ed66a6c, 0x41cd5133,
	0x12c70000, 0x36b92527, 0x55f4471c, 0x6dc962f1,
	0x1f1930fb, 0xf9b80c8b, 0xd4e1e707, 0xb3c0c3a9,
	0x7c297640, 0x7fd77f61, 0x78837d89, 0x66ce70e1,
	0x9931a57e, 0x877c8f1e, 0x80288276, 0x83d6809e,
	0x4c3f5a81, 0x2b1e3c56, 0x064718f8, 0xe0e6f374,
	0x923689bf, 0xaa0b9d0e, 0xc946b8e3, 0xed38dad8,
	0xbe32cf04, 0xa129aecc, 0x8c4a9593, 0x81638583,
	0x12c70000, 0x36b92527, 0x55f4471c, 0x6dc962f1,
	0x81638001, 0x8c4a8583, 0xa1299593, 0xbe32aecc,
	0x7c297640, 0x7fd77f61, 0x78837d89, 0x66ce70e1,
	0xe0e6cf04, 0x0647f374, 0x2b1e18f8, 0x4c3f3c56,
	0x4c3f5a81, 0x2b1e3c56, 0x064718f8, 0xe0e6f374,
	0x66ce5a81, 0x788370e1, 0x7fd77d89, 0x7c297f61,
	0xbe32cf04, 0xa129aecc, 0x8c4a9593, 0x81638583,
	0x6dc97640, 0x55f462f1, 0x36b9471c, 0x12c72527,
	0x7fa67fff, 0x7ce27e9c, 0x776b7a7c, 0x6f5e73b5,
	0x096a0000, 0x1c0b12c7, 0x2e102527, 0x3f1636b9,
	0x64e76a6c, 0x58425ed6, 0x49b35133, 0x398c41cd,
	0x4ebf471c, 0x5cb355f4, 0x68a562f1, 0x72546dc9,
	0x282630fb, 0x15e11f19, 0x03240c8b, 0xf054f9b8,
	0x79897640, 0x7e1c7c29, 0x7ff57f61, 0x7f087fd7,
	0xdddce707, 0xcc21d4e1, 0xbb85c3a9, 0xac65b3c0,
	0x7b5c7d89, 0x75037883, 0x6c2370e1, 0x60eb66ce,
	0x9f14a57e, 0x93dc9931, 0x8afc8f1e, 0x84a3877c,
	0x539a5a81, 0x447a4c3f, 0x33de3c56, 0x22232b1e,
	0x80f78276, 0x800a8028, 0x81e3809e, 0x867683d6,
	0x0fab18f8, 0xfcdb0647, 0xea1ef374, 0xd7d9e0e6,
	0x8dab89bf, 0x975a9236, 0xa34c9d0e, 0xb140aa0b,
	0xc673cf04, 0xb64cbe32, 0xa7bdaecc, 0x9b18a129,
	0xc0e9b8e3, 0xd1efc946, 0xe3f4dad8, 0xf695ed38,
	0x90a19593, 0x88948c4a, 0x831d8583, 0x80598163,
	0x096a0000, 0x1c0b12c7, 0x2e102527, 0x3f1636b9,
	0x80598001, 0x831d8163, 0x88948583, 0x90a18c4a,
	0x4ebf471c, 0x5cb355f4, 0x68a562f1, 0x72546dc9,
	0x9b189593, 0xa7bda129, 0xb64caecc, 0xc673be32,
	0x79897640, 0x7e1c7c29, 0x7ff57f61, 0x7f087fd7,
	0xd7d9cf04, 0xea1ee0e6, 0xfcdbf374, 0x0fab0647,
	0x7b5c7d89, 0x75037883, 0x6c2370e1, 0x60eb66ce,
	0x222318f8, 0x33de2b1e, 0x447a3c56, 0x539a4c3f,
	0x539a5a81, 0x447a4c3f, 0x33de3c56, 0x22232b1e,
	0x60eb5a81, 0x6c2366ce, 0x750370e1, 0x7b5c7883,
	0x0fab18f8, 0xfcdb0647, 0xea1ef374, 0xd7d9e0e6,
	0x7f087d89, 0x7ff57fd7, 0x7e1c7f61, 0x79897c29,
	0xc673cf04, 0xb64cbe32, 0xa7bdaecc, 0x9b18a129,
	0x72547640, 0x68a56dc9, 0x5cb362f1, 0x4ebf55f4,
	0x90a19593, 0x88948c4a, 0x831d8583, 0x80598163,
	0x3f16471c, 0x2e1036b9, 0x1c0b2527, 0x096a12c7,
	0x7fe87fff, 0x7f377fa6, 0x7dd57e9c, 0x7bc47ce2,
	0x04b60000, 0x0e1b096a, 0x176d12c7, 0x209f1c0b,
	0x79087a7c, 0x75a4776b, 0x719d73b5, 0x6cf86f5e,
	0x29a32527, 0x326d2e10, 0x3af236b9, 0x43253f16,
	0x67bc6a6c, 0x61f064e7, 0x5b9c5ed6, 0x54c95842,
	0x4afa471c, 0x52684ebf, 0x596355f4, 0x5fe25cb3,
	0x4d805133, 0x45cc49b3, 0x3db741cd, 0x354d398c,
	0x65dd62f1, 0x6b4a68a5, 0x70226dc9, 0x745e7254,
	0x2c9830fb, 0x23a62826, 0x1a821f19, 0x113915e1,
	0x77f97640, 0x7aee7989, 0x7d387c29, 0x7ed47e1c,
	0x07d90c8b, 0xfe6d0324, 0xf504f9b8, 0xebaaf054,
	0x7fc17f61, 0x7ffc7ff5, 0x7f867fd7, 0x7e5e7f08,
	0xe26ce707, 0xd958dddc, 0xd079d4e1, 0xc7dbcc21,
	0x7c887d89, 0x7a047b5c, 0x76d87883, 0x73067503,
	0xbf8cc3a9, 0xb796bb85, 0xb005b3c0, 0xa8e2ac65,
	0x6e9570e1, 0x698b6c23, 0x63ee66ce, 0x5dc660eb,
	0xa239a57e, 0x9c119f14, 0x96749931, 0x916a93dc,
	0x571d5a81, 0x4ffa539a, 0x48694c3f, 0x4073447a,
	0x8cf98f1e, 0x89278afc, 0x85fb877c, 0x837784a3,
	0x38243c56, 0x2f8633de, 0x26a72b1e, 0x1d932223,
	0x81a18276, 0x807980f7, 0x80038028, 0x803e800a,
	0x145518f8, 0x0afb0fab, 0x01920647, 0xf826fcdb,
	0x812b809e, 0x82c781e3, 0x851183d6, 0x88068676,
	0xeec6f374, 0xe57dea1e, 0xdc59e0e6, 0xd367d7d9,
	0x8ba189bf, 0x8fdd8dab, 0x94b59236, 0x9a22975a,
	0xcab2cf04, 0xc248c673, 0xba33be32, 0xb27fb64c,
	0xa01d9d0e, 0xa69ca34c, 0xad97aa0b, 0xb505b140,
	0xab36aecc, 0xa463a7bd, 0x9e0fa129, 0x98439b18,
	0xbcdab8e3, 0xc50dc0e9, 0xcd92c946, 0xd65cd1ef,
	0x93079593, 0x8e6290a1, 0x8a5b8c4a, 0x86f78894,
	0xdf60dad8, 0xe892e3f4, 0xf1e4ed38, 0xfb49f695,
	0x843b8583, 0x822a831d, 0x80c88163, 0x80178059,
	0x04b60000, 0x0e1b096a, 0x176d12c7, 0x209f1c0b,
	0x80178001, 0x80c88059, 0x822a8163, 0x843b831d,
	0x29a32527, 0x326d2e10, 0x3af236b9, 0x43253f16,
	0x86f78583, 0x8a5b8894, 0x8e628c4a, 0x930790a1,
	0x4afa471c, 0x52684ebf, 0x596355f4, 0x5fe25cb3,
	0x98439593, 0x9e0f9b18, 0xa463a129, 0xab36a7bd,
	0x65dd62f1, 0x6b4a68a5, 0x70226dc9, 0x745e7254,
	0xb27faecc, 0xba33b64c, 0xc248be32, 0xcab2c673,
	0x77f97640, 0x7aee7989, 0x7d387c29, 0x7ed47e1c,
	0xd367cf04, 0xdc59d7d9, 0xe57de0e6, 0xeec6ea1e,
	0x7fc17f61, 0x7ffc7ff5, 0x7f867fd7, 0x7e5e7f08,
	0xf826f374, 0x0192fcdb, 0x0afb0647, 0x14550fab,
	0x7c887d89, 0x7a047b5c, 0x76d87883, 0x73067503,
	0x1d9318f8, 0x26a72223, 0x2f862b1e, 0x382433de,
	0x6e9570e1, 0x698b6c23, 0x63ee66ce, 0x5dc660eb,
	0x40733c56, 0x4869447a, 0x4ffa4c3f, 0x571d539a,
	0x571d5a81, 0x4ffa539a, 0x48694c3f, 0x4073447a,
	0x5dc65a81, 0x63ee60eb, 0x698b66ce, 0x6e956c23,
	0x38243c56, 0x2f8633de, 0x26a72b1e, 0x1d932223,
	0x730670e1, 0x76d87503, 0x7a047883, 0x7c887b5c,
	0x145518f8, 0x0afb0fab, 0x01920647, 0xf826fcdb,
	0x7e5e7d89, 0x7f867f08, 0x7ffc7fd7, 0x7fc17ff5,
	0xeec6f374, 0xe57dea1e, 0xdc59e0e6, 0xd367d7d9,
	0x7ed47f61, 0x7d387e1c, 0x7aee7c29, 0x77f97989,
	0xcab2cf04, 0xc248c673, 0xba33be32, 0xb27fb64c,
	0x745e7640, 0x70227254, 0x6b4a6dc9, 0x65dd68a5,
	0xab36aecc, 0xa463a7bd, 0x9e0fa129, 0x98439b18,
	0x5fe262f1, 0x59635cb3, 0x526855f4, 0x4afa4ebf,
	0x93079593, 0x8e6290a1, 0x8a5b8c4a, 0x86f78894,
	0x4325471c, 0x3af23f16, 0x326d36b9, 0x29a32e10,
	0x843b8583, 0x822a831d, 0x80c88163, 0x80178059,
	0x209f2527, 0x176d1c0b, 0x0e1b12c7, 0x04b6096a,
	0x7ff97fff, 0x7fcd7fe8, 0x7f747fa6, 0x7eef7f37,
	0x025b0000, 0x071004b6, 0x0bc3096a, 0x10720e1b,
	0x7e3e7e9c, 0x7d617dd5, 0x7c597ce2, 0x7b257bc4,
	0x151b12c7, 0x19bd176d, 0x1e561c0b, 0x22e4209f,
	0x79c77a7c, 0x783f7908, 0x768d776b, 0x74b175a4,
	0x27672527, 0x2bdb29a3, 0x30412e10, 0x3496326d,
	0x72ae73b5, 0x7082719d, 0x6e306f5e, 0x6bb76cf8,
	0x38d836b9, 0x3d073af2, 0x41203f16, 0x45234325,
	0x69196a6c, 0x665667bc, 0x637064e7, 0x606761f0,
	0x490e471c, 0x4ce04afa, 0x50974ebf, 0x54325268,
	0x5d3d5ed6, 0x59f35b9c, 0x56895842, 0x530154c9,
	0x57b055f4, 0x5b0f5963, 0x5e4f5cb3, 0x616e5fe2,
	0x4f5d5133, 0x4b9d4d80, 0x47c349b3, 0x43d045cc,
	0x646b62f1, 0x674565dd, 0x69fc68a5, 0x6c8e6b4a,
	0x3fc541cd, 0x3ba43db7, 0x376f398c, 0x3326354d,
	0x6efa6dc9, 0x71407022, 0x735e7254, 0x7554745e,
	0x2ecc30fb, 0x2a612c98, 0x25e72826, 0x216123a6,
	0x77227640, 0x78c677f9, 0x7a417989, 0x7b917aee,
	0x1ccf1f19, 0x18331a82, 0x138e15e1, 0x0ee31139,
	0x7cb67c29, 0x7db07d38, 0x7e7e7e1c, 0x7f207ed4,
	0x0a320c8b, 0x057e07d9, 0x00c90324, 0xfc12fe6d,
	0x7f967f61, 0x7fe07fc1, 0x7ffe7ff5, 0x7fef7ffc,
	0xf75ef9b8, 0xf2acf504, 0xedfff054, 0xe958ebaa,
	0x7fb47fd7, 0x7f4c7f86, 0x7eb97f08, 0x7df97e5e,
	0xe4b9e707, 0xe023e26c, 0xdb98dddc, 0xd71ad958,
	0x7d0e7d89, 0x7bf77c88, 0x7ab57b5c, 0x79497a04,
	0xd2abd4e1, 0xce4bd079, 0xc9fccc21, 0xc5c0c7db,
	0x77b37883, 0x75f376d8, 0x740a7503, 0x71f97306,
	0xc198c3a9, 0xbd86bf8c, 0xb98bbb85, 0xb5a8b796,
	0x6fc070e1, 0x6d616e95, 0x6adb6c23, 0x6831698b,
	0xb1dfb3c0, 0xae31b005, 0xaaa0ac65, 0xa72ca8e2,
	0x656266ce, 0x627163ee, 0x5f5d60eb, 0x5c285dc6,
	0xa3d7a57e, 0xa0a2a239, 0x9d8e9f14, 0x9a9d9c11,
	0x58d35a81, 0x555f571d, 0x51ce539a, 0x4e204ffa,
	0x97ce9931, 0x95249674, 0x929e93dc, 0x903f916a,
	0x4a574c3f, 0x46744869, 0x4279447a, 0x3e674073,
	0x8e068f1e, 0x8bf58cf9, 0x8a0c8afc, 0x884c8927,
	0x3a3f3c56, 0x36033824, 0x31b433de, 0x2d542f86,
	0x86b6877c, 0x854a85fb, 0x840884a3, 0x82f18377,
	0x28e52b1e, 0x246726a7, 0x1fdc2223, 0x1b461d93,
	0x82068276, 0x814681a1, 0x80b380f7, 0x804b8079,
	0x16a718f8, 0x12001455, 0x0d530fab, 0x08a10afb,
	0x80108028, 0x80018003, 0x801f800a, 0x8069803e,
	0x03ed0647, 0xff360192, 0xfa81fcdb, 0xf5cdf826,
	0x80df809e, 0x8181812b, 0x824f81e3, 0x834982c7,
	0xf11cf374, 0xec71eec6, 0xe7ccea1e, 0xe330e57d,
	0x846e83d6, 0x85be8511, 0x87398676, 0x88dd8806,
	0xde9ee0e6, 0xda18dc59, 0xd59ed7d9, 0xd133d367,
	0x8aab89bf, 0x8ca18ba1, 0x8ebf8dab, 0x91058fdd,
	0xccd9cf04, 0xc890cab2, 0xc45bc673, 0xc03ac248,
	0x93719236, 0x960394b5, 0x98ba975a, 0x9b949a22,
	0xbc2fbe32, 0xb83cba33, 0xb462b64c, 0xb0a2b27f,
	0x9e919d0e, 0xa1b0a01d, 0xa4f0a34c, 0xa84fa69c,
	0xacfeaecc, 0xa976ab36, 0xa60ca7bd, 0xa2c2a463,
	0xabcdaa0b, 0xaf68ad97, 0xb31fb140, 0xb6f1b505,
	0x9f98a129, 0x9c8f9e0f, 0x99a99b18, 0x96e69843,
	0xbadcb8e3, 0xbedfbcda, 0xc2f8c0e9, 0xc727c50d,
	0x94489593, 0x91cf9307, 0x8f7d90a1, 0x8d518e62,
	0xcb69c946, 0xcfbecd92, 0xd424d1ef, 0xd898d65c,
	0x8b4e8c4a, 0x89728a5b, 0x87c08894, 0x863886f7,
	0xdd1bdad8, 0xe1a9df60, 0xe642e3f4, 0xeae4e892,
	0x84da8583, 0x83a6843b, 0x829e831d, 0x81c1822a,
	0xef8ded38, 0xf43cf1e4, 0xf8eff695, 0xfda4fb49,
	0x81108163, 0x808b80c8, 0x80328059, 0x80068017,
	0x025b0000, 0x071004b6, 0x0bc3096a, 0x10720e1b,
	0x80068001, 0x80328017, 0x808b8059, 0x811080c8,
	0x151b12c7, 0x19bd176d, 0x1e561c0b, 0x22e4209f,
	0x81c18163, 0x829e822a, 0x83a6831d, 0x84da843b,
	0x27672527, 0x2bdb29a3, 0x30412e10, 0x3496326d,
	0x86388583, 0x87c086f7, 0x89728894, 0x8b4e8a5b,
	0x38d836b9, 0x3d073af2, 0x41203f16, 0x45234325,
	0x8d518c4a, 0x8f7d8e62, 0x91cf90a1, 0x94489307,
	0x490e471c, 0x4ce04afa, 0x50974ebf, 0x54325268,
	0x96e69593, 0x99a99843, 0x9c8f9b18, 0x9f989e0f,
	0x57b055f4, 0x5b0f5963, 0x5e4f5cb3, 0x616e5fe2,
	0xa2c2a129, 0xa60ca463, 0xa976a7bd, 0xacfeab36,
	0x646b62f1, 0x674565dd, 0x69fc68a5, 0x6c8e6b4a,
	0xb0a2aecc, 0xb462b27f, 0xb83cb64c, 0xbc2fba33,
	0x6efa6dc9, 0x71407022, 0x735e7254, 0x7554745e,
	0xc03abe32, 0xc45bc248, 0xc890c673, 0xccd9cab2,
	0x77227640, 0x78c677f9, 0x7a417989, 0x7b917aee,
	0xd133cf04, 0xd59ed367, 0xda18d7d9, 0xde9edc59,
	0x7cb67c29, 0x7db07d38, 0x7e7e7e1c, 0x7f207ed4,
	0xe330e0e6, 0xe7cce57d, 0xec71ea1e, 0xf11ceec6,
	0x7f967f61, 0x7fe07fc1, 0x7ffe7ff5, 0x7fef7ffc,
	0xf5cdf374, 0xfa81f826, 0xff36fcdb, 0x03ed0192,
	0x7fb47fd7, 0x7f4c7f86, 0x7eb97f08, 0x7df97e5e,
	0x08a10647, 0x0d530afb, 0x12000fab, 0x16a71455,
	0x7d0e7d89, 0x7bf77c88, 0x7ab57b5c, 0x79497a04,
	0x1b4618f8, 0x1fdc1d93, 0x24672223, 0x28e526a7,
	0x77b37883, 0x75f376d8, 0x740a7503, 0x71f97306,
	0x2d542b1e, 0x31b42f86, 0x360333de, 0x3a3f3824,
	0x6fc070e1, 0x6d616e95, 0x6adb6c23, 0x6831698b,
	0x3e673c56, 0x42794073, 0x4674447a, 0x4a574869,
	0x656266ce, 0x627163ee, 0x5f5d60eb, 0x5c285dc6,
	0x4e204c3f, 0x51ce4ffa, 0x555f539a, 0x58d3571d,
	0x58d35a81, 0x555f571d, 0x51ce539a, 0x4e204ffa,
	0x5c285a81, 0x5f5d5dc6, 0x627160eb, 0x656263ee,
	0x4a574c3f, 0x46744869, 0x4279447a, 0x3e674073,
	0x683166ce, 0x6adb698b, 0x6d616c23, 0x6fc06e95,
	0x3a3f3c56, 0x36033824, 0x31b433de, 0x2d542f86,
	0x71f970e1, 0x740a7306, 0x75f37503, 0x77b376d8,
	0x28e52b1e, 0x246726a7, 0x1fdc2223, 0x1b461d93,
	0x79497883, 0x7ab57a04, 0x7bf77b5c, 0x7d0e7c88,
	0x16a718f8, 0x12001455, 0x0d530fab, 0x08a10afb,
	0x7df97d89, 0x7eb97e5e, 0x7f4c7f08, 0x7fb47f86,
	0x03ed0647, 0xff360192, 0xfa81fcdb, 0xf5cdf826,
	0x7fef7fd7, 0x7ffe7ffc, 0x7fe07ff5, 0x7f967fc1,
	0xf11cf374, 0xec71eec6, 0xe7ccea1e, 0xe330e57d,
	0x7f207f61, 0x7e7e7ed4, 0x7db07e1c, 0x7cb67d38,
	0xde9ee0e6, 0xda18dc59, 0xd59ed7d9, 0xd133d367,
	0x7b917c29, 0x7a417aee, 0x78c67989, 0x772277f9,
	0xccd9cf04, 0xc890cab2, 0xc45bc673, 0xc03ac248,
	0x75547640, 0x735e745e, 0x71407254, 0x6efa7022,
	0xbc2fbe32, 0xb83cba33, 0xb462b64c, 0xb0a2b27f,
	0x6c8e6dc9, 0x69fc6b4a, 0x674568a5, 0x646b65dd,
	0xacfeaecc, 0xa976ab36, 0xa60ca7bd, 0xa2c2a463,
	0x616e62f1, 0x5e4f5fe2, 0x5b0f5cb3, 0x57b05963,
	0x9f98a129, 0x9c8f9e0f, 0x99a99b18, 0x96e69843,
	0x543255f4, 0x50975268, 0x4ce04ebf, 0x490e4afa,
	0x94489593, 0x91cf9307, 0x8f7d90a1, 0x8d518e62,
	0x4523471c, 0x41204325, 0x3d073f16, 0x38d83af2,
	0x8b4e8c4a, 0x89728a5b, 0x87c08894, 0x863886f7,
	0x349636b9, 0x3041326d, 0x2bdb2e10, 0x276729a3,
	0x84da8583, 0x83a6843b, 0x829e831d, 0x81c1822a,
	0x22e42527, 0x1e56209f, 0x19bd1c0b, 0x151b176d,
	0x81108163, 0x808b80c8, 0x80328059, 0x80068017,
	0x107212c7, 0x0bc30e1b, 0x0710096a, 0x025b04b6,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_1D_M[] = {
	0x00000000, 0x00000000, 0x2d403fff, 0xd2bf0000,
	0x00000000, 0x00000000, 0xd2bf0000, 0xd2bfc000,
	0x3b203fff, 0x187d2d40, 0xe7820000, 0xc4dfd2bf,
	0xe7820000, 0xc4dfd2bf, 0xc4dfc000, 0xe782d2bf,
	0x3ec43fff, 0x35363b20, 0x238e2d40, 0x0c7c187d,
	0xf3830000, 0xdc71e782, 0xcac9d2bf, 0xc13bc4df,
	0xf3830000, 0xdc71e782, 0xcac9d2bf, 0xc13bc4df,
	0xc13bc000, 0xcac9c4df, 0xdc71d2bf, 0xf383e782,
	0x3fb03fff, 0x3d3e3ec4, 0x38703b20, 0x31783536,
	0xf9ba0000, 0xed6cf383, 0xe1d4e782, 0xd766dc71,
	0x28992d40, 0x1e2b238e, 0x1293187d, 0x06450c7c,
	0xce87d2bf, 0xc78fcac9, 0xc2c1c4df, 0xc04fc13b,
	0xf9ba0000, 0xed6cf383, 0xe1d4e782, 0xd766dc71,
	0xc04fc000, 0xc2c1c13b, 0xc78fc4df, 0xce87cac9,
	0xce87d2bf, 0xc78fcac9, 0xc2c1c4df, 0xc04fc13b,
	0xd766d2bf, 0xe1d4dc71, 0xed6ce782, 0xf9baf383,
	0x3feb3fff, 0x3f4e3fb0, 0x3e143ec4, 0x3c413d3e,
	0xfcdc0000, 0xf69cf9ba, 0xf073f383, 0xea70ed6c,
	0x39da3b20, 0x36e43870, 0x33673536, 0x2f6b3178,
	0xe4a3e782, 0xdf19e1d4, 0xd9e0dc71, 0xd505d766,
	0x2afa2d40, 0x261f2899, 0x20e6238e, 0x1b5c1e2b,
	0xd094d2bf, 0xcc98ce87, 0xc91bcac9, 0xc625c78f,
	0x158f187d, 0x0f8c1293, 0x09630c7c, 0x03230645,
	0xc3bec4df, 0xc1ebc2c1, 0xc0b1c13b, 0xc014c04f,
	0xfcdc0000, 0xf69cf9ba, 0xf073f383, 0xea70ed6c,
	0xc014c000, 0xc0b1c04f, 0xc1ebc13b, 0xc3bec2c1,
	0xe4a3e782, 0xdf19e1d4, 0xd9e0dc71, 0xd505d766,
	0xc625c4df, 0xc91bc78f, 0xcc98cac9, 0xd094ce87,
	0xd094d2bf, 0xcc98ce87, 0xc91bcac9, 0xc625c78f,
	0xd505d2bf, 0xd9e0d766, 0xdf19dc71, 0xe4a3e1d4,
	0xc3bec4df, 0xc1ebc2c1, 0xc0b1c13b, 0xc014c04f,
	0xea70e782, 0xf073ed6c, 0xf69cf383, 0xfcdcf9ba,
	0x3ffa3fff, 0x3fd33feb, 0x3f843fb0, 0x3f0e3f4e,
	0xfe6d0000, 0xfb4afcdc, 0xf82af9ba, 0xf50ff69c,
	0x3e713ec4, 0x3dae3e14, 0x3cc43d3e, 0x3bb53c41,
	0xf1faf383, 0xeeeef073, 0xebeced6c, 0xe8f7ea70,
	0x3a813b20, 0x392a39da, 0x37af3870, 0x361136e4,
	0xe610e782, 0xe339e4a3, 0xe074e1d4, 0xddc2df19,
	0x34523536, 0x32733367, 0x30753178, 0x2e592f6b,
	0xdb26dc71, 0xd8a0d9e0, 0xd632d766, 0xd3ded505,
	0x2c212d40, 0x29cd2afa, 0x275f2899, 0x24d9261f,
	0xd1a6d2bf, 0xcf8ad094, 0xcd8cce87, 0xcbadcc98,
	0x223d238e, 0x1f8b20e6, 0x1cc61e2b, 0x19ef1b5c,
	0xc9eecac9, 0xc850c91b, 0xc6d5c78f, 0xc57ec625,
	0x1708187d, 0x1413158f, 0x11111293, 0x0e050f8c,
	0xc44ac4df, 0xc33bc3be, 0xc251c2c1, 0xc18ec1eb,
	0x0af00c7c, 0x07d50963, 0x04b50645, 0x01920323,
	0xc0f1c13b, 0xc07bc0b1, 0xc02cc04f, 0xc005c014,
	0xfe6d0000, 0xfb4afcdc, 0xf82af9ba, 0xf50ff69c,
	0xc005c000, 0xc02cc014, 0xc07bc04f, 0xc0f1c0b1,
	0xf1faf383, 0xeeeef073, 0xebeced6c, 0xe8f7ea70,
	0xc18ec13b, 0xc251c1eb, 0xc33bc2c1, 0xc44ac3be,
	0xe610e782, 0xe339e4a3, 0xe074e1d4, 0xddc2df19,
	0xc57ec4df, 0xc6d5c625, 0xc850c78f, 0xc9eec91b,
	0xdb26dc71, 0xd8a0d9e0, 0xd632d766, 0xd3ded505,
	0xcbadcac9, 0xcd8ccc98, 0xcf8ace87, 0xd1a6d094,
	0xd1a6d2bf, 0xcf8ad094, 0xcd8cce87, 0xcbadcc98,
	0xd3ded2bf, 0xd632d505, 0xd8a0d766, 0xdb26d9e0,
	0xc9eecac9, 0xc850c91b, 0xc6d5c78f, 0xc57ec625,
	0xddc2dc71, 0xe074df19, 0xe339e1d4, 0xe610e4a3,
	0xc44ac4df, 0xc33bc3be, 0xc251c2c1, 0xc18ec1eb,
	0xe8f7e782, 0xebecea70, 0xeeeeed6c, 0xf1faf073,
	0xc0f1c13b, 0xc07bc0b1, 0xc02cc04f, 0xc005c014,
	0xf50ff383, 0xf82af69c, 0xfb4af9ba, 0xfe6dfcdc,
	0x3ffe3fff, 0x3ff43ffa, 0x3fe03feb, 0x3fc33fd3,
	0xff360000, 0xfda4fe6d, 0xfc13fcdc, 0xfa82fb4a,
	0x3f9b3fb0, 0x3f6a3f84, 0x3f2f3f4e, 0x3eea3f0e,
	0xf8f2f9ba, 0xf763f82a, 0xf5d5f69c, 0xf449f50f,
	0x3e9c3ec4, 0x3e443e71, 0x3de23e14, 0x3d773dae,
	0xf2bef383, 0xf136f1fa, 0xefb0f073, 0xee2ceeee,
	0x3d023d3e, 0x3c843cc4, 0x3bfc3c41, 0x3b6c3bb5,
	0xecaced6c, 0xeb2eebec, 0xe9b3ea70, 0xe83ce8f7,
	0x3ad23b20, 0x3a2f3a81, 0x398339da, 0x38ce392a,
	0xe6c9e782, 0xe559e610, 0xe3ede4a3, 0xe286e339,
	0x38113870, 0x374a37af, 0x367c36e4, 0x35a53611,
	0xe124e1d4, 0xdfc6e074, 0xde6ddf19, 0xdd19ddc2,
	0x34c53536, 0x33de3452, 0x32ee3367, 0x31f73273,
	0xdbcbdc71, 0xda82db26, 0xd93fd9e0, 0xd802d8a0,
	0x30f83178, 0x2ff13075, 0x2ee32f6b, 0x2dce2e59,
	0xd6cbd766, 0xd59bd632, 0xd471d505, 0xd34ed3de,
	0x2cb12d40, 0x2b8e2c21, 0x2a642afa, 0x293429cd,
	0xd231d2bf, 0xd11cd1a6, 0xd00ed094, 0xcf07cf8a,
	0x27fd2899, 0x26c0275f, 0x257d261f, 0x243424d9,
	0xce08ce87, 0xcd11cd8c, 0xcc21cc98, 0xcb3acbad,
	0x22e6238e, 0x2192223d, 0x203920e6, 0x1edb1f8b,
	0xca5acac9, 0xc983c9ee, 0xc8b5c91b, 0xc7eec850,
	0x1d791e2b, 0x1c121cc6, 0x1aa61b5c, 0x193619ef,
	0xc731c78f, 0xc67cc6d5, 0xc5d0c625, 0xc52dc57e,
	0x17c3187d, 0x164c1708, 0x14d1158f, 0x13531413,
	0xc493c4df, 0xc403c44a, 0xc37bc3be, 0xc2fdc33b,
	0x11d31293, 0x104f1111, 0x0ec90f8c, 0x0d410e05,
	0xc288c2c1, 0xc21dc251, 0xc1bbc1eb, 0xc163c18e,
	0x0bb60c7c, 0x0a2a0af0, 0x089c0963, 0x070d07d5,
	0xc115c13b, 0xc0d0c0f1, 0xc095c0b1, 0xc064c07b,
	0x057d0645, 0x03ec04b5, 0x025b0323, 0x00c90192,
	0xc03cc04f, 0xc01fc02c, 0xc00bc014, 0xc001c005,
	0xff360000, 0xfda4fe6d, 0xfc13fcdc, 0xfa82fb4a,
	0xc001c000, 0xc00bc005, 0xc01fc014, 0xc03cc02c,
	0xf8f2f9ba, 0xf763f82a, 0xf5d5f69c, 0xf449f50f,
	0xc064c04f, 0xc095c07b, 0xc0d0c0b1, 0xc115c0f1,
	0xf2bef383, 0xf136f1fa, 0xefb0f073, 0xee2ceeee,
	0xc163c13b, 0xc1bbc18e, 0xc21dc1eb, 0xc288c251,
	0xecaced6c, 0xeb2eebec, 0xe9b3ea70, 0xe83ce8f7,
	0xc2fdc2c1, 0xc37bc33b, 0xc403c3be, 0xc493c44a,
	0xe6c9e782, 0xe559e610, 0xe3ede4a3, 0xe286e339,
	0xc52dc4df, 0xc5d0c57e, 0xc67cc625, 0xc731c6d5,
	0xe124e1d4, 0xdfc6e074, 0xde6ddf19, 0xdd19ddc2,
	0xc7eec78f, 0xc8b5c850, 0xc983c91b, 0xca5ac9ee,
	0xdbcbdc71, 0xda82db26, 0xd93fd9e0, 0xd802d8a0,
	0xcb3acac9, 0xcc21cbad, 0xcd11cc98, 0xce08cd8c,
	0xd6cbd766, 0xd59bd632, 0xd471d505, 0xd34ed3de,
	0xcf07ce87, 0xd00ecf8a, 0xd11cd094, 0xd231d1a6,
	0xd231d2bf, 0xd11cd1a6, 0xd00ed094, 0xcf07cf8a,
	0xd34ed2bf, 0xd471d3de, 0xd59bd505, 0xd6cbd632,
	0xce08ce87, 0xcd11cd8c, 0xcc21cc98, 0xcb3acbad,
	0xd802d766, 0xd93fd8a0, 0xda82d9e0, 0xdbcbdb26,
	0xca5acac9, 0xc983c9ee, 0xc8b5c91b, 0xc7eec850,
	0xdd19dc71, 0xde6dddc2, 0xdfc6df19, 0xe124e074,
	0xc731c78f, 0xc67cc6d5, 0xc5d0c625, 0xc52dc57e,
	0xe286e1d4, 0xe3ede339, 0xe559e4a3, 0xe6c9e610,
	0xc493c4df, 0xc403c44a, 0xc37bc3be, 0xc2fdc33b,
	0xe83ce782, 0xe9b3e8f7, 0xeb2eea70, 0xecacebec,
	0xc288c2c1, 0xc21dc251, 0xc1bbc1eb, 0xc163c18e,
	0xee2ced6c, 0xefb0eeee, 0xf136f073, 0xf2bef1fa,
	0xc115c13b, 0xc0d0c0f1, 0xc095c0b1, 0xc064c07b,
	0xf449f383, 0xf5d5f50f, 0xf763f69c, 0xf8f2f82a,
	0xc03cc04f, 0xc01fc02c, 0xc00bc014, 0xc001c005,
	0xfa82f9ba, 0xfc13fb4a, 0xfda4fcdc, 0xff36fe6d,
	0x3fff3fff, 0x3ffc3ffe, 0x3ff73ffa, 0x3ff03ff4,
	0xff9b0000, 0xfed2ff36, 0xfe09fe6d, 0xfd40fda4,
	0x3fe63feb, 0x3fda3fe0, 0x3fcb3fd3, 0x3fba3fc3,
	0xfc77fcdc, 0xfbaffc13, 0xfae6fb4a, 0xfa1efa82,
	0x3fa63fb0, 0x3f903f9b, 0x3f773f84, 0x3f5c3f6a,
	0xf956f9ba, 0xf88ef8f2, 0xf7c6f82a, 0xf6fff763,
	0x3f3f3f4e, 0x3f1f3f2f, 0x3efc3f0e, 0x3ed83eea,
	0xf638f69c, 0xf572f5d5, 0xf4acf50f, 0xf3e6f449,
	0x3eb03ec4, 0x3e873e9c, 0x3e5b3e71, 0x3e2c3e44,
	0xf321f383, 0xf25cf2be, 0xf198f1fa, 0xf0d4f136,
	0x3dfb3e14, 0x3dc83de2, 0x3d923dae, 0x3d5a3d77,
	0xf011f073, 0xef4fefb0, 0xee8deeee, 0xedccee2c,
	0x3d203d3e, 0x3ce33d02, 0x3ca43cc4, 0x3c633c84,
	0xed0ced6c, 0xec4cecac, 0xeb8debec, 0xeacfeb2e,
	0x3c1f3c41, 0x3bd93bfc, 0x3b913bb5, 0x3b463b6c,
	0xea12ea70, 0xe955e9b3, 0xe899e8f7, 0xe7dfe83c,
	0x3af93b20, 0x3aaa3ad2, 0x3a583a81, 0x3a053a2f,
	0xe725e782, 0xe66ce6c9, 0xe5b4e610, 0xe4fee559,
	0x39af39da, 0x39573983, 0x38fc392a, 0x38a038ce,
	0xe448e4a3, 0xe393e3ed, 0xe2e0e339, 0xe22de286,
	0x38413870, 0x37e03811, 0x377d37af, 0x3718374a,
	0xe17ce1d4, 0xe0cce124, 0xe01de074, 0xdf6fdfc6,
	0x36b036e4, 0x3647367c, 0x35db3611, 0x356d35a5,
	0xdec3df19, 0xde17de6d, 0xdd6eddc2, 0xdcc5dd19,
	0x34fe3536, 0x348c34c5, 0x34183452, 0x33a233de,
	0xdc1edc71, 0xdb78dbcb, 0xdad4db26, 0xda31da82,
	0x332b3367, 0x32b132ee, 0x32353273, 0x31b831f7,
	0xd98fd9e0, 0xd8efd93f, 0xd851d8a0, 0xd7b4d802,
	0x31383178, 0x30b730f8, 0x30333075, 0x2fae2ff1,
	0xd718d766, 0xd67fd6cb, 0xd5e6d632, 0xd550d59b,
	0x2f272f6b, 0x2e9e2ee3, 0x2e142e59, 0x2d872dce,
	0xd4bbd505, 0xd427d471, 0xd396d3de, 0xd306d34e,
	0x2cf92d40, 0x2c692cb1, 0x2bd82c21, 0x2b442b8e,
	0xd278d2bf, 0xd1ebd231, 0xd161d1a6, 0xd0d8d11c,
	0x2aaf2afa, 0x2a192a64, 0x298029cd, 0x28e72934,
	0xd051d094, 0xcfccd00e, 0xcf48cf8a, 0xcec7cf07,
	0x284b2899, 0x27ae27fd, 0x2710275f, 0x267026c0,
	0xce47ce87, 0xcdcace08, 0xcd4ecd8c, 0xccd4cd11,
	0x25ce261f, 0x252b257d, 0x248724d9, 0x23e12434,
	0xcc5dcc98, 0xcbe7cc21, 0xcb73cbad, 0xcb01cb3a,
	0x233a238e, 0x229122e6, 0x21e8223d, 0x213c2192,
	0xca92cac9, 0xca24ca5a, 0xc9b8c9ee, 0xc94fc983,
	0x209020e6, 0x1fe22039, 0x1f331f8b, 0x1e831edb,
	0xc8e7c91b, 0xc882c8b5, 0xc81fc850, 0xc7bec7ee,
	0x1dd21e2b, 0x1d1f1d79, 0x1c6c1cc6, 0x1bb71c12,
	0xc75fc78f, 0xc703c731, 0xc6a8c6d5, 0xc650c67c,
	0x1b011b5c, 0x1a4b1aa6, 0x199319ef, 0x18da1936,
	0xc5fac625, 0xc5a7c5d0, 0xc555c57e, 0xc506c52d,
	0x1820187d, 0x176617c3, 0x16aa1708, 0x15ed164c,
	0xc4b9c4df, 0xc46ec493, 0xc426c44a, 0xc3e0c403,
	0x1530158f, 0x147214d1, 0x13b31413, 0x12f31353,
	0xc39cc3be, 0xc35bc37b, 0xc31cc33b, 0xc2dfc2fd,
	0x12331293, 0x117211d3, 0x10b01111, 0x0fee104f,
	0xc2a5c2c1, 0xc26dc288, 0xc237c251, 0xc204c21d,
	0x0f2b0f8c, 0x0e670ec9, 0x0da30e05, 0x0cde0d41,
	0xc1d3c1eb, 0xc1a4c1bb, 0xc178c18e, 0xc14fc163,
	0x0c190c7c, 0x0b530bb6, 0x0a8d0af0, 0x09c70a2a,
	0xc127c13b, 0xc103c115, 0xc0e0c0f1, 0xc0c0c0d0,
	0x09000963, 0x0839089c, 0x077107d5, 0x06a9070d,
	0xc0a3c0b1, 0xc088c095, 0xc06fc07b, 0xc059c064,
	0x05e10645, 0x0519057d, 0x045004b5, 0x038803ec,
	0xc045c04f, 0xc034c03c, 0xc025c02c, 0xc019c01f,
	0x02bf0323, 0x01f6025b, 0x012d0192, 0x006400c9,
	0xc00fc014, 0xc008c00b, 0xc003c005, 0xc000c001,
	0xff9b0000, 0xfed2ff36, 0xfe09fe6d, 0xfd40fda4,
	0xc000c000, 0xc003c001, 0xc008c005, 0xc00fc00b,
	0xfc77fcdc, 0xfbaffc13, 0xfae6fb4a, 0xfa1efa82,
	0xc019c014, 0xc025c01f, 0xc034c02c, 0xc045c03c,
	0xf956f9ba, 0xf88ef8f2, 0xf7c6f82a, 0xf6fff763,
	0xc059c04f, 0xc06fc064, 0xc088c07b, 0xc0a3c095,
	0xf638f69c, 0xf572f5d5, 0xf4acf50f, 0xf3e6f449,
	0xc0c0c0b1, 0xc0e0c0d0, 0xc103c0f1, 0xc127c115,
	0xf321f383, 0xf25cf2be, 0xf198f1fa, 0xf0d4f136,
	0xc14fc13b, 0xc178c163, 0xc1a4c18e, 0xc1d3c1bb,
	0xf011f073, 0xef4fefb0, 0xee8deeee, 0xedccee2c,
	0xc204c1eb, 0xc237c21d, 0xc26dc251, 0xc2a5c288,
	0xed0ced6c, 0xec4cecac, 0xeb8debec, 0xeacfeb2e,
	0xc2dfc2c1, 0xc31cc2fd, 0xc35bc33b, 0xc39cc37b,
	0xea12ea70, 0xe955e9b3, 0xe899e8f7, 0xe7dfe83c,
	0xc3e0c3be, 0xc426c403, 0xc46ec44a, 0xc4b9c493,
	0xe725e782, 0xe66ce6c9, 0xe5b4e610, 0xe4fee559,
	0xc506c4df, 0xc555c52d, 0xc5a7c57e, 0xc5fac5d0,
	0xe448e4a3, 0xe393e3ed, 0xe2e0e339, 0xe22de286,
	0xc650c625, 0xc6a8c67c, 0xc703c6d5, 0xc75fc731,
	0xe17ce1d4, 0xe0cce124, 0xe01de074, 0xdf6fdfc6,
	0xc7bec78f, 0xc81fc7ee, 0xc882c850, 0xc8e7c8b5,
	0xdec3df19, 0xde17de6d, 0xdd6eddc2, 0xdcc5dd19,
	0xc94fc91b, 0xc9b8c983, 0xca24c9ee, 0xca92ca5a,
	0xdc1edc71, 0xdb78dbcb, 0xdad4db26, 0xda31da82,
	0xcb01cac9, 0xcb73cb3a, 0xcbe7cbad, 0xcc5dcc21,
	0xd98fd9e0, 0xd8efd93f, 0xd851d8a0, 0xd7b4d802,
	0xccd4cc98, 0xcd4ecd11, 0xcdcacd8c, 0xce47ce08,
	0xd718d766, 0xd67fd6cb, 0xd5e6d632, 0xd550d59b,
	0xcec7ce87, 0xcf48cf07, 0xcfcccf8a, 0xd051d00e,
	0xd4bbd505, 0xd427d471, 0xd396d3de, 0xd306d34e,
	0xd0d8d094, 0xd161d11c, 0xd1ebd1a6, 0xd278d231,
	0xd278d2bf, 0xd1ebd231, 0xd161d1a6, 0xd0d8d11c,
	0xd306d2bf, 0xd396d34e, 0xd427d3de, 0xd4bbd471,
	0xd051d094, 0xcfccd00e, 0xcf48cf8a, 0xcec7cf07,
	0xd550d505, 0xd5e6d59b, 0xd67fd632, 0xd718d6cb,
	0xce47ce87, 0xcdcace08, 0xcd4ecd8c, 0xccd4cd11,
	0xd7b4d766, 0xd851d802, 0xd8efd8a0, 0xd98fd93f,
	0xcc5dcc98, 0xcbe7cc21, 0xcb73cbad, 0xcb01cb3a,
	0xda31d9e0, 0xdad4da82, 0xdb78db26, 0xdc1edbcb,
	0xca92cac9, 0xca24ca5a, 0xc9b8c9ee, 0xc94fc983,
	0xdcc5dc71, 0xdd6edd19, 0xde17ddc2, 0xdec3de6d,
	0xc8e7c91b, 0xc882c8b5, 0xc81fc850, 0xc7bec7ee,
	0xdf6fdf19, 0xe01ddfc6, 0xe0cce074, 0xe17ce124,
	0xc75fc78f, 0xc703c731, 0xc6a8c6d5, 0xc650c67c,
	0xe22de1d4, 0xe2e0e286, 0xe393e339, 0xe448e3ed,
	0xc5fac625, 0xc5a7c5d0, 0xc555c57e, 0xc506c52d,
	0xe4fee4a3, 0xe5b4e559, 0xe66ce610, 0xe725e6c9,
	0xc4b9c4df, 0xc46ec493, 0xc426c44a, 0xc3e0c403,
	0xe7dfe782, 0xe899e83c, 0xe955e8f7, 0xea12e9b3,
	0xc39cc3be, 0xc35bc37b, 0xc31cc33b, 0xc2dfc2fd,
	0xeacfea70, 0xeb8deb2e, 0xec4cebec, 0xed0cecac,
	0xc2a5c2c1, 0xc26dc288, 0xc237c251, 0xc204c21d,
	0xedcced6c, 0xee8dee2c, 0xef4feeee, 0xf011efb0,
	0xc1d3c1eb, 0xc1a4c1bb, 0xc178c18e, 0xc14fc163,
	0xf0d4f073, 0xf198f136, 0xf25cf1fa, 0xf321f2be,
	0xc127c13b, 0xc103c115, 0xc0e0c0f1, 0xc0c0c0d0,
	0xf3e6f383, 0xf4acf449, 0xf572f50f, 0xf638f5d5,
	0xc0a3c0b1, 0xc088c095, 0xc06fc07b, 0xc059c064,
	0xf6fff69c, 0xf7c6f763, 0xf88ef82a, 0xf956f8f2,
	0xc045c04f, 0xc034c03c, 0xc025c02c, 0xc019c01f,
	0xfa1ef9ba, 0xfae6fa82, 0xfbaffb4a, 0xfc77fc13,
	0xc00fc014, 0xc008c00b, 0xc003c005, 0xc000c001,
	0xfd40fcdc, 0xfe09fda4, 0xfed2fe6d, 0xff9bff36,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_1D_P[] = {
	0x00000000, 0x00000000, 0x2d403fff, 0xd2bf0000,
	0x00000000, 0x00000000, 0x2d400000, 0x2d403fff,
	0x3b203fff, 0x187d2d40, 0xe7820000, 0xc4dfd2bf,
	0x187d0000, 0x3b202d40, 0x3b203fff, 0x187d2d40,
	0x3ec43fff, 0x35363b20, 0x238e2d40, 0x0c7c187d,
	0x0c7c0000, 0x238e187d, 0x35362d40, 0x3ec43b20,
	0xf3830000, 0xdc71e782, 0xcac9d2bf, 0xc13bc4df,
	0x3ec43fff, 0x35363b20, 0x238e2d40, 0x0c7c187d,
	0x3fb03fff, 0x3d3e3ec4, 0x38703b20, 0x31783536,
	0x06450000, 0x12930c7c, 0x1e2b187d, 0x2899238e,
	0x28992d40, 0x1e2b238e, 0x1293187d, 0x06450c7c,
	0x31782d40, 0x38703536, 0x3d3e3b20, 0x3fb03ec4,
	0xf9ba0000, 0xed6cf383, 0xe1d4e782, 0xd766dc71,
	0x3fb03fff, 0x3d3e3ec4, 0x38703b20, 0x31783536,
	0xce87d2bf, 0xc78fcac9, 0xc2c1c4df, 0xc04fc13b,
	0x28992d40, 0x1e2b238e, 0x1293187d, 0x06450c7c,
	0x3feb3fff, 0x3f4e3fb0, 0x3e143ec4, 0x3c413d3e,
	0x03230000, 0x09630645, 0x0f8c0c7c, 0x158f1293,
	0x39da3b20, 0x36e43870, 0x33673536, 0x2f6b3178,
	0x1b5c187d, 0x20e61e2b, 0x261f238e, 0x2afa2899,
	0x2afa2d40, 0x261f2899, 0x20e6238e, 0x1b5c1e2b,
	0x2f6b2d40, 0x33673178, 0x36e43536, 0x39da3870,
	0x158f187d, 0x0f8c1293, 0x09630c7c, 0x03230645,
	0x3c413b20, 0x3e143d3e, 0x3f4e3ec4, 0x3feb3fb0,
	0xfcdc0000, 0xf69cf9ba, 0xf073f383, 0xea70ed6c,
	0x3feb3fff, 0x3f4e3fb0, 0x3e143ec4, 0x3c413d3e,
	0xe4a3e782, 0xdf19e1d4, 0xd9e0dc71, 0xd505d766,
	0x39da3b20, 0x36e43870, 0x33673536, 0x2f6b3178,
	0xd094d2bf, 0xcc98ce87, 0xc91bcac9, 0xc625c78f,
	0x2afa2d40, 0x261f2899, 0x20e6238e, 0x1b5c1e2b,
	0xc3bec4df, 0xc1ebc2c1, 0xc0b1c13b, 0xc014c04f,
	0x158f187d, 0x0f8c1293, 0x09630c7c, 0x03230645,
	0x3ffa3fff, 0x3fd33feb, 0x3f843fb0, 0x3f0e3f4e,
	0x01920000, 0x04b50323, 0x07d50645, 0x0af00963,
	0x3e713ec4, 0x3dae3e14, 0x3cc43d3e, 0x3bb53c41,
	0x0e050c7c, 0x11110f8c, 0x14131293, 0x1708158f,
	0x3a813b20, 0x392a39da, 0x37af3870, 0x361136e4,
	0x19ef187d, 0x1cc61b5c, 0x1f8b1e2b, 0x223d20e6,
	0x34523536, 0x32733367, 0x30753178, 0x2e592f6b,
	0x24d9238e, 0x275f261f, 0x29cd2899, 0x2c212afa,
	0x2c212d40, 0x29cd2afa, 0x275f2899, 0x24d9261f,
	0x2e592d40, 0x30752f6b, 0x32733178, 0x34523367,
	0x223d238e, 0x1f8b20e6, 0x1cc61e2b, 0x19ef1b5c,
	0x36113536, 0x37af36e4, 0x392a3870, 0x3a8139da,
	0x1708187d, 0x1413158f, 0x11111293, 0x0e050f8c,
	0x3bb53b20, 0x3cc43c41, 0x3dae3d3e, 0x3e713e14,
	0x0af00c7c, 0x07d50963, 0x04b50645, 0x01920323,
	0x3f0e3ec4, 0x3f843f4e, 0x3fd33fb0, 0x3ffa3feb,
	0xfe6d0000, 0xfb4afcdc, 0xf82af9ba, 0xf50ff69c,
	0x3ffa3fff, 0x3fd33feb, 0x3f843fb0, 0x3f0e3f4e,
	0xf1faf383, 0xeeeef073, 0xebeced6c, 0xe8f7ea70,
	0x3e713ec4, 0x3dae3e14, 0x3cc43d3e, 0x3bb53c41,
	0xe610e782, 0xe339e4a3, 0xe074e1d4, 0xddc2df19,
	0x3a813b20, 0x392a39da, 0x37af3870, 0x361136e4,
	0xdb26dc71, 0xd8a0d9e0, 0xd632d766, 0xd3ded505,
	0x34523536, 0x32733367, 0x30753178, 0x2e592f6b,
	0xd1a6d2bf, 0xcf8ad094, 0xcd8cce87, 0xcbadcc98,
	0x2c212d40, 0x29cd2afa, 0x275f2899, 0x24d9261f,
	0xc9eecac9, 0xc850c91b, 0xc6d5c78f, 0xc57ec625,
	0x223d238e, 0x1f8b20e6, 0x1cc61e2b, 0x19ef1b5c,
	0xc44ac4df, 0xc33bc3be, 0xc251c2c1, 0xc18ec1eb,
	0x1708187d, 0x1413158f, 0x11111293, 0x0e050f8c,
	0xc0f1c13b, 0xc07bc0b1, 0xc02cc04f, 0xc005c014,
	0x0af00c7c, 0x07d50963, 0x04b50645, 0x01920323,
	0x3ffe3fff, 0x3ff43ffa, 0x3fe03feb, 0x3fc33fd3,
	0x00c90000, 0x025b0192, 0x03ec0323, 0x057d04b5,
	0x3f9b3fb0, 0x3f6a3f84, 0x3f2f3f4e, 0x3eea3f0e,
	0x070d0645, 0x089c07d5, 0x0a2a0963, 0x0bb60af0,
	0x3e9c3ec4, 0x3e443e71, 0x3de23e14, 0x3d773dae,
	0x0d410c7c, 0x0ec90e05, 0x104f0f8c, 0x11d31111,
	0x3d023d3e, 0x3c843cc4, 0x3bfc3c41, 0x3b6c3bb5,
	0x13531293, 0x14d11413, 0x164c158f, 0x17c31708,
	0x3ad23b20, 0x3a2f3a81, 0x398339da, 0x38ce392a,
	0x1936187d, 0x1aa619ef, 0x1c121b5c, 0x1d791cc6,
	0x38113870, 0x374a37af, 0x367c36e4, 0x35a53611,
	0x1edb1e2b, 0x20391f8b, 0x219220e6, 0x22e6223d,
	0x34c53536, 0x33de3452, 0x32ee3367, 0x31f73273,
	0x2434238e, 0x257d24d9, 0x26c0261f, 0x27fd275f,
	0x30f83178, 0x2ff13075, 0x2ee32f6b, 0x2dce2e59,
	0x29342899, 0x2a6429cd, 0x2b8e2afa, 0x2cb12c21,
	0x2cb12d40, 0x2b8e2c21, 0x2a642afa, 0x293429cd,
	0x2dce2d40, 0x2ee32e59, 0x2ff12f6b, 0x30f83075,
	0x27fd2899, 0x26c0275f, 0x257d261f, 0x243424d9,
	0x31f73178, 0x32ee3273, 0x33de3367, 0x34c53452,
	0x22e6238e, 0x2192223d, 0x203920e6, 0x1edb1f8b,
	0x35a53536, 0x367c3611, 0x374a36e4, 0x381137af,
	0x1d791e2b, 0x1c121cc6, 0x1aa61b5c, 0x193619ef,
	0x38ce3870, 0x3983392a, 0x3a2f39da, 0x3ad23a81,
	0x17c3187d, 0x164c1708, 0x14d1158f, 0x13531413,
	0x3b6c3b20, 0x3bfc3bb5, 0x3c843c41, 0x3d023cc4,
	0x11d31293, 0x104f1111, 0x0ec90f8c, 0x0d410e05,
	0x3d773d3e, 0x3de23dae, 0x3e443e14, 0x3e9c3e71,
	0x0bb60c7c, 0x0a2a0af0, 0x089c0963, 0x070d07d5,
	0x3eea3ec4, 0x3f2f3f0e, 0x3f6a3f4e, 0x3f9b3f84,
	0x057d0645, 0x03ec04b5, 0x025b0323, 0x00c90192,
	0x3fc33fb0, 0x3fe03fd3, 0x3ff43feb, 0x3ffe3ffa,
	0xff360000, 0xfda4fe6d, 0xfc13fcdc, 0xfa82fb4a,
	0x3ffe3fff, 0x3ff43ffa, 0x3fe03feb, 0x3fc33fd3,
	0xf8f2f9ba, 0xf763f82a, 0xf5d5f69c, 0xf449f50f,
	0x3f9b3fb0, 0x3f6a3f84, 0x3f2f3f4e, 0x3eea3f0e,
	0xf2bef383, 0xf136f1fa, 0xefb0f073, 0xee2ceeee,
	0x3e9c3ec4, 0x3e443e71, 0x3de23e14, 0x3d773dae,
	0xecaced6c, 0xeb2eebec, 0xe9b3ea70, 0xe83ce8f7,
	0x3d023d3e, 0x3c843cc4, 0x3bfc3c41, 0x3b6c3bb5,
	0xe6c9e782, 0xe559e610, 0xe3ede4a3, 0xe286e339,
	0x3ad23b20, 0x3a2f3a81, 0x398339da, 0x38ce392a,
	0xe124e1d4, 0xdfc6e074, 0xde6ddf19, 0xdd19ddc2,
	0x38113870, 0x374a37af, 0x367c36e4, 0x35a53611,
	0xdbcbdc71, 0xda82db26, 0xd93fd9e0, 0xd802d8a0,
	0x34c53536, 0x33de3452, 0x32ee3367, 0x31f73273,
	0xd6cbd766, 0xd59bd632, 0xd471d505, 0xd34ed3de,
	0x30f83178, 0x2ff13075, 0x2ee32f6b, 0x2dce2e59,
	0xd231d2bf, 0xd11cd1a6, 0xd00ed094, 0xcf07cf8a,
	0x2cb12d40, 0x2b8e2c21, 0x2a642afa, 0x293429cd,
	0xce08ce87, 0xcd11cd8c, 0xcc21cc98, 0xcb3acbad,
	0x27fd2899, 0x26c0275f, 0x257d261f, 0x243424d9,
	0xca5acac9, 0xc983c9ee, 0xc8b5c91b, 0xc7eec850,
	0x22e6238e, 0x2192223d, 0x203920e6, 0x1edb1f8b,
	0xc731c78f, 0xc67cc6d5, 0xc5d0c625, 0xc52dc57e,
	0x1d791e2b, 0x1c121cc6, 0x1aa61b5c, 0x193619ef,
	0xc493c4df, 0xc403c44a, 0xc37bc3be, 0xc2fdc33b,
	0x17c3187d, 0x164c1708, 0x14d1158f, 0x13531413,
	0xc288c2c1, 0xc21dc251, 0xc1bbc1eb, 0xc163c18e,
	0x11d31293, 0x104f1111, 0x0ec90f8c, 0x0d410e05,
	0xc115c13b, 0xc0d0c0f1, 0xc095c0b1, 0xc064c07b,
	0x0bb60c7c, 0x0a2a0af0, 0x089c0963, 0x070d07d5,
	0xc03cc04f, 0xc01fc02c, 0xc00bc014, 0xc001c005,
	0x057d0645, 0x03ec04b5, 0x025b0323, 0x00c90192,
	0x3fff3fff, 0x3ffc3ffe, 0x3ff73ffa, 0x3ff03ff4,
	0x00640000, 0x012d00c9, 0x01f60192, 0x02bf025b,
	0x3fe63feb, 0x3fda3fe0, 0x3fcb3fd3, 0x3fba3fc3,
	0x03880323, 0x045003ec, 0x051904b5, 0x05e1057d,
	0x3fa63fb0, 0x3f903f9b, 0x3f773f84, 0x3f5c3f6a,
	0x06a90645, 0x0771070d, 0x083907d5, 0x0900089c,
	0x3f3f3f4e, 0x3f1f3f2f, 0x3efc3f0e, 0x3ed83eea,
	0x09c70963, 0x0a8d0a2a, 0x0b530af0, 0x0c190bb6,
	0x3eb03ec4, 0x3e873e9c, 0x3e5b3e71, 0x3e2c3e44,
	0x0cde0c7c, 0x0da30d41, 0x0e670e05, 0x0f2b0ec9,
	0x3dfb3e14, 0x3dc83de2, 0x3d923dae, 0x3d5a3d77,
	0x0fee0f8c, 0x10b0104f, 0x11721111, 0x123311d3,
	0x3d203d3e, 0x3ce33d02, 0x3ca43cc4, 0x3c633c84,
	0x12f31293, 0x13b31353, 0x14721413, 0x153014d1,
	0x3c1f3c41, 0x3bd93bfc, 0x3b913bb5, 0x3b463b6c,
	0x15ed158f, 0x16aa164c, 0x17661708, 0x182017c3,
	0x3af93b20, 0x3aaa3ad2, 0x3a583a81, 0x3a053a2f,
	0x18da187d, 0x19931936, 0x1a4b19ef, 0x1b011aa6,
	0x39af39da, 0x39573983, 0x38fc392a, 0x38a038ce,
	0x1bb71b5c, 0x1c6c1c12, 0x1d1f1cc6, 0x1dd21d79,
	0x38413870, 0x37e03811, 0x377d37af, 0x3718374a,
	0x1e831e2b, 0x1f331edb, 0x1fe21f8b, 0x20902039,
	0x36b036e4, 0x3647367c, 0x35db3611, 0x356d35a5,
	0x213c20e6, 0x21e82192, 0x2291223d, 0x233a22e6,
	0x34fe3536, 0x348c34c5, 0x34183452, 0x33a233de,
	0x23e1238e, 0x24872434, 0x252b24d9, 0x25ce257d,
	0x332b3367, 0x32b132ee, 0x32353273, 0x31b831f7,
	0x2670261f, 0x271026c0, 0x27ae275f, 0x284b27fd,
	0x31383178, 0x30b730f8, 0x30333075, 0x2fae2ff1,
	0x28e72899, 0x29802934, 0x2a1929cd, 0x2aaf2a64,
	0x2f272f6b, 0x2e9e2ee3, 0x2e142e59, 0x2d872dce,
	0x2b442afa, 0x2bd82b8e, 0x2c692c21, 0x2cf92cb1,
	0x2cf92d40, 0x2c692cb1, 0x2bd82c21, 0x2b442b8e,
	0x2d872d40, 0x2e142dce, 0x2e9e2e59, 0x2f272ee3,
	0x2aaf2afa, 0x2a192a64, 0x298029cd, 0x28e72934,
	0x2fae2f6b, 0x30332ff1, 0x30b73075, 0x313830f8,
	0x284b2899, 0x27ae27fd, 0x2710275f, 0x267026c0,
	0x31b83178, 0x323531f7, 0x32b13273, 0x332b32ee,
	0x25ce261f, 0x252b257d, 0x248724d9, 0x23e12434,
	0x33a23367, 0x341833de, 0x348c3452, 0x34fe34c5,
	0x233a238e, 0x229122e6, 0x21e8223d, 0x213c2192,
	0x356d3536, 0x35db35a5, 0x36473611, 0x36b0367c,
	0x209020e6, 0x1fe22039, 0x1f331f8b, 0x1e831edb,
	0x371836e4, 0x377d374a, 0x37e037af, 0x38413811,
	0x1dd21e2b, 0x1d1f1d79, 0x1c6c1cc6, 0x1bb71c12,
	0x38a03870, 0x38fc38ce, 0x3957392a, 0x39af3983,
	0x1b011b5c, 0x1a4b1aa6, 0x199319ef, 0x18da1936,
	0x3a0539da, 0x3a583a2f, 0x3aaa3a81, 0x3af93ad2,
	0x1820187d, 0x176617c3, 0x16aa1708, 0x15ed164c,
	0x3b463b20, 0x3b913b6c, 0x3bd93bb5, 0x3c1f3bfc,
	0x1530158f, 0x147214d1, 0x13b31413, 0x12f31353,
	0x3c633c41, 0x3ca43c84, 0x3ce33cc4, 0x3d203d02,
	0x12331293, 0x117211d3, 0x10b01111, 0x0fee104f,
	0x3d5a3d3e, 0x3d923d77, 0x3dc83dae, 0x3dfb3de2,
	0x0f2b0f8c, 0x0e670ec9, 0x0da30e05, 0x0cde0d41,
	0x3e2c3e14, 0x3e5b3e44, 0x3e873e71, 0x3eb03e9c,
	0x0c190c7c, 0x0b530bb6, 0x0a8d0af0, 0x09c70a2a,
	0x3ed83ec4, 0x3efc3eea, 0x3f1f3f0e, 0x3f3f3f2f,
	0x09000963, 0x0839089c, 0x077107d5, 0x06a9070d,
	0x3f5c3f4e, 0x3f773f6a, 0x3f903f84, 0x3fa63f9b,
	0x05e10645, 0x0519057d, 0x045004b5, 0x038803ec,
	0x3fba3fb0, 0x3fcb3fc3, 0x3fda3fd3, 0x3fe63fe0,
	0x02bf0323, 0x01f6025b, 0x012d0192, 0x006400c9,
	0x3ff03feb, 0x3ff73ff4, 0x3ffc3ffa, 0x3fff3ffe,
	0xff9b0000, 0xfed2ff36, 0xfe09fe6d, 0xfd40fda4,
	0x3fff3fff, 0x3ffc3ffe, 0x3ff73ffa, 0x3ff03ff4,
	0xfc77fcdc, 0xfbaffc13, 0xfae6fb4a, 0xfa1efa82,
	0x3fe63feb, 0x3fda3fe0, 0x3fcb3fd3, 0x3fba3fc3,
	0xf956f9ba, 0xf88ef8f2, 0xf7c6f82a, 0xf6fff763,
	0x3fa63fb0, 0x3f903f9b, 0x3f773f84, 0x3f5c3f6a,
	0xf638f69c, 0xf572f5d5, 0xf4acf50f, 0xf3e6f449,
	0x3f3f3f4e, 0x3f1f3f2f, 0x3efc3f0e, 0x3ed83eea,
	0xf321f383, 0xf25cf2be, 0xf198f1fa, 0xf0d4f136,
	0x3eb03ec4, 0x3e873e9c, 0x3e5b3e71, 0x3e2c3e44,
	0xf011f073, 0xef4fefb0, 0xee8deeee, 0xedccee2c,
	0x3dfb3e14, 0x3dc83de2, 0x3d923dae, 0x3d5a3d77,
	0xed0ced6c, 0xec4cecac, 0xeb8debec, 0xeacfeb2e,
	0x3d203d3e, 0x3ce33d02, 0x3ca43cc4, 0x3c633c84,
	0xea12ea70, 0xe955e9b3, 0xe899e8f7, 0xe7dfe83c,
	0x3c1f3c41, 0x3bd93bfc, 0x3b913bb5, 0x3b463b6c,
	0xe725e782, 0xe66ce6c9, 0xe5b4e610, 0xe4fee559,
	0x3af93b20, 0x3aaa3ad2, 0x3a583a81, 0x3a053a2f,
	0xe448e4a3, 0xe393e3ed, 0xe2e0e339, 0xe22de286,
	0x39af39da, 0x39573983, 0x38fc392a, 0x38a038ce,
	0xe17ce1d4, 0xe0cce124, 0xe01de074, 0xdf6fdfc6,
	0x38413870, 0x37e03811, 0x377d37af, 0x3718374a,
	0xdec3df19, 0xde17de6d, 0xdd6eddc2, 0xdcc5dd19,
	0x36b036e4, 0x3647367c, 0x35db3611, 0x356d35a5,
	0xdc1edc71, 0xdb78dbcb, 0xdad4db26, 0xda31da82,
	0x34fe3536, 0x348c34c5, 0x34183452, 0x33a233de,
	0xd98fd9e0, 0xd8efd93f, 0xd851d8a0, 0xd7b4d802,
	0x332b3367, 0x32b132ee, 0x32353273, 0x31b831f7,
	0xd718d766, 0xd67fd6cb, 0xd5e6d632, 0xd550d59b,
	0x31383178, 0x30b730f8, 0x30333075, 0x2fae2ff1,
	0xd4bbd505, 0xd427d471, 0xd396d3de, 0xd306d34e,
	0x2f272f6b, 0x2e9e2ee3, 0x2e142e59, 0x2d872dce,
	0xd278d2bf, 0xd1ebd231, 0xd161d1a6, 0xd0d8d11c,
	0x2cf92d40, 0x2c692cb1, 0x2bd82c21, 0x2b442b8e,
	0xd051d094, 0xcfccd00e, 0xcf48cf8a, 0xcec7cf07,
	0x2aaf2afa, 0x2a192a64, 0x298029cd, 0x28e72934,
	0xce47ce87, 0xcdcace08, 0xcd4ecd8c, 0xccd4cd11,
	0x284b2899, 0x27ae27fd, 0x2710275f, 0x267026c0,
	0xcc5dcc98, 0xcbe7cc21, 0xcb73cbad, 0xcb01cb3a,
	0x25ce261f, 0x252b257d, 0x248724d9, 0x23e12434,
	0xca92cac9, 0xca24ca5a, 0xc9b8c9ee, 0xc94fc983,
	0x233a238e, 0x229122e6, 0x21e8223d, 0x213c2192,
	0xc8e7c91b, 0xc882c8b5, 0xc81fc850, 0xc7bec7ee,
	0x209020e6, 0x1fe22039, 0x1f331f8b, 0x1e831edb,
	0xc75fc78f, 0xc703c731, 0xc6a8c6d5, 0xc650c67c,
	0x1dd21e2b, 0x1d1f1d79, 0x1c6c1cc6, 0x1bb71c12,
	0xc5fac625, 0xc5a7c5d0, 0xc555c57e, 0xc506c52d,
	0x1b011b5c, 0x1a4b1aa6, 0x199319ef, 0x18da1936,
	0xc4b9c4df, 0xc46ec493, 0xc426c44a, 0xc3e0c403,
	0x1820187d, 0x176617c3, 0x16aa1708, 0x15ed164c,
	0xc39cc3be, 0xc35bc37b, 0xc31cc33b, 0xc2dfc2fd,
	0x1530158f, 0x147214d1, 0x13b31413, 0x12f31353,
	0xc2a5c2c1, 0xc26dc288, 0xc237c251, 0xc204c21d,
	0x12331293, 0x117211d3, 0x10b01111, 0x0fee104f,
	0xc1d3c1eb, 0xc1a4c1bb, 0xc178c18e, 0xc14fc163,
	0x0f2b0f8c, 0x0e670ec9, 0x0da30e05, 0x0cde0d41,
	0xc127c13b, 0xc103c115, 0xc0e0c0f1, 0xc0c0c0d0,
	0x0c190c7c, 0x0b530bb6, 0x0a8d0af0, 0x09c70a2a,
	0xc0a3c0b1, 0xc088c095, 0xc06fc07b, 0xc059c064,
	0x09000963, 0x0839089c, 0x077107d5, 0x06a9070d,
	0xc045c04f, 0xc034c03c, 0xc025c02c, 0xc019c01f,
	0x05e10645, 0x0519057d, 0x045004b5, 0x038803ec,
	0xc00fc014, 0xc008c00b, 0xc003c005, 0xc000c001,
	0x02bf0323, 0x01f6025b, 0x012d0192, 0x006400c9,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_3D_M[] = {
	0x00000000, 0x00000000, 0xd2bf3fff, 0x2d400000,
	0x00000000, 0x00000000, 0xd2bf0000, 0xd2bf3fff,
	0x187d3fff, 0xc4dfd2bf, 0x3b200000, 0xe7822d40,
	0xc4df0000, 0x187dd2bf, 0x187d3fff, 0xc4dfd2bf,
	0x35363fff, 0xf383187d, 0xc13bd2bf, 0xdc71c4df,
	0xdc710000, 0xc13bc4df, 0xf383d2bf, 0x3536187d,
	0x238e0000, 0x3ec43b20, 0x0c7c2d40, 0xcac9e782,
	0x35363fff, 0xf383187d, 0xc13bd2bf, 0xdc71c4df,
	0x3d3e3fff, 0x28993536, 0x0645187d, 0xe1d4f383,
	0xed6c0000, 0xce87dc71, 0xc04fc4df, 0xc78fc13b,
	0xc78fd2bf, 0xc04fc13b, 0xce87c4df, 0xed6cdc71,
	0xe1d4d2bf, 0x0645f383, 0x2899187d, 0x3d3e3536,
	0x12930000, 0x3178238e, 0x3fb03b20, 0x38703ec4,
	0x3d3e3fff, 0x28993536, 0x0645187d, 0xe1d4f383,
	0x1e2b2d40, 0xf9ba0c7c, 0xd766e782, 0xc2c1cac9,
	0xc78fd2bf, 0xc04fc13b, 0xce87c4df, 0xed6cdc71,
	0x3f4e3fff, 0x39da3d3e, 0x2f6b3536, 0x20e62899,
	0xf69c0000, 0xe4a3ed6c, 0xd505dc71, 0xc91bce87,
	0x0f8c187d, 0xfcdc0645, 0xea70f383, 0xd9e0e1d4,
	0xc1ebc4df, 0xc014c04f, 0xc3bec13b, 0xcc98c78f,
	0xcc98d2bf, 0xc3bec78f, 0xc014c13b, 0xc1ebc04f,
	0xd9e0d2bf, 0xea70e1d4, 0xfcdcf383, 0x0f8c0645,
	0xc91bc4df, 0xd505ce87, 0xe4a3dc71, 0xf69ced6c,
	0x20e6187d, 0x2f6b2899, 0x39da3536, 0x3f4e3d3e,
	0x09630000, 0x1b5c1293, 0x2afa238e, 0x36e43178,
	0x3f4e3fff, 0x39da3d3e, 0x2f6b3536, 0x20e62899,
	0x3e143b20, 0x3feb3fb0, 0x3c413ec4, 0x33673870,
	0x0f8c187d, 0xfcdc0645, 0xea70f383, 0xd9e0e1d4,
	0x261f2d40, 0x158f1e2b, 0x03230c7c, 0xf073f9ba,
	0xcc98d2bf, 0xc3bec78f, 0xc014c13b, 0xc1ebc04f,
	0xdf19e782, 0xd094d766, 0xc625cac9, 0xc0b1c2c1,
	0xc91bc4df, 0xd505ce87, 0xe4a3dc71, 0xf69ced6c,
	0x3fd33fff, 0x3e713f4e, 0x3bb53d3e, 0x37af39da,
	0xfb4a0000, 0xf1faf69c, 0xe8f7ed6c, 0xe074e4a3,
	0x32733536, 0x2c212f6b, 0x24d92899, 0x1cc620e6,
	0xd8a0dc71, 0xd1a6d505, 0xcbadce87, 0xc6d5c91b,
	0x1413187d, 0x0af00f8c, 0x01920645, 0xf82afcdc,
	0xc33bc4df, 0xc0f1c1eb, 0xc005c04f, 0xc07bc014,
	0xeeeef383, 0xe610ea70, 0xddc2e1d4, 0xd632d9e0,
	0xc251c13b, 0xc57ec3be, 0xc9eec78f, 0xcf8acc98,
	0xcf8ad2bf, 0xc9eecc98, 0xc57ec78f, 0xc251c3be,
	0xd632d2bf, 0xddc2d9e0, 0xe610e1d4, 0xeeeeea70,
	0xc07bc13b, 0xc005c014, 0xc0f1c04f, 0xc33bc1eb,
	0xf82af383, 0x0192fcdc, 0x0af00645, 0x14130f8c,
	0xc6d5c4df, 0xcbadc91b, 0xd1a6ce87, 0xd8a0d505,
	0x1cc6187d, 0x24d920e6, 0x2c212899, 0x32732f6b,
	0xe074dc71, 0xe8f7e4a3, 0xf1faed6c, 0xfb4af69c,
	0x37af3536, 0x3bb539da, 0x3e713d3e, 0x3fd33f4e,
	0x04b50000, 0x0e050963, 0x17081293, 0x1f8b1b5c,
	0x3fd33fff, 0x3e713f4e, 0x3bb53d3e, 0x37af39da,
	0x275f238e, 0x2e592afa, 0x34523178, 0x392a36e4,
	0x32733536, 0x2c212f6b, 0x24d92899, 0x1cc620e6,
	0x3cc43b20, 0x3f0e3e14, 0x3ffa3fb0, 0x3f843feb,
	0x1413187d, 0x0af00f8c, 0x01920645, 0xf82afcdc,
	0x3dae3ec4, 0x3a813c41, 0x36113870, 0x30753367,
	0xeeeef383, 0xe610ea70, 0xddc2e1d4, 0xd632d9e0,
	0x29cd2d40, 0x223d261f, 0x19ef1e2b, 0x1111158f,
	0xcf8ad2bf, 0xc9eecc98, 0xc57ec78f, 0xc251c3be,
	0x07d50c7c, 0xfe6d0323, 0xf50ff9ba, 0xebecf073,
	0xc07bc13b, 0xc005c014, 0xc0f1c04f, 0xc33bc1eb,
	0xe339e782, 0xdb26df19, 0xd3ded766, 0xcd8cd094,
	0xc6d5c4df, 0xcbadc91b, 0xd1a6ce87, 0xd8a0d505,
	0xc850cac9, 0xc44ac625, 0xc18ec2c1, 0xc02cc0b1,
	0xe074dc71, 0xe8f7e4a3, 0xf1faed6c, 0xfb4af69c,
	0x3ff43fff, 0x3f9b3fd3, 0x3eea3f4e, 0x3de23e71,
	0xfda40000, 0xf8f2fb4a, 0xf449f69c, 0xefb0f1fa,
	0x3c843d3e, 0x3ad23bb5, 0x38ce39da, 0x367c37af,
	0xeb2eed6c, 0xe6c9e8f7, 0xe286e4a3, 0xde6de074,
	0x33de3536, 0x30f83273, 0x2dce2f6b, 0x2a642c21,
	0xda82dc71, 0xd6cbd8a0, 0xd34ed505, 0xd00ed1a6,
	0x26c02899, 0x22e624d9, 0x1edb20e6, 0x1aa61cc6,
	0xcd11ce87, 0xca5acbad, 0xc7eec91b, 0xc5d0c6d5,
	0x164c187d, 0x11d31413, 0x0d410f8c, 0x089c0af0,
	0xc403c4df, 0xc288c33b, 0xc163c1eb, 0xc095c0f1,
	0x03ec0645, 0xff360192, 0xfa82fcdc, 0xf5d5f82a,
	0xc01fc04f, 0xc001c005, 0xc03cc014, 0xc0d0c07b,
	0xf136f383, 0xecaceeee, 0xe83cea70, 0xe3ede610,
	0xc1bbc13b, 0xc2fdc251, 0xc493c3be, 0xc67cc57e,
	0xdfc6e1d4, 0xdbcbddc2, 0xd802d9e0, 0xd471d632,
	0xc8b5c78f, 0xcb3ac9ee, 0xce08cc98, 0xd11ccf8a,
	0xd11cd2bf, 0xce08cf8a, 0xcb3acc98, 0xc8b5c9ee,
	0xd471d2bf, 0xd802d632, 0xdbcbd9e0, 0xdfc6ddc2,
	0xc67cc78f, 0xc493c57e, 0xc2fdc3be, 0xc1bbc251,
	0xe3ede1d4, 0xe83ce610, 0xecacea70, 0xf136eeee,
	0xc0d0c13b, 0xc03cc07b, 0xc001c014, 0xc01fc005,
	0xf5d5f383, 0xfa82f82a, 0xff36fcdc, 0x03ec0192,
	0xc095c04f, 0xc163c0f1, 0xc288c1eb, 0xc403c33b,
	0x089c0645, 0x0d410af0, 0x11d30f8c, 0x164c1413,
	0xc5d0c4df, 0xc7eec6d5, 0xca5ac91b, 0xcd11cbad,
	0x1aa6187d, 0x1edb1cc6, 0x22e620e6, 0x26c024d9,
	0xd00ece87, 0xd34ed1a6, 0xd6cbd505, 0xda82d8a0,
	0x2a642899, 0x2dce2c21, 0x30f82f6b, 0x33de3273,
	0xde6ddc71, 0xe286e074, 0xe6c9e4a3, 0xeb2ee8f7,
	0x367c3536, 0x38ce37af, 0x3ad239da, 0x3c843bb5,
	0xefb0ed6c, 0xf449f1fa, 0xf8f2f69c, 0xfda4fb4a,
	0x3de23d3e, 0x3eea3e71, 0x3f9b3f4e, 0x3ff43fd3,
	0x025b0000, 0x070d04b5, 0x0bb60963, 0x104f0e05,
	0x3ff43fff, 0x3f9b3fd3, 0x3eea3f4e, 0x3de23e71,
	0x14d11293, 0x19361708, 0x1d791b5c, 0x21921f8b,
	0x3c843d3e, 0x3ad23bb5, 0x38ce39da, 0x367c37af,
	0x257d238e, 0x2934275f, 0x2cb12afa, 0x2ff12e59,
	0x33de3536, 0x30f83273, 0x2dce2f6b, 0x2a642c21,
	0x32ee3178, 0x35a53452, 0x381136e4, 0x3a2f392a,
	0x26c02899, 0x22e624d9, 0x1edb20e6, 0x1aa61cc6,
	0x3bfc3b20, 0x3d773cc4, 0x3e9c3e14, 0x3f6a3f0e,
	0x164c187d, 0x11d31413, 0x0d410f8c, 0x089c0af0,
	0x3fe03fb0, 0x3ffe3ffa, 0x3fc33feb, 0x3f2f3f84,
	0x03ec0645, 0xff360192, 0xfa82fcdc, 0xf5d5f82a,
	0x3e443ec4, 0x3d023dae, 0x3b6c3c41, 0x39833a81,
	0xf136f383, 0xecaceeee, 0xe83cea70, 0xe3ede610,
	0x374a3870, 0x34c53611, 0x31f73367, 0x2ee33075,
	0xdfc6e1d4, 0xdbcbddc2, 0xd802d9e0, 0xd471d632,
	0x2b8e2d40, 0x27fd29cd, 0x2434261f, 0x2039223d,
	0xd11cd2bf, 0xce08cf8a, 0xcb3acc98, 0xc8b5c9ee,
	0x1c121e2b, 0x17c319ef, 0x1353158f, 0x0ec91111,
	0xc67cc78f, 0xc493c57e, 0xc2fdc3be, 0xc1bbc251,
	0x0a2a0c7c, 0x057d07d5, 0x00c90323, 0xfc13fe6d,
	0xc0d0c13b, 0xc03cc07b, 0xc001c014, 0xc01fc005,
	0xf763f9ba, 0xf2bef50f, 0xee2cf073, 0xe9b3ebec,
	0xc095c04f, 0xc163c0f1, 0xc288c1eb, 0xc403c33b,
	0xe559e782, 0xe124e339, 0xdd19df19, 0xd93fdb26,
	0xc5d0c4df, 0xc7eec6d5, 0xca5ac91b, 0xcd11cbad,
	0xd59bd766, 0xd231d3de, 0xcf07d094, 0xcc21cd8c,
	0xd00ece87, 0xd34ed1a6, 0xd6cbd505, 0xda82d8a0,
	0xc983cac9, 0xc731c850, 0xc52dc625, 0xc37bc44a,
	0xde6ddc71, 0xe286e074, 0xe6c9e4a3, 0xeb2ee8f7,
	0xc21dc2c1, 0xc115c18e, 0xc064c0b1, 0xc00bc02c,
	0xefb0ed6c, 0xf449f1fa, 0xf8f2f69c, 0xfda4fb4a,
	0x3ffc3fff, 0x3fe63ff4, 0x3fba3fd3, 0x3f773f9b,
	0xfed20000, 0xfc77fda4, 0xfa1efb4a, 0xf7c6f8f2,
	0x3f1f3f4e, 0x3eb03eea, 0x3e2c3e71, 0x3d923de2,
	0xf572f69c, 0xf321f449, 0xf0d4f1fa, 0xee8defb0,
	0x3ce33d3e, 0x3c1f3c84, 0x3b463bb5, 0x3a583ad2,
	0xec4ced6c, 0xea12eb2e, 0xe7dfe8f7, 0xe5b4e6c9,
	0x395739da, 0x384138ce, 0x371837af, 0x35db367c,
	0xe393e4a3, 0xe17ce286, 0xdf6fe074, 0xdd6ede6d,
	0x348c3536, 0x332b33de, 0x31b83273, 0x303330f8,
	0xdb78dc71, 0xd98fda82, 0xd7b4d8a0, 0xd5e6d6cb,
	0x2e9e2f6b, 0x2cf92dce, 0x2b442c21, 0x29802a64,
	0xd427d505, 0xd278d34e, 0xd0d8d1a6, 0xcf48d00e,
	0x27ae2899, 0x25ce26c0, 0x23e124d9, 0x21e822e6,
	0xcdcace87, 0xcc5dcd11, 0xcb01cbad, 0xc9b8ca5a,
	0x1fe220e6, 0x1dd21edb, 0x1bb71cc6, 0x19931aa6,
	0xc882c91b, 0xc75fc7ee, 0xc650c6d5, 0xc555c5d0,
	0x1766187d, 0x1530164c, 0x12f31413, 0x10b011d3,
	0xc46ec4df, 0xc39cc403, 0xc2dfc33b, 0xc237c288,
	0x0e670f8c, 0x0c190d41, 0x09c70af0, 0x0771089c,
	0xc1a4c1eb, 0xc127c163, 0xc0c0c0f1, 0xc06fc095,
	0x05190645, 0x02bf03ec, 0x00640192, 0xfe09ff36,
	0xc034c04f, 0xc00fc01f, 0xc000c005, 0xc008c001,
	0xfbaffcdc, 0xf956fa82, 0xf6fff82a, 0xf4acf5d5,
	0xc025c014, 0xc059c03c, 0xc0a3c07b, 0xc103c0d0,
	0xf25cf383, 0xf011f136, 0xedcceeee, 0xeb8decac,
	0xc178c13b, 0xc204c1bb, 0xc2a5c251, 0xc35bc2fd,
	0xe955ea70, 0xe725e83c, 0xe4fee610, 0xe2e0e3ed,
	0xc426c3be, 0xc506c493, 0xc5fac57e, 0xc703c67c,
	0xe0cce1d4, 0xdec3dfc6, 0xdcc5ddc2, 0xdad4dbcb,
	0xc81fc78f, 0xc94fc8b5, 0xca92c9ee, 0xcbe7cb3a,
	0xd8efd9e0, 0xd718d802, 0xd550d632, 0xd396d471,
	0xcd4ecc98, 0xcec7ce08, 0xd051cf8a, 0xd1ebd11c,
	0xd1ebd2bf, 0xd051d11c, 0xcec7cf8a, 0xcd4ece08,
	0xd396d2bf, 0xd550d471, 0xd718d632, 0xd8efd802,
	0xcbe7cc98, 0xca92cb3a, 0xc94fc9ee, 0xc81fc8b5,
	0xdad4d9e0, 0xdcc5dbcb, 0xdec3ddc2, 0xe0ccdfc6,
	0xc703c78f, 0xc5fac67c, 0xc506c57e, 0xc426c493,
	0xe2e0e1d4, 0xe4fee3ed, 0xe725e610, 0xe955e83c,
	0xc35bc3be, 0xc2a5c2fd, 0xc204c251, 0xc178c1bb,
	0xeb8dea70, 0xedccecac, 0xf011eeee, 0xf25cf136,
	0xc103c13b, 0xc0a3c0d0, 0xc059c07b, 0xc025c03c,
	0xf4acf383, 0xf6fff5d5, 0xf956f82a, 0xfbaffa82,
	0xc008c014, 0xc000c001, 0xc00fc005, 0xc034c01f,
	0xfe09fcdc, 0x0064ff36, 0x02bf0192, 0x051903ec,
	0xc06fc04f, 0xc0c0c095, 0xc127c0f1, 0xc1a4c163,
	0x07710645, 0x09c7089c, 0x0c190af0, 0x0e670d41,
	0xc237c1eb, 0xc2dfc288, 0xc39cc33b, 0xc46ec403,
	0x10b00f8c, 0x12f311d3, 0x15301413, 0x1766164c,
	0xc555c4df, 0xc650c5d0, 0xc75fc6d5, 0xc882c7ee,
	0x1993187d, 0x1bb71aa6, 0x1dd21cc6, 0x1fe21edb,
	0xc9b8c91b, 0xcb01ca5a, 0xcc5dcbad, 0xcdcacd11,
	0x21e820e6, 0x23e122e6, 0x25ce24d9, 0x27ae26c0,
	0xcf48ce87, 0xd0d8d00e, 0xd278d1a6, 0xd427d34e,
	0x29802899, 0x2b442a64, 0x2cf92c21, 0x2e9e2dce,
	0xd5e6d505, 0xd7b4d6cb, 0xd98fd8a0, 0xdb78da82,
	0x30332f6b, 0x31b830f8, 0x332b3273, 0x348c33de,
	0xdd6edc71, 0xdf6fde6d, 0xe17ce074, 0xe393e286,
	0x35db3536, 0x3718367c, 0x384137af, 0x395738ce,
	0xe5b4e4a3, 0xe7dfe6c9, 0xea12e8f7, 0xec4ceb2e,
	0x3a5839da, 0x3b463ad2, 0x3c1f3bb5, 0x3ce33c84,
	0xee8ded6c, 0xf0d4efb0, 0xf321f1fa, 0xf572f449,
	0x3d923d3e, 0x3e2c3de2, 0x3eb03e71, 0x3f1f3eea,
	0xf7c6f69c, 0xfa1ef8f2, 0xfc77fb4a, 0xfed2fda4,
	0x3f773f4e, 0x3fba3f9b, 0x3fe63fd3, 0x3ffc3ff4,
	0x012d0000, 0x0388025b, 0x05e104b5, 0x0839070d,
	0x3ffc3fff, 0x3fe63ff4, 0x3fba3fd3, 0x3f773f9b,
	0x0a8d0963, 0x0cde0bb6, 0x0f2b0e05, 0x1172104f,
	0x3f1f3f4e, 0x3eb03eea, 0x3e2c3e71, 0x3d923de2,
	0x13b31293, 0x15ed14d1, 0x18201708, 0x1a4b1936,
	0x3ce33d3e, 0x3c1f3c84, 0x3b463bb5, 0x3a583ad2,
	0x1c6c1b5c, 0x1e831d79, 0x20901f8b, 0x22912192,
	0x395739da, 0x384138ce, 0x371837af, 0x35db367c,
	0x2487238e, 0x2670257d, 0x284b275f, 0x2a192934,
	0x348c3536, 0x332b33de, 0x31b83273, 0x303330f8,
	0x2bd82afa, 0x2d872cb1, 0x2f272e59, 0x30b72ff1,
	0x2e9e2f6b, 0x2cf92dce, 0x2b442c21, 0x29802a64,
	0x32353178, 0x33a232ee, 0x34fe3452, 0x364735a5,
	0x27ae2899, 0x25ce26c0, 0x23e124d9, 0x21e822e6,
	0x377d36e4, 0x38a03811, 0x39af392a, 0x3aaa3a2f,
	0x1fe220e6, 0x1dd21edb, 0x1bb71cc6, 0x19931aa6,
	0x3b913b20, 0x3c633bfc, 0x3d203cc4, 0x3dc83d77,
	0x1766187d, 0x1530164c, 0x12f31413, 0x10b011d3,
	0x3e5b3e14, 0x3ed83e9c, 0x3f3f3f0e, 0x3f903f6a,
	0x0e670f8c, 0x0c190d41, 0x09c70af0, 0x0771089c,
	0x3fcb3fb0, 0x3ff03fe0, 0x3fff3ffa, 0x3ff73ffe,
	0x05190645, 0x02bf03ec, 0x00640192, 0xfe09ff36,
	0x3fda3feb, 0x3fa63fc3, 0x3f5c3f84, 0x3efc3f2f,
	0xfbaffcdc, 0xf956fa82, 0xf6fff82a, 0xf4acf5d5,
	0x3e873ec4, 0x3dfb3e44, 0x3d5a3dae, 0x3ca43d02,
	0xf25cf383, 0xf011f136, 0xedcceeee, 0xeb8decac,
	0x3bd93c41, 0x3af93b6c, 0x3a053a81, 0x38fc3983,
	0xe955ea70, 0xe725e83c, 0xe4fee610, 0xe2e0e3ed,
	0x37e03870, 0x36b0374a, 0x356d3611, 0x341834c5,
	0xe0cce1d4, 0xdec3dfc6, 0xdcc5ddc2, 0xdad4dbcb,
	0x32b13367, 0x313831f7, 0x2fae3075, 0x2e142ee3,
	0xd8efd9e0, 0xd718d802, 0xd550d632, 0xd396d471,
	0x2c692d40, 0x2aaf2b8e, 0x28e729cd, 0x271027fd,
	0xd1ebd2bf, 0xd051d11c, 0xcec7cf8a, 0xcd4ece08,
	0x252b261f, 0x233a2434, 0x213c223d, 0x1f332039,
	0xcbe7cc98, 0xca92cb3a, 0xc94fc9ee, 0xc81fc8b5,
	0x1d1f1e2b, 0x1b011c12, 0x18da19ef, 0x16aa17c3,
	0xc703c78f, 0xc5fac67c, 0xc506c57e, 0xc426c493,
	0x1472158f, 0x12331353, 0x0fee1111, 0x0da30ec9,
	0xc35bc3be, 0xc2a5c2fd, 0xc204c251, 0xc178c1bb,
	0x0b530c7c, 0x09000a2a, 0x06a907d5, 0x0450057d,
	0xc103c13b, 0xc0a3c0d0, 0xc059c07b, 0xc025c03c,
	0x01f60323, 0xff9b00c9, 0xfd40fe6d, 0xfae6fc13,
	0xc008c014, 0xc000c001, 0xc00fc005, 0xc034c01f,
	0xf88ef9ba, 0xf638f763, 0xf3e6f50f, 0xf198f2be,
	0xc06fc04f, 0xc0c0c095, 0xc127c0f1, 0xc1a4c163,
	0xef4ff073, 0xed0cee2c, 0xeacfebec, 0xe899e9b3,
	0xc237c1eb, 0xc2dfc288, 0xc39cc33b, 0xc46ec403,
	0xe66ce782, 0xe448e559, 0xe22de339, 0xe01de124,
	0xc555c4df, 0xc650c5d0, 0xc75fc6d5, 0xc882c7ee,
	0xde17df19, 0xdc1edd19, 0xda31db26, 0xd851d93f,
	0xc9b8c91b, 0xcb01ca5a, 0xcc5dcbad, 0xcdcacd11,
	0xd67fd766, 0xd4bbd59b, 0xd306d3de, 0xd161d231,
	0xcf48ce87, 0xd0d8d00e, 0xd278d1a6, 0xd427d34e,
	0xcfccd094, 0xce47cf07, 0xccd4cd8c, 0xcb73cc21,
	0xd5e6d505, 0xd7b4d6cb, 0xd98fd8a0, 0xdb78da82,
	0xca24cac9, 0xc8e7c983, 0xc7bec850, 0xc6a8c731,
	0xdd6edc71, 0xdf6fde6d, 0xe17ce074, 0xe393e286,
	0xc5a7c625, 0xc4b9c52d, 0xc3e0c44a, 0xc31cc37b,
	0xe5b4e4a3, 0xe7dfe6c9, 0xea12e8f7, 0xec4ceb2e,
	0xc26dc2c1, 0xc1d3c21d, 0xc14fc18e, 0xc0e0c115,
	0xee8ded6c, 0xf0d4efb0, 0xf321f1fa, 0xf572f449,
	0xc088c0b1, 0xc045c064, 0xc019c02c, 0xc003c00b,
	0xf7c6f69c, 0xfa1ef8f2, 0xfc77fb4a, 0xfed2fda4,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_3D_P[] = {
	0x00000000, 0x00000000, 0xd2bf3fff, 0x2d400000,
	0x00000000, 0x00000000, 0x2d400000, 0x2d40c000,
	0x187d3fff, 0xc4dfd2bf, 0x3b200000, 0xe7822d40,
	0x3b200000, 0xe7822d40, 0xe782c000, 0x3b202d40,
	0x35363fff, 0xf383187d, 0xc13bd2bf, 0xdc71c4df,
	0x238e0000, 0x3ec43b20, 0x0c7c2d40, 0xcac9e782,
	0x238e0000, 0x3ec43b20, 0x0c7c2d40, 0xcac9e782,
	0xcac9c000, 0x0c7ce782, 0x3ec42d40, 0x238e3b20,
	0x3d3e3fff, 0x28993536, 0x0645187d, 0xe1d4f383,
	0x12930000, 0x3178238e, 0x3fb03b20, 0x38703ec4,
	0xc78fd2bf, 0xc04fc13b, 0xce87c4df, 0xed6cdc71,
	0x1e2b2d40, 0xf9ba0c7c, 0xd766e782, 0xc2c1cac9,
	0x12930000, 0x3178238e, 0x3fb03b20, 0x38703ec4,
	0xc2c1c000, 0xd766cac9, 0xf9bae782, 0x1e2b0c7c,
	0x1e2b2d40, 0xf9ba0c7c, 0xd766e782, 0xc2c1cac9,
	0x38702d40, 0x3fb03ec4, 0x31783b20, 0x1293238e,
	0x3f4e3fff, 0x39da3d3e, 0x2f6b3536, 0x20e62899,
	0x09630000, 0x1b5c1293, 0x2afa238e, 0x36e43178,
	0x0f8c187d, 0xfcdc0645, 0xea70f383, 0xd9e0e1d4,
	0x3e143b20, 0x3feb3fb0, 0x3c413ec4, 0x33673870,
	0xcc98d2bf, 0xc3bec78f, 0xc014c13b, 0xc1ebc04f,
	0x261f2d40, 0x158f1e2b, 0x03230c7c, 0xf073f9ba,
	0xc91bc4df, 0xd505ce87, 0xe4a3dc71, 0xf69ced6c,
	0xdf19e782, 0xd094d766, 0xc625cac9, 0xc0b1c2c1,
	0x09630000, 0x1b5c1293, 0x2afa238e, 0x36e43178,
	0xc0b1c000, 0xc625c2c1, 0xd094cac9, 0xdf19d766,
	0x3e143b20, 0x3feb3fb0, 0x3c413ec4, 0x33673870,
	0xf073e782, 0x0323f9ba, 0x158f0c7c, 0x261f1e2b,
	0x261f2d40, 0x158f1e2b, 0x03230c7c, 0xf073f9ba,
	0x33672d40, 0x3c413870, 0x3feb3ec4, 0x3e143fb0,
	0xdf19e782, 0xd094d766, 0xc625cac9, 0xc0b1c2c1,
	0x36e43b20, 0x2afa3178, 0x1b5c238e, 0x09631293,
	0x3fd33fff, 0x3e713f4e, 0x3bb53d3e, 0x37af39da,
	0x04b50000, 0x0e050963, 0x17081293, 0x1f8b1b5c,
	0x32733536, 0x2c212f6b, 0x24d92899, 0x1cc620e6,
	0x275f238e, 0x2e592afa, 0x34523178, 0x392a36e4,
	0x1413187d, 0x0af00f8c, 0x01920645, 0xf82afcdc,
	0x3cc43b20, 0x3f0e3e14, 0x3ffa3fb0, 0x3f843feb,
	0xeeeef383, 0xe610ea70, 0xddc2e1d4, 0xd632d9e0,
	0x3dae3ec4, 0x3a813c41, 0x36113870, 0x30753367,
	0xcf8ad2bf, 0xc9eecc98, 0xc57ec78f, 0xc251c3be,
	0x29cd2d40, 0x223d261f, 0x19ef1e2b, 0x1111158f,
	0xc07bc13b, 0xc005c014, 0xc0f1c04f, 0xc33bc1eb,
	0x07d50c7c, 0xfe6d0323, 0xf50ff9ba, 0xebecf073,
	0xc6d5c4df, 0xcbadc91b, 0xd1a6ce87, 0xd8a0d505,
	0xe339e782, 0xdb26df19, 0xd3ded766, 0xcd8cd094,
	0xe074dc71, 0xe8f7e4a3, 0xf1faed6c, 0xfb4af69c,
	0xc850cac9, 0xc44ac625, 0xc18ec2c1, 0xc02cc0b1,
	0x04b50000, 0x0e050963, 0x17081293, 0x1f8b1b5c,
	0xc02cc000, 0xc18ec0b1, 0xc44ac2c1, 0xc850c625,
	0x275f238e, 0x2e592afa, 0x34523178, 0x392a36e4,
	0xcd8ccac9, 0xd3ded094, 0xdb26d766, 0xe339df19,
	0x3cc43b20, 0x3f0e3e14, 0x3ffa3fb0, 0x3f843feb,
	0xebece782, 0xf50ff073, 0xfe6df9ba, 0x07d50323,
	0x3dae3ec4, 0x3a813c41, 0x36113870, 0x30753367,
	0x11110c7c, 0x19ef158f, 0x223d1e2b, 0x29cd261f,
	0x29cd2d40, 0x223d261f, 0x19ef1e2b, 0x1111158f,
	0x30752d40, 0x36113367, 0x3a813870, 0x3dae3c41,
	0x07d50c7c, 0xfe6d0323, 0xf50ff9ba, 0xebecf073,
	0x3f843ec4, 0x3ffa3feb, 0x3f0e3fb0, 0x3cc43e14,
	0xe339e782, 0xdb26df19, 0xd3ded766, 0xcd8cd094,
	0x392a3b20, 0x345236e4, 0x2e593178, 0x275f2afa,
	0xc850cac9, 0xc44ac625, 0xc18ec2c1, 0xc02cc0b1,
	0x1f8b238e, 0x17081b5c, 0x0e051293, 0x04b50963,
	0x3ff43fff, 0x3f9b3fd3, 0x3eea3f4e, 0x3de23e71,
	0x025b0000, 0x070d04b5, 0x0bb60963, 0x104f0e05,
	0x3c843d3e, 0x3ad23bb5, 0x38ce39da, 0x367c37af,
	0x14d11293, 0x19361708, 0x1d791b5c, 0x21921f8b,
	0x33de3536, 0x30f83273, 0x2dce2f6b, 0x2a642c21,
	0x257d238e, 0x2934275f, 0x2cb12afa, 0x2ff12e59,
	0x26c02899, 0x22e624d9, 0x1edb20e6, 0x1aa61cc6,
	0x32ee3178, 0x35a53452, 0x381136e4, 0x3a2f392a,
	0x164c187d, 0x11d31413, 0x0d410f8c, 0x089c0af0,
	0x3bfc3b20, 0x3d773cc4, 0x3e9c3e14, 0x3f6a3f0e,
	0x03ec0645, 0xff360192, 0xfa82fcdc, 0xf5d5f82a,
	0x3fe03fb0, 0x3ffe3ffa, 0x3fc33feb, 0x3f2f3f84,
	0xf136f383, 0xecaceeee, 0xe83cea70, 0xe3ede610,
	0x3e443ec4, 0x3d023dae, 0x3b6c3c41, 0x39833a81,
	0xdfc6e1d4, 0xdbcbddc2, 0xd802d9e0, 0xd471d632,
	0x374a3870, 0x34c53611, 0x31f73367, 0x2ee33075,
	0xd11cd2bf, 0xce08cf8a, 0xcb3acc98, 0xc8b5c9ee,
	0x2b8e2d40, 0x27fd29cd, 0x2434261f, 0x2039223d,
	0xc67cc78f, 0xc493c57e, 0xc2fdc3be, 0xc1bbc251,
	0x1c121e2b, 0x17c319ef, 0x1353158f, 0x0ec91111,
	0xc0d0c13b, 0xc03cc07b, 0xc001c014, 0xc01fc005,
	0x0a2a0c7c, 0x057d07d5, 0x00c90323, 0xfc13fe6d,
	0xc095c04f, 0xc163c0f1, 0xc288c1eb, 0xc403c33b,
	0xf763f9ba, 0xf2bef50f, 0xee2cf073, 0xe9b3ebec,
	0xc5d0c4df, 0xc7eec6d5, 0xca5ac91b, 0xcd11cbad,
	0xe559e782, 0xe124e339, 0xdd19df19, 0xd93fdb26,
	0xd00ece87, 0xd34ed1a6, 0xd6cbd505, 0xda82d8a0,
	0xd59bd766, 0xd231d3de, 0xcf07d094, 0xcc21cd8c,
	0xde6ddc71, 0xe286e074, 0xe6c9e4a3, 0xeb2ee8f7,
	0xc983cac9, 0xc731c850, 0xc52dc625, 0xc37bc44a,
	0xefb0ed6c, 0xf449f1fa, 0xf8f2f69c, 0xfda4fb4a,
	0xc21dc2c1, 0xc115c18e, 0xc064c0b1, 0xc00bc02c,
	0x025b0000, 0x070d04b5, 0x0bb60963, 0x104f0e05,
	0xc00bc000, 0xc064c02c, 0xc115c0b1, 0xc21dc18e,
	0x14d11293, 0x19361708, 0x1d791b5c, 0x21921f8b,
	0xc37bc2c1, 0xc52dc44a, 0xc731c625, 0xc983c850,
	0x257d238e, 0x2934275f, 0x2cb12afa, 0x2ff12e59,
	0xcc21cac9, 0xcf07cd8c, 0xd231d094, 0xd59bd3de,
	0x32ee3178, 0x35a53452, 0x381136e4, 0x3a2f392a,
	0xd93fd766, 0xdd19db26, 0xe124df19, 0xe559e339,
	0x3bfc3b20, 0x3d773cc4, 0x3e9c3e14, 0x3f6a3f0e,
	0xe9b3e782, 0xee2cebec, 0xf2bef073, 0xf763f50f,
	0x3fe03fb0, 0x3ffe3ffa, 0x3fc33feb, 0x3f2f3f84,
	0xfc13f9ba, 0x00c9fe6d, 0x057d0323, 0x0a2a07d5,
	0x3e443ec4, 0x3d023dae, 0x3b6c3c41, 0x39833a81,
	0x0ec90c7c, 0x13531111, 0x17c3158f, 0x1c1219ef,
	0x374a3870, 0x34c53611, 0x31f73367, 0x2ee33075,
	0x20391e2b, 0x2434223d, 0x27fd261f, 0x2b8e29cd,
	0x2b8e2d40, 0x27fd29cd, 0x2434261f, 0x2039223d,
	0x2ee32d40, 0x31f73075, 0x34c53367, 0x374a3611,
	0x1c121e2b, 0x17c319ef, 0x1353158f, 0x0ec91111,
	0x39833870, 0x3b6c3a81, 0x3d023c41, 0x3e443dae,
	0x0a2a0c7c, 0x057d07d5, 0x00c90323, 0xfc13fe6d,
	0x3f2f3ec4, 0x3fc33f84, 0x3ffe3feb, 0x3fe03ffa,
	0xf763f9ba, 0xf2bef50f, 0xee2cf073, 0xe9b3ebec,
	0x3f6a3fb0, 0x3e9c3f0e, 0x3d773e14, 0x3bfc3cc4,
	0xe559e782, 0xe124e339, 0xdd19df19, 0xd93fdb26,
	0x3a2f3b20, 0x3811392a, 0x35a536e4, 0x32ee3452,
	0xd59bd766, 0xd231d3de, 0xcf07d094, 0xcc21cd8c,
	0x2ff13178, 0x2cb12e59, 0x29342afa, 0x257d275f,
	0xc983cac9, 0xc731c850, 0xc52dc625, 0xc37bc44a,
	0x2192238e, 0x1d791f8b, 0x19361b5c, 0x14d11708,
	0xc21dc2c1, 0xc115c18e, 0xc064c0b1, 0xc00bc02c,
	0x104f1293, 0x0bb60e05, 0x070d0963, 0x025b04b5,
	0x3ffc3fff, 0x3fe63ff4, 0x3fba3fd3, 0x3f773f9b,
	0x012d0000, 0x0388025b, 0x05e104b5, 0x0839070d,
	0x3f1f3f4e, 0x3eb03eea, 0x3e2c3e71, 0x3d923de2,
	0x0a8d0963, 0x0cde0bb6, 0x0f2b0e05, 0x1172104f,
	0x3ce33d3e, 0x3c1f3c84, 0x3b463bb5, 0x3a583ad2,
	0x13b31293, 0x15ed14d1, 0x18201708, 0x1a4b1936,
	0x395739da, 0x384138ce, 0x371837af, 0x35db367c,
	0x1c6c1b5c, 0x1e831d79, 0x20901f8b, 0x22912192,
	0x348c3536, 0x332b33de, 0x31b83273, 0x303330f8,
	0x2487238e, 0x2670257d, 0x284b275f, 0x2a192934,
	0x2e9e2f6b, 0x2cf92dce, 0x2b442c21, 0x29802a64,
	0x2bd82afa, 0x2d872cb1, 0x2f272e59, 0x30b72ff1,
	0x27ae2899, 0x25ce26c0, 0x23e124d9, 0x21e822e6,
	0x32353178, 0x33a232ee, 0x34fe3452, 0x364735a5,
	0x1fe220e6, 0x1dd21edb, 0x1bb71cc6, 0x19931aa6,
	0x377d36e4, 0x38a03811, 0x39af392a, 0x3aaa3a2f,
	0x1766187d, 0x1530164c, 0x12f31413, 0x10b011d3,
	0x3b913b20, 0x3c633bfc, 0x3d203cc4, 0x3dc83d77,
	0x0e670f8c, 0x0c190d41, 0x09c70af0, 0x0771089c,
	0x3e5b3e14, 0x3ed83e9c, 0x3f3f3f0e, 0x3f903f6a,
	0x05190645, 0x02bf03ec, 0x00640192, 0xfe09ff36,
	0x3fcb3fb0, 0x3ff03fe0, 0x3fff3ffa, 0x3ff73ffe,
	0xfbaffcdc, 0xf956fa82, 0xf6fff82a, 0xf4acf5d5,
	0x3fda3feb, 0x3fa63fc3, 0x3f5c3f84, 0x3efc3f2f,
	0xf25cf383, 0xf011f136, 0xedcceeee, 0xeb8decac,
	0x3e873ec4, 0x3dfb3e44, 0x3d5a3dae, 0x3ca43d02,
	0xe955ea70, 0xe725e83c, 0xe4fee610, 0xe2e0e3ed,
	0x3bd93c41, 0x3af93b6c, 0x3a053a81, 0x38fc3983,
	0xe0cce1d4, 0xdec3dfc6, 0xdcc5ddc2, 0xdad4dbcb,
	0x37e03870, 0x36b0374a, 0x356d3611, 0x341834c5,
	0xd8efd9e0, 0xd718d802, 0xd550d632, 0xd396d471,
	0x32b13367, 0x313831f7, 0x2fae3075, 0x2e142ee3,
	0xd1ebd2bf, 0xd051d11c, 0xcec7cf8a, 0xcd4ece08,
	0x2c692d40, 0x2aaf2b8e, 0x28e729cd, 0x271027fd,
	0xcbe7cc98, 0xca92cb3a, 0xc94fc9ee, 0xc81fc8b5,
	0x252b261f, 0x233a2434, 0x213c223d, 0x1f332039,
	0xc703c78f, 0xc5fac67c, 0xc506c57e, 0xc426c493,
	0x1d1f1e2b, 0x1b011c12, 0x18da19ef, 0x16aa17c3,
	0xc35bc3be, 0xc2a5c2fd, 0xc204c251, 0xc178c1bb,
	0x1472158f, 0x12331353, 0x0fee1111, 0x0da30ec9,
	0xc103c13b, 0xc0a3c0d0, 0xc059c07b, 0xc025c03c,
	0x0b530c7c, 0x09000a2a, 0x06a907d5, 0x0450057d,
	0xc008c014, 0xc000c001, 0xc00fc005, 0xc034c01f,
	0x01f60323, 0xff9b00c9, 0xfd40fe6d, 0xfae6fc13,
	0xc06fc04f, 0xc0c0c095, 0xc127c0f1, 0xc1a4c163,
	0xf88ef9ba, 0xf638f763, 0xf3e6f50f, 0xf198f2be,
	0xc237c1eb, 0xc2dfc288, 0xc39cc33b, 0xc46ec403,
	0xef4ff073, 0xed0cee2c, 0xeacfebec, 0xe899e9b3,
	0xc555c4df, 0xc650c5d0, 0xc75fc6d5, 0xc882c7ee,
	0xe66ce782, 0xe448e559, 0xe22de339, 0xe01de124,
	0xc9b8c91b, 0xcb01ca5a, 0xcc5dcbad, 0xcdcacd11,
	0xde17df19, 0xdc1edd19, 0xda31db26, 0xd851d93f,
	0xcf48ce87, 0xd0d8d00e, 0xd278d1a6, 0xd427d34e,
	0xd67fd766, 0xd4bbd59b, 0xd306d3de, 0xd161d231,
	0xd5e6d505, 0xd7b4d6cb, 0xd98fd8a0, 0xdb78da82,
	0xcfccd094, 0xce47cf07, 0xccd4cd8c, 0xcb73cc21,
	0xdd6edc71, 0xdf6fde6d, 0xe17ce074, 0xe393e286,
	0xca24cac9, 0xc8e7c983, 0xc7bec850, 0xc6a8c731,
	0xe5b4e4a3, 0xe7dfe6c9, 0xea12e8f7, 0xec4ceb2e,
	0xc5a7c625, 0xc4b9c52d, 0xc3e0c44a, 0xc31cc37b,
	0xee8ded6c, 0xf0d4efb0, 0xf321f1fa, 0xf572f449,
	0xc26dc2c1, 0xc1d3c21d, 0xc14fc18e, 0xc0e0c115,
	0xf7c6f69c, 0xfa1ef8f2, 0xfc77fb4a, 0xfed2fda4,
	0xc088c0b1, 0xc045c064, 0xc019c02c, 0xc003c00b,
	0x012d0000, 0x0388025b, 0x05e104b5, 0x0839070d,
	0xc003c000, 0xc019c00b, 0xc045c02c, 0xc088c064,
	0x0a8d0963, 0x0cde0bb6, 0x0f2b0e05, 0x1172104f,
	0xc0e0c0b1, 0xc14fc115, 0xc1d3c18e, 0xc26dc21d,
	0x13b31293, 0x15ed14d1, 0x18201708, 0x1a4b1936,
	0xc31cc2c1, 0xc3e0c37b, 0xc4b9c44a, 0xc5a7c52d,
	0x1c6c1b5c, 0x1e831d79, 0x20901f8b, 0x22912192,
	0xc6a8c625, 0xc7bec731, 0xc8e7c850, 0xca24c983,
	0x2487238e, 0x2670257d, 0x284b275f, 0x2a192934,
	0xcb73cac9, 0xccd4cc21, 0xce47cd8c, 0xcfcccf07,
	0x2bd82afa, 0x2d872cb1, 0x2f272e59, 0x30b72ff1,
	0xd161d094, 0xd306d231, 0xd4bbd3de, 0xd67fd59b,
	0x32353178, 0x33a232ee, 0x34fe3452, 0x364735a5,
	0xd851d766, 0xda31d93f, 0xdc1edb26, 0xde17dd19,
	0x377d36e4, 0x38a03811, 0x39af392a, 0x3aaa3a2f,
	0xe01ddf19, 0xe22de124, 0xe448e339, 0xe66ce559,
	0x3b913b20, 0x3c633bfc, 0x3d203cc4, 0x3dc83d77,
	0xe899e782, 0xeacfe9b3, 0xed0cebec, 0xef4fee2c,
	0x3e5b3e14, 0x3ed83e9c, 0x3f3f3f0e, 0x3f903f6a,
	0xf198f073, 0xf3e6f2be, 0xf638f50f, 0xf88ef763,
	0x3fcb3fb0, 0x3ff03fe0, 0x3fff3ffa, 0x3ff73ffe,
	0xfae6f9ba, 0xfd40fc13, 0xff9bfe6d, 0x01f600c9,
	0x3fda3feb, 0x3fa63fc3, 0x3f5c3f84, 0x3efc3f2f,
	0x04500323, 0x06a9057d, 0x090007d5, 0x0b530a2a,
	0x3e873ec4, 0x3dfb3e44, 0x3d5a3dae, 0x3ca43d02,
	0x0da30c7c, 0x0fee0ec9, 0x12331111, 0x14721353,
	0x3bd93c41, 0x3af93b6c, 0x3a053a81, 0x38fc3983,
	0x16aa158f, 0x18da17c3, 0x1b0119ef, 0x1d1f1c12,
	0x37e03870, 0x36b0374a, 0x356d3611, 0x341834c5,
	0x1f331e2b, 0x213c2039, 0x233a223d, 0x252b2434,
	0x32b13367, 0x313831f7, 0x2fae3075, 0x2e142ee3,
	0x2710261f, 0x28e727fd, 0x2aaf29cd, 0x2c692b8e,
	0x2c692d40, 0x2aaf2b8e, 0x28e729cd, 0x271027fd,
	0x2e142d40, 0x2fae2ee3, 0x31383075, 0x32b131f7,
	0x252b261f, 0x233a2434, 0x213c223d, 0x1f332039,
	0x34183367, 0x356d34c5, 0x36b03611, 0x37e0374a,
	0x1d1f1e2b, 0x1b011c12, 0x18da19ef, 0x16aa17c3,
	0x38fc3870, 0x3a053983, 0x3af93a81, 0x3bd93b6c,
	0x1472158f, 0x12331353, 0x0fee1111, 0x0da30ec9,
	0x3ca43c41, 0x3d5a3d02, 0x3dfb3dae, 0x3e873e44,
	0x0b530c7c, 0x09000a2a, 0x06a907d5, 0x0450057d,
	0x3efc3ec4, 0x3f5c3f2f, 0x3fa63f84, 0x3fda3fc3,
	0x01f60323, 0xff9b00c9, 0xfd40fe6d, 0xfae6fc13,
	0x3ff73feb, 0x3fff3ffe, 0x3ff03ffa, 0x3fcb3fe0,
	0xf88ef9ba, 0xf638f763, 0xf3e6f50f, 0xf198f2be,
	0x3f903fb0, 0x3f3f3f6a, 0x3ed83f0e, 0x3e5b3e9c,
	0xef4ff073, 0xed0cee2c, 0xeacfebec, 0xe899e9b3,
	0x3dc83e14, 0x3d203d77, 0x3c633cc4, 0x3b913bfc,
	0xe66ce782, 0xe448e559, 0xe22de339, 0xe01de124,
	0x3aaa3b20, 0x39af3a2f, 0x38a0392a, 0x377d3811,
	0xde17df19, 0xdc1edd19, 0xda31db26, 0xd851d93f,
	0x364736e4, 0x34fe35a5, 0x33a23452, 0x323532ee,
	0xd67fd766, 0xd4bbd59b, 0xd306d3de, 0xd161d231,
	0x30b73178, 0x2f272ff1, 0x2d872e59, 0x2bd82cb1,
	0xcfccd094, 0xce47cf07, 0xccd4cd8c, 0xcb73cc21,
	0x2a192afa, 0x284b2934, 0x2670275f, 0x2487257d,
	0xca24cac9, 0xc8e7c983, 0xc7bec850, 0xc6a8c731,
	0x2291238e, 0x20902192, 0x1e831f8b, 0x1c6c1d79,
	0xc5a7c625, 0xc4b9c52d, 0xc3e0c44a, 0xc31cc37b,
	0x1a4b1b5c, 0x18201936, 0x15ed1708, 0x13b314d1,
	0xc26dc2c1, 0xc1d3c21d, 0xc14fc18e, 0xc0e0c115,
	0x11721293, 0x0f2b104f, 0x0cde0e05, 0x0a8d0bb6,
	0xc088c0b1, 0xc045c064, 0xc019c02c, 0xc003c00b,
	0x08390963, 0x05e1070d, 0x038804b5, 0x012d025b,
};


#if 0
const mlib_u32 mlib_s_fftK_WTable_1_M[] = {
	0x00000000, 0x00000000, 0x0000a57e, 0x7fff5a81,
	0x00000000, 0x00000000, 0x8001a57e, 0x0000a57e,
	0x5a8130fb, 0x7fff7640, 0xa57e89bf, 0x0000cf04,
	0xa57e89bf, 0x0000cf04, 0xa57ecf04, 0x800189bf,
	0x76406a6c, 0x7fff7d89, 0x30fb18f8, 0x5a81471c,
	0xcf04b8e3, 0x0000e707, 0x89bf8276, 0xa57e9593,
	0xcf04b8e3, 0x0000e707, 0x89bf8276, 0xa57e9593,
	0x89bf9593, 0x80018276, 0xcf04e707, 0xa57eb8e3,
	0x7d897a7c, 0x7fff7f61, 0x6a6c62f1, 0x764070e1,
	0xe707dad8, 0x0000f374, 0xb8e3aecc, 0xcf04c3a9,
	0x471c3c56, 0x5a815133, 0x18f80c8b, 0x30fb2527,
	0x95938f1e, 0xa57e9d0e, 0x8276809e, 0x89bf8583,
	0xe707dad8, 0x0000f374, 0xb8e3aecc, 0xcf04c3a9,
	0x82768583, 0x8001809e, 0x95939d0e, 0x89bf8f1e,
	0x95938f1e, 0xa57e9d0e, 0x8276809e, 0x89bf8583,
	0xb8e3c3a9, 0xa57eaecc, 0xe707f374, 0xcf04dad8,
	0x7f617e9c, 0x7fff7fd7, 0x7a7c7883, 0x7d897c29,
	0xf374ed38, 0x0000f9b8, 0xdad8d4e1, 0xe707e0e6,
	0x70e16dc9, 0x764073b5, 0x62f15ed6, 0x6a6c66ce,
	0xc3a9be32, 0xcf04c946, 0xaeccaa0b, 0xb8e3b3c0,
	0x51334c3f, 0x5a8155f4, 0x3c5636b9, 0x471c41cd,
	0x9d0e9931, 0xa57ea129, 0x8f1e8c4a, 0x95939236,
	0x25271f19, 0x30fb2b1e, 0x0c8b0647, 0x18f812c7,
	0x858383d6, 0x89bf877c, 0x809e8028, 0x82768163,
	0xf374ed38, 0x0000f9b8, 0xdad8d4e1, 0xe707e0e6,
	0x809e8163, 0x80018028, 0x8583877c, 0x827683d6,
	0xc3a9be32, 0xcf04c946, 0xaeccaa0b, 0xb8e3b3c0,
	0x8f1e9236, 0x89bf8c4a, 0x9d0ea129, 0x95939931,
	0x9d0e9931, 0xa57ea129, 0x8f1e8c4a, 0x95939236,
	0xaeccb3c0, 0xa57eaa0b, 0xc3a9c946, 0xb8e3be32,
	0x858383d6, 0x89bf877c, 0x809e8028, 0x82768163,
	0xdad8e0e6, 0xcf04d4e1, 0xf374f9b8, 0xe707ed38,
	0x7fd77fa6, 0x7fff7ff5, 0x7e9c7e1c, 0x7f617f08,
	0xf9b8f695, 0x0000fcdb, 0xed38ea1e, 0xf374f054,
	0x7c297b5c, 0x7d897ce2, 0x7883776b, 0x7a7c7989,
	0xe0e6dddc, 0xe707e3f4, 0xd4e1d1ef, 0xdad8d7d9,
	0x73b57254, 0x76407503, 0x6dc96c23, 0x70e16f5e,
	0xc946c673, 0xcf04cc21, 0xbe32bb85, 0xc3a9c0e9,
	0x66ce64e7, 0x6a6c68a5, 0x5ed65cb3, 0x62f160eb,
	0xb3c0b140, 0xb8e3b64c, 0xaa0ba7bd, 0xaeccac65,
	0x55f4539a, 0x5a815842, 0x4c3f49b3, 0x51334ebf,
	0xa1299f14, 0xa57ea34c, 0x9931975a, 0x9d0e9b18,
	0x41cd3f16, 0x471c447a, 0x36b933de, 0x3c56398c,
	0x923690a1, 0x959393dc, 0x8c4a8afc, 0x8f1e8dab,
	0x2b1e2826, 0x30fb2e10, 0x1f191c0b, 0x25272223,
	0x877c8676, 0x89bf8894, 0x83d6831d, 0x858384a3,
	0x12c70fab, 0x18f815e1, 0x06470324, 0x0c8b096a,
	0x816380f7, 0x827681e3, 0x8028800a, 0x809e8059,
	0xf9b8f695, 0x0000fcdb, 0xed38ea1e, 0xf374f054,
	0x80288059, 0x8001800a, 0x816381e3, 0x809e80f7,
	0xe0e6dddc, 0xe707e3f4, 0xd4e1d1ef, 0xdad8d7d9,
	0x83d684a3, 0x8276831d, 0x877c8894, 0x85838676,
	0xc946c673, 0xcf04cc21, 0xbe32bb85, 0xc3a9c0e9,
	0x8c4a8dab, 0x89bf8afc, 0x923693dc, 0x8f1e90a1,
	0xb3c0b140, 0xb8e3b64c, 0xaa0ba7bd, 0xaeccac65,
	0x99319b18, 0x9593975a, 0xa129a34c, 0x9d0e9f14,
	0xa1299f14, 0xa57ea34c, 0x9931975a, 0x9d0e9b18,
	0xaa0bac65, 0xa57ea7bd, 0xb3c0b64c, 0xaeccb140,
	0x923690a1, 0x959393dc, 0x8c4a8afc, 0x8f1e8dab,
	0xbe32c0e9, 0xb8e3bb85, 0xc946cc21, 0xc3a9c673,
	0x877c8676, 0x89bf8894, 0x83d6831d, 0x858384a3,
	0xd4e1d7d9, 0xcf04d1ef, 0xe0e6e3f4, 0xdad8dddc,
	0x816380f7, 0x827681e3, 0x8028800a, 0x809e8059,
	0xed38f054, 0xe707ea1e, 0xf9b8fcdb, 0xf374f695,
	0x7ff57fe8, 0x7fff7ffc, 0x7fa67f86, 0x7fd77fc1,
	0xfcdbfb49, 0x0000fe6d, 0xf695f504, 0xf9b8f826,
	0x7f087ed4, 0x7f617f37, 0x7e1c7dd5, 0x7e9c7e5e,
	0xf054eec6, 0xf374f1e4, 0xea1ee892, 0xed38ebaa,
	0x7ce27c88, 0x7d897d38, 0x7b5c7aee, 0x7c297bc4,
	0xe3f4e26c, 0xe707e57d, 0xdddcdc59, 0xe0e6df60,
	0x79897908, 0x7a7c7a04, 0x776b76d8, 0x788377f9,
	0xd7d9d65c, 0xdad8d958, 0xd1efd079, 0xd4e1d367,
	0x7503745e, 0x764075a4, 0x7254719d, 0x73b57306,
	0xcc21cab2, 0xcf04cd92, 0xc673c50d, 0xc946c7db,
	0x6f5e6e95, 0x70e17022, 0x6c236b4a, 0x6dc96cf8,
	0xc0e9bf8c, 0xc3a9c248, 0xbb85ba33, 0xbe32bcda,
	0x68a567bc, 0x6a6c698b, 0x64e763ee, 0x66ce65dd,
	0xb64cb505, 0xb8e3b796, 0xb140b005, 0xb3c0b27f,
	0x60eb5fe2, 0x62f161f0, 0x5cb35b9c, 0x5ed65dc6,
	0xac65ab36, 0xaeccad97, 0xa7bda69c, 0xaa0ba8e2,
	0x5842571d, 0x5a815963, 0x539a5268, 0x55f454c9,
	0xa34ca239, 0xa57ea463, 0x9f149e0f, 0xa129a01d,
	0x4ebf4d80, 0x51334ffa, 0x49b34869, 0x4c3f4afa,
	0x9b189a22, 0x9d0e9c11, 0x975a9674, 0x99319843,
	0x447a4325, 0x471c45cc, 0x3f163db7, 0x41cd4073,
	0x93dc9307, 0x959394b5, 0x90a18fdd, 0x9236916a,
	0x398c3824, 0x3c563af2, 0x33de326d, 0x36b9354d,
	0x8dab8cf9, 0x8f1e8e62, 0x8afc8a5b, 0x8c4a8ba1,
	0x2e102c98, 0x30fb2f86, 0x282626a7, 0x2b1e29a3,
	0x88948806, 0x89bf8927, 0x867685fb, 0x877c86f7,
	0x2223209f, 0x252723a6, 0x1c0b1a82, 0x1f191d93,
	0x84a3843b, 0x85838511, 0x831d82c7, 0x83d68377,
	0x15e11455, 0x18f8176d, 0x0fab0e1b, 0x12c71139,
	0x81e381a1, 0x8276822a, 0x80f780c8, 0x8163812b,
	0x096a07d9, 0x0c8b0afb, 0x03240192, 0x064704b6,
	0x8059803e, 0x809e8079, 0x800a8003, 0x80288017,
	0xfcdbfb49, 0x0000fe6d, 0xf695f504, 0xf9b8f826,
	0x800a8017, 0x80018003, 0x80598079, 0x8028803e,
	0xf054eec6, 0xf374f1e4, 0xea1ee892, 0xed38ebaa,
	0x80f7812b, 0x809e80c8, 0x81e3822a, 0x816381a1,
	0xe3f4e26c, 0xe707e57d, 0xdddcdc59, 0xe0e6df60,
	0x831d8377, 0x827682c7, 0x84a38511, 0x83d6843b,
	0xd7d9d65c, 0xdad8d958, 0xd1efd079, 0xd4e1d367,
	0x867686f7, 0x858385fb, 0x88948927, 0x877c8806,
	0xcc21cab2, 0xcf04cd92, 0xc673c50d, 0xc946c7db,
	0x8afc8ba1, 0x89bf8a5b, 0x8dab8e62, 0x8c4a8cf9,
	0xc0e9bf8c, 0xc3a9c248, 0xbb85ba33, 0xbe32bcda,
	0x90a1916a, 0x8f1e8fdd, 0x93dc94b5, 0x92369307,
	0xb64cb505, 0xb8e3b796, 0xb140b005, 0xb3c0b27f,
	0x975a9843, 0x95939674, 0x9b189c11, 0x99319a22,
	0xac65ab36, 0xaeccad97, 0xa7bda69c, 0xaa0ba8e2,
	0x9f14a01d, 0x9d0e9e0f, 0xa34ca463, 0xa129a239,
	0xa34ca239, 0xa57ea463, 0x9f149e0f, 0xa129a01d,
	0xa7bda8e2, 0xa57ea69c, 0xac65ad97, 0xaa0bab36,
	0x9b189a22, 0x9d0e9c11, 0x975a9674, 0x99319843,
	0xb140b27f, 0xaeccb005, 0xb64cb796, 0xb3c0b505,
	0x93dc9307, 0x959394b5, 0x90a18fdd, 0x9236916a,
	0xbb85bcda, 0xb8e3ba33, 0xc0e9c248, 0xbe32bf8c,
	0x8dab8cf9, 0x8f1e8e62, 0x8afc8a5b, 0x8c4a8ba1,
	0xc673c7db, 0xc3a9c50d, 0xcc21cd92, 0xc946cab2,
	0x88948806, 0x89bf8927, 0x867685fb, 0x877c86f7,
	0xd1efd367, 0xcf04d079, 0xd7d9d958, 0xd4e1d65c,
	0x84a3843b, 0x85838511, 0x831d82c7, 0x83d68377,
	0xdddcdf60, 0xdad8dc59, 0xe3f4e57d, 0xe0e6e26c,
	0x81e381a1, 0x8276822a, 0x80f780c8, 0x8163812b,
	0xea1eebaa, 0xe707e892, 0xf054f1e4, 0xed38eec6,
	0x8059803e, 0x809e8079, 0x800a8003, 0x80288017,
	0xf695f826, 0xf374f504, 0xfcdbfe6d, 0xf9b8fb49,
	0x7ffc7ff9, 0x7fff7ffe, 0x7fe87fe0, 0x7ff57fef,
	0xfe6dfda4, 0x0000ff36, 0xfb49fa81, 0xfcdbfc12,
	0x7fc17fb4, 0x7fd77fcd, 0x7f867f74, 0x7fa67f96,
	0xf826f75e, 0xf9b8f8ef, 0xf504f43c, 0xf695f5cd,
	0x7f377f20, 0x7f617f4c, 0x7ed47eb9, 0x7f087eef,
	0xf1e4f11c, 0xf374f2ac, 0xeec6edff, 0xf054ef8d,
	0x7e5e7e3e, 0x7e9c7e7e, 0x7dd57db0, 0x7e1c7df9,
	0xebaaeae4, 0xed38ec71, 0xe892e7cc, 0xea1ee958,
	0x7d387d0e, 0x7d897d61, 0x7c887c59, 0x7ce27cb6,
	0xe57de4b9, 0xe707e642, 0xe26ce1a9, 0xe3f4e330,
	0x7bc47b91, 0x7c297bf7, 0x7aee7ab5, 0x7b5c7b25,
	0xdf60de9e, 0xe0e6e023, 0xdc59db98, 0xdddcdd1b,
	0x7a0479c7, 0x7a7c7a41, 0x790878c6, 0x79897949,
	0xd958d898, 0xdad8da18, 0xd65cd59e, 0xd7d9d71a,
	0x77f977b3, 0x7883783f, 0x76d8768d, 0x776b7722,
	0xd367d2ab, 0xd4e1d424, 0xd079cfbe, 0xd1efd133,
	0x75a47554, 0x764075f3, 0x745e740a, 0x750374b1,
	0xcd92ccd9, 0xcf04ce4b, 0xcab2c9fc, 0xcc21cb69,
	0x730672ae, 0x73b5735e, 0x719d7140, 0x725471f9,
	0xc7dbc727, 0xc946c890, 0xc50dc45b, 0xc673c5c0,
	0x70226fc0, 0x70e17082, 0x6e956e30, 0x6f5e6efa,
	0xc248c198, 0xc3a9c2f8, 0xbf8cbedf, 0xc0e9c03a,
	0x6cf86c8e, 0x6dc96d61, 0x6b4a6adb, 0x6c236bb7,
	0xbcdabc2f, 0xbe32bd86, 0xba33b98b, 0xbb85badc,
	0x698b6919, 0x6a6c69fc, 0x67bc6745, 0x68a56831,
	0xb796b6f1, 0xb8e3b83c, 0xb505b462, 0xb64cb5a8,
	0x65dd6562, 0x66ce6656, 0x63ee6370, 0x64e7646b,
	0xb27fb1df, 0xb3c0b31f, 0xb005af68, 0xb140b0a2,
	0x61f0616e, 0x62f16271, 0x5fe25f5d, 0x60eb6067,
	0xad97acfe, 0xaeccae31, 0xab36aaa0, 0xac65abcd,
	0x5dc65d3d, 0x5ed65e4f, 0x5b9c5b0f, 0x5cb35c28,
	0xa8e2a84f, 0xaa0ba976, 0xa69ca60c, 0xa7bda72c,
	0x596358d3, 0x5a8159f3, 0x571d5689, 0x584257b0,
	0xa463a3d7, 0xa57ea4f0, 0xa239a1b0, 0xa34ca2c2,
	0x54c95432, 0x55f4555f, 0x526851ce, 0x539a5301,
	0xa01d9f98, 0xa129a0a2, 0x9e0f9d8e, 0x9f149e91,
	0x4ffa4f5d, 0x51335097, 0x4d804ce0, 0x4ebf4e20,
	0x9c119b94, 0x9d0e9c8f, 0x9a2299a9, 0x9b189a9d,
	0x4afa4a57, 0x4c3f4b9d, 0x486947c3, 0x49b3490e,
	0x984397ce, 0x993198ba, 0x96749603, 0x975a96e6,
	0x45cc4523, 0x471c4674, 0x43254279, 0x447a43d0,
	0x94b59448, 0x95939524, 0x9307929e, 0x93dc9371,
	0x40733fc5, 0x41cd4120, 0x3db73d07, 0x3f163e67,
	0x916a9105, 0x923691cf, 0x8fdd8f7d, 0x90a1903f,
	0x3af23a3f, 0x3c563ba4, 0x3824376f, 0x398c38d8,
	0x8e628e06, 0x8f1e8ebf, 0x8cf98ca1, 0x8dab8d51,
	0x354d3496, 0x36b93603, 0x326d31b4, 0x33de3326,
	0x8ba18b4e, 0x8c4a8bf5, 0x8a5b8a0c, 0x8afc8aab,
	0x2f862ecc, 0x30fb3041, 0x2c982bdb, 0x2e102d54,
	0x892788dd, 0x89bf8972, 0x880687c0, 0x8894884c,
	0x29a328e5, 0x2b1e2a61, 0x26a725e7, 0x28262767,
	0x86f786b6, 0x877c8739, 0x85fb85be, 0x86768638,
	0x23a622e4, 0x25272467, 0x209f1fdc, 0x22232161,
	0x851184da, 0x8583854a, 0x843b8408, 0x84a3846e,
	0x1d931ccf, 0x1f191e56, 0x1a8219bd, 0x1c0b1b46,
	0x83778349, 0x83d683a6, 0x82c7829e, 0x831d82f1,
	0x176d16a7, 0x18f81833, 0x1455138e, 0x15e1151b,
	0x822a8206, 0x8276824f, 0x81a18181, 0x81e381c1,
	0x11391072, 0x12c71200, 0x0e1b0d53, 0x0fab0ee3,
	0x812b8110, 0x81638146, 0x80c880b3, 0x80f780df,
	0x0afb0a32, 0x0c8b0bc3, 0x07d90710, 0x096a08a1,
	0x80798069, 0x809e808b, 0x803e8032, 0x8059804b,
	0x04b603ed, 0x0647057e, 0x019200c9, 0x0324025b,
	0x80178010, 0x8028801f, 0x80038001, 0x800a8006,
	0xfe6dfda4, 0x0000ff36, 0xfb49fa81, 0xfcdbfc12,
	0x80038006, 0x80018001, 0x8017801f, 0x800a8010,
	0xf826f75e, 0xf9b8f8ef, 0xf504f43c, 0xf695f5cd,
	0x803e804b, 0x80288032, 0x8079808b, 0x80598069,
	0xf1e4f11c, 0xf374f2ac, 0xeec6edff, 0xf054ef8d,
	0x80c880df, 0x809e80b3, 0x812b8146, 0x80f78110,
	0xebaaeae4, 0xed38ec71, 0xe892e7cc, 0xea1ee958,
	0x81a181c1, 0x81638181, 0x822a824f, 0x81e38206,
	0xe57de4b9, 0xe707e642, 0xe26ce1a9, 0xe3f4e330,
	0x82c782f1, 0x8276829e, 0x837783a6, 0x831d8349,
	0xdf60de9e, 0xe0e6e023, 0xdc59db98, 0xdddcdd1b,
	0x843b846e, 0x83d68408, 0x8511854a, 0x84a384da,
	0xd958d898, 0xdad8da18, 0xd65cd59e, 0xd7d9d71a,
	0x85fb8638, 0x858385be, 0x86f78739, 0x867686b6,
	0xd367d2ab, 0xd4e1d424, 0xd079cfbe, 0xd1efd133,
	0x8806884c, 0x877c87c0, 0x89278972, 0x889488dd,
	0xcd92ccd9, 0xcf04ce4b, 0xcab2c9fc, 0xcc21cb69,
	0x8a5b8aab, 0x89bf8a0c, 0x8ba18bf5, 0x8afc8b4e,
	0xc7dbc727, 0xc946c890, 0xc50dc45b, 0xc673c5c0,
	0x8cf98d51, 0x8c4a8ca1, 0x8e628ebf, 0x8dab8e06,
	0xc248c198, 0xc3a9c2f8, 0xbf8cbedf, 0xc0e9c03a,
	0x8fdd903f, 0x8f1e8f7d, 0x916a91cf, 0x90a19105,
	0xbcdabc2f, 0xbe32bd86, 0xba33b98b, 0xbb85badc,
	0x93079371, 0x9236929e, 0x94b59524, 0x93dc9448,
	0xb796b6f1, 0xb8e3b83c, 0xb505b462, 0xb64cb5a8,
	0x967496e6, 0x95939603, 0x984398ba, 0x975a97ce,
	0xb27fb1df, 0xb3c0b31f, 0xb005af68, 0xb140b0a2,
	0x9a229a9d, 0x993199a9, 0x9c119c8f, 0x9b189b94,
	0xad97acfe, 0xaeccae31, 0xab36aaa0, 0xac65abcd,
	0x9e0f9e91, 0x9d0e9d8e, 0xa01da0a2, 0x9f149f98,
	0xa8e2a84f, 0xaa0ba976, 0xa69ca60c, 0xa7bda72c,
	0xa239a2c2, 0xa129a1b0, 0xa463a4f0, 0xa34ca3d7,
	0xa463a3d7, 0xa57ea4f0, 0xa239a1b0, 0xa34ca2c2,
	0xa69ca72c, 0xa57ea60c, 0xa8e2a976, 0xa7bda84f,
	0xa01d9f98, 0xa129a0a2, 0x9e0f9d8e, 0x9f149e91,
	0xab36abcd, 0xaa0baaa0, 0xad97ae31, 0xac65acfe,
	0x9c119b94, 0x9d0e9c8f, 0x9a2299a9, 0x9b189a9d,
	0xb005b0a2, 0xaeccaf68, 0xb27fb31f, 0xb140b1df,
	0x984397ce, 0x993198ba, 0x96749603, 0x975a96e6,
	0xb505b5a8, 0xb3c0b462, 0xb796b83c, 0xb64cb6f1,
	0x94b59448, 0x95939524, 0x9307929e, 0x93dc9371,
	0xba33badc, 0xb8e3b98b, 0xbcdabd86, 0xbb85bc2f,
	0x916a9105, 0x923691cf, 0x8fdd8f7d, 0x90a1903f,
	0xbf8cc03a, 0xbe32bedf, 0xc248c2f8, 0xc0e9c198,
	0x8e628e06, 0x8f1e8ebf, 0x8cf98ca1, 0x8dab8d51,
	0xc50dc5c0, 0xc3a9c45b, 0xc7dbc890, 0xc673c727,
	0x8ba18b4e, 0x8c4a8bf5, 0x8a5b8a0c, 0x8afc8aab,
	0xcab2cb69, 0xc946c9fc, 0xcd92ce4b, 0xcc21ccd9,
	0x892788dd, 0x89bf8972, 0x880687c0, 0x8894884c,
	0xd079d133, 0xcf04cfbe, 0xd367d424, 0xd1efd2ab,
	0x86f786b6, 0x877c8739, 0x85fb85be, 0x86768638,
	0xd65cd71a, 0xd4e1d59e, 0xd958da18, 0xd7d9d898,
	0x851184da, 0x8583854a, 0x843b8408, 0x84a3846e,
	0xdc59dd1b, 0xdad8db98, 0xdf60e023, 0xdddcde9e,
	0x83778349, 0x83d683a6, 0x82c7829e, 0x831d82f1,
	0xe26ce330, 0xe0e6e1a9, 0xe57de642, 0xe3f4e4b9,
	0x822a8206, 0x8276824f, 0x81a18181, 0x81e381c1,
	0xe892e958, 0xe707e7cc, 0xebaaec71, 0xea1eeae4,
	0x812b8110, 0x81638146, 0x80c880b3, 0x80f780df,
	0xeec6ef8d, 0xed38edff, 0xf1e4f2ac, 0xf054f11c,
	0x80798069, 0x809e808b, 0x803e8032, 0x8059804b,
	0xf504f5cd, 0xf374f43c, 0xf826f8ef, 0xf695f75e,
	0x80178010, 0x8028801f, 0x80038001, 0x800a8006,
	0xfb49fc12, 0xf9b8fa81, 0xfe6dff36, 0xfcdbfda4,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_1_P[] = {
	0x00000000, 0x00000000, 0x0000a57e, 0x7fff5a81,
	0x00000000, 0x00000000, 0x7fff5a81, 0x00005a81,
	0x5a8130fb, 0x7fff7640, 0xa57e89bf, 0x0000cf04,
	0x5a817640, 0x000030fb, 0x5a8130fb, 0x7fff7640,
	0x76406a6c, 0x7fff7d89, 0x30fb18f8, 0x5a81471c,
	0x30fb471c, 0x000018f8, 0x76407d89, 0x5a816a6c,
	0xcf04b8e3, 0x0000e707, 0x89bf8276, 0xa57e9593,
	0x76406a6c, 0x7fff7d89, 0x30fb18f8, 0x5a81471c,
	0x7d897a7c, 0x7fff7f61, 0x6a6c62f1, 0x764070e1,
	0x18f82527, 0x00000c8b, 0x471c5133, 0x30fb3c56,
	0x471c3c56, 0x5a815133, 0x18f80c8b, 0x30fb2527,
	0x6a6c70e1, 0x5a8162f1, 0x7d897f61, 0x76407a7c,
	0xe707dad8, 0x0000f374, 0xb8e3aecc, 0xcf04c3a9,
	0x7d897a7c, 0x7fff7f61, 0x6a6c62f1, 0x764070e1,
	0x95938f1e, 0xa57e9d0e, 0x8276809e, 0x89bf8583,
	0x471c3c56, 0x5a815133, 0x18f80c8b, 0x30fb2527,
	0x7f617e9c, 0x7fff7fd7, 0x7a7c7883, 0x7d897c29,
	0x0c8b12c7, 0x00000647, 0x25272b1e, 0x18f81f19,
	0x70e16dc9, 0x764073b5, 0x62f15ed6, 0x6a6c66ce,
	0x3c5641cd, 0x30fb36b9, 0x513355f4, 0x471c4c3f,
	0x51334c3f, 0x5a8155f4, 0x3c5636b9, 0x471c41cd,
	0x62f166ce, 0x5a815ed6, 0x70e173b5, 0x6a6c6dc9,
	0x25271f19, 0x30fb2b1e, 0x0c8b0647, 0x18f812c7,
	0x7a7c7c29, 0x76407883, 0x7f617fd7, 0x7d897e9c,
	0xf374ed38, 0x0000f9b8, 0xdad8d4e1, 0xe707e0e6,
	0x7f617e9c, 0x7ffe7fd7, 0x7a7c7883, 0x7d897c29,
	0xc3a9be32, 0xcf04c946, 0xaeccaa0b, 0xb8e3b3c0,
	0x70e16dc9, 0x764073b5, 0x62f15ed6, 0x6a6c66ce,
	0x9d0e9931, 0xa57ea129, 0x8f1e8c4a, 0x95939236,
	0x51334c3f, 0x5a8155f4, 0x3c5636b9, 0x471c41cd,
	0x858383d6, 0x89bf877c, 0x809e8028, 0x82768163,
	0x25271f19, 0x30fb2b1e, 0x0c8b0647, 0x18f812c7,
	0x7fd77fa6, 0x7fff7ff5, 0x7e9c7e1c, 0x7f617f08,
	0x0647096a, 0x00000324, 0x12c715e1, 0x0c8b0fab,
	0x7c297b5c, 0x7d897ce2, 0x7883776b, 0x7a7c7989,
	0x1f192223, 0x18f81c0b, 0x2b1e2e10, 0x25272826,
	0x73b57254, 0x76407503, 0x6dc96c23, 0x70e16f5e,
	0x36b9398c, 0x30fb33de, 0x41cd447a, 0x3c563f16,
	0x66ce64e7, 0x6a6c68a5, 0x5ed65cb3, 0x62f160eb,
	0x4c3f4ebf, 0x471c49b3, 0x55f45842, 0x5133539a,
	0x55f4539a, 0x5a815842, 0x4c3f49b3, 0x51334ebf,
	0x5ed660eb, 0x5a815cb3, 0x66ce68a5, 0x62f164e7,
	0x41cd3f16, 0x471c447a, 0x36b933de, 0x3c56398c,
	0x6dc96f5e, 0x6a6c6c23, 0x73b57503, 0x70e17254,
	0x2b1e2826, 0x30fb2e10, 0x1f191c0b, 0x25272223,
	0x78837989, 0x7640776b, 0x7c297ce2, 0x7a7c7b5c,
	0x12c70fab, 0x18f815e1, 0x06470324, 0x0c8b096a,
	0x7e9c7f08, 0x7d897e1c, 0x7fd77ff5, 0x7f617fa6,
	0xf9b8f695, 0x0000fcdb, 0xed38ea1e, 0xf374f054,
	0x7fd77fa6, 0x7fff7ff5, 0x7e9c7e1c, 0x7f617f08,
	0xe0e6dddc, 0xe707e3f4, 0xd4e1d1ef, 0xdad8d7d9,
	0x7c297b5c, 0x7d897ce2, 0x7883776b, 0x7a7c7989,
	0xc946c673, 0xcf04cc21, 0xbe32bb85, 0xc3a9c0e9,
	0x73b57254, 0x76407503, 0x6dc96c23, 0x70e16f5e,
	0xb3c0b140, 0xb8e3b64c, 0xaa0ba7bd, 0xaeccac65,
	0x66ce64e7, 0x6a6c68a5, 0x5ed65cb3, 0x62f160eb,
	0xa1299f14, 0xa57ea34c, 0x9931975a, 0x9d0e9b18,
	0x55f4539a, 0x5a815842, 0x4c3f49b3, 0x51334ebf,
	0x923690a1, 0x959393dc, 0x8c4a8afc, 0x8f1e8dab,
	0x41cd3f16, 0x471c447a, 0x36b933de, 0x3c56398c,
	0x877c8676, 0x89bf8894, 0x83d6831d, 0x858384a3,
	0x2b1e2826, 0x30fb2e10, 0x1f191c0b, 0x25272223,
	0x816380f7, 0x827681e3, 0x8028800a, 0x809e8059,
	0x12c70fab, 0x18f815e1, 0x06470324, 0x0c8b096a,
	0x7ff57fe8, 0x7fff7ffc, 0x7fa67f86, 0x7fd77fc1,
	0x032404b6, 0x00000192, 0x096a0afb, 0x064707d9,
	0x7f087ed4, 0x7f617f37, 0x7e1c7dd5, 0x7e9c7e5e,
	0x0fab1139, 0x0c8b0e1b, 0x15e1176d, 0x12c71455,
	0x7ce27c88, 0x7d897d38, 0x7b5c7aee, 0x7c297bc4,
	0x1c0b1d93, 0x18f81a82, 0x222323a6, 0x1f19209f,
	0x79897908, 0x7a7c7a04, 0x776b76d8, 0x788377f9,
	0x282629a3, 0x252726a7, 0x2e102f86, 0x2b1e2c98,
	0x7503745e, 0x764075a4, 0x7254719d, 0x73b57306,
	0x33de354d, 0x30fb326d, 0x398c3af2, 0x36b93824,
	0x6f5e6e95, 0x70e17022, 0x6c236b4a, 0x6dc96cf8,
	0x3f164073, 0x3c563db7, 0x447a45cc, 0x41cd4325,
	0x68a567bc, 0x6a6c698b, 0x64e763ee, 0x66ce65dd,
	0x49b34afa, 0x471c4869, 0x4ebf4ffa, 0x4c3f4d80,
	0x60eb5fe2, 0x62f161f0, 0x5cb35b9c, 0x5ed65dc6,
	0x539a54c9, 0x51335268, 0x58425963, 0x55f4571d,
	0x5842571d, 0x5a815963, 0x539a5268, 0x55f454c9,
	0x5cb35dc6, 0x5a815b9c, 0x60eb61f0, 0x5ed65fe2,
	0x4ebf4d80, 0x51334ffa, 0x49b34869, 0x4c3f4afa,
	0x64e765dd, 0x62f163ee, 0x68a5698b, 0x66ce67bc,
	0x447a4325, 0x471c45cc, 0x3f163db7, 0x41cd4073,
	0x6c236cf8, 0x6a6c6b4a, 0x6f5e7022, 0x6dc96e95,
	0x398c3824, 0x3c563af2, 0x33de326d, 0x36b9354d,
	0x72547306, 0x70e1719d, 0x750375a4, 0x73b5745e,
	0x2e102c98, 0x30fb2f86, 0x282626a7, 0x2b1e29a3,
	0x776b77f9, 0x764076d8, 0x79897a04, 0x78837908,
	0x2223209f, 0x252723a6, 0x1c0b1a82, 0x1f191d93,
	0x7b5c7bc4, 0x7a7c7aee, 0x7ce27d38, 0x7c297c88,
	0x15e11455, 0x18f8176d, 0x0fab0e1b, 0x12c71139,
	0x7e1c7e5e, 0x7d897dd5, 0x7f087f37, 0x7e9c7ed4,
	0x096a07d9, 0x0c8b0afb, 0x03240192, 0x064704b6,
	0x7fa67fc1, 0x7f617f86, 0x7ff57ffc, 0x7fd77fe8,
	0xfcdbfb49, 0x0000fe6d, 0xf695f504, 0xf9b8f826,
	0x7ff57fe8, 0x7fff7ffc, 0x7fa67f86, 0x7fd77fc1,
	0xf054eec6, 0xf374f1e4, 0xea1ee892, 0xed38ebaa,
	0x7f087ed4, 0x7f617f37, 0x7e1c7dd5, 0x7e9c7e5e,
	0xe3f4e26c, 0xe707e57d, 0xdddcdc59, 0xe0e6df60,
	0x7ce27c88, 0x7d897d38, 0x7b5c7aee, 0x7c297bc4,
	0xd7d9d65c, 0xdad8d958, 0xd1efd079, 0xd4e1d367,
	0x79897908, 0x7a7c7a04, 0x776b76d8, 0x788377f9,
	0xcc21cab2, 0xcf04cd92, 0xc673c50d, 0xc946c7db,
	0x7503745e, 0x764075a4, 0x7254719d, 0x73b57306,
	0xc0e9bf8c, 0xc3a9c248, 0xbb85ba33, 0xbe32bcda,
	0x6f5e6e95, 0x70e17022, 0x6c236b4a, 0x6dc96cf8,
	0xb64cb505, 0xb8e3b796, 0xb140b005, 0xb3c0b27f,
	0x68a567bc, 0x6a6c698b, 0x64e763ee, 0x66ce65dd,
	0xac65ab36, 0xaeccad97, 0xa7bda69c, 0xaa0ba8e2,
	0x60eb5fe2, 0x62f161f0, 0x5cb35b9c, 0x5ed65dc6,
	0xa34ca239, 0xa57ea463, 0x9f149e0f, 0xa129a01d,
	0x5842571d, 0x5a815963, 0x539a5268, 0x55f454c9,
	0x9b189a22, 0x9d0e9c11, 0x975a9674, 0x99319843,
	0x4ebf4d80, 0x51334ffa, 0x49b34869, 0x4c3f4afa,
	0x93dc9307, 0x959394b5, 0x90a18fdd, 0x9236916a,
	0x447a4325, 0x471c45cc, 0x3f163db7, 0x41cd4073,
	0x8dab8cf9, 0x8f1e8e62, 0x8afc8a5b, 0x8c4a8ba1,
	0x398c3824, 0x3c563af2, 0x33de326d, 0x36b9354d,
	0x88948806, 0x89bf8927, 0x867685fb, 0x877c86f7,
	0x2e102c98, 0x30fb2f86, 0x282626a7, 0x2b1e29a3,
	0x84a3843b, 0x85838511, 0x831d82c7, 0x83d68377,
	0x2223209f, 0x252723a6, 0x1c0b1a82, 0x1f191d93,
	0x81e381a1, 0x8276822a, 0x80f780c8, 0x8163812b,
	0x15e11455, 0x18f8176d, 0x0fab0e1b, 0x12c71139,
	0x8059803e, 0x809e8079, 0x800a8003, 0x80288017,
	0x096a07d9, 0x0c8b0afb, 0x03240192, 0x064704b6,
	0x7ffc7ff9, 0x7fff7ffe, 0x7fe87fe0, 0x7ff57fef,
	0x0192025b, 0x000000c9, 0x04b6057e, 0x032403ed,
	0x7fc17fb4, 0x7fd77fcd, 0x7f867f74, 0x7fa67f96,
	0x07d908a1, 0x06470710, 0x0afb0bc3, 0x096a0a32,
	0x7f377f20, 0x7f617f4c, 0x7ed47eb9, 0x7f087eef,
	0x0e1b0ee3, 0x0c8b0d53, 0x11391200, 0x0fab1072,
	0x7e5e7e3e, 0x7e9c7e7e, 0x7dd57db0, 0x7e1c7df9,
	0x1455151b, 0x12c7138e, 0x176d1833, 0x15e116a7,
	0x7d387d0e, 0x7d897d61, 0x7c887c59, 0x7ce27cb6,
	0x1a821b46, 0x18f819bd, 0x1d931e56, 0x1c0b1ccf,
	0x7bc47b91, 0x7c297bf7, 0x7aee7ab5, 0x7b5c7b25,
	0x209f2161, 0x1f191fdc, 0x23a62467, 0x222322e4,
	0x7a0479c7, 0x7a7c7a41, 0x790878c6, 0x79897949,
	0x26a72767, 0x252725e7, 0x29a32a61, 0x282628e5,
	0x77f977b3, 0x7883783f, 0x76d8768d, 0x776b7722,
	0x2c982d54, 0x2b1e2bdb, 0x2f863041, 0x2e102ecc,
	0x75a47554, 0x764075f3, 0x745e740a, 0x750374b1,
	0x326d3326, 0x30fb31b4, 0x354d3603, 0x33de3496,
	0x730672ae, 0x73b5735e, 0x719d7140, 0x725471f9,
	0x382438d8, 0x36b9376f, 0x3af23ba4, 0x398c3a3f,
	0x70226fc0, 0x70e17082, 0x6e956e30, 0x6f5e6efa,
	0x3db73e67, 0x3c563d07, 0x40734120, 0x3f163fc5,
	0x6cf86c8e, 0x6dc96d61, 0x6b4a6adb, 0x6c236bb7,
	0x432543d0, 0x41cd4279, 0x45cc4674, 0x447a4523,
	0x698b6919, 0x6a6c69fc, 0x67bc6745, 0x68a56831,
	0x4869490e, 0x471c47c3, 0x4afa4b9d, 0x49b34a57,
	0x65dd6562, 0x66ce6656, 0x63ee6370, 0x64e7646b,
	0x4d804e20, 0x4c3f4ce0, 0x4ffa5097, 0x4ebf4f5d,
	0x61f0616e, 0x62f16271, 0x5fe25f5d, 0x60eb6067,
	0x52685301, 0x513351ce, 0x54c9555f, 0x539a5432,
	0x5dc65d3d, 0x5ed65e4f, 0x5b9c5b0f, 0x5cb35c28,
	0x571d57b0, 0x55f45689, 0x596359f3, 0x584258d3,
	0x596358d3, 0x5a8159f3, 0x571d5689, 0x584257b0,
	0x5b9c5c28, 0x5a815b0f, 0x5dc65e4f, 0x5cb35d3d,
	0x54c95432, 0x55f4555f, 0x526851ce, 0x539a5301,
	0x5fe26067, 0x5ed65f5d, 0x61f06271, 0x60eb616e,
	0x4ffa4f5d, 0x51335097, 0x4d804ce0, 0x4ebf4e20,
	0x63ee646b, 0x62f16370, 0x65dd6656, 0x64e76562,
	0x4afa4a57, 0x4c3f4b9d, 0x486947c3, 0x49b3490e,
	0x67bc6831, 0x66ce6745, 0x698b69fc, 0x68a56919,
	0x45cc4523, 0x471c4674, 0x43254279, 0x447a43d0,
	0x6b4a6bb7, 0x6a6c6adb, 0x6cf86d61, 0x6c236c8e,
	0x40733fc5, 0x41cd4120, 0x3db73d07, 0x3f163e67,
	0x6e956efa, 0x6dc96e30, 0x70227082, 0x6f5e6fc0,
	0x3af23a3f, 0x3c563ba4, 0x3824376f, 0x398c38d8,
	0x719d71f9, 0x70e17140, 0x7306735e, 0x725472ae,
	0x354d3496, 0x36b93603, 0x326d31b4, 0x33de3326,
	0x745e74b1, 0x73b5740a, 0x75a475f3, 0x75037554,
	0x2f862ecc, 0x30fb3041, 0x2c982bdb, 0x2e102d54,
	0x76d87722, 0x7640768d, 0x77f9783f, 0x776b77b3,
	0x29a328e5, 0x2b1e2a61, 0x26a725e7, 0x28262767,
	0x79087949, 0x788378c6, 0x7a047a41, 0x798979c7,
	0x23a622e4, 0x25272467, 0x209f1fdc, 0x22232161,
	0x7aee7b25, 0x7a7c7ab5, 0x7bc47bf7, 0x7b5c7b91,
	0x1d931ccf, 0x1f191e56, 0x1a8219bd, 0x1c0b1b46,
	0x7c887cb6, 0x7c297c59, 0x7d387d61, 0x7ce27d0e,
	0x176d16a7, 0x18f81833, 0x1455138e, 0x15e1151b,
	0x7dd57df9, 0x7d897db0, 0x7e5e7e7e, 0x7e1c7e3e,
	0x11391072, 0x12c71200, 0x0e1b0d53, 0x0fab0ee3,
	0x7ed47eef, 0x7e9c7eb9, 0x7f377f4c, 0x7f087f20,
	0x0afb0a32, 0x0c8b0bc3, 0x07d90710, 0x096a08a1,
	0x7f867f96, 0x7f617f74, 0x7fc17fcd, 0x7fa67fb4,
	0x04b603ed, 0x0647057e, 0x019200c9, 0x0324025b,
	0x7fe87fef, 0x7fd77fe0, 0x7ffc7ffe, 0x7ff57ff9,
	0xfe6dfda4, 0x0000ff36, 0xfb49fa81, 0xfcdbfc12,
	0x7ffc7ff9, 0x7fff7ffe, 0x7fe87fe0, 0x7ff57fef,
	0xf826f75e, 0xf9b8f8ef, 0xf504f43c, 0xf695f5cd,
	0x7fc17fb4, 0x7fd77fcd, 0x7f867f74, 0x7fa67f96,
	0xf1e4f11c, 0xf374f2ac, 0xeec6edff, 0xf054ef8d,
	0x7f377f20, 0x7f617f4c, 0x7ed47eb9, 0x7f087eef,
	0xebaaeae4, 0xed38ec71, 0xe892e7cc, 0xea1ee958,
	0x7e5e7e3e, 0x7e9c7e7e, 0x7dd57db0, 0x7e1c7df9,
	0xe57de4b9, 0xe707e642, 0xe26ce1a9, 0xe3f4e330,
	0x7d387d0e, 0x7d897d61, 0x7c887c59, 0x7ce27cb6,
	0xdf60de9e, 0xe0e6e023, 0xdc59db98, 0xdddcdd1b,
	0x7bc47b91, 0x7c297bf7, 0x7aee7ab5, 0x7b5c7b25,
	0xd958d898, 0xdad8da18, 0xd65cd59e, 0xd7d9d71a,
	0x7a0479c7, 0x7a7c7a41, 0x790878c6, 0x79897949,
	0xd367d2ab, 0xd4e1d424, 0xd079cfbe, 0xd1efd133,
	0x77f977b3, 0x7883783f, 0x76d8768d, 0x776b7722,
	0xcd92ccd9, 0xcf04ce4b, 0xcab2c9fc, 0xcc21cb69,
	0x75a47554, 0x764075f3, 0x745e740a, 0x750374b1,
	0xc7dbc727, 0xc946c890, 0xc50dc45b, 0xc673c5c0,
	0x730672ae, 0x73b5735e, 0x719d7140, 0x725471f9,
	0xc248c198, 0xc3a9c2f8, 0xbf8cbedf, 0xc0e9c03a,
	0x70226fc0, 0x70e17082, 0x6e956e30, 0x6f5e6efa,
	0xbcdabc2f, 0xbe32bd86, 0xba33b98b, 0xbb85badc,
	0x6cf86c8e, 0x6dc96d61, 0x6b4a6adb, 0x6c236bb7,
	0xb796b6f1, 0xb8e3b83c, 0xb505b462, 0xb64cb5a8,
	0x698b6919, 0x6a6c69fc, 0x67bc6745, 0x68a56831,
	0xb27fb1df, 0xb3c0b31f, 0xb005af68, 0xb140b0a2,
	0x65dd6562, 0x66ce6656, 0x63ee6370, 0x64e7646b,
	0xad97acfe, 0xaeccae31, 0xab36aaa0, 0xac65abcd,
	0x61f0616e, 0x62f16271, 0x5fe25f5d, 0x60eb6067,
	0xa8e2a84f, 0xaa0ba976, 0xa69ca60c, 0xa7bda72c,
	0x5dc65d3d, 0x5ed65e4f, 0x5b9c5b0f, 0x5cb35c28,
	0xa463a3d7, 0xa57ea4f0, 0xa239a1b0, 0xa34ca2c2,
	0x596358d3, 0x5a8159f3, 0x571d5689, 0x584257b0,
	0xa01d9f98, 0xa129a0a2, 0x9e0f9d8e, 0x9f149e91,
	0x54c95432, 0x55f4555f, 0x526851ce, 0x539a5301,
	0x9c119b94, 0x9d0e9c8f, 0x9a2299a9, 0x9b189a9d,
	0x4ffa4f5d, 0x51335097, 0x4d804ce0, 0x4ebf4e20,
	0x984397ce, 0x993198ba, 0x96749603, 0x975a96e6,
	0x4afa4a57, 0x4c3f4b9d, 0x486947c3, 0x49b3490e,
	0x94b59448, 0x95939524, 0x9307929e, 0x93dc9371,
	0x45cc4523, 0x471c4674, 0x43254279, 0x447a43d0,
	0x916a9105, 0x923691cf, 0x8fdd8f7d, 0x90a1903f,
	0x40733fc5, 0x41cd4120, 0x3db73d07, 0x3f163e67,
	0x8e628e06, 0x8f1e8ebf, 0x8cf98ca1, 0x8dab8d51,
	0x3af23a3f, 0x3c563ba4, 0x3824376f, 0x398c38d8,
	0x8ba18b4e, 0x8c4a8bf5, 0x8a5b8a0c, 0x8afc8aab,
	0x354d3496, 0x36b93603, 0x326d31b4, 0x33de3326,
	0x892788dd, 0x89bf8972, 0x880687c0, 0x8894884c,
	0x2f862ecc, 0x30fb3041, 0x2c982bdb, 0x2e102d54,
	0x86f786b6, 0x877c8739, 0x85fb85be, 0x86768638,
	0x29a328e5, 0x2b1e2a61, 0x26a725e7, 0x28262767,
	0x851184da, 0x8583854a, 0x843b8408, 0x84a3846e,
	0x23a622e4, 0x25272467, 0x209f1fdc, 0x22232161,
	0x83778349, 0x83d683a6, 0x82c7829e, 0x831d82f1,
	0x1d931ccf, 0x1f191e56, 0x1a8219bd, 0x1c0b1b46,
	0x822a8206, 0x8276824f, 0x81a18181, 0x81e381c1,
	0x176d16a7, 0x18f81833, 0x1455138e, 0x15e1151b,
	0x812b8110, 0x81638146, 0x80c880b3, 0x80f780df,
	0x11391072, 0x12c71200, 0x0e1b0d53, 0x0fab0ee3,
	0x80798069, 0x809e808b, 0x803e8032, 0x8059804b,
	0x0afb0a32, 0x0c8b0bc3, 0x07d90710, 0x096a08a1,
	0x80178010, 0x8028801f, 0x80038001, 0x800a8006,
	0x04b603ed, 0x0647057e, 0x019200c9, 0x0324025b,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_3_M[] = {
	0x00000000, 0x00000000, 0x00005a81, 0x7fffa57e,
	0x00000000, 0x00000000, 0x7fffa57e, 0x0000a57e,
	0xa57e89bf, 0x7fff30fb, 0x5a81cf04, 0x00007640,
	0xa57e30fb, 0x000089bf, 0xa57e89bf, 0x7fff30fb,
	0x30fbe707, 0x7fff6a6c, 0x89bfb8e3, 0xa57e8276,
	0x89bf8276, 0x0000b8e3, 0x30fb6a6c, 0xa57ee707,
	0x76407d89, 0x0000471c, 0xcf049593, 0x5a8118f8,
	0x30fbe707, 0x7fff6a6c, 0x89bfb8e3, 0xa57e8276,
	0x6a6c5133, 0x7fff7a7c, 0xe707c3a9, 0x30fb0c8b,
	0xb8e39d0e, 0x0000dad8, 0x82768f1e, 0x89bf809e,
	0x8276809e, 0xa57e8f1e, 0xb8e3dad8, 0x89bf9d0e,
	0xe7070c8b, 0xa57ec3a9, 0x6a6c7a7c, 0x30fb5133,
	0x471c62f1, 0x00002527, 0x7d8970e1, 0x76407f61,
	0x6a6c5133, 0x7ffe7a7c, 0xe707c3a9, 0x30fb0c8b,
	0x18f8f374, 0x5a813c56, 0x95938583, 0xcf04aecc,
	0x8276809e, 0xa57e8f1e, 0xb8e3dad8, 0x89bf9d0e,
	0x7a7c73b5, 0x7fff7e9c, 0x513341cd, 0x6a6c5ed6,
	0xdad8c946, 0x0000ed38, 0x9d0e9236, 0xb8e3aa0b,
	0x0c8bf9b8, 0x30fb1f19, 0xc3a9b3c0, 0xe707d4e1,
	0x809e8028, 0x89bf83d6, 0x8f1e9931, 0x8276877c,
	0x8f1e877c, 0xa57e9931, 0x809e83d6, 0x82768028,
	0xc3a9d4e1, 0xa57eb3c0, 0x0c8b1f19, 0xe707f9b8,
	0x9d0eaa0b, 0x89bf9236, 0xdad8ed38, 0xb8e3c946,
	0x51335ed6, 0x30fb41cd, 0x7a7c7e9c, 0x6a6c73b5,
	0x252736b9, 0x000012c7, 0x62f16dc9, 0x471c55f4,
	0x7a7c73b5, 0x7ffe7e9c, 0x513341cd, 0x6a6c5ed6,
	0x7f617fd7, 0x76407c29, 0x70e166ce, 0x7d897883,
	0x0c8bf9b8, 0x30fb1f19, 0xc3a9b3c0, 0xe707d4e1,
	0x3c562b1e, 0x5a814c3f, 0xf374e0e6, 0x18f80647,
	0x8f1e877c, 0xa57e9931, 0x809e83d6, 0x82768028,
	0xaecca129, 0xcf04be32, 0x85838163, 0x95938c4a,
	0x9d0eaa0b, 0x89bf9236, 0xdad8ed38, 0xb8e3c946,
	0x7e9c7ce2, 0x7fff7fa6, 0x73b56f5e, 0x7a7c776b,
	0xed38e3f4, 0x0000f695, 0xc946c0e9, 0xdad8d1ef,
	0x5ed65842, 0x6a6c64e7, 0x41cd398c, 0x513349b3,
	0xaa0ba34c, 0xb8e3b140, 0x92368dab, 0x9d0e975a,
	0x1f1915e1, 0x30fb2826, 0xf9b8f054, 0x0c8b0324,
	0x83d681e3, 0x89bf8676, 0x802880f7, 0x809e800a,
	0xd4e1cc21, 0xe707dddc, 0xb3c0ac65, 0xc3a9bb85,
	0x877c8afc, 0x827684a3, 0x99319f14, 0x8f1e93dc,
	0x993193dc, 0xa57e9f14, 0x877c84a3, 0x8f1e8afc,
	0xb3c0bb85, 0xa57eac65, 0xd4e1dddc, 0xc3a9cc21,
	0x8028800a, 0x827680f7, 0x83d68676, 0x809e81e3,
	0xf9b80324, 0xe707f054, 0x1f192826, 0x0c8b15e1,
	0x9236975a, 0x89bf8dab, 0xaa0bb140, 0x9d0ea34c,
	0x41cd49b3, 0x30fb398c, 0x5ed664e7, 0x51335842,
	0xc946d1ef, 0xb8e3c0e9, 0xed38f695, 0xdad8e3f4,
	0x73b5776b, 0x6a6c6f5e, 0x7e9c7fa6, 0x7a7c7ce2,
	0x12c71c0b, 0x0000096a, 0x36b93f16, 0x25272e10,
	0x7e9c7ce2, 0x7fff7fa6, 0x73b56f5e, 0x7a7c776b,
	0x55f45cb3, 0x471c4ebf, 0x6dc97254, 0x62f168a5,
	0x5ed65842, 0x6a6c64e7, 0x41cd398c, 0x513349b3,
	0x7c297e1c, 0x76407989, 0x7fd77f08, 0x7f617ff5,
	0x1f1915e1, 0x30fb2826, 0xf9b8f054, 0x0c8b0324,
	0x78837503, 0x7d897b5c, 0x66ce60eb, 0x70e16c23,
	0xd4e1cc21, 0xe707dddc, 0xb3c0ac65, 0xc3a9bb85,
	0x4c3f447a, 0x5a81539a, 0x2b1e2223, 0x3c5633de,
	0x993193dc, 0xa57e9f14, 0x877c84a3, 0x8f1e8afc,
	0x0647fcdb, 0x18f80fab, 0xe0e6d7d9, 0xf374ea1e,
	0x8028800a, 0x827680f7, 0x83d68676, 0x809e81e3,
	0xbe32b64c, 0xcf04c673, 0xa1299b18, 0xaecca7bd,
	0x9236975a, 0x89bf8dab, 0xaa0bb140, 0x9d0ea34c,
	0x8c4a8894, 0x959390a1, 0x81638059, 0x8583831d,
	0xc946d1ef, 0xb8e3c0e9, 0xed38f695, 0xdad8e3f4,
	0x7fa67f37, 0x7fff7fe8, 0x7ce27bc4, 0x7e9c7dd5,
	0xf695f1e4, 0x0000fb49, 0xe3f4df60, 0xed38e892,
	0x776b75a4, 0x7a7c7908, 0x6f5e6cf8, 0x73b5719d,
	0xd1efcd92, 0xdad8d65c, 0xc0e9bcda, 0xc946c50d,
	0x64e761f0, 0x6a6c67bc, 0x584254c9, 0x5ed65b9c,
	0xb140ad97, 0xb8e3b505, 0xa34ca01d, 0xaa0ba69c,
	0x49b345cc, 0x51334d80, 0x398c354d, 0x41cd3db7,
	0x975a94b5, 0x9d0e9a22, 0x8dab8ba1, 0x92368fdd,
	0x282623a6, 0x30fb2c98, 0x15e11139, 0x1f191a82,
	0x86768511, 0x89bf8806, 0x81e3812b, 0x83d682c7,
	0x0324fe6d, 0x0c8b07d9, 0xf054ebaa, 0xf9b8f504,
	0x800a8003, 0x809e803e, 0x80f781a1, 0x80288079,
	0xdddcd958, 0xe707e26c, 0xcc21c7db, 0xd4e1d079,
	0x84a385fb, 0x82768377, 0x8afc8cf9, 0x877c8927,
	0xbb85b796, 0xc3a9bf8c, 0xac65a8e2, 0xb3c0b005,
	0x93dc9674, 0x8f1e916a, 0x9f14a239, 0x99319c11,
	0x9f149c11, 0xa57ea239, 0x93dc916a, 0x99319674,
	0xac65b005, 0xa57ea8e2, 0xbb85bf8c, 0xb3c0b796,
	0x8afc8927, 0x8f1e8cf9, 0x84a38377, 0x877c85fb,
	0xcc21d079, 0xc3a9c7db, 0xdddce26c, 0xd4e1d958,
	0x80f78079, 0x827681a1, 0x800a803e, 0x80288003,
	0xf054f504, 0xe707ebaa, 0x032407d9, 0xf9b8fe6d,
	0x81e382c7, 0x809e812b, 0x86768806, 0x83d68511,
	0x15e11a82, 0x0c8b1139, 0x28262c98, 0x1f1923a6,
	0x8dab8fdd, 0x89bf8ba1, 0x975a9a22, 0x923694b5,
	0x398c3db7, 0x30fb354d, 0x49b34d80, 0x41cd45cc,
	0xa34ca69c, 0x9d0ea01d, 0xb140b505, 0xaa0bad97,
	0x58425b9c, 0x513354c9, 0x64e767bc, 0x5ed661f0,
	0xc0e9c50d, 0xb8e3bcda, 0xd1efd65c, 0xc946cd92,
	0x6f5e719d, 0x6a6c6cf8, 0x776b7908, 0x73b575a4,
	0xe3f4e892, 0xdad8df60, 0xf695fb49, 0xed38f1e4,
	0x7ce27dd5, 0x7a7c7bc4, 0x7fa67fe8, 0x7e9c7f37,
	0x096a0e1b, 0x000004b6, 0x1c0b209f, 0x12c7176d,
	0x7fa67f37, 0x7fff7fe8, 0x7ce27bc4, 0x7e9c7dd5,
	0x2e10326d, 0x252729a3, 0x3f164325, 0x36b93af2,
	0x776b75a4, 0x7a7c7908, 0x6f5e6cf8, 0x73b5719d,
	0x4ebf5268, 0x471c4afa, 0x5cb35fe2, 0x55f45963,
	0x64e761f0, 0x6a6c67bc, 0x584254c9, 0x5ed65b9c,
	0x68a56b4a, 0x62f165dd, 0x7254745e, 0x6dc97022,
	0x49b345cc, 0x51334d80, 0x398c354d, 0x41cd3db7,
	0x79897aee, 0x764077f9, 0x7e1c7ed4, 0x7c297d38,
	0x282623a6, 0x30fb2c98, 0x15e11139, 0x1f191a82,
	0x7ff57ffc, 0x7f617fc1, 0x7f087e5e, 0x7fd77f86,
	0x0324fe6d, 0x0c8b07d9, 0xf054ebaa, 0xf9b8f504,
	0x7b5c7a04, 0x7d897c88, 0x75037306, 0x788376d8,
	0xdddcd958, 0xe707e26c, 0xcc21c7db, 0xd4e1d079,
	0x6c23698b, 0x70e16e95, 0x60eb5dc6, 0x66ce63ee,
	0xbb85b796, 0xc3a9bf8c, 0xac65a8e2, 0xb3c0b005,
	0x539a4ffa, 0x5a81571d, 0x447a4073, 0x4c3f4869,
	0x9f149c11, 0xa57ea239, 0x93dc916a, 0x99319674,
	0x33de2f86, 0x3c563824, 0x22231d93, 0x2b1e26a7,
	0x8afc8927, 0x8f1e8cf9, 0x84a38377, 0x877c85fb,
	0x0fab0afb, 0x18f81455, 0xfcdbf826, 0x06470192,
	0x80f78079, 0x827681a1, 0x800a803e, 0x80288003,
	0xea1ee57d, 0xf374eec6, 0xd7d9d367, 0xe0e6dc59,
	0x81e382c7, 0x809e812b, 0x86768806, 0x83d68511,
	0xc673c248, 0xcf04cab2, 0xb64cb27f, 0xbe32ba33,
	0x8dab8fdd, 0x89bf8ba1, 0x975a9a22, 0x923694b5,
	0xa7bda463, 0xaeccab36, 0x9b189843, 0xa1299e0f,
	0xa34ca69c, 0x9d0ea01d, 0xb140b505, 0xaa0bad97,
	0x90a18e62, 0x95939307, 0x889486f7, 0x8c4a8a5b,
	0xc0e9c50d, 0xb8e3bcda, 0xd1efd65c, 0xc946cd92,
	0x831d822a, 0x8583843b, 0x80598017, 0x816380c8,
	0xe3f4e892, 0xdad8df60, 0xf695fb49, 0xed38f1e4,
	0x7fe87fcd, 0x7fff7ff9, 0x7f377eef, 0x7fa67f74,
	0xfb49f8ef, 0x0000fda4, 0xf1e4ef8d, 0xf695f43c,
	0x7dd57d61, 0x7e9c7e3e, 0x7bc47b25, 0x7ce27c59,
	0xe892e642, 0xed38eae4, 0xdf60dd1b, 0xe3f4e1a9,
	0x7908783f, 0x7a7c79c7, 0x75a474b1, 0x776b768d,
	0xd65cd424, 0xdad8d898, 0xcd92cb69, 0xd1efcfbe,
	0x719d7082, 0x73b572ae, 0x6cf86bb7, 0x6f5e6e30,
	0xc50dc2f8, 0xc946c727, 0xbcdabadc, 0xc0e9bedf,
	0x67bc6656, 0x6a6c6919, 0x61f06067, 0x64e76370,
	0xb505b31f, 0xb8e3b6f1, 0xad97abcd, 0xb140af68,
	0x5b9c59f3, 0x5ed65d3d, 0x54c95301, 0x58425689,
	0xa69ca4f0, 0xaa0ba84f, 0xa01d9e91, 0xa34ca1b0,
	0x4d804b9d, 0x51334f5d, 0x45cc43d0, 0x49b347c3,
	0x9a2298ba, 0x9d0e9b94, 0x94b59371, 0x975a9603,
	0x3db73ba4, 0x41cd3fc5, 0x354d3326, 0x398c376f,
	0x8fdd8ebf, 0x92369105, 0x8ba18aab, 0x8dab8ca1,
	0x2c982a61, 0x30fb2ecc, 0x23a62161, 0x282625e7,
	0x88068739, 0x89bf88dd, 0x8511846e, 0x867685be,
	0x1a821833, 0x1f191ccf, 0x11390ee3, 0x15e1138e,
	0x82c7824f, 0x83d68349, 0x812b80df, 0x81e38181,
	0x07d9057e, 0x0c8b0a32, 0xfe6dfc12, 0x032400c9,
	0x803e801f, 0x809e8069, 0x80038010, 0x800a8001,
	0xf504f2ac, 0xf9b8f75e, 0xebaae958, 0xf054edff,
	0x807980b3, 0x8028804b, 0x81a18206, 0x80f78146,
	0xe26ce023, 0xe707e4b9, 0xd958d71a, 0xdddcdb98,
	0x83778408, 0x827682f1, 0x85fb86b6, 0x84a3854a,
	0xd079ce4b, 0xd4e1d2ab, 0xc7dbc5c0, 0xcc21c9fc,
	0x89278a0c, 0x877c884c, 0x8cf98e06, 0x8afc8bf5,
	0xbf8cbd86, 0xc3a9c198, 0xb796b5a8, 0xbb85b98b,
	0x916a929e, 0x8f1e903f, 0x967497ce, 0x93dc9524,
	0xb005ae31, 0xb3c0b1df, 0xa8e2a72c, 0xac65aaa0,
	0x9c119d8e, 0x99319a9d, 0xa239a3d7, 0x9f14a0a2,
	0xa239a0a2, 0xa57ea3d7, 0x9c119a9d, 0x9f149d8e,
	0xa8e2aaa0, 0xa57ea72c, 0xb005b1df, 0xac65ae31,
	0x96749524, 0x993197ce, 0x916a903f, 0x93dc929e,
	0xb796b98b, 0xb3c0b5a8, 0xbf8cc198, 0xbb85bd86,
	0x8cf98bf5, 0x8f1e8e06, 0x8927884c, 0x8afc8a0c,
	0xc7dbc9fc, 0xc3a9c5c0, 0xd079d2ab, 0xcc21ce4b,
	0x85fb854a, 0x877c86b6, 0x837782f1, 0x84a38408,
	0xd958db98, 0xd4e1d71a, 0xe26ce4b9, 0xdddce023,
	0x81a18146, 0x82768206, 0x8079804b, 0x80f780b3,
	0xebaaedff, 0xe707e958, 0xf504f75e, 0xf054f2ac,
	0x80038001, 0x80288010, 0x803e8069, 0x800a801f,
	0xfe6d00c9, 0xf9b8fc12, 0x07d90a32, 0x0324057e,
	0x812b8181, 0x809e80df, 0x82c78349, 0x81e3824f,
	0x1139138e, 0x0c8b0ee3, 0x1a821ccf, 0x15e11833,
	0x851185be, 0x83d6846e, 0x880688dd, 0x86768739,
	0x23a625e7, 0x1f192161, 0x2c982ecc, 0x28262a61,
	0x8ba18ca1, 0x89bf8aab, 0x8fdd9105, 0x8dab8ebf,
	0x354d376f, 0x30fb3326, 0x3db73fc5, 0x398c3ba4,
	0x94b59603, 0x92369371, 0x9a229b94, 0x975a98ba,
	0x45cc47c3, 0x41cd43d0, 0x4d804f5d, 0x49b34b9d,
	0xa01da1b0, 0x9d0e9e91, 0xa69ca84f, 0xa34ca4f0,
	0x54c95689, 0x51335301, 0x5b9c5d3d, 0x584259f3,
	0xad97af68, 0xaa0babcd, 0xb505b6f1, 0xb140b31f,
	0x61f06370, 0x5ed66067, 0x67bc6919, 0x64e76656,
	0xbcdabedf, 0xb8e3badc, 0xc50dc727, 0xc0e9c2f8,
	0x6cf86e30, 0x6a6c6bb7, 0x719d72ae, 0x6f5e7082,
	0xcd92cfbe, 0xc946cb69, 0xd65cd898, 0xd1efd424,
	0x75a4768d, 0x73b574b1, 0x790879c7, 0x776b783f,
	0xdf60e1a9, 0xdad8dd1b, 0xe892eae4, 0xe3f4e642,
	0x7bc47c59, 0x7a7c7b25, 0x7dd57e3e, 0x7ce27d61,
	0xf1e4f43c, 0xed38ef8d, 0xfb49fda4, 0xf695f8ef,
	0x7f377f74, 0x7e9c7eef, 0x7fe87ff9, 0x7fa67fcd,
	0x04b60710, 0x0000025b, 0x0e1b1072, 0x096a0bc3,
	0x7fe87fcd, 0x7fff7ff9, 0x7f377eef, 0x7fa67f74,
	0x176d19bd, 0x12c7151b, 0x209f22e4, 0x1c0b1e56,
	0x7dd57d61, 0x7e9c7e3e, 0x7bc47b25, 0x7ce27c59,
	0x29a32bdb, 0x25272767, 0x326d3496, 0x2e103041,
	0x7908783f, 0x7a7c79c7, 0x75a474b1, 0x776b768d,
	0x3af23d07, 0x36b938d8, 0x43254523, 0x3f164120,
	0x719d7082, 0x73b572ae, 0x6cf86bb7, 0x6f5e6e30,
	0x4afa4ce0, 0x471c490e, 0x52685432, 0x4ebf5097,
	0x67bc6656, 0x6a6c6919, 0x61f06067, 0x64e76370,
	0x59635b0f, 0x55f457b0, 0x5fe2616e, 0x5cb35e4f,
	0x5b9c59f3, 0x5ed65d3d, 0x54c95301, 0x58425689,
	0x65dd6745, 0x62f1646b, 0x6b4a6c8e, 0x68a569fc,
	0x4d804b9d, 0x51334f5d, 0x45cc43d0, 0x49b347c3,
	0x70227140, 0x6dc96efa, 0x745e7554, 0x7254735e,
	0x3db73ba4, 0x41cd3fc5, 0x354d3326, 0x398c376f,
	0x77f978c6, 0x76407722, 0x7aee7b91, 0x79897a41,
	0x2c982a61, 0x30fb2ecc, 0x23a62161, 0x282625e7,
	0x7d387db0, 0x7c297cb6, 0x7ed47f20, 0x7e1c7e7e,
	0x1a821833, 0x1f191ccf, 0x11390ee3, 0x15e1138e,
	0x7fc17fe0, 0x7f617f96, 0x7ffc7fef, 0x7ff57ffe,
	0x07d9057e, 0x0c8b0a32, 0xfe6dfc12, 0x032400c9,
	0x7f867f4c, 0x7fd77fb4, 0x7e5e7df9, 0x7f087eb9,
	0xf504f2ac, 0xf9b8f75e, 0xebaae958, 0xf054edff,
	0x7c887bf7, 0x7d897d0e, 0x7a047949, 0x7b5c7ab5,
	0xe26ce023, 0xe707e4b9, 0xd958d71a, 0xdddcdb98,
	0x76d875f3, 0x788377b3, 0x730671f9, 0x7503740a,
	0xd079ce4b, 0xd4e1d2ab, 0xc7dbc5c0, 0xcc21c9fc,
	0x6e956d61, 0x70e16fc0, 0x698b6831, 0x6c236adb,
	0xbf8cbd86, 0xc3a9c198, 0xb796b5a8, 0xbb85b98b,
	0x63ee6271, 0x66ce6562, 0x5dc65c28, 0x60eb5f5d,
	0xb005ae31, 0xb3c0b1df, 0xa8e2a72c, 0xac65aaa0,
	0x571d555f, 0x5a8158d3, 0x4ffa4e20, 0x539a51ce,
	0xa239a0a2, 0xa57ea3d7, 0x9c119a9d, 0x9f149d8e,
	0x48694674, 0x4c3f4a57, 0x40733e67, 0x447a4279,
	0x96749524, 0x993197ce, 0x916a903f, 0x93dc929e,
	0x38243603, 0x3c563a3f, 0x2f862d54, 0x33de31b4,
	0x8cf98bf5, 0x8f1e8e06, 0x8927884c, 0x8afc8a0c,
	0x26a72467, 0x2b1e28e5, 0x1d931b46, 0x22231fdc,
	0x85fb854a, 0x877c86b6, 0x837782f1, 0x84a38408,
	0x14551200, 0x18f816a7, 0x0afb08a1, 0x0fab0d53,
	0x81a18146, 0x82768206, 0x8079804b, 0x80f780b3,
	0x0192ff36, 0x064703ed, 0xf826f5cd, 0xfcdbfa81,
	0x80038001, 0x80288010, 0x803e8069, 0x800a801f,
	0xeec6ec71, 0xf374f11c, 0xe57de330, 0xea1ee7cc,
	0x812b8181, 0x809e80df, 0x82c78349, 0x81e3824f,
	0xdc59da18, 0xe0e6de9e, 0xd367d133, 0xd7d9d59e,
	0x851185be, 0x83d6846e, 0x880688dd, 0x86768739,
	0xcab2c890, 0xcf04ccd9, 0xc248c03a, 0xc673c45b,
	0x8ba18ca1, 0x89bf8aab, 0x8fdd9105, 0x8dab8ebf,
	0xba33b83c, 0xbe32bc2f, 0xb27fb0a2, 0xb64cb462,
	0x94b59603, 0x92369371, 0x9a229b94, 0x975a98ba,
	0xab36a976, 0xaeccacfe, 0xa463a2c2, 0xa7bda60c,
	0xa01da1b0, 0x9d0e9e91, 0xa69ca84f, 0xa34ca4f0,
	0x9e0f9c8f, 0xa1299f98, 0x984396e6, 0x9b1899a9,
	0xad97af68, 0xaa0babcd, 0xb505b6f1, 0xb140b31f,
	0x930791cf, 0x95939448, 0x8e628d51, 0x90a18f7d,
	0xbcdabedf, 0xb8e3badc, 0xc50dc727, 0xc0e9c2f8,
	0x8a5b8972, 0x8c4a8b4e, 0x86f78638, 0x889487c0,
	0xcd92cfbe, 0xc946cb69, 0xd65cd898, 0xd1efd424,
	0x843b83a6, 0x858384da, 0x822a81c1, 0x831d829e,
	0xdf60e1a9, 0xdad8dd1b, 0xe892eae4, 0xe3f4e642,
	0x80c8808b, 0x81638110, 0x80178006, 0x80598032,
	0xf1e4f43c, 0xed38ef8d, 0xfb49fda4, 0xf695f8ef,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_3_P[] = {
	0x00000000, 0x00000000, 0x00005a81, 0x7fffa57e,
	0x00000000, 0x00000000, 0x80015a81, 0x00005a81,
	0xa57e89bf, 0x7fff30fb, 0x5a81cf04, 0x00007640,
	0x5a81cf04, 0x00007640, 0x5a817640, 0x8001cf04,
	0x30fbe707, 0x7fff6a6c, 0x89bfb8e3, 0xa57e8276,
	0x76407d89, 0x0000471c, 0xcf049593, 0x5a8118f8,
	0x76407d89, 0x0000471c, 0xcf049593, 0x5a8118f8,
	0xcf0418f8, 0x80019593, 0x7640471c, 0x5a817d89,
	0x6a6c5133, 0x7fff7a7c, 0xe707c3a9, 0x30fb0c8b,
	0x471c62f1, 0x00002527, 0x7d8970e1, 0x76407f61,
	0x8276809e, 0xa57e8f1e, 0xb8e3dad8, 0x89bf9d0e,
	0x18f8f374, 0x5a813c56, 0x95938583, 0xcf04aecc,
	0x471c62f1, 0x00002527, 0x7d8970e1, 0x76407f61,
	0x9593aecc, 0x80018583, 0x18f83c56, 0xcf04f374,
	0x18f8f374, 0x5a813c56, 0x95938583, 0xcf04aecc,
	0x7d897f61, 0x5a8170e1, 0x471c2527, 0x764062f1,
	0x7a7c73b5, 0x7fff7e9c, 0x513341cd, 0x6a6c5ed6,
	0x252736b9, 0x000012c7, 0x62f16dc9, 0x471c55f4,
	0x0c8bf9b8, 0x30fb1f19, 0xc3a9b3c0, 0xe707d4e1,
	0x7f617fd7, 0x76407c29, 0x70e166ce, 0x7d897883,
	0x8f1e877c, 0xa57e9931, 0x809e83d6, 0x82768028,
	0x3c562b1e, 0x5a814c3f, 0xf374e0e6, 0x18f80647,
	0x9d0eaa0b, 0x89bf9236, 0xdad8ed38, 0xb8e3c946,
	0xaecca129, 0xcf04be32, 0x85838163, 0x95938c4a,
	0x252736b9, 0x000012c7, 0x62f16dc9, 0x471c55f4,
	0x85838c4a, 0x80018163, 0xaeccbe32, 0x9593a129,
	0x7f617fd7, 0x76407c29, 0x70e166ce, 0x7d897883,
	0xf3740647, 0xcf04e0e6, 0x3c564c3f, 0x18f82b1e,
	0x3c562b1e, 0x5a814c3f, 0xf374e0e6, 0x18f80647,
	0x70e17883, 0x5a8166ce, 0x7f617c29, 0x7d897fd7,
	0xaecca129, 0xcf04be32, 0x85838163, 0x95938c4a,
	0x62f155f4, 0x76406dc9, 0x252712c7, 0x471c36b9,
	0x7e9c7ce2, 0x7fff7fa6, 0x73b56f5e, 0x7a7c776b,
	0x12c71c0b, 0x0000096a, 0x36b93f16, 0x25272e10,
	0x5ed65842, 0x6a6c64e7, 0x41cd398c, 0x513349b3,
	0x55f45cb3, 0x471c4ebf, 0x6dc97254, 0x62f168a5,
	0x1f1915e1, 0x30fb2826, 0xf9b8f054, 0x0c8b0324,
	0x7c297e1c, 0x76407989, 0x7fd77f08, 0x7f617ff5,
	0xd4e1cc21, 0xe707dddc, 0xb3c0ac65, 0xc3a9bb85,
	0x78837503, 0x7d897b5c, 0x66ce60eb, 0x70e16c23,
	0x993193dc, 0xa57e9f14, 0x877c84a3, 0x8f1e8afc,
	0x4c3f447a, 0x5a81539a, 0x2b1e2223, 0x3c5633de,
	0x8028800a, 0x827680f7, 0x83d68676, 0x809e81e3,
	0x0647fcdb, 0x18f80fab, 0xe0e6d7d9, 0xf374ea1e,
	0x9236975a, 0x89bf8dab, 0xaa0bb140, 0x9d0ea34c,
	0xbe32b64c, 0xcf04c673, 0xa1299b18, 0xaecca7bd,
	0xc946d1ef, 0xb8e3c0e9, 0xed38f695, 0xdad8e3f4,
	0x8c4a8894, 0x959390a1, 0x81638059, 0x8583831d,
	0x12c71c0b, 0x0000096a, 0x36b93f16, 0x25272e10,
	0x8163831d, 0x80018059, 0x8c4a90a1, 0x85838894,
	0x55f45cb3, 0x471c4ebf, 0x6dc97254, 0x62f168a5,
	0xa129a7bd, 0x95939b18, 0xbe32c673, 0xaeccb64c,
	0x7c297e1c, 0x76407989, 0x7fd77f08, 0x7f617ff5,
	0xe0e6ea1e, 0xcf04d7d9, 0x06470fab, 0xf374fcdb,
	0x78837503, 0x7d897b5c, 0x66ce60eb, 0x70e16c23,
	0x2b1e33de, 0x18f82223, 0x4c3f539a, 0x3c56447a,
	0x4c3f447a, 0x5a81539a, 0x2b1e2223, 0x3c5633de,
	0x66ce6c23, 0x5a8160eb, 0x78837b5c, 0x70e17503,
	0x0647fcdb, 0x18f80fab, 0xe0e6d7d9, 0xf374ea1e,
	0x7fd77ff5, 0x7d897f08, 0x7c297989, 0x7f617e1c,
	0xbe32b64c, 0xcf04c673, 0xa1299b18, 0xaecca7bd,
	0x6dc968a5, 0x76407254, 0x55f44ebf, 0x62f15cb3,
	0x8c4a8894, 0x959390a1, 0x81638059, 0x8583831d,
	0x36b92e10, 0x471c3f16, 0x12c7096a, 0x25271c0b,
	0x7fa67f37, 0x7fff7fe8, 0x7ce27bc4, 0x7e9c7dd5,
	0x096a0e1b, 0x000004b6, 0x1c0b209f, 0x12c7176d,
	0x776b75a4, 0x7a7c7908, 0x6f5e6cf8, 0x73b5719d,
	0x2e10326d, 0x252729a3, 0x3f164325, 0x36b93af2,
	0x64e761f0, 0x6a6c67bc, 0x584254c9, 0x5ed65b9c,
	0x4ebf5268, 0x471c4afa, 0x5cb35fe2, 0x55f45963,
	0x49b345cc, 0x51334d80, 0x398c354d, 0x41cd3db7,
	0x68a56b4a, 0x62f165dd, 0x7254745e, 0x6dc97022,
	0x282623a6, 0x30fb2c98, 0x15e11139, 0x1f191a82,
	0x79897aee, 0x764077f9, 0x7e1c7ed4, 0x7c297d38,
	0x0324fe6d, 0x0c8b07d9, 0xf054ebaa, 0xf9b8f504,
	0x7ff57ffc, 0x7f617fc1, 0x7f087e5e, 0x7fd77f86,
	0xdddcd958, 0xe707e26c, 0xcc21c7db, 0xd4e1d079,
	0x7b5c7a04, 0x7d897c88, 0x75037306, 0x788376d8,
	0xbb85b796, 0xc3a9bf8c, 0xac65a8e2, 0xb3c0b005,
	0x6c23698b, 0x70e16e95, 0x60eb5dc6, 0x66ce63ee,
	0x9f149c11, 0xa57ea239, 0x93dc916a, 0x99319674,
	0x539a4ffa, 0x5a81571d, 0x447a4073, 0x4c3f4869,
	0x8afc8927, 0x8f1e8cf9, 0x84a38377, 0x877c85fb,
	0x33de2f86, 0x3c563824, 0x22231d93, 0x2b1e26a7,
	0x80f78079, 0x827681a1, 0x800a803e, 0x80288003,
	0x0fab0afb, 0x18f81455, 0xfcdbf826, 0x06470192,
	0x81e382c7, 0x809e812b, 0x86768806, 0x83d68511,
	0xea1ee57d, 0xf374eec6, 0xd7d9d367, 0xe0e6dc59,
	0x8dab8fdd, 0x89bf8ba1, 0x975a9a22, 0x923694b5,
	0xc673c248, 0xcf04cab2, 0xb64cb27f, 0xbe32ba33,
	0xa34ca69c, 0x9d0ea01d, 0xb140b505, 0xaa0bad97,
	0xa7bda463, 0xaeccab36, 0x9b189843, 0xa1299e0f,
	0xc0e9c50d, 0xb8e3bcda, 0xd1efd65c, 0xc946cd92,
	0x90a18e62, 0x95939307, 0x889486f7, 0x8c4a8a5b,
	0xe3f4e892, 0xdad8df60, 0xf695fb49, 0xed38f1e4,
	0x831d822a, 0x8583843b, 0x80598017, 0x816380c8,
	0x096a0e1b, 0x000004b6, 0x1c0b209f, 0x12c7176d,
	0x805980c8, 0x80018017, 0x831d843b, 0x8163822a,
	0x2e10326d, 0x252729a3, 0x3f164325, 0x36b93af2,
	0x88948a5b, 0x858386f7, 0x90a19307, 0x8c4a8e62,
	0x4ebf5268, 0x471c4afa, 0x5cb35fe2, 0x55f45963,
	0x9b189e0f, 0x95939843, 0xa7bdab36, 0xa129a463,
	0x68a56b4a, 0x62f165dd, 0x7254745e, 0x6dc97022,
	0xb64cba33, 0xaeccb27f, 0xc673cab2, 0xbe32c248,
	0x79897aee, 0x764077f9, 0x7e1c7ed4, 0x7c297d38,
	0xd7d9dc59, 0xcf04d367, 0xea1eeec6, 0xe0e6e57d,
	0x7ff57ffc, 0x7f617fc1, 0x7f087e5e, 0x7fd77f86,
	0xfcdb0192, 0xf374f826, 0x0fab1455, 0x06470afb,
	0x7b5c7a04, 0x7d897c88, 0x75037306, 0x788376d8,
	0x222326a7, 0x18f81d93, 0x33de3824, 0x2b1e2f86,
	0x6c23698b, 0x70e16e95, 0x60eb5dc6, 0x66ce63ee,
	0x447a4869, 0x3c564073, 0x539a571d, 0x4c3f4ffa,
	0x539a4ffa, 0x5a81571d, 0x447a4073, 0x4c3f4869,
	0x60eb63ee, 0x5a815dc6, 0x6c236e95, 0x66ce698b,
	0x33de2f86, 0x3c563824, 0x22231d93, 0x2b1e26a7,
	0x750376d8, 0x70e17306, 0x7b5c7c88, 0x78837a04,
	0x0fab0afb, 0x18f81455, 0xfcdbf826, 0x06470192,
	0x7f087f86, 0x7d897e5e, 0x7ff57fc1, 0x7fd77ffc,
	0xea1ee57d, 0xf374eec6, 0xd7d9d367, 0xe0e6dc59,
	0x7e1c7d38, 0x7f617ed4, 0x798977f9, 0x7c297aee,
	0xc673c248, 0xcf04cab2, 0xb64cb27f, 0xbe32ba33,
	0x72547022, 0x7640745e, 0x68a565dd, 0x6dc96b4a,
	0xa7bda463, 0xaeccab36, 0x9b189843, 0xa1299e0f,
	0x5cb35963, 0x62f15fe2, 0x4ebf4afa, 0x55f45268,
	0x90a18e62, 0x95939307, 0x889486f7, 0x8c4a8a5b,
	0x3f163af2, 0x471c4325, 0x2e1029a3, 0x36b9326d,
	0x831d822a, 0x8583843b, 0x80598017, 0x816380c8,
	0x1c0b176d, 0x2527209f, 0x096a04b6, 0x12c70e1b,
	0x7fe87fcd, 0x7fff7ff9, 0x7f377eef, 0x7fa67f74,
	0x04b60710, 0x0000025b, 0x0e1b1072, 0x096a0bc3,
	0x7dd57d61, 0x7e9c7e3e, 0x7bc47b25, 0x7ce27c59,
	0x176d19bd, 0x12c7151b, 0x209f22e4, 0x1c0b1e56,
	0x7908783f, 0x7a7c79c7, 0x75a474b1, 0x776b768d,
	0x29a32bdb, 0x25272767, 0x326d3496, 0x2e103041,
	0x719d7082, 0x73b572ae, 0x6cf86bb7, 0x6f5e6e30,
	0x3af23d07, 0x36b938d8, 0x43254523, 0x3f164120,
	0x67bc6656, 0x6a6c6919, 0x61f06067, 0x64e76370,
	0x4afa4ce0, 0x471c490e, 0x52685432, 0x4ebf5097,
	0x5b9c59f3, 0x5ed65d3d, 0x54c95301, 0x58425689,
	0x59635b0f, 0x55f457b0, 0x5fe2616e, 0x5cb35e4f,
	0x4d804b9d, 0x51334f5d, 0x45cc43d0, 0x49b347c3,
	0x65dd6745, 0x62f1646b, 0x6b4a6c8e, 0x68a569fc,
	0x3db73ba4, 0x41cd3fc5, 0x354d3326, 0x398c376f,
	0x70227140, 0x6dc96efa, 0x745e7554, 0x7254735e,
	0x2c982a61, 0x30fb2ecc, 0x23a62161, 0x282625e7,
	0x77f978c6, 0x76407722, 0x7aee7b91, 0x79897a41,
	0x1a821833, 0x1f191ccf, 0x11390ee3, 0x15e1138e,
	0x7d387db0, 0x7c297cb6, 0x7ed47f20, 0x7e1c7e7e,
	0x07d9057e, 0x0c8b0a32, 0xfe6dfc12, 0x032400c9,
	0x7fc17fe0, 0x7f617f96, 0x7ffc7fef, 0x7ff57ffe,
	0xf504f2ac, 0xf9b8f75e, 0xebaae958, 0xf054edff,
	0x7f867f4c, 0x7fd77fb4, 0x7e5e7df9, 0x7f087eb9,
	0xe26ce023, 0xe707e4b9, 0xd958d71a, 0xdddcdb98,
	0x7c887bf7, 0x7d897d0e, 0x7a047949, 0x7b5c7ab5,
	0xd079ce4b, 0xd4e1d2ab, 0xc7dbc5c0, 0xcc21c9fc,
	0x76d875f3, 0x788377b3, 0x730671f9, 0x7503740a,
	0xbf8cbd86, 0xc3a9c198, 0xb796b5a8, 0xbb85b98b,
	0x6e956d61, 0x70e16fc0, 0x698b6831, 0x6c236adb,
	0xb005ae31, 0xb3c0b1df, 0xa8e2a72c, 0xac65aaa0,
	0x63ee6271, 0x66ce6562, 0x5dc65c28, 0x60eb5f5d,
	0xa239a0a2, 0xa57ea3d7, 0x9c119a9d, 0x9f149d8e,
	0x571d555f, 0x5a8158d3, 0x4ffa4e20, 0x539a51ce,
	0x96749524, 0x993197ce, 0x916a903f, 0x93dc929e,
	0x48694674, 0x4c3f4a57, 0x40733e67, 0x447a4279,
	0x8cf98bf5, 0x8f1e8e06, 0x8927884c, 0x8afc8a0c,
	0x38243603, 0x3c563a3f, 0x2f862d54, 0x33de31b4,
	0x85fb854a, 0x877c86b6, 0x837782f1, 0x84a38408,
	0x26a72467, 0x2b1e28e5, 0x1d931b46, 0x22231fdc,
	0x81a18146, 0x82768206, 0x8079804b, 0x80f780b3,
	0x14551200, 0x18f816a7, 0x0afb08a1, 0x0fab0d53,
	0x80038001, 0x80288010, 0x803e8069, 0x800a801f,
	0x0192ff36, 0x064703ed, 0xf826f5cd, 0xfcdbfa81,
	0x812b8181, 0x809e80df, 0x82c78349, 0x81e3824f,
	0xeec6ec71, 0xf374f11c, 0xe57de330, 0xea1ee7cc,
	0x851185be, 0x83d6846e, 0x880688dd, 0x86768739,
	0xdc59da18, 0xe0e6de9e, 0xd367d133, 0xd7d9d59e,
	0x8ba18ca1, 0x89bf8aab, 0x8fdd9105, 0x8dab8ebf,
	0xcab2c890, 0xcf04ccd9, 0xc248c03a, 0xc673c45b,
	0x94b59603, 0x92369371, 0x9a229b94, 0x975a98ba,
	0xba33b83c, 0xbe32bc2f, 0xb27fb0a2, 0xb64cb462,
	0xa01da1b0, 0x9d0e9e91, 0xa69ca84f, 0xa34ca4f0,
	0xab36a976, 0xaeccacfe, 0xa463a2c2, 0xa7bda60c,
	0xad97af68, 0xaa0babcd, 0xb505b6f1, 0xb140b31f,
	0x9e0f9c8f, 0xa1299f98, 0x984396e6, 0x9b1899a9,
	0xbcdabedf, 0xb8e3badc, 0xc50dc727, 0xc0e9c2f8,
	0x930791cf, 0x95939448, 0x8e628d51, 0x90a18f7d,
	0xcd92cfbe, 0xc946cb69, 0xd65cd898, 0xd1efd424,
	0x8a5b8972, 0x8c4a8b4e, 0x86f78638, 0x889487c0,
	0xdf60e1a9, 0xdad8dd1b, 0xe892eae4, 0xe3f4e642,
	0x843b83a6, 0x858384da, 0x822a81c1, 0x831d829e,
	0xf1e4f43c, 0xed38ef8d, 0xfb49fda4, 0xf695f8ef,
	0x80c8808b, 0x81638110, 0x80178006, 0x80598032,
	0x04b60710, 0x0000025b, 0x0e1b1072, 0x096a0bc3,
	0x80178032, 0x80018006, 0x80c88110, 0x8059808b,
	0x176d19bd, 0x12c7151b, 0x209f22e4, 0x1c0b1e56,
	0x822a829e, 0x816381c1, 0x843b84da, 0x831d83a6,
	0x29a32bdb, 0x25272767, 0x326d3496, 0x2e103041,
	0x86f787c0, 0x85838638, 0x8a5b8b4e, 0x88948972,
	0x3af23d07, 0x36b938d8, 0x43254523, 0x3f164120,
	0x8e628f7d, 0x8c4a8d51, 0x93079448, 0x90a191cf,
	0x4afa4ce0, 0x471c490e, 0x52685432, 0x4ebf5097,
	0x984399a9, 0x959396e6, 0x9e0f9f98, 0x9b189c8f,
	0x59635b0f, 0x55f457b0, 0x5fe2616e, 0x5cb35e4f,
	0xa463a60c, 0xa129a2c2, 0xab36acfe, 0xa7bda976,
	0x65dd6745, 0x62f1646b, 0x6b4a6c8e, 0x68a569fc,
	0xb27fb462, 0xaeccb0a2, 0xba33bc2f, 0xb64cb83c,
	0x70227140, 0x6dc96efa, 0x745e7554, 0x7254735e,
	0xc248c45b, 0xbe32c03a, 0xcab2ccd9, 0xc673c890,
	0x77f978c6, 0x76407722, 0x7aee7b91, 0x79897a41,
	0xd367d59e, 0xcf04d133, 0xdc59de9e, 0xd7d9da18,
	0x7d387db0, 0x7c297cb6, 0x7ed47f20, 0x7e1c7e7e,
	0xe57de7cc, 0xe0e6e330, 0xeec6f11c, 0xea1eec71,
	0x7fc17fe0, 0x7f617f96, 0x7ffc7fef, 0x7ff57ffe,
	0xf826fa81, 0xf374f5cd, 0x019203ed, 0xfcdbff36,
	0x7f867f4c, 0x7fd77fb4, 0x7e5e7df9, 0x7f087eb9,
	0x0afb0d53, 0x064708a1, 0x145516a7, 0x0fab1200,
	0x7c887bf7, 0x7d897d0e, 0x7a047949, 0x7b5c7ab5,
	0x1d931fdc, 0x18f81b46, 0x26a728e5, 0x22232467,
	0x76d875f3, 0x788377b3, 0x730671f9, 0x7503740a,
	0x2f8631b4, 0x2b1e2d54, 0x38243a3f, 0x33de3603,
	0x6e956d61, 0x70e16fc0, 0x698b6831, 0x6c236adb,
	0x40734279, 0x3c563e67, 0x48694a57, 0x447a4674,
	0x63ee6271, 0x66ce6562, 0x5dc65c28, 0x60eb5f5d,
	0x4ffa51ce, 0x4c3f4e20, 0x571d58d3, 0x539a555f,
	0x571d555f, 0x5a8158d3, 0x4ffa4e20, 0x539a51ce,
	0x5dc65f5d, 0x5a815c28, 0x63ee6562, 0x60eb6271,
	0x48694674, 0x4c3f4a57, 0x40733e67, 0x447a4279,
	0x698b6adb, 0x66ce6831, 0x6e956fc0, 0x6c236d61,
	0x38243603, 0x3c563a3f, 0x2f862d54, 0x33de31b4,
	0x7306740a, 0x70e171f9, 0x76d877b3, 0x750375f3,
	0x26a72467, 0x2b1e28e5, 0x1d931b46, 0x22231fdc,
	0x7a047ab5, 0x78837949, 0x7c887d0e, 0x7b5c7bf7,
	0x14551200, 0x18f816a7, 0x0afb08a1, 0x0fab0d53,
	0x7e5e7eb9, 0x7d897df9, 0x7f867fb4, 0x7f087f4c,
	0x0192ff36, 0x064703ed, 0xf826f5cd, 0xfcdbfa81,
	0x7ffc7ffe, 0x7fd77fef, 0x7fc17f96, 0x7ff57fe0,
	0xeec6ec71, 0xf374f11c, 0xe57de330, 0xea1ee7cc,
	0x7ed47e7e, 0x7f617f20, 0x7d387cb6, 0x7e1c7db0,
	0xdc59da18, 0xe0e6de9e, 0xd367d133, 0xd7d9d59e,
	0x7aee7a41, 0x7c297b91, 0x77f97722, 0x798978c6,
	0xcab2c890, 0xcf04ccd9, 0xc248c03a, 0xc673c45b,
	0x745e735e, 0x76407554, 0x70226efa, 0x72547140,
	0xba33b83c, 0xbe32bc2f, 0xb27fb0a2, 0xb64cb462,
	0x6b4a69fc, 0x6dc96c8e, 0x65dd646b, 0x68a56745,
	0xab36a976, 0xaeccacfe, 0xa463a2c2, 0xa7bda60c,
	0x5fe25e4f, 0x62f1616e, 0x596357b0, 0x5cb35b0f,
	0x9e0f9c8f, 0xa1299f98, 0x984396e6, 0x9b1899a9,
	0x52685097, 0x55f45432, 0x4afa490e, 0x4ebf4ce0,
	0x930791cf, 0x95939448, 0x8e628d51, 0x90a18f7d,
	0x43254120, 0x471c4523, 0x3af238d8, 0x3f163d07,
	0x8a5b8972, 0x8c4a8b4e, 0x86f78638, 0x889487c0,
	0x326d3041, 0x36b93496, 0x29a32767, 0x2e102bdb,
	0x843b83a6, 0x858384da, 0x822a81c1, 0x831d829e,
	0x209f1e56, 0x252722e4, 0x176d151b, 0x1c0b19bd,
	0x80c8808b, 0x81638110, 0x80178006, 0x80598032,
	0x0e1b0bc3, 0x12c71072, 0x04b6025b, 0x096a0710,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_1D_M[] = {
	0x00000000, 0x00000000, 0x0000d2bf, 0x3fff2d40,
	0x00000000, 0x00000000, 0xc000d2bf, 0x0000d2bf,
	0x2d40187d, 0x3fff3b20, 0xd2bfc4df, 0x0000e782,
	0xd2bfc4df, 0x0000e782, 0xd2bfe782, 0xc000c4df,
	0x3b203536, 0x3fff3ec4, 0x187d0c7c, 0x2d40238e,
	0xe782dc71, 0x0000f383, 0xc4dfc13b, 0xd2bfcac9,
	0xe782dc71, 0x0000f383, 0xc4dfc13b, 0xd2bfcac9,
	0xc4dfcac9, 0xc000c13b, 0xe782f383, 0xd2bfdc71,
	0x3ec43d3e, 0x3fff3fb0, 0x35363178, 0x3b203870,
	0xf383ed6c, 0x0000f9ba, 0xdc71d766, 0xe782e1d4,
	0x238e1e2b, 0x2d402899, 0x0c7c0645, 0x187d1293,
	0xcac9c78f, 0xd2bfce87, 0xc13bc04f, 0xc4dfc2c1,
	0xf383ed6c, 0x0000f9ba, 0xdc71d766, 0xe782e1d4,
	0xc13bc2c1, 0xc000c04f, 0xcac9ce87, 0xc4dfc78f,
	0xcac9c78f, 0xd2bfce87, 0xc13bc04f, 0xc4dfc2c1,
	0xdc71e1d4, 0xd2bfd766, 0xf383f9ba, 0xe782ed6c,
	0x3fb03f4e, 0x3fff3feb, 0x3d3e3c41, 0x3ec43e14,
	0xf9baf69c, 0x0000fcdc, 0xed6cea70, 0xf383f073,
	0x387036e4, 0x3b2039da, 0x31782f6b, 0x35363367,
	0xe1d4df19, 0xe782e4a3, 0xd766d505, 0xdc71d9e0,
	0x2899261f, 0x2d402afa, 0x1e2b1b5c, 0x238e20e6,
	0xce87cc98, 0xd2bfd094, 0xc78fc625, 0xcac9c91b,
	0x12930f8c, 0x187d158f, 0x06450323, 0x0c7c0963,
	0xc2c1c1eb, 0xc4dfc3be, 0xc04fc014, 0xc13bc0b1,
	0xf9baf69c, 0x0000fcdc, 0xed6cea70, 0xf383f073,
	0xc04fc0b1, 0xc000c014, 0xc2c1c3be, 0xc13bc1eb,
	0xe1d4df19, 0xe782e4a3, 0xd766d505, 0xdc71d9e0,
	0xc78fc91b, 0xc4dfc625, 0xce87d094, 0xcac9cc98,
	0xce87cc98, 0xd2bfd094, 0xc78fc625, 0xcac9c91b,
	0xd766d9e0, 0xd2bfd505, 0xe1d4e4a3, 0xdc71df19,
	0xc2c1c1eb, 0xc4dfc3be, 0xc04fc014, 0xc13bc0b1,
	0xed6cf073, 0xe782ea70, 0xf9bafcdc, 0xf383f69c,
	0x3feb3fd3, 0x3fff3ffa, 0x3f4e3f0e, 0x3fb03f84,
	0xfcdcfb4a, 0x0000fe6d, 0xf69cf50f, 0xf9baf82a,
	0x3e143dae, 0x3ec43e71, 0x3c413bb5, 0x3d3e3cc4,
	0xf073eeee, 0xf383f1fa, 0xea70e8f7, 0xed6cebec,
	0x39da392a, 0x3b203a81, 0x36e43611, 0x387037af,
	0xe4a3e339, 0xe782e610, 0xdf19ddc2, 0xe1d4e074,
	0x33673273, 0x35363452, 0x2f6b2e59, 0x31783075,
	0xd9e0d8a0, 0xdc71db26, 0xd505d3de, 0xd766d632,
	0x2afa29cd, 0x2d402c21, 0x261f24d9, 0x2899275f,
	0xd094cf8a, 0xd2bfd1a6, 0xcc98cbad, 0xce87cd8c,
	0x20e61f8b, 0x238e223d, 0x1b5c19ef, 0x1e2b1cc6,
	0xc91bc850, 0xcac9c9ee, 0xc625c57e, 0xc78fc6d5,
	0x158f1413, 0x187d1708, 0x0f8c0e05, 0x12931111,
	0xc3bec33b, 0xc4dfc44a, 0xc1ebc18e, 0xc2c1c251,
	0x096307d5, 0x0c7c0af0, 0x03230192, 0x064504b5,
	0xc0b1c07b, 0xc13bc0f1, 0xc014c005, 0xc04fc02c,
	0xfcdcfb4a, 0x0000fe6d, 0xf69cf50f, 0xf9baf82a,
	0xc014c02c, 0xc000c005, 0xc0b1c0f1, 0xc04fc07b,
	0xf073eeee, 0xf383f1fa, 0xea70e8f7, 0xed6cebec,
	0xc1ebc251, 0xc13bc18e, 0xc3bec44a, 0xc2c1c33b,
	0xe4a3e339, 0xe782e610, 0xdf19ddc2, 0xe1d4e074,
	0xc625c6d5, 0xc4dfc57e, 0xc91bc9ee, 0xc78fc850,
	0xd9e0d8a0, 0xdc71db26, 0xd505d3de, 0xd766d632,
	0xcc98cd8c, 0xcac9cbad, 0xd094d1a6, 0xce87cf8a,
	0xd094cf8a, 0xd2bfd1a6, 0xcc98cbad, 0xce87cd8c,
	0xd505d632, 0xd2bfd3de, 0xd9e0db26, 0xd766d8a0,
	0xc91bc850, 0xcac9c9ee, 0xc625c57e, 0xc78fc6d5,
	0xdf19e074, 0xdc71ddc2, 0xe4a3e610, 0xe1d4e339,
	0xc3bec33b, 0xc4dfc44a, 0xc1ebc18e, 0xc2c1c251,
	0xea70ebec, 0xe782e8f7, 0xf073f1fa, 0xed6ceeee,
	0xc0b1c07b, 0xc13bc0f1, 0xc014c005, 0xc04fc02c,
	0xf69cf82a, 0xf383f50f, 0xfcdcfe6d, 0xf9bafb4a,
	0x3ffa3ff4, 0x3fff3ffe, 0x3fd33fc3, 0x3feb3fe0,
	0xfe6dfda4, 0x0000ff36, 0xfb4afa82, 0xfcdcfc13,
	0x3f843f6a, 0x3fb03f9b, 0x3f0e3eea, 0x3f4e3f2f,
	0xf82af763, 0xf9baf8f2, 0xf50ff449, 0xf69cf5d5,
	0x3e713e44, 0x3ec43e9c, 0x3dae3d77, 0x3e143de2,
	0xf1faf136, 0xf383f2be, 0xeeeeee2c, 0xf073efb0,
	0x3cc43c84, 0x3d3e3d02, 0x3bb53b6c, 0x3c413bfc,
	0xebeceb2e, 0xed6cecac, 0xe8f7e83c, 0xea70e9b3,
	0x3a813a2f, 0x3b203ad2, 0x392a38ce, 0x39da3983,
	0xe610e559, 0xe782e6c9, 0xe339e286, 0xe4a3e3ed,
	0x37af374a, 0x38703811, 0x361135a5, 0x36e4367c,
	0xe074dfc6, 0xe1d4e124, 0xddc2dd19, 0xdf19de6d,
	0x345233de, 0x353634c5, 0x327331f7, 0x336732ee,
	0xdb26da82, 0xdc71dbcb, 0xd8a0d802, 0xd9e0d93f,
	0x30752ff1, 0x317830f8, 0x2e592dce, 0x2f6b2ee3,
	0xd632d59b, 0xd766d6cb, 0xd3ded34e, 0xd505d471,
	0x2c212b8e, 0x2d402cb1, 0x29cd2934, 0x2afa2a64,
	0xd1a6d11c, 0xd2bfd231, 0xcf8acf07, 0xd094d00e,
	0x275f26c0, 0x289927fd, 0x24d92434, 0x261f257d,
	0xcd8ccd11, 0xce87ce08, 0xcbadcb3a, 0xcc98cc21,
	0x223d2192, 0x238e22e6, 0x1f8b1edb, 0x20e62039,
	0xc9eec983, 0xcac9ca5a, 0xc850c7ee, 0xc91bc8b5,
	0x1cc61c12, 0x1e2b1d79, 0x19ef1936, 0x1b5c1aa6,
	0xc6d5c67c, 0xc78fc731, 0xc57ec52d, 0xc625c5d0,
	0x1708164c, 0x187d17c3, 0x14131353, 0x158f14d1,
	0xc44ac403, 0xc4dfc493, 0xc33bc2fd, 0xc3bec37b,
	0x1111104f, 0x129311d3, 0x0e050d41, 0x0f8c0ec9,
	0xc251c21d, 0xc2c1c288, 0xc18ec163, 0xc1ebc1bb,
	0x0af00a2a, 0x0c7c0bb6, 0x07d5070d, 0x0963089c,
	0xc0f1c0d0, 0xc13bc115, 0xc07bc064, 0xc0b1c095,
	0x04b503ec, 0x0645057d, 0x019200c9, 0x0323025b,
	0xc02cc01f, 0xc04fc03c, 0xc005c001, 0xc014c00b,
	0xfe6dfda4, 0x0000ff36, 0xfb4afa82, 0xfcdcfc13,
	0xc005c00b, 0xc000c001, 0xc02cc03c, 0xc014c01f,
	0xf82af763, 0xf9baf8f2, 0xf50ff449, 0xf69cf5d5,
	0xc07bc095, 0xc04fc064, 0xc0f1c115, 0xc0b1c0d0,
	0xf1faf136, 0xf383f2be, 0xeeeeee2c, 0xf073efb0,
	0xc18ec1bb, 0xc13bc163, 0xc251c288, 0xc1ebc21d,
	0xebeceb2e, 0xed6cecac, 0xe8f7e83c, 0xea70e9b3,
	0xc33bc37b, 0xc2c1c2fd, 0xc44ac493, 0xc3bec403,
	0xe610e559, 0xe782e6c9, 0xe339e286, 0xe4a3e3ed,
	0xc57ec5d0, 0xc4dfc52d, 0xc6d5c731, 0xc625c67c,
	0xe074dfc6, 0xe1d4e124, 0xddc2dd19, 0xdf19de6d,
	0xc850c8b5, 0xc78fc7ee, 0xc9eeca5a, 0xc91bc983,
	0xdb26da82, 0xdc71dbcb, 0xd8a0d802, 0xd9e0d93f,
	0xcbadcc21, 0xcac9cb3a, 0xcd8cce08, 0xcc98cd11,
	0xd632d59b, 0xd766d6cb, 0xd3ded34e, 0xd505d471,
	0xcf8ad00e, 0xce87cf07, 0xd1a6d231, 0xd094d11c,
	0xd1a6d11c, 0xd2bfd231, 0xcf8acf07, 0xd094d00e,
	0xd3ded471, 0xd2bfd34e, 0xd632d6cb, 0xd505d59b,
	0xcd8ccd11, 0xce87ce08, 0xcbadcb3a, 0xcc98cc21,
	0xd8a0d93f, 0xd766d802, 0xdb26dbcb, 0xd9e0da82,
	0xc9eec983, 0xcac9ca5a, 0xc850c7ee, 0xc91bc8b5,
	0xddc2de6d, 0xdc71dd19, 0xe074e124, 0xdf19dfc6,
	0xc6d5c67c, 0xc78fc731, 0xc57ec52d, 0xc625c5d0,
	0xe339e3ed, 0xe1d4e286, 0xe610e6c9, 0xe4a3e559,
	0xc44ac403, 0xc4dfc493, 0xc33bc2fd, 0xc3bec37b,
	0xe8f7e9b3, 0xe782e83c, 0xebececac, 0xea70eb2e,
	0xc251c21d, 0xc2c1c288, 0xc18ec163, 0xc1ebc1bb,
	0xeeeeefb0, 0xed6cee2c, 0xf1faf2be, 0xf073f136,
	0xc0f1c0d0, 0xc13bc115, 0xc07bc064, 0xc0b1c095,
	0xf50ff5d5, 0xf383f449, 0xf82af8f2, 0xf69cf763,
	0xc02cc01f, 0xc04fc03c, 0xc005c001, 0xc014c00b,
	0xfb4afc13, 0xf9bafa82, 0xfe6dff36, 0xfcdcfda4,
	0x3ffe3ffc, 0x3fff3fff, 0x3ff43ff0, 0x3ffa3ff7,
	0xff36fed2, 0x0000ff9b, 0xfda4fd40, 0xfe6dfe09,
	0x3fe03fda, 0x3feb3fe6, 0x3fc33fba, 0x3fd33fcb,
	0xfc13fbaf, 0xfcdcfc77, 0xfa82fa1e, 0xfb4afae6,
	0x3f9b3f90, 0x3fb03fa6, 0x3f6a3f5c, 0x3f843f77,
	0xf8f2f88e, 0xf9baf956, 0xf763f6ff, 0xf82af7c6,
	0x3f2f3f1f, 0x3f4e3f3f, 0x3eea3ed8, 0x3f0e3efc,
	0xf5d5f572, 0xf69cf638, 0xf449f3e6, 0xf50ff4ac,
	0x3e9c3e87, 0x3ec43eb0, 0x3e443e2c, 0x3e713e5b,
	0xf2bef25c, 0xf383f321, 0xf136f0d4, 0xf1faf198,
	0x3de23dc8, 0x3e143dfb, 0x3d773d5a, 0x3dae3d92,
	0xefb0ef4f, 0xf073f011, 0xee2cedcc, 0xeeeeee8d,
	0x3d023ce3, 0x3d3e3d20, 0x3c843c63, 0x3cc43ca4,
	0xecacec4c, 0xed6ced0c, 0xeb2eeacf, 0xebeceb8d,
	0x3bfc3bd9, 0x3c413c1f, 0x3b6c3b46, 0x3bb53b91,
	0xe9b3e955, 0xea70ea12, 0xe83ce7df, 0xe8f7e899,
	0x3ad23aaa, 0x3b203af9, 0x3a2f3a05, 0x3a813a58,
	0xe6c9e66c, 0xe782e725, 0xe559e4fe, 0xe610e5b4,
	0x39833957, 0x39da39af, 0x38ce38a0, 0x392a38fc,
	0xe3ede393, 0xe4a3e448, 0xe286e22d, 0xe339e2e0,
	0x381137e0, 0x38703841, 0x374a3718, 0x37af377d,
	0xe124e0cc, 0xe1d4e17c, 0xdfc6df6f, 0xe074e01d,
	0x367c3647, 0x36e436b0, 0x35a5356d, 0x361135db,
	0xde6dde17, 0xdf19dec3, 0xdd19dcc5, 0xddc2dd6e,
	0x34c5348c, 0x353634fe, 0x33de33a2, 0x34523418,
	0xdbcbdb78, 0xdc71dc1e, 0xda82da31, 0xdb26dad4,
	0x32ee32b1, 0x3367332b, 0x31f731b8, 0x32733235,
	0xd93fd8ef, 0xd9e0d98f, 0xd802d7b4, 0xd8a0d851,
	0x30f830b7, 0x31783138, 0x2ff12fae, 0x30753033,
	0xd6cbd67f, 0xd766d718, 0xd59bd550, 0xd632d5e6,
	0x2ee32e9e, 0x2f6b2f27, 0x2dce2d87, 0x2e592e14,
	0xd471d427, 0xd505d4bb, 0xd34ed306, 0xd3ded396,
	0x2cb12c69, 0x2d402cf9, 0x2b8e2b44, 0x2c212bd8,
	0xd231d1eb, 0xd2bfd278, 0xd11cd0d8, 0xd1a6d161,
	0x2a642a19, 0x2afa2aaf, 0x293428e7, 0x29cd2980,
	0xd00ecfcc, 0xd094d051, 0xcf07cec7, 0xcf8acf48,
	0x27fd27ae, 0x2899284b, 0x26c02670, 0x275f2710,
	0xce08cdca, 0xce87ce47, 0xcd11ccd4, 0xcd8ccd4e,
	0x257d252b, 0x261f25ce, 0x243423e1, 0x24d92487,
	0xcc21cbe7, 0xcc98cc5d, 0xcb3acb01, 0xcbadcb73,
	0x22e62291, 0x238e233a, 0x2192213c, 0x223d21e8,
	0xca5aca24, 0xcac9ca92, 0xc983c94f, 0xc9eec9b8,
	0x20391fe2, 0x20e62090, 0x1edb1e83, 0x1f8b1f33,
	0xc8b5c882, 0xc91bc8e7, 0xc7eec7be, 0xc850c81f,
	0x1d791d1f, 0x1e2b1dd2, 0x1c121bb7, 0x1cc61c6c,
	0xc731c703, 0xc78fc75f, 0xc67cc650, 0xc6d5c6a8,
	0x1aa61a4b, 0x1b5c1b01, 0x193618da, 0x19ef1993,
	0xc5d0c5a7, 0xc625c5fa, 0xc52dc506, 0xc57ec555,
	0x17c31766, 0x187d1820, 0x164c15ed, 0x170816aa,
	0xc493c46e, 0xc4dfc4b9, 0xc403c3e0, 0xc44ac426,
	0x14d11472, 0x158f1530, 0x135312f3, 0x141313b3,
	0xc37bc35b, 0xc3bec39c, 0xc2fdc2df, 0xc33bc31c,
	0x11d31172, 0x12931233, 0x104f0fee, 0x111110b0,
	0xc288c26d, 0xc2c1c2a5, 0xc21dc204, 0xc251c237,
	0x0ec90e67, 0x0f8c0f2b, 0x0d410cde, 0x0e050da3,
	0xc1bbc1a4, 0xc1ebc1d3, 0xc163c14f, 0xc18ec178,
	0x0bb60b53, 0x0c7c0c19, 0x0a2a09c7, 0x0af00a8d,
	0xc115c103, 0xc13bc127, 0xc0d0c0c0, 0xc0f1c0e0,
	0x089c0839, 0x09630900, 0x070d06a9, 0x07d50771,
	0xc095c088, 0xc0b1c0a3, 0xc064c059, 0xc07bc06f,
	0x057d0519, 0x064505e1, 0x03ec0388, 0x04b50450,
	0xc03cc034, 0xc04fc045, 0xc01fc019, 0xc02cc025,
	0x025b01f6, 0x032302bf, 0x00c90064, 0x0192012d,
	0xc00bc008, 0xc014c00f, 0xc001c000, 0xc005c003,
	0xff36fed2, 0x0000ff9b, 0xfda4fd40, 0xfe6dfe09,
	0xc001c003, 0xc000c000, 0xc00bc00f, 0xc005c008,
	0xfc13fbaf, 0xfcdcfc77, 0xfa82fa1e, 0xfb4afae6,
	0xc01fc025, 0xc014c019, 0xc03cc045, 0xc02cc034,
	0xf8f2f88e, 0xf9baf956, 0xf763f6ff, 0xf82af7c6,
	0xc064c06f, 0xc04fc059, 0xc095c0a3, 0xc07bc088,
	0xf5d5f572, 0xf69cf638, 0xf449f3e6, 0xf50ff4ac,
	0xc0d0c0e0, 0xc0b1c0c0, 0xc115c127, 0xc0f1c103,
	0xf2bef25c, 0xf383f321, 0xf136f0d4, 0xf1faf198,
	0xc163c178, 0xc13bc14f, 0xc1bbc1d3, 0xc18ec1a4,
	0xefb0ef4f, 0xf073f011, 0xee2cedcc, 0xeeeeee8d,
	0xc21dc237, 0xc1ebc204, 0xc288c2a5, 0xc251c26d,
	0xecacec4c, 0xed6ced0c, 0xeb2eeacf, 0xebeceb8d,
	0xc2fdc31c, 0xc2c1c2df, 0xc37bc39c, 0xc33bc35b,
	0xe9b3e955, 0xea70ea12, 0xe83ce7df, 0xe8f7e899,
	0xc403c426, 0xc3bec3e0, 0xc493c4b9, 0xc44ac46e,
	0xe6c9e66c, 0xe782e725, 0xe559e4fe, 0xe610e5b4,
	0xc52dc555, 0xc4dfc506, 0xc5d0c5fa, 0xc57ec5a7,
	0xe3ede393, 0xe4a3e448, 0xe286e22d, 0xe339e2e0,
	0xc67cc6a8, 0xc625c650, 0xc731c75f, 0xc6d5c703,
	0xe124e0cc, 0xe1d4e17c, 0xdfc6df6f, 0xe074e01d,
	0xc7eec81f, 0xc78fc7be, 0xc8b5c8e7, 0xc850c882,
	0xde6dde17, 0xdf19dec3, 0xdd19dcc5, 0xddc2dd6e,
	0xc983c9b8, 0xc91bc94f, 0xca5aca92, 0xc9eeca24,
	0xdbcbdb78, 0xdc71dc1e, 0xda82da31, 0xdb26dad4,
	0xcb3acb73, 0xcac9cb01, 0xcc21cc5d, 0xcbadcbe7,
	0xd93fd8ef, 0xd9e0d98f, 0xd802d7b4, 0xd8a0d851,
	0xcd11cd4e, 0xcc98ccd4, 0xce08ce47, 0xcd8ccdca,
	0xd6cbd67f, 0xd766d718, 0xd59bd550, 0xd632d5e6,
	0xcf07cf48, 0xce87cec7, 0xd00ed051, 0xcf8acfcc,
	0xd471d427, 0xd505d4bb, 0xd34ed306, 0xd3ded396,
	0xd11cd161, 0xd094d0d8, 0xd231d278, 0xd1a6d1eb,
	0xd231d1eb, 0xd2bfd278, 0xd11cd0d8, 0xd1a6d161,
	0xd34ed396, 0xd2bfd306, 0xd471d4bb, 0xd3ded427,
	0xd00ecfcc, 0xd094d051, 0xcf07cec7, 0xcf8acf48,
	0xd59bd5e6, 0xd505d550, 0xd6cbd718, 0xd632d67f,
	0xce08cdca, 0xce87ce47, 0xcd11ccd4, 0xcd8ccd4e,
	0xd802d851, 0xd766d7b4, 0xd93fd98f, 0xd8a0d8ef,
	0xcc21cbe7, 0xcc98cc5d, 0xcb3acb01, 0xcbadcb73,
	0xda82dad4, 0xd9e0da31, 0xdbcbdc1e, 0xdb26db78,
	0xca5aca24, 0xcac9ca92, 0xc983c94f, 0xc9eec9b8,
	0xdd19dd6e, 0xdc71dcc5, 0xde6ddec3, 0xddc2de17,
	0xc8b5c882, 0xc91bc8e7, 0xc7eec7be, 0xc850c81f,
	0xdfc6e01d, 0xdf19df6f, 0xe124e17c, 0xe074e0cc,
	0xc731c703, 0xc78fc75f, 0xc67cc650, 0xc6d5c6a8,
	0xe286e2e0, 0xe1d4e22d, 0xe3ede448, 0xe339e393,
	0xc5d0c5a7, 0xc625c5fa, 0xc52dc506, 0xc57ec555,
	0xe559e5b4, 0xe4a3e4fe, 0xe6c9e725, 0xe610e66c,
	0xc493c46e, 0xc4dfc4b9, 0xc403c3e0, 0xc44ac426,
	0xe83ce899, 0xe782e7df, 0xe9b3ea12, 0xe8f7e955,
	0xc37bc35b, 0xc3bec39c, 0xc2fdc2df, 0xc33bc31c,
	0xeb2eeb8d, 0xea70eacf, 0xecaced0c, 0xebecec4c,
	0xc288c26d, 0xc2c1c2a5, 0xc21dc204, 0xc251c237,
	0xee2cee8d, 0xed6cedcc, 0xefb0f011, 0xeeeeef4f,
	0xc1bbc1a4, 0xc1ebc1d3, 0xc163c14f, 0xc18ec178,
	0xf136f198, 0xf073f0d4, 0xf2bef321, 0xf1faf25c,
	0xc115c103, 0xc13bc127, 0xc0d0c0c0, 0xc0f1c0e0,
	0xf449f4ac, 0xf383f3e6, 0xf5d5f638, 0xf50ff572,
	0xc095c088, 0xc0b1c0a3, 0xc064c059, 0xc07bc06f,
	0xf763f7c6, 0xf69cf6ff, 0xf8f2f956, 0xf82af88e,
	0xc03cc034, 0xc04fc045, 0xc01fc019, 0xc02cc025,
	0xfa82fae6, 0xf9bafa1e, 0xfc13fc77, 0xfb4afbaf,
	0xc00bc008, 0xc014c00f, 0xc001c000, 0xc005c003,
	0xfda4fe09, 0xfcdcfd40, 0xff36ff9b, 0xfe6dfed2,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_1D_P[] = {
	0x00000000, 0x00000000, 0x0000d2bf, 0x3fff2d40,
	0x00000000, 0x00000000, 0x3fff2d40, 0x00002d40,
	0x2d40187d, 0x3fff3b20, 0xd2bfc4df, 0x0000e782,
	0x2d403b20, 0x0000187d, 0x2d40187d, 0x3fff3b20,
	0x3b203536, 0x3fff3ec4, 0x187d0c7c, 0x2d40238e,
	0x187d238e, 0x00000c7c, 0x3b203ec4, 0x2d403536,
	0xe782dc71, 0x0000f383, 0xc4dfc13b, 0xd2bfcac9,
	0x3b203536, 0x3fff3ec4, 0x187d0c7c, 0x2d40238e,
	0x3ec43d3e, 0x3fff3fb0, 0x35363178, 0x3b203870,
	0x0c7c1293, 0x00000645, 0x238e2899, 0x187d1e2b,
	0x238e1e2b, 0x2d402899, 0x0c7c0645, 0x187d1293,
	0x35363870, 0x2d403178, 0x3ec43fb0, 0x3b203d3e,
	0xf383ed6c, 0x0000f9ba, 0xdc71d766, 0xe782e1d4,
	0x3ec43d3e, 0x3fff3fb0, 0x35363178, 0x3b203870,
	0xcac9c78f, 0xd2bfce87, 0xc13bc04f, 0xc4dfc2c1,
	0x238e1e2b, 0x2d402899, 0x0c7c0645, 0x187d1293,
	0x3fb03f4e, 0x3fff3feb, 0x3d3e3c41, 0x3ec43e14,
	0x06450963, 0x00000323, 0x1293158f, 0x0c7c0f8c,
	0x387036e4, 0x3b2039da, 0x31782f6b, 0x35363367,
	0x1e2b20e6, 0x187d1b5c, 0x28992afa, 0x238e261f,
	0x2899261f, 0x2d402afa, 0x1e2b1b5c, 0x238e20e6,
	0x31783367, 0x2d402f6b, 0x387039da, 0x353636e4,
	0x12930f8c, 0x187d158f, 0x06450323, 0x0c7c0963,
	0x3d3e3e14, 0x3b203c41, 0x3fb03feb, 0x3ec43f4e,
	0xf9baf69c, 0x0000fcdc, 0xed6cea70, 0xf383f073,
	0x3fb03f4e, 0x3fff3feb, 0x3d3e3c41, 0x3ec43e14,
	0xe1d4df19, 0xe782e4a3, 0xd766d505, 0xdc71d9e0,
	0x387036e4, 0x3b2039da, 0x31782f6b, 0x35363367,
	0xce87cc98, 0xd2bfd094, 0xc78fc625, 0xcac9c91b,
	0x2899261f, 0x2d402afa, 0x1e2b1b5c, 0x238e20e6,
	0xc2c1c1eb, 0xc4dfc3be, 0xc04fc014, 0xc13bc0b1,
	0x12930f8c, 0x187d158f, 0x06450323, 0x0c7c0963,
	0x3feb3fd3, 0x3fff3ffa, 0x3f4e3f0e, 0x3fb03f84,
	0x032304b5, 0x00000192, 0x09630af0, 0x064507d5,
	0x3e143dae, 0x3ec43e71, 0x3c413bb5, 0x3d3e3cc4,
	0x0f8c1111, 0x0c7c0e05, 0x158f1708, 0x12931413,
	0x39da392a, 0x3b203a81, 0x36e43611, 0x387037af,
	0x1b5c1cc6, 0x187d19ef, 0x20e6223d, 0x1e2b1f8b,
	0x33673273, 0x35363452, 0x2f6b2e59, 0x31783075,
	0x261f275f, 0x238e24d9, 0x2afa2c21, 0x289929cd,
	0x2afa29cd, 0x2d402c21, 0x261f24d9, 0x2899275f,
	0x2f6b3075, 0x2d402e59, 0x33673452, 0x31783273,
	0x20e61f8b, 0x238e223d, 0x1b5c19ef, 0x1e2b1cc6,
	0x36e437af, 0x35363611, 0x39da3a81, 0x3870392a,
	0x158f1413, 0x187d1708, 0x0f8c0e05, 0x12931111,
	0x3c413cc4, 0x3b203bb5, 0x3e143e71, 0x3d3e3dae,
	0x096307d5, 0x0c7c0af0, 0x03230192, 0x064504b5,
	0x3f4e3f84, 0x3ec43f0e, 0x3feb3ffa, 0x3fb03fd3,
	0xfcdcfb4a, 0x0000fe6d, 0xf69cf50f, 0xf9baf82a,
	0x3feb3fd3, 0x3fff3ffa, 0x3f4e3f0e, 0x3fb03f84,
	0xf073eeee, 0xf383f1fa, 0xea70e8f7, 0xed6cebec,
	0x3e143dae, 0x3ec43e71, 0x3c413bb5, 0x3d3e3cc4,
	0xe4a3e339, 0xe782e610, 0xdf19ddc2, 0xe1d4e074,
	0x39da392a, 0x3b203a81, 0x36e43611, 0x387037af,
	0xd9e0d8a0, 0xdc71db26, 0xd505d3de, 0xd766d632,
	0x33673273, 0x35363452, 0x2f6b2e59, 0x31783075,
	0xd094cf8a, 0xd2bfd1a6, 0xcc98cbad, 0xce87cd8c,
	0x2afa29cd, 0x2d402c21, 0x261f24d9, 0x2899275f,
	0xc91bc850, 0xcac9c9ee, 0xc625c57e, 0xc78fc6d5,
	0x20e61f8b, 0x238e223d, 0x1b5c19ef, 0x1e2b1cc6,
	0xc3bec33b, 0xc4dfc44a, 0xc1ebc18e, 0xc2c1c251,
	0x158f1413, 0x187d1708, 0x0f8c0e05, 0x12931111,
	0xc0b1c07b, 0xc13bc0f1, 0xc014c005, 0xc04fc02c,
	0x096307d5, 0x0c7c0af0, 0x03230192, 0x064504b5,
	0x3ffa3ff4, 0x3fff3ffe, 0x3fd33fc3, 0x3feb3fe0,
	0x0192025b, 0x000000c9, 0x04b5057d, 0x032303ec,
	0x3f843f6a, 0x3fb03f9b, 0x3f0e3eea, 0x3f4e3f2f,
	0x07d5089c, 0x0645070d, 0x0af00bb6, 0x09630a2a,
	0x3e713e44, 0x3ec43e9c, 0x3dae3d77, 0x3e143de2,
	0x0e050ec9, 0x0c7c0d41, 0x111111d3, 0x0f8c104f,
	0x3cc43c84, 0x3d3e3d02, 0x3bb53b6c, 0x3c413bfc,
	0x141314d1, 0x12931353, 0x170817c3, 0x158f164c,
	0x3a813a2f, 0x3b203ad2, 0x392a38ce, 0x39da3983,
	0x19ef1aa6, 0x187d1936, 0x1cc61d79, 0x1b5c1c12,
	0x37af374a, 0x38703811, 0x361135a5, 0x36e4367c,
	0x1f8b2039, 0x1e2b1edb, 0x223d22e6, 0x20e62192,
	0x345233de, 0x353634c5, 0x327331f7, 0x336732ee,
	0x24d9257d, 0x238e2434, 0x275f27fd, 0x261f26c0,
	0x30752ff1, 0x317830f8, 0x2e592dce, 0x2f6b2ee3,
	0x29cd2a64, 0x28992934, 0x2c212cb1, 0x2afa2b8e,
	0x2c212b8e, 0x2d402cb1, 0x29cd2934, 0x2afa2a64,
	0x2e592ee3, 0x2d402dce, 0x307530f8, 0x2f6b2ff1,
	0x275f26c0, 0x289927fd, 0x24d92434, 0x261f257d,
	0x327332ee, 0x317831f7, 0x345234c5, 0x336733de,
	0x223d2192, 0x238e22e6, 0x1f8b1edb, 0x20e62039,
	0x3611367c, 0x353635a5, 0x37af3811, 0x36e4374a,
	0x1cc61c12, 0x1e2b1d79, 0x19ef1936, 0x1b5c1aa6,
	0x392a3983, 0x387038ce, 0x3a813ad2, 0x39da3a2f,
	0x1708164c, 0x187d17c3, 0x14131353, 0x158f14d1,
	0x3bb53bfc, 0x3b203b6c, 0x3cc43d02, 0x3c413c84,
	0x1111104f, 0x129311d3, 0x0e050d41, 0x0f8c0ec9,
	0x3dae3de2, 0x3d3e3d77, 0x3e713e9c, 0x3e143e44,
	0x0af00a2a, 0x0c7c0bb6, 0x07d5070d, 0x0963089c,
	0x3f0e3f2f, 0x3ec43eea, 0x3f843f9b, 0x3f4e3f6a,
	0x04b503ec, 0x0645057d, 0x019200c9, 0x0323025b,
	0x3fd33fe0, 0x3fb03fc3, 0x3ffa3ffe, 0x3feb3ff4,
	0xfe6dfda4, 0x0000ff36, 0xfb4afa82, 0xfcdcfc13,
	0x3ffa3ff4, 0x3fff3ffe, 0x3fd33fc3, 0x3feb3fe0,
	0xf82af763, 0xf9baf8f2, 0xf50ff449, 0xf69cf5d5,
	0x3f843f6a, 0x3fb03f9b, 0x3f0e3eea, 0x3f4e3f2f,
	0xf1faf136, 0xf383f2be, 0xeeeeee2c, 0xf073efb0,
	0x3e713e44, 0x3ec43e9c, 0x3dae3d77, 0x3e143de2,
	0xebeceb2e, 0xed6cecac, 0xe8f7e83c, 0xea70e9b3,
	0x3cc43c84, 0x3d3e3d02, 0x3bb53b6c, 0x3c413bfc,
	0xe610e559, 0xe782e6c9, 0xe339e286, 0xe4a3e3ed,
	0x3a813a2f, 0x3b203ad2, 0x392a38ce, 0x39da3983,
	0xe074dfc6, 0xe1d4e124, 0xddc2dd19, 0xdf19de6d,
	0x37af374a, 0x38703811, 0x361135a5, 0x36e4367c,
	0xdb26da82, 0xdc71dbcb, 0xd8a0d802, 0xd9e0d93f,
	0x345233de, 0x353634c5, 0x327331f7, 0x336732ee,
	0xd632d59b, 0xd766d6cb, 0xd3ded34e, 0xd505d471,
	0x30752ff1, 0x317830f8, 0x2e592dce, 0x2f6b2ee3,
	0xd1a6d11c, 0xd2bfd231, 0xcf8acf07, 0xd094d00e,
	0x2c212b8e, 0x2d402cb1, 0x29cd2934, 0x2afa2a64,
	0xcd8ccd11, 0xce87ce08, 0xcbadcb3a, 0xcc98cc21,
	0x275f26c0, 0x289927fd, 0x24d92434, 0x261f257d,
	0xc9eec983, 0xcac9ca5a, 0xc850c7ee, 0xc91bc8b5,
	0x223d2192, 0x238e22e6, 0x1f8b1edb, 0x20e62039,
	0xc6d5c67c, 0xc78fc731, 0xc57ec52d, 0xc625c5d0,
	0x1cc61c12, 0x1e2b1d79, 0x19ef1936, 0x1b5c1aa6,
	0xc44ac403, 0xc4dfc493, 0xc33bc2fd, 0xc3bec37b,
	0x1708164c, 0x187d17c3, 0x14131353, 0x158f14d1,
	0xc251c21d, 0xc2c1c288, 0xc18ec163, 0xc1ebc1bb,
	0x1111104f, 0x129311d3, 0x0e050d41, 0x0f8c0ec9,
	0xc0f1c0d0, 0xc13bc115, 0xc07bc064, 0xc0b1c095,
	0x0af00a2a, 0x0c7c0bb6, 0x07d5070d, 0x0963089c,
	0xc02cc01f, 0xc04fc03c, 0xc005c001, 0xc014c00b,
	0x04b503ec, 0x0645057d, 0x019200c9, 0x0323025b,
	0x3ffe3ffc, 0x3fff3fff, 0x3ff43ff0, 0x3ffa3ff7,
	0x00c9012d, 0x00000064, 0x025b02bf, 0x019201f6,
	0x3fe03fda, 0x3feb3fe6, 0x3fc33fba, 0x3fd33fcb,
	0x03ec0450, 0x03230388, 0x057d05e1, 0x04b50519,
	0x3f9b3f90, 0x3fb03fa6, 0x3f6a3f5c, 0x3f843f77,
	0x070d0771, 0x064506a9, 0x089c0900, 0x07d50839,
	0x3f2f3f1f, 0x3f4e3f3f, 0x3eea3ed8, 0x3f0e3efc,
	0x0a2a0a8d, 0x096309c7, 0x0bb60c19, 0x0af00b53,
	0x3e9c3e87, 0x3ec43eb0, 0x3e443e2c, 0x3e713e5b,
	0x0d410da3, 0x0c7c0cde, 0x0ec90f2b, 0x0e050e67,
	0x3de23dc8, 0x3e143dfb, 0x3d773d5a, 0x3dae3d92,
	0x104f10b0, 0x0f8c0fee, 0x11d31233, 0x11111172,
	0x3d023ce3, 0x3d3e3d20, 0x3c843c63, 0x3cc43ca4,
	0x135313b3, 0x129312f3, 0x14d11530, 0x14131472,
	0x3bfc3bd9, 0x3c413c1f, 0x3b6c3b46, 0x3bb53b91,
	0x164c16aa, 0x158f15ed, 0x17c31820, 0x17081766,
	0x3ad23aaa, 0x3b203af9, 0x3a2f3a05, 0x3a813a58,
	0x19361993, 0x187d18da, 0x1aa61b01, 0x19ef1a4b,
	0x39833957, 0x39da39af, 0x38ce38a0, 0x392a38fc,
	0x1c121c6c, 0x1b5c1bb7, 0x1d791dd2, 0x1cc61d1f,
	0x381137e0, 0x38703841, 0x374a3718, 0x37af377d,
	0x1edb1f33, 0x1e2b1e83, 0x20392090, 0x1f8b1fe2,
	0x367c3647, 0x36e436b0, 0x35a5356d, 0x361135db,
	0x219221e8, 0x20e6213c, 0x22e6233a, 0x223d2291,
	0x34c5348c, 0x353634fe, 0x33de33a2, 0x34523418,
	0x24342487, 0x238e23e1, 0x257d25ce, 0x24d9252b,
	0x32ee32b1, 0x3367332b, 0x31f731b8, 0x32733235,
	0x26c02710, 0x261f2670, 0x27fd284b, 0x275f27ae,
	0x30f830b7, 0x31783138, 0x2ff12fae, 0x30753033,
	0x29342980, 0x289928e7, 0x2a642aaf, 0x29cd2a19,
	0x2ee32e9e, 0x2f6b2f27, 0x2dce2d87, 0x2e592e14,
	0x2b8e2bd8, 0x2afa2b44, 0x2cb12cf9, 0x2c212c69,
	0x2cb12c69, 0x2d402cf9, 0x2b8e2b44, 0x2c212bd8,
	0x2dce2e14, 0x2d402d87, 0x2ee32f27, 0x2e592e9e,
	0x2a642a19, 0x2afa2aaf, 0x293428e7, 0x29cd2980,
	0x2ff13033, 0x2f6b2fae, 0x30f83138, 0x307530b7,
	0x27fd27ae, 0x2899284b, 0x26c02670, 0x275f2710,
	0x31f73235, 0x317831b8, 0x32ee332b, 0x327332b1,
	0x257d252b, 0x261f25ce, 0x243423e1, 0x24d92487,
	0x33de3418, 0x336733a2, 0x34c534fe, 0x3452348c,
	0x22e62291, 0x238e233a, 0x2192213c, 0x223d21e8,
	0x35a535db, 0x3536356d, 0x367c36b0, 0x36113647,
	0x20391fe2, 0x20e62090, 0x1edb1e83, 0x1f8b1f33,
	0x374a377d, 0x36e43718, 0x38113841, 0x37af37e0,
	0x1d791d1f, 0x1e2b1dd2, 0x1c121bb7, 0x1cc61c6c,
	0x38ce38fc, 0x387038a0, 0x398339af, 0x392a3957,
	0x1aa61a4b, 0x1b5c1b01, 0x193618da, 0x19ef1993,
	0x3a2f3a58, 0x39da3a05, 0x3ad23af9, 0x3a813aaa,
	0x17c31766, 0x187d1820, 0x164c15ed, 0x170816aa,
	0x3b6c3b91, 0x3b203b46, 0x3bfc3c1f, 0x3bb53bd9,
	0x14d11472, 0x158f1530, 0x135312f3, 0x141313b3,
	0x3c843ca4, 0x3c413c63, 0x3d023d20, 0x3cc43ce3,
	0x11d31172, 0x12931233, 0x104f0fee, 0x111110b0,
	0x3d773d92, 0x3d3e3d5a, 0x3de23dfb, 0x3dae3dc8,
	0x0ec90e67, 0x0f8c0f2b, 0x0d410cde, 0x0e050da3,
	0x3e443e5b, 0x3e143e2c, 0x3e9c3eb0, 0x3e713e87,
	0x0bb60b53, 0x0c7c0c19, 0x0a2a09c7, 0x0af00a8d,
	0x3eea3efc, 0x3ec43ed8, 0x3f2f3f3f, 0x3f0e3f1f,
	0x089c0839, 0x09630900, 0x070d06a9, 0x07d50771,
	0x3f6a3f77, 0x3f4e3f5c, 0x3f9b3fa6, 0x3f843f90,
	0x057d0519, 0x064505e1, 0x03ec0388, 0x04b50450,
	0x3fc33fcb, 0x3fb03fba, 0x3fe03fe6, 0x3fd33fda,
	0x025b01f6, 0x032302bf, 0x00c90064, 0x0192012d,
	0x3ff43ff7, 0x3feb3ff0, 0x3ffe3fff, 0x3ffa3ffc,
	0xff36fed2, 0x0000ff9b, 0xfda4fd40, 0xfe6dfe09,
	0x3ffe3ffc, 0x3fff3fff, 0x3ff43ff0, 0x3ffa3ff7,
	0xfc13fbaf, 0xfcdcfc77, 0xfa82fa1e, 0xfb4afae6,
	0x3fe03fda, 0x3feb3fe6, 0x3fc33fba, 0x3fd33fcb,
	0xf8f2f88e, 0xf9baf956, 0xf763f6ff, 0xf82af7c6,
	0x3f9b3f90, 0x3fb03fa6, 0x3f6a3f5c, 0x3f843f77,
	0xf5d5f572, 0xf69cf638, 0xf449f3e6, 0xf50ff4ac,
	0x3f2f3f1f, 0x3f4e3f3f, 0x3eea3ed8, 0x3f0e3efc,
	0xf2bef25c, 0xf383f321, 0xf136f0d4, 0xf1faf198,
	0x3e9c3e87, 0x3ec43eb0, 0x3e443e2c, 0x3e713e5b,
	0xefb0ef4f, 0xf073f011, 0xee2cedcc, 0xeeeeee8d,
	0x3de23dc8, 0x3e143dfb, 0x3d773d5a, 0x3dae3d92,
	0xecacec4c, 0xed6ced0c, 0xeb2eeacf, 0xebeceb8d,
	0x3d023ce3, 0x3d3e3d20, 0x3c843c63, 0x3cc43ca4,
	0xe9b3e955, 0xea70ea12, 0xe83ce7df, 0xe8f7e899,
	0x3bfc3bd9, 0x3c413c1f, 0x3b6c3b46, 0x3bb53b91,
	0xe6c9e66c, 0xe782e725, 0xe559e4fe, 0xe610e5b4,
	0x3ad23aaa, 0x3b203af9, 0x3a2f3a05, 0x3a813a58,
	0xe3ede393, 0xe4a3e448, 0xe286e22d, 0xe339e2e0,
	0x39833957, 0x39da39af, 0x38ce38a0, 0x392a38fc,
	0xe124e0cc, 0xe1d4e17c, 0xdfc6df6f, 0xe074e01d,
	0x381137e0, 0x38703841, 0x374a3718, 0x37af377d,
	0xde6dde17, 0xdf19dec3, 0xdd19dcc5, 0xddc2dd6e,
	0x367c3647, 0x36e436b0, 0x35a5356d, 0x361135db,
	0xdbcbdb78, 0xdc71dc1e, 0xda82da31, 0xdb26dad4,
	0x34c5348c, 0x353634fe, 0x33de33a2, 0x34523418,
	0xd93fd8ef, 0xd9e0d98f, 0xd802d7b4, 0xd8a0d851,
	0x32ee32b1, 0x3367332b, 0x31f731b8, 0x32733235,
	0xd6cbd67f, 0xd766d718, 0xd59bd550, 0xd632d5e6,
	0x30f830b7, 0x31783138, 0x2ff12fae, 0x30753033,
	0xd471d427, 0xd505d4bb, 0xd34ed306, 0xd3ded396,
	0x2ee32e9e, 0x2f6b2f27, 0x2dce2d87, 0x2e592e14,
	0xd231d1eb, 0xd2bfd278, 0xd11cd0d8, 0xd1a6d161,
	0x2cb12c69, 0x2d402cf9, 0x2b8e2b44, 0x2c212bd8,
	0xd00ecfcc, 0xd094d051, 0xcf07cec7, 0xcf8acf48,
	0x2a642a19, 0x2afa2aaf, 0x293428e7, 0x29cd2980,
	0xce08cdca, 0xce87ce47, 0xcd11ccd4, 0xcd8ccd4e,
	0x27fd27ae, 0x2899284b, 0x26c02670, 0x275f2710,
	0xcc21cbe7, 0xcc98cc5d, 0xcb3acb01, 0xcbadcb73,
	0x257d252b, 0x261f25ce, 0x243423e1, 0x24d92487,
	0xca5aca24, 0xcac9ca92, 0xc983c94f, 0xc9eec9b8,
	0x22e62291, 0x238e233a, 0x2192213c, 0x223d21e8,
	0xc8b5c882, 0xc91bc8e7, 0xc7eec7be, 0xc850c81f,
	0x20391fe2, 0x20e62090, 0x1edb1e83, 0x1f8b1f33,
	0xc731c703, 0xc78fc75f, 0xc67cc650, 0xc6d5c6a8,
	0x1d791d1f, 0x1e2b1dd2, 0x1c121bb7, 0x1cc61c6c,
	0xc5d0c5a7, 0xc625c5fa, 0xc52dc506, 0xc57ec555,
	0x1aa61a4b, 0x1b5c1b01, 0x193618da, 0x19ef1993,
	0xc493c46e, 0xc4dfc4b9, 0xc403c3e0, 0xc44ac426,
	0x17c31766, 0x187d1820, 0x164c15ed, 0x170816aa,
	0xc37bc35b, 0xc3bec39c, 0xc2fdc2df, 0xc33bc31c,
	0x14d11472, 0x158f1530, 0x135312f3, 0x141313b3,
	0xc288c26d, 0xc2c1c2a5, 0xc21dc204, 0xc251c237,
	0x11d31172, 0x12931233, 0x104f0fee, 0x111110b0,
	0xc1bbc1a4, 0xc1ebc1d3, 0xc163c14f, 0xc18ec178,
	0x0ec90e67, 0x0f8c0f2b, 0x0d410cde, 0x0e050da3,
	0xc115c103, 0xc13bc127, 0xc0d0c0c0, 0xc0f1c0e0,
	0x0bb60b53, 0x0c7c0c19, 0x0a2a09c7, 0x0af00a8d,
	0xc095c088, 0xc0b1c0a3, 0xc064c059, 0xc07bc06f,
	0x089c0839, 0x09630900, 0x070d06a9, 0x07d50771,
	0xc03cc034, 0xc04fc045, 0xc01fc019, 0xc02cc025,
	0x057d0519, 0x064505e1, 0x03ec0388, 0x04b50450,
	0xc00bc008, 0xc014c00f, 0xc001c000, 0xc005c003,
	0x025b01f6, 0x032302bf, 0x00c90064, 0x0192012d,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_3D_M[] = {
	0x00000000, 0x00000000, 0x00002d40, 0x3fffd2bf,
	0x00000000, 0x00000000, 0x3fffd2bf, 0x0000d2bf,
	0xd2bfc4df, 0x3fff187d, 0x2d40e782, 0x00003b20,
	0xd2bf187d, 0x0000c4df, 0xd2bfc4df, 0x3fff187d,
	0x187df383, 0x3fff3536, 0xc4dfdc71, 0xd2bfc13b,
	0xc4dfc13b, 0x0000dc71, 0x187d3536, 0xd2bff383,
	0x3b203ec4, 0x0000238e, 0xe782cac9, 0x2d400c7c,
	0x187df383, 0x3fff3536, 0xc4dfdc71, 0xd2bfc13b,
	0x35362899, 0x3fff3d3e, 0xf383e1d4, 0x187d0645,
	0xdc71ce87, 0x0000ed6c, 0xc13bc78f, 0xc4dfc04f,
	0xc13bc04f, 0xd2bfc78f, 0xdc71ed6c, 0xc4dfce87,
	0xf3830645, 0xd2bfe1d4, 0x35363d3e, 0x187d2899,
	0x238e3178, 0x00001293, 0x3ec43870, 0x3b203fb0,
	0x35362899, 0x3fff3d3e, 0xf383e1d4, 0x187d0645,
	0x0c7cf9ba, 0x2d401e2b, 0xcac9c2c1, 0xe782d766,
	0xc13bc04f, 0xd2bfc78f, 0xdc71ed6c, 0xc4dfce87,
	0x3d3e39da, 0x3fff3f4e, 0x289920e6, 0x35362f6b,
	0xed6ce4a3, 0x0000f69c, 0xce87c91b, 0xdc71d505,
	0x0645fcdc, 0x187d0f8c, 0xe1d4d9e0, 0xf383ea70,
	0xc04fc014, 0xc4dfc1eb, 0xc78fcc98, 0xc13bc3be,
	0xc78fc3be, 0xd2bfcc98, 0xc04fc1eb, 0xc13bc014,
	0xe1d4ea70, 0xd2bfd9e0, 0x06450f8c, 0xf383fcdc,
	0xce87d505, 0xc4dfc91b, 0xed6cf69c, 0xdc71e4a3,
	0x28992f6b, 0x187d20e6, 0x3d3e3f4e, 0x353639da,
	0x12931b5c, 0x00000963, 0x317836e4, 0x238e2afa,
	0x3d3e39da, 0x3fff3f4e, 0x289920e6, 0x35362f6b,
	0x3fb03feb, 0x3b203e14, 0x38703367, 0x3ec43c41,
	0x0645fcdc, 0x187d0f8c, 0xe1d4d9e0, 0xf383ea70,
	0x1e2b158f, 0x2d40261f, 0xf9baf073, 0x0c7c0323,
	0xc78fc3be, 0xd2bfcc98, 0xc04fc1eb, 0xc13bc014,
	0xd766d094, 0xe782df19, 0xc2c1c0b1, 0xcac9c625,
	0xce87d505, 0xc4dfc91b, 0xed6cf69c, 0xdc71e4a3,
	0x3f4e3e71, 0x3fff3fd3, 0x39da37af, 0x3d3e3bb5,
	0xf69cf1fa, 0x0000fb4a, 0xe4a3e074, 0xed6ce8f7,
	0x2f6b2c21, 0x35363273, 0x20e61cc6, 0x289924d9,
	0xd505d1a6, 0xdc71d8a0, 0xc91bc6d5, 0xce87cbad,
	0x0f8c0af0, 0x187d1413, 0xfcdcf82a, 0x06450192,
	0xc1ebc0f1, 0xc4dfc33b, 0xc014c07b, 0xc04fc005,
	0xea70e610, 0xf383eeee, 0xd9e0d632, 0xe1d4ddc2,
	0xc3bec57e, 0xc13bc251, 0xcc98cf8a, 0xc78fc9ee,
	0xcc98c9ee, 0xd2bfcf8a, 0xc3bec251, 0xc78fc57e,
	0xd9e0ddc2, 0xd2bfd632, 0xea70eeee, 0xe1d4e610,
	0xc014c005, 0xc13bc07b, 0xc1ebc33b, 0xc04fc0f1,
	0xfcdc0192, 0xf383f82a, 0x0f8c1413, 0x06450af0,
	0xc91bcbad, 0xc4dfc6d5, 0xd505d8a0, 0xce87d1a6,
	0x20e624d9, 0x187d1cc6, 0x2f6b3273, 0x28992c21,
	0xe4a3e8f7, 0xdc71e074, 0xf69cfb4a, 0xed6cf1fa,
	0x39da3bb5, 0x353637af, 0x3f4e3fd3, 0x3d3e3e71,
	0x09630e05, 0x000004b5, 0x1b5c1f8b, 0x12931708,
	0x3f4e3e71, 0x3fff3fd3, 0x39da37af, 0x3d3e3bb5,
	0x2afa2e59, 0x238e275f, 0x36e4392a, 0x31783452,
	0x2f6b2c21, 0x35363273, 0x20e61cc6, 0x289924d9,
	0x3e143f0e, 0x3b203cc4, 0x3feb3f84, 0x3fb03ffa,
	0x0f8c0af0, 0x187d1413, 0xfcdcf82a, 0x06450192,
	0x3c413a81, 0x3ec43dae, 0x33673075, 0x38703611,
	0xea70e610, 0xf383eeee, 0xd9e0d632, 0xe1d4ddc2,
	0x261f223d, 0x2d4029cd, 0x158f1111, 0x1e2b19ef,
	0xcc98c9ee, 0xd2bfcf8a, 0xc3bec251, 0xc78fc57e,
	0x0323fe6d, 0x0c7c07d5, 0xf073ebec, 0xf9baf50f,
	0xc014c005, 0xc13bc07b, 0xc1ebc33b, 0xc04fc0f1,
	0xdf19db26, 0xe782e339, 0xd094cd8c, 0xd766d3de,
	0xc91bcbad, 0xc4dfc6d5, 0xd505d8a0, 0xce87d1a6,
	0xc625c44a, 0xcac9c850, 0xc0b1c02c, 0xc2c1c18e,
	0xe4a3e8f7, 0xdc71e074, 0xf69cfb4a, 0xed6cf1fa,
	0x3fd33f9b, 0x3fff3ff4, 0x3e713de2, 0x3f4e3eea,
	0xfb4af8f2, 0x0000fda4, 0xf1faefb0, 0xf69cf449,
	0x3bb53ad2, 0x3d3e3c84, 0x37af367c, 0x39da38ce,
	0xe8f7e6c9, 0xed6ceb2e, 0xe074de6d, 0xe4a3e286,
	0x327330f8, 0x353633de, 0x2c212a64, 0x2f6b2dce,
	0xd8a0d6cb, 0xdc71da82, 0xd1a6d00e, 0xd505d34e,
	0x24d922e6, 0x289926c0, 0x1cc61aa6, 0x20e61edb,
	0xcbadca5a, 0xce87cd11, 0xc6d5c5d0, 0xc91bc7ee,
	0x141311d3, 0x187d164c, 0x0af0089c, 0x0f8c0d41,
	0xc33bc288, 0xc4dfc403, 0xc0f1c095, 0xc1ebc163,
	0x0192ff36, 0x064503ec, 0xf82af5d5, 0xfcdcfa82,
	0xc005c001, 0xc04fc01f, 0xc07bc0d0, 0xc014c03c,
	0xeeeeecac, 0xf383f136, 0xe610e3ed, 0xea70e83c,
	0xc251c2fd, 0xc13bc1bb, 0xc57ec67c, 0xc3bec493,
	0xddc2dbcb, 0xe1d4dfc6, 0xd632d471, 0xd9e0d802,
	0xc9eecb3a, 0xc78fc8b5, 0xcf8ad11c, 0xcc98ce08,
	0xcf8ace08, 0xd2bfd11c, 0xc9eec8b5, 0xcc98cb3a,
	0xd632d802, 0xd2bfd471, 0xddc2dfc6, 0xd9e0dbcb,
	0xc57ec493, 0xc78fc67c, 0xc251c1bb, 0xc3bec2fd,
	0xe610e83c, 0xe1d4e3ed, 0xeeeef136, 0xea70ecac,
	0xc07bc03c, 0xc13bc0d0, 0xc005c01f, 0xc014c001,
	0xf82afa82, 0xf383f5d5, 0x019203ec, 0xfcdcff36,
	0xc0f1c163, 0xc04fc095, 0xc33bc403, 0xc1ebc288,
	0x0af00d41, 0x0645089c, 0x1413164c, 0x0f8c11d3,
	0xc6d5c7ee, 0xc4dfc5d0, 0xcbadcd11, 0xc91bca5a,
	0x1cc61edb, 0x187d1aa6, 0x24d926c0, 0x20e622e6,
	0xd1a6d34e, 0xce87d00e, 0xd8a0da82, 0xd505d6cb,
	0x2c212dce, 0x28992a64, 0x327333de, 0x2f6b30f8,
	0xe074e286, 0xdc71de6d, 0xe8f7eb2e, 0xe4a3e6c9,
	0x37af38ce, 0x3536367c, 0x3bb53c84, 0x39da3ad2,
	0xf1faf449, 0xed6cefb0, 0xfb4afda4, 0xf69cf8f2,
	0x3e713eea, 0x3d3e3de2, 0x3fd33ff4, 0x3f4e3f9b,
	0x04b5070d, 0x0000025b, 0x0e05104f, 0x09630bb6,
	0x3fd33f9b, 0x3fff3ff4, 0x3e713de2, 0x3f4e3eea,
	0x17081936, 0x129314d1, 0x1f8b2192, 0x1b5c1d79,
	0x3bb53ad2, 0x3d3e3c84, 0x37af367c, 0x39da38ce,
	0x275f2934, 0x238e257d, 0x2e592ff1, 0x2afa2cb1,
	0x327330f8, 0x353633de, 0x2c212a64, 0x2f6b2dce,
	0x345235a5, 0x317832ee, 0x392a3a2f, 0x36e43811,
	0x24d922e6, 0x289926c0, 0x1cc61aa6, 0x20e61edb,
	0x3cc43d77, 0x3b203bfc, 0x3f0e3f6a, 0x3e143e9c,
	0x141311d3, 0x187d164c, 0x0af0089c, 0x0f8c0d41,
	0x3ffa3ffe, 0x3fb03fe0, 0x3f843f2f, 0x3feb3fc3,
	0x0192ff36, 0x064503ec, 0xf82af5d5, 0xfcdcfa82,
	0x3dae3d02, 0x3ec43e44, 0x3a813983, 0x3c413b6c,
	0xeeeeecac, 0xf383f136, 0xe610e3ed, 0xea70e83c,
	0x361134c5, 0x3870374a, 0x30752ee3, 0x336731f7,
	0xddc2dbcb, 0xe1d4dfc6, 0xd632d471, 0xd9e0d802,
	0x29cd27fd, 0x2d402b8e, 0x223d2039, 0x261f2434,
	0xcf8ace08, 0xd2bfd11c, 0xc9eec8b5, 0xcc98cb3a,
	0x19ef17c3, 0x1e2b1c12, 0x11110ec9, 0x158f1353,
	0xc57ec493, 0xc78fc67c, 0xc251c1bb, 0xc3bec2fd,
	0x07d5057d, 0x0c7c0a2a, 0xfe6dfc13, 0x032300c9,
	0xc07bc03c, 0xc13bc0d0, 0xc005c01f, 0xc014c001,
	0xf50ff2be, 0xf9baf763, 0xebece9b3, 0xf073ee2c,
	0xc0f1c163, 0xc04fc095, 0xc33bc403, 0xc1ebc288,
	0xe339e124, 0xe782e559, 0xdb26d93f, 0xdf19dd19,
	0xc6d5c7ee, 0xc4dfc5d0, 0xcbadcd11, 0xc91bca5a,
	0xd3ded231, 0xd766d59b, 0xcd8ccc21, 0xd094cf07,
	0xd1a6d34e, 0xce87d00e, 0xd8a0da82, 0xd505d6cb,
	0xc850c731, 0xcac9c983, 0xc44ac37b, 0xc625c52d,
	0xe074e286, 0xdc71de6d, 0xe8f7eb2e, 0xe4a3e6c9,
	0xc18ec115, 0xc2c1c21d, 0xc02cc00b, 0xc0b1c064,
	0xf1faf449, 0xed6cefb0, 0xfb4afda4, 0xf69cf8f2,
	0x3ff43fe6, 0x3fff3ffc, 0x3f9b3f77, 0x3fd33fba,
	0xfda4fc77, 0x0000fed2, 0xf8f2f7c6, 0xfb4afa1e,
	0x3eea3eb0, 0x3f4e3f1f, 0x3de23d92, 0x3e713e2c,
	0xf449f321, 0xf69cf572, 0xefb0ee8d, 0xf1faf0d4,
	0x3c843c1f, 0x3d3e3ce3, 0x3ad23a58, 0x3bb53b46,
	0xeb2eea12, 0xed6cec4c, 0xe6c9e5b4, 0xe8f7e7df,
	0x38ce3841, 0x39da3957, 0x367c35db, 0x37af3718,
	0xe286e17c, 0xe4a3e393, 0xde6ddd6e, 0xe074df6f,
	0x33de332b, 0x3536348c, 0x30f83033, 0x327331b8,
	0xda82d98f, 0xdc71db78, 0xd6cbd5e6, 0xd8a0d7b4,
	0x2dce2cf9, 0x2f6b2e9e, 0x2a642980, 0x2c212b44,
	0xd34ed278, 0xd505d427, 0xd00ecf48, 0xd1a6d0d8,
	0x26c025ce, 0x289927ae, 0x22e621e8, 0x24d923e1,
	0xcd11cc5d, 0xce87cdca, 0xca5ac9b8, 0xcbadcb01,
	0x1edb1dd2, 0x20e61fe2, 0x1aa61993, 0x1cc61bb7,
	0xc7eec75f, 0xc91bc882, 0xc5d0c555, 0xc6d5c650,
	0x164c1530, 0x187d1766, 0x11d310b0, 0x141312f3,
	0xc403c39c, 0xc4dfc46e, 0xc288c237, 0xc33bc2df,
	0x0d410c19, 0x0f8c0e67, 0x089c0771, 0x0af009c7,
	0xc163c127, 0xc1ebc1a4, 0xc095c06f, 0xc0f1c0c0,
	0x03ec02bf, 0x06450519, 0xff36fe09, 0x01920064,
	0xc01fc00f, 0xc04fc034, 0xc001c008, 0xc005c000,
	0xfa82f956, 0xfcdcfbaf, 0xf5d5f4ac, 0xf82af6ff,
	0xc03cc059, 0xc014c025, 0xc0d0c103, 0xc07bc0a3,
	0xf136f011, 0xf383f25c, 0xecaceb8d, 0xeeeeedcc,
	0xc1bbc204, 0xc13bc178, 0xc2fdc35b, 0xc251c2a5,
	0xe83ce725, 0xea70e955, 0xe3ede2e0, 0xe610e4fe,
	0xc493c506, 0xc3bec426, 0xc67cc703, 0xc57ec5fa,
	0xdfc6dec3, 0xe1d4e0cc, 0xdbcbdad4, 0xddc2dcc5,
	0xc8b5c94f, 0xc78fc81f, 0xcb3acbe7, 0xc9eeca92,
	0xd802d718, 0xd9e0d8ef, 0xd471d396, 0xd632d550,
	0xce08cec7, 0xcc98cd4e, 0xd11cd1eb, 0xcf8ad051,
	0xd11cd051, 0xd2bfd1eb, 0xce08cd4e, 0xcf8acec7,
	0xd471d550, 0xd2bfd396, 0xd802d8ef, 0xd632d718,
	0xcb3aca92, 0xcc98cbe7, 0xc8b5c81f, 0xc9eec94f,
	0xdbcbdcc5, 0xd9e0dad4, 0xdfc6e0cc, 0xddc2dec3,
	0xc67cc5fa, 0xc78fc703, 0xc493c426, 0xc57ec506,
	0xe3ede4fe, 0xe1d4e2e0, 0xe83ce955, 0xe610e725,
	0xc2fdc2a5, 0xc3bec35b, 0xc1bbc178, 0xc251c204,
	0xecacedcc, 0xea70eb8d, 0xf136f25c, 0xeeeef011,
	0xc0d0c0a3, 0xc13bc103, 0xc03cc025, 0xc07bc059,
	0xf5d5f6ff, 0xf383f4ac, 0xfa82fbaf, 0xf82af956,
	0xc001c000, 0xc014c008, 0xc01fc034, 0xc005c00f,
	0xff360064, 0xfcdcfe09, 0x03ec0519, 0x019202bf,
	0xc095c0c0, 0xc04fc06f, 0xc163c1a4, 0xc0f1c127,
	0x089c09c7, 0x06450771, 0x0d410e67, 0x0af00c19,
	0xc288c2df, 0xc1ebc237, 0xc403c46e, 0xc33bc39c,
	0x11d312f3, 0x0f8c10b0, 0x164c1766, 0x14131530,
	0xc5d0c650, 0xc4dfc555, 0xc7eec882, 0xc6d5c75f,
	0x1aa61bb7, 0x187d1993, 0x1edb1fe2, 0x1cc61dd2,
	0xca5acb01, 0xc91bc9b8, 0xcd11cdca, 0xcbadcc5d,
	0x22e623e1, 0x20e621e8, 0x26c027ae, 0x24d925ce,
	0xd00ed0d8, 0xce87cf48, 0xd34ed427, 0xd1a6d278,
	0x2a642b44, 0x28992980, 0x2dce2e9e, 0x2c212cf9,
	0xd6cbd7b4, 0xd505d5e6, 0xda82db78, 0xd8a0d98f,
	0x30f831b8, 0x2f6b3033, 0x33de348c, 0x3273332b,
	0xde6ddf6f, 0xdc71dd6e, 0xe286e393, 0xe074e17c,
	0x367c3718, 0x353635db, 0x38ce3957, 0x37af3841,
	0xe6c9e7df, 0xe4a3e5b4, 0xeb2eec4c, 0xe8f7ea12,
	0x3ad23b46, 0x39da3a58, 0x3c843ce3, 0x3bb53c1f,
	0xefb0f0d4, 0xed6cee8d, 0xf449f572, 0xf1faf321,
	0x3de23e2c, 0x3d3e3d92, 0x3eea3f1f, 0x3e713eb0,
	0xf8f2fa1e, 0xf69cf7c6, 0xfda4fed2, 0xfb4afc77,
	0x3f9b3fba, 0x3f4e3f77, 0x3ff43ffc, 0x3fd33fe6,
	0x025b0388, 0x0000012d, 0x070d0839, 0x04b505e1,
	0x3ff43fe6, 0x3fff3ffc, 0x3f9b3f77, 0x3fd33fba,
	0x0bb60cde, 0x09630a8d, 0x104f1172, 0x0e050f2b,
	0x3eea3eb0, 0x3f4e3f1f, 0x3de23d92, 0x3e713e2c,
	0x14d115ed, 0x129313b3, 0x19361a4b, 0x17081820,
	0x3c843c1f, 0x3d3e3ce3, 0x3ad23a58, 0x3bb53b46,
	0x1d791e83, 0x1b5c1c6c, 0x21922291, 0x1f8b2090,
	0x38ce3841, 0x39da3957, 0x367c35db, 0x37af3718,
	0x257d2670, 0x238e2487, 0x29342a19, 0x275f284b,
	0x33de332b, 0x3536348c, 0x30f83033, 0x327331b8,
	0x2cb12d87, 0x2afa2bd8, 0x2ff130b7, 0x2e592f27,
	0x2dce2cf9, 0x2f6b2e9e, 0x2a642980, 0x2c212b44,
	0x32ee33a2, 0x31783235, 0x35a53647, 0x345234fe,
	0x26c025ce, 0x289927ae, 0x22e621e8, 0x24d923e1,
	0x381138a0, 0x36e4377d, 0x3a2f3aaa, 0x392a39af,
	0x1edb1dd2, 0x20e61fe2, 0x1aa61993, 0x1cc61bb7,
	0x3bfc3c63, 0x3b203b91, 0x3d773dc8, 0x3cc43d20,
	0x164c1530, 0x187d1766, 0x11d310b0, 0x141312f3,
	0x3e9c3ed8, 0x3e143e5b, 0x3f6a3f90, 0x3f0e3f3f,
	0x0d410c19, 0x0f8c0e67, 0x089c0771, 0x0af009c7,
	0x3fe03ff0, 0x3fb03fcb, 0x3ffe3ff7, 0x3ffa3fff,
	0x03ec02bf, 0x06450519, 0xff36fe09, 0x01920064,
	0x3fc33fa6, 0x3feb3fda, 0x3f2f3efc, 0x3f843f5c,
	0xfa82f956, 0xfcdcfbaf, 0xf5d5f4ac, 0xf82af6ff,
	0x3e443dfb, 0x3ec43e87, 0x3d023ca4, 0x3dae3d5a,
	0xf136f011, 0xf383f25c, 0xecaceb8d, 0xeeeeedcc,
	0x3b6c3af9, 0x3c413bd9, 0x398338fc, 0x3a813a05,
	0xe83ce725, 0xea70e955, 0xe3ede2e0, 0xe610e4fe,
	0x374a36b0, 0x387037e0, 0x34c53418, 0x3611356d,
	0xdfc6dec3, 0xe1d4e0cc, 0xdbcbdad4, 0xddc2dcc5,
	0x31f73138, 0x336732b1, 0x2ee32e14, 0x30752fae,
	0xd802d718, 0xd9e0d8ef, 0xd471d396, 0xd632d550,
	0x2b8e2aaf, 0x2d402c69, 0x27fd2710, 0x29cd28e7,
	0xd11cd051, 0xd2bfd1eb, 0xce08cd4e, 0xcf8acec7,
	0x2434233a, 0x261f252b, 0x20391f33, 0x223d213c,
	0xcb3aca92, 0xcc98cbe7, 0xc8b5c81f, 0xc9eec94f,
	0x1c121b01, 0x1e2b1d1f, 0x17c316aa, 0x19ef18da,
	0xc67cc5fa, 0xc78fc703, 0xc493c426, 0xc57ec506,
	0x13531233, 0x158f1472, 0x0ec90da3, 0x11110fee,
	0xc2fdc2a5, 0xc3bec35b, 0xc1bbc178, 0xc251c204,
	0x0a2a0900, 0x0c7c0b53, 0x057d0450, 0x07d506a9,
	0xc0d0c0a3, 0xc13bc103, 0xc03cc025, 0xc07bc059,
	0x00c9ff9b, 0x032301f6, 0xfc13fae6, 0xfe6dfd40,
	0xc001c000, 0xc014c008, 0xc01fc034, 0xc005c00f,
	0xf763f638, 0xf9baf88e, 0xf2bef198, 0xf50ff3e6,
	0xc095c0c0, 0xc04fc06f, 0xc163c1a4, 0xc0f1c127,
	0xee2ced0c, 0xf073ef4f, 0xe9b3e899, 0xebeceacf,
	0xc288c2df, 0xc1ebc237, 0xc403c46e, 0xc33bc39c,
	0xe559e448, 0xe782e66c, 0xe124e01d, 0xe339e22d,
	0xc5d0c650, 0xc4dfc555, 0xc7eec882, 0xc6d5c75f,
	0xdd19dc1e, 0xdf19de17, 0xd93fd851, 0xdb26da31,
	0xca5acb01, 0xc91bc9b8, 0xcd11cdca, 0xcbadcc5d,
	0xd59bd4bb, 0xd766d67f, 0xd231d161, 0xd3ded306,
	0xd00ed0d8, 0xce87cf48, 0xd34ed427, 0xd1a6d278,
	0xcf07ce47, 0xd094cfcc, 0xcc21cb73, 0xcd8cccd4,
	0xd6cbd7b4, 0xd505d5e6, 0xda82db78, 0xd8a0d98f,
	0xc983c8e7, 0xcac9ca24, 0xc731c6a8, 0xc850c7be,
	0xde6ddf6f, 0xdc71dd6e, 0xe286e393, 0xe074e17c,
	0xc52dc4b9, 0xc625c5a7, 0xc37bc31c, 0xc44ac3e0,
	0xe6c9e7df, 0xe4a3e5b4, 0xeb2eec4c, 0xe8f7ea12,
	0xc21dc1d3, 0xc2c1c26d, 0xc115c0e0, 0xc18ec14f,
	0xefb0f0d4, 0xed6cee8d, 0xf449f572, 0xf1faf321,
	0xc064c045, 0xc0b1c088, 0xc00bc003, 0xc02cc019,
	0xf8f2fa1e, 0xf69cf7c6, 0xfda4fed2, 0xfb4afc77,
};

/* *********************************************************** */

const mlib_u32 mlib_s_fftK_WTable_3D_P[] = {
	0x00000000, 0x00000000, 0x00002d40, 0x3fffd2bf,
	0x00000000, 0x00000000, 0xc0002d40, 0x00002d40,
	0xd2bfc4df, 0x3fff187d, 0x2d40e782, 0x00003b20,
	0x2d40e782, 0x00003b20, 0x2d403b20, 0xc000e782,
	0x187df383, 0x3fff3536, 0xc4dfdc71, 0xd2bfc13b,
	0x3b203ec4, 0x0000238e, 0xe782cac9, 0x2d400c7c,
	0x3b203ec4, 0x0000238e, 0xe782cac9, 0x2d400c7c,
	0xe7820c7c, 0xc000cac9, 0x3b20238e, 0x2d403ec4,
	0x35362899, 0x3fff3d3e, 0xf383e1d4, 0x187d0645,
	0x238e3178, 0x00001293, 0x3ec43870, 0x3b203fb0,
	0xc13bc04f, 0xd2bfc78f, 0xdc71ed6c, 0xc4dfce87,
	0x0c7cf9ba, 0x2d401e2b, 0xcac9c2c1, 0xe782d766,
	0x238e3178, 0x00001293, 0x3ec43870, 0x3b203fb0,
	0xcac9d766, 0xc000c2c1, 0x0c7c1e2b, 0xe782f9ba,
	0x0c7cf9ba, 0x2d401e2b, 0xcac9c2c1, 0xe782d766,
	0x3ec43fb0, 0x2d403870, 0x238e1293, 0x3b203178,
	0x3d3e39da, 0x3fff3f4e, 0x289920e6, 0x35362f6b,
	0x12931b5c, 0x00000963, 0x317836e4, 0x238e2afa,
	0x0645fcdc, 0x187d0f8c, 0xe1d4d9e0, 0xf383ea70,
	0x3fb03feb, 0x3b203e14, 0x38703367, 0x3ec43c41,
	0xc78fc3be, 0xd2bfcc98, 0xc04fc1eb, 0xc13bc014,
	0x1e2b158f, 0x2d40261f, 0xf9baf073, 0x0c7c0323,
	0xce87d505, 0xc4dfc91b, 0xed6cf69c, 0xdc71e4a3,
	0xd766d094, 0xe782df19, 0xc2c1c0b1, 0xcac9c625,
	0x12931b5c, 0x00000963, 0x317836e4, 0x238e2afa,
	0xc2c1c625, 0xc000c0b1, 0xd766df19, 0xcac9d094,
	0x3fb03feb, 0x3b203e14, 0x38703367, 0x3ec43c41,
	0xf9ba0323, 0xe782f073, 0x1e2b261f, 0x0c7c158f,
	0x1e2b158f, 0x2d40261f, 0xf9baf073, 0x0c7c0323,
	0x38703c41, 0x2d403367, 0x3fb03e14, 0x3ec43feb,
	0xd766d094, 0xe782df19, 0xc2c1c0b1, 0xcac9c625,
	0x31782afa, 0x3b2036e4, 0x12930963, 0x238e1b5c,
	0x3f4e3e71, 0x3fff3fd3, 0x39da37af, 0x3d3e3bb5,
	0x09630e05, 0x000004b5, 0x1b5c1f8b, 0x12931708,
	0x2f6b2c21, 0x35363273, 0x20e61cc6, 0x289924d9,
	0x2afa2e59, 0x238e275f, 0x36e4392a, 0x31783452,
	0x0f8c0af0, 0x187d1413, 0xfcdcf82a, 0x06450192,
	0x3e143f0e, 0x3b203cc4, 0x3feb3f84, 0x3fb03ffa,
	0xea70e610, 0xf383eeee, 0xd9e0d632, 0xe1d4ddc2,
	0x3c413a81, 0x3ec43dae, 0x33673075, 0x38703611,
	0xcc98c9ee, 0xd2bfcf8a, 0xc3bec251, 0xc78fc57e,
	0x261f223d, 0x2d4029cd, 0x158f1111, 0x1e2b19ef,
	0xc014c005, 0xc13bc07b, 0xc1ebc33b, 0xc04fc0f1,
	0x0323fe6d, 0x0c7c07d5, 0xf073ebec, 0xf9baf50f,
	0xc91bcbad, 0xc4dfc6d5, 0xd505d8a0, 0xce87d1a6,
	0xdf19db26, 0xe782e339, 0xd094cd8c, 0xd766d3de,
	0xe4a3e8f7, 0xdc71e074, 0xf69cfb4a, 0xed6cf1fa,
	0xc625c44a, 0xcac9c850, 0xc0b1c02c, 0xc2c1c18e,
	0x09630e05, 0x000004b5, 0x1b5c1f8b, 0x12931708,
	0xc0b1c18e, 0xc000c02c, 0xc625c850, 0xc2c1c44a,
	0x2afa2e59, 0x238e275f, 0x36e4392a, 0x31783452,
	0xd094d3de, 0xcac9cd8c, 0xdf19e339, 0xd766db26,
	0x3e143f0e, 0x3b203cc4, 0x3feb3f84, 0x3fb03ffa,
	0xf073f50f, 0xe782ebec, 0x032307d5, 0xf9bafe6d,
	0x3c413a81, 0x3ec43dae, 0x33673075, 0x38703611,
	0x158f19ef, 0x0c7c1111, 0x261f29cd, 0x1e2b223d,
	0x261f223d, 0x2d4029cd, 0x158f1111, 0x1e2b19ef,
	0x33673611, 0x2d403075, 0x3c413dae, 0x38703a81,
	0x0323fe6d, 0x0c7c07d5, 0xf073ebec, 0xf9baf50f,
	0x3feb3ffa, 0x3ec43f84, 0x3e143cc4, 0x3fb03f0e,
	0xdf19db26, 0xe782e339, 0xd094cd8c, 0xd766d3de,
	0x36e43452, 0x3b20392a, 0x2afa275f, 0x31782e59,
	0xc625c44a, 0xcac9c850, 0xc0b1c02c, 0xc2c1c18e,
	0x1b5c1708, 0x238e1f8b, 0x096304b5, 0x12930e05,
	0x3fd33f9b, 0x3fff3ff4, 0x3e713de2, 0x3f4e3eea,
	0x04b5070d, 0x0000025b, 0x0e05104f, 0x09630bb6,
	0x3bb53ad2, 0x3d3e3c84, 0x37af367c, 0x39da38ce,
	0x17081936, 0x129314d1, 0x1f8b2192, 0x1b5c1d79,
	0x327330f8, 0x353633de, 0x2c212a64, 0x2f6b2dce,
	0x275f2934, 0x238e257d, 0x2e592ff1, 0x2afa2cb1,
	0x24d922e6, 0x289926c0, 0x1cc61aa6, 0x20e61edb,
	0x345235a5, 0x317832ee, 0x392a3a2f, 0x36e43811,
	0x141311d3, 0x187d164c, 0x0af0089c, 0x0f8c0d41,
	0x3cc43d77, 0x3b203bfc, 0x3f0e3f6a, 0x3e143e9c,
	0x0192ff36, 0x064503ec, 0xf82af5d5, 0xfcdcfa82,
	0x3ffa3ffe, 0x3fb03fe0, 0x3f843f2f, 0x3feb3fc3,
	0xeeeeecac, 0xf383f136, 0xe610e3ed, 0xea70e83c,
	0x3dae3d02, 0x3ec43e44, 0x3a813983, 0x3c413b6c,
	0xddc2dbcb, 0xe1d4dfc6, 0xd632d471, 0xd9e0d802,
	0x361134c5, 0x3870374a, 0x30752ee3, 0x336731f7,
	0xcf8ace08, 0xd2bfd11c, 0xc9eec8b5, 0xcc98cb3a,
	0x29cd27fd, 0x2d402b8e, 0x223d2039, 0x261f2434,
	0xc57ec493, 0xc78fc67c, 0xc251c1bb, 0xc3bec2fd,
	0x19ef17c3, 0x1e2b1c12, 0x11110ec9, 0x158f1353,
	0xc07bc03c, 0xc13bc0d0, 0xc005c01f, 0xc014c001,
	0x07d5057d, 0x0c7c0a2a, 0xfe6dfc13, 0x032300c9,
	0xc0f1c163, 0xc04fc095, 0xc33bc403, 0xc1ebc288,
	0xf50ff2be, 0xf9baf763, 0xebece9b3, 0xf073ee2c,
	0xc6d5c7ee, 0xc4dfc5d0, 0xcbadcd11, 0xc91bca5a,
	0xe339e124, 0xe782e559, 0xdb26d93f, 0xdf19dd19,
	0xd1a6d34e, 0xce87d00e, 0xd8a0da82, 0xd505d6cb,
	0xd3ded231, 0xd766d59b, 0xcd8ccc21, 0xd094cf07,
	0xe074e286, 0xdc71de6d, 0xe8f7eb2e, 0xe4a3e6c9,
	0xc850c731, 0xcac9c983, 0xc44ac37b, 0xc625c52d,
	0xf1faf449, 0xed6cefb0, 0xfb4afda4, 0xf69cf8f2,
	0xc18ec115, 0xc2c1c21d, 0xc02cc00b, 0xc0b1c064,
	0x04b5070d, 0x0000025b, 0x0e05104f, 0x09630bb6,
	0xc02cc064, 0xc000c00b, 0xc18ec21d, 0xc0b1c115,
	0x17081936, 0x129314d1, 0x1f8b2192, 0x1b5c1d79,
	0xc44ac52d, 0xc2c1c37b, 0xc850c983, 0xc625c731,
	0x275f2934, 0x238e257d, 0x2e592ff1, 0x2afa2cb1,
	0xcd8ccf07, 0xcac9cc21, 0xd3ded59b, 0xd094d231,
	0x345235a5, 0x317832ee, 0x392a3a2f, 0x36e43811,
	0xdb26dd19, 0xd766d93f, 0xe339e559, 0xdf19e124,
	0x3cc43d77, 0x3b203bfc, 0x3f0e3f6a, 0x3e143e9c,
	0xebecee2c, 0xe782e9b3, 0xf50ff763, 0xf073f2be,
	0x3ffa3ffe, 0x3fb03fe0, 0x3f843f2f, 0x3feb3fc3,
	0xfe6d00c9, 0xf9bafc13, 0x07d50a2a, 0x0323057d,
	0x3dae3d02, 0x3ec43e44, 0x3a813983, 0x3c413b6c,
	0x11111353, 0x0c7c0ec9, 0x19ef1c12, 0x158f17c3,
	0x361134c5, 0x3870374a, 0x30752ee3, 0x336731f7,
	0x223d2434, 0x1e2b2039, 0x29cd2b8e, 0x261f27fd,
	0x29cd27fd, 0x2d402b8e, 0x223d2039, 0x261f2434,
	0x307531f7, 0x2d402ee3, 0x3611374a, 0x336734c5,
	0x19ef17c3, 0x1e2b1c12, 0x11110ec9, 0x158f1353,
	0x3a813b6c, 0x38703983, 0x3dae3e44, 0x3c413d02,
	0x07d5057d, 0x0c7c0a2a, 0xfe6dfc13, 0x032300c9,
	0x3f843fc3, 0x3ec43f2f, 0x3ffa3fe0, 0x3feb3ffe,
	0xf50ff2be, 0xf9baf763, 0xebece9b3, 0xf073ee2c,
	0x3f0e3e9c, 0x3fb03f6a, 0x3cc43bfc, 0x3e143d77,
	0xe339e124, 0xe782e559, 0xdb26d93f, 0xdf19dd19,
	0x392a3811, 0x3b203a2f, 0x345232ee, 0x36e435a5,
	0xd3ded231, 0xd766d59b, 0xcd8ccc21, 0xd094cf07,
	0x2e592cb1, 0x31782ff1, 0x275f257d, 0x2afa2934,
	0xc850c731, 0xcac9c983, 0xc44ac37b, 0xc625c52d,
	0x1f8b1d79, 0x238e2192, 0x170814d1, 0x1b5c1936,
	0xc18ec115, 0xc2c1c21d, 0xc02cc00b, 0xc0b1c064,
	0x0e050bb6, 0x1293104f, 0x04b5025b, 0x0963070d,
	0x3ff43fe6, 0x3fff3ffc, 0x3f9b3f77, 0x3fd33fba,
	0x025b0388, 0x0000012d, 0x070d0839, 0x04b505e1,
	0x3eea3eb0, 0x3f4e3f1f, 0x3de23d92, 0x3e713e2c,
	0x0bb60cde, 0x09630a8d, 0x104f1172, 0x0e050f2b,
	0x3c843c1f, 0x3d3e3ce3, 0x3ad23a58, 0x3bb53b46,
	0x14d115ed, 0x129313b3, 0x19361a4b, 0x17081820,
	0x38ce3841, 0x39da3957, 0x367c35db, 0x37af3718,
	0x1d791e83, 0x1b5c1c6c, 0x21922291, 0x1f8b2090,
	0x33de332b, 0x3536348c, 0x30f83033, 0x327331b8,
	0x257d2670, 0x238e2487, 0x29342a19, 0x275f284b,
	0x2dce2cf9, 0x2f6b2e9e, 0x2a642980, 0x2c212b44,
	0x2cb12d87, 0x2afa2bd8, 0x2ff130b7, 0x2e592f27,
	0x26c025ce, 0x289927ae, 0x22e621e8, 0x24d923e1,
	0x32ee33a2, 0x31783235, 0x35a53647, 0x345234fe,
	0x1edb1dd2, 0x20e61fe2, 0x1aa61993, 0x1cc61bb7,
	0x381138a0, 0x36e4377d, 0x3a2f3aaa, 0x392a39af,
	0x164c1530, 0x187d1766, 0x11d310b0, 0x141312f3,
	0x3bfc3c63, 0x3b203b91, 0x3d773dc8, 0x3cc43d20,
	0x0d410c19, 0x0f8c0e67, 0x089c0771, 0x0af009c7,
	0x3e9c3ed8, 0x3e143e5b, 0x3f6a3f90, 0x3f0e3f3f,
	0x03ec02bf, 0x06450519, 0xff36fe09, 0x01920064,
	0x3fe03ff0, 0x3fb03fcb, 0x3ffe3ff7, 0x3ffa3fff,
	0xfa82f956, 0xfcdcfbaf, 0xf5d5f4ac, 0xf82af6ff,
	0x3fc33fa6, 0x3feb3fda, 0x3f2f3efc, 0x3f843f5c,
	0xf136f011, 0xf383f25c, 0xecaceb8d, 0xeeeeedcc,
	0x3e443dfb, 0x3ec43e87, 0x3d023ca4, 0x3dae3d5a,
	0xe83ce725, 0xea70e955, 0xe3ede2e0, 0xe610e4fe,
	0x3b6c3af9, 0x3c413bd9, 0x398338fc, 0x3a813a05,
	0xdfc6dec3, 0xe1d4e0cc, 0xdbcbdad4, 0xddc2dcc5,
	0x374a36b0, 0x387037e0, 0x34c53418, 0x3611356d,
	0xd802d718, 0xd9e0d8ef, 0xd471d396, 0xd632d550,
	0x31f73138, 0x336732b1, 0x2ee32e14, 0x30752fae,
	0xd11cd051, 0xd2bfd1eb, 0xce08cd4e, 0xcf8acec7,
	0x2b8e2aaf, 0x2d402c69, 0x27fd2710, 0x29cd28e7,
	0xcb3aca92, 0xcc98cbe7, 0xc8b5c81f, 0xc9eec94f,
	0x2434233a, 0x261f252b, 0x20391f33, 0x223d213c,
	0xc67cc5fa, 0xc78fc703, 0xc493c426, 0xc57ec506,
	0x1c121b01, 0x1e2b1d1f, 0x17c316aa, 0x19ef18da,
	0xc2fdc2a5, 0xc3bec35b, 0xc1bbc178, 0xc251c204,
	0x13531233, 0x158f1472, 0x0ec90da3, 0x11110fee,
	0xc0d0c0a3, 0xc13bc103, 0xc03cc025, 0xc07bc059,
	0x0a2a0900, 0x0c7c0b53, 0x057d0450, 0x07d506a9,
	0xc001c000, 0xc014c008, 0xc01fc034, 0xc005c00f,
	0x00c9ff9b, 0x032301f6, 0xfc13fae6, 0xfe6dfd40,
	0xc095c0c0, 0xc04fc06f, 0xc163c1a4, 0xc0f1c127,
	0xf763f638, 0xf9baf88e, 0xf2bef198, 0xf50ff3e6,
	0xc288c2df, 0xc1ebc237, 0xc403c46e, 0xc33bc39c,
	0xee2ced0c, 0xf073ef4f, 0xe9b3e899, 0xebeceacf,
	0xc5d0c650, 0xc4dfc555, 0xc7eec882, 0xc6d5c75f,
	0xe559e448, 0xe782e66c, 0xe124e01d, 0xe339e22d,
	0xca5acb01, 0xc91bc9b8, 0xcd11cdca, 0xcbadcc5d,
	0xdd19dc1e, 0xdf19de17, 0xd93fd851, 0xdb26da31,
	0xd00ed0d8, 0xce87cf48, 0xd34ed427, 0xd1a6d278,
	0xd59bd4bb, 0xd766d67f, 0xd231d161, 0xd3ded306,
	0xd6cbd7b4, 0xd505d5e6, 0xda82db78, 0xd8a0d98f,
	0xcf07ce47, 0xd094cfcc, 0xcc21cb73, 0xcd8cccd4,
	0xde6ddf6f, 0xdc71dd6e, 0xe286e393, 0xe074e17c,
	0xc983c8e7, 0xcac9ca24, 0xc731c6a8, 0xc850c7be,
	0xe6c9e7df, 0xe4a3e5b4, 0xeb2eec4c, 0xe8f7ea12,
	0xc52dc4b9, 0xc625c5a7, 0xc37bc31c, 0xc44ac3e0,
	0xefb0f0d4, 0xed6cee8d, 0xf449f572, 0xf1faf321,
	0xc21dc1d3, 0xc2c1c26d, 0xc115c0e0, 0xc18ec14f,
	0xf8f2fa1e, 0xf69cf7c6, 0xfda4fed2, 0xfb4afc77,
	0xc064c045, 0xc0b1c088, 0xc00bc003, 0xc02cc019,
	0x025b0388, 0x0000012d, 0x070d0839, 0x04b505e1,
	0xc00bc019, 0xc000c003, 0xc064c088, 0xc02cc045,
	0x0bb60cde, 0x09630a8d, 0x104f1172, 0x0e050f2b,
	0xc115c14f, 0xc0b1c0e0, 0xc21dc26d, 0xc18ec1d3,
	0x14d115ed, 0x129313b3, 0x19361a4b, 0x17081820,
	0xc37bc3e0, 0xc2c1c31c, 0xc52dc5a7, 0xc44ac4b9,
	0x1d791e83, 0x1b5c1c6c, 0x21922291, 0x1f8b2090,
	0xc731c7be, 0xc625c6a8, 0xc983ca24, 0xc850c8e7,
	0x257d2670, 0x238e2487, 0x29342a19, 0x275f284b,
	0xcc21ccd4, 0xcac9cb73, 0xcf07cfcc, 0xcd8cce47,
	0x2cb12d87, 0x2afa2bd8, 0x2ff130b7, 0x2e592f27,
	0xd231d306, 0xd094d161, 0xd59bd67f, 0xd3ded4bb,
	0x32ee33a2, 0x31783235, 0x35a53647, 0x345234fe,
	0xd93fda31, 0xd766d851, 0xdd19de17, 0xdb26dc1e,
	0x381138a0, 0x36e4377d, 0x3a2f3aaa, 0x392a39af,
	0xe124e22d, 0xdf19e01d, 0xe559e66c, 0xe339e448,
	0x3bfc3c63, 0x3b203b91, 0x3d773dc8, 0x3cc43d20,
	0xe9b3eacf, 0xe782e899, 0xee2cef4f, 0xebeced0c,
	0x3e9c3ed8, 0x3e143e5b, 0x3f6a3f90, 0x3f0e3f3f,
	0xf2bef3e6, 0xf073f198, 0xf763f88e, 0xf50ff638,
	0x3fe03ff0, 0x3fb03fcb, 0x3ffe3ff7, 0x3ffa3fff,
	0xfc13fd40, 0xf9bafae6, 0x00c901f6, 0xfe6dff9b,
	0x3fc33fa6, 0x3feb3fda, 0x3f2f3efc, 0x3f843f5c,
	0x057d06a9, 0x03230450, 0x0a2a0b53, 0x07d50900,
	0x3e443dfb, 0x3ec43e87, 0x3d023ca4, 0x3dae3d5a,
	0x0ec90fee, 0x0c7c0da3, 0x13531472, 0x11111233,
	0x3b6c3af9, 0x3c413bd9, 0x398338fc, 0x3a813a05,
	0x17c318da, 0x158f16aa, 0x1c121d1f, 0x19ef1b01,
	0x374a36b0, 0x387037e0, 0x34c53418, 0x3611356d,
	0x2039213c, 0x1e2b1f33, 0x2434252b, 0x223d233a,
	0x31f73138, 0x336732b1, 0x2ee32e14, 0x30752fae,
	0x27fd28e7, 0x261f2710, 0x2b8e2c69, 0x29cd2aaf,
	0x2b8e2aaf, 0x2d402c69, 0x27fd2710, 0x29cd28e7,
	0x2ee32fae, 0x2d402e14, 0x31f732b1, 0x30753138,
	0x2434233a, 0x261f252b, 0x20391f33, 0x223d213c,
	0x34c5356d, 0x33673418, 0x374a37e0, 0x361136b0,
	0x1c121b01, 0x1e2b1d1f, 0x17c316aa, 0x19ef18da,
	0x39833a05, 0x387038fc, 0x3b6c3bd9, 0x3a813af9,
	0x13531233, 0x158f1472, 0x0ec90da3, 0x11110fee,
	0x3d023d5a, 0x3c413ca4, 0x3e443e87, 0x3dae3dfb,
	0x0a2a0900, 0x0c7c0b53, 0x057d0450, 0x07d506a9,
	0x3f2f3f5c, 0x3ec43efc, 0x3fc33fda, 0x3f843fa6,
	0x00c9ff9b, 0x032301f6, 0xfc13fae6, 0xfe6dfd40,
	0x3ffe3fff, 0x3feb3ff7, 0x3fe03fcb, 0x3ffa3ff0,
	0xf763f638, 0xf9baf88e, 0xf2bef198, 0xf50ff3e6,
	0x3f6a3f3f, 0x3fb03f90, 0x3e9c3e5b, 0x3f0e3ed8,
	0xee2ced0c, 0xf073ef4f, 0xe9b3e899, 0xebeceacf,
	0x3d773d20, 0x3e143dc8, 0x3bfc3b91, 0x3cc43c63,
	0xe559e448, 0xe782e66c, 0xe124e01d, 0xe339e22d,
	0x3a2f39af, 0x3b203aaa, 0x3811377d, 0x392a38a0,
	0xdd19dc1e, 0xdf19de17, 0xd93fd851, 0xdb26da31,
	0x35a534fe, 0x36e43647, 0x32ee3235, 0x345233a2,
	0xd59bd4bb, 0xd766d67f, 0xd231d161, 0xd3ded306,
	0x2ff12f27, 0x317830b7, 0x2cb12bd8, 0x2e592d87,
	0xcf07ce47, 0xd094cfcc, 0xcc21cb73, 0xcd8cccd4,
	0x2934284b, 0x2afa2a19, 0x257d2487, 0x275f2670,
	0xc983c8e7, 0xcac9ca24, 0xc731c6a8, 0xc850c7be,
	0x21922090, 0x238e2291, 0x1d791c6c, 0x1f8b1e83,
	0xc52dc4b9, 0xc625c5a7, 0xc37bc31c, 0xc44ac3e0,
	0x19361820, 0x1b5c1a4b, 0x14d113b3, 0x170815ed,
	0xc21dc1d3, 0xc2c1c26d, 0xc115c0e0, 0xc18ec14f,
	0x104f0f2b, 0x12931172, 0x0bb60a8d, 0x0e050cde,
	0xc064c045, 0xc0b1c088, 0xc00bc003, 0xc02cc019,
	0x070d05e1, 0x09630839, 0x025b012d, 0x04b50388,
};
#endif

/* *********************************************************** */

const mlib_u16 mlib_fft_rot[] = {

	0x4040, 0x4040, 0x0000, 0x0000,
	0x4000, 0x2D41, 0x0000, 0xD2BF,
	0x0000, 0x2D41, 0x4000, 0x2D41,
	0x4000, 0x3B21, 0x2D41, 0x187E,
	0x0000, 0xE782, 0xD2BF, 0xC4DF,
	0x4000, 0x187E, 0xD2BF, 0xC4DF,
	0x0000, 0xC4DF, 0xD2BF, 0x187E,
	0x7FFF, 0x5A82, 0x0000, 0xA57E,
	0x0000, 0x5A82, 0x7FFF, 0x5A82,
	0x7FFF, 0x7642, 0x5A82, 0x30FC,
	0x0000, 0xCF04, 0xA57E, 0x89BE,
	0x7FFF, 0x30FC, 0xA57E, 0x89BE,
	0x0000, 0x89BE, 0xA57E, 0x30FC,
};
