/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)__vexpf.c	1.9	06/01/31 SMI"

#ifdef __RESTRICT
#define restrict _Restrict
#else
#define restrict
#endif

/* float expf(float x)
 *
 * Method :
 *	1. Special cases:
 *		for x > 88.722839355...(0x42B17218) => Inf + overflow;
 *		for x < -103.97207642..(0xc2CFF1B4) => 0 + underflow;
 *		for x = Inf			    => Inf;
 *		for x = -Inf			    => 0;
 *		for x = +-NaN			    => QNaN.
 *	2. Computes exponential from:
 *		exp(x) = 2**a  *  2**(k/256)  *  2**(y/256)
 *	Where:
 *		a    =    int  ( 256 * log2(e) * x ) >> 8;
 *		k    =    int  ( 256 * log2(e) * x ) & 0xFF;
 *		y    =    frac ( 256 * x * log2(e)).
 *	Note that:
 *		k = 0, 1, ..., 255;
 *		y = (-1, 1).
 *	Then:
 *		2**(k/256) is looked up in a table of 2**0, 2**1/256, ...
 *		2**(y/256) is computed using approximation:
 *			2**(y/256) =  a0 + a1 * y + a2 * y**2
 *		Multiplication by 2**a is done by adding "a" to
 *		the biased exponent.
 * Accuracy:
 *	The maximum relative error for the approximating
 *	polynomial is 2**(-29.18).  All calculations are of
 *	double precision.
 *	Maximum error observed: less than 0.528 ulp for the whole
 *	float type range.
 *
 * NOTE: This implementation has been modified for SPARC to deliver
 * zero instead of a subnormal result whenever the argument is less
 * than log(2^-126).  Therefore the worst case relative error is 1.
 */

static const double __TBL_exp2f[] = {
	/* 2^(i/256) - (((i & 0xff) << 44), i = [0, 255] */
1.000000000000000000e+00, 9.994025125251012609e-01, 9.988087005564013632e-01,
9.982185740592087742e-01, 9.976321430258502376e-01, 9.970494174757447148e-01,
9.964704074554765478e-01, 9.958951230388689568e-01, 9.953235743270583136e-01,
9.947557714485678604e-01, 9.941917245593818730e-01, 9.936314438430204898e-01,
9.930749395106142074e-01, 9.925222218009785990e-01, 9.919733009806893653e-01,
9.914281873441580517e-01, 9.908868912137068774e-01, 9.903494229396448967e-01,
9.898157929003436051e-01, 9.892860115023132117e-01, 9.887600891802785785e-01,
9.882380363972563808e-01, 9.877198636446310465e-01, 9.872055814422322495e-01,
9.866952003384118486e-01, 9.861887309101209365e-01, 9.856861837629877776e-01,
9.851875695313955239e-01, 9.846928988785599302e-01, 9.842021824966076249e-01,
9.837154311066546031e-01, 9.832326554588848300e-01, 9.827538663326288448e-01,
9.822790745364429199e-01, 9.818082909081884413e-01, 9.813415263151109569e-01,
9.808787916539204454e-01, 9.804200978508705866e-01, 9.799654558618393629e-01,
9.795148766724087741e-01, 9.790683712979462161e-01, 9.786259507836846394e-01,
9.781876262048033732e-01, 9.777534086665099489e-01, 9.773233093041209241e-01,
9.768973392831440394e-01, 9.764755097993595978e-01, 9.760578320789027318e-01,
9.756443173783457823e-01, 9.752349769847807881e-01, 9.748298222159020865e-01,
9.744288644200894689e-01, 9.740321149764913367e-01, 9.736395852951079677e-01,
9.732512868168755604e-01, 9.728672310137493895e-01, 9.724874293887887378e-01,
9.721118934762408292e-01, 9.717406348416250950e-01, 9.713736650818186602e-01,
9.710109958251406104e-01, 9.706526387314379223e-01, 9.702986054921705072e-01,
9.699489078304969203e-01, 9.696035575013605134e-01, 9.692625662915755891e-01,
9.689259460199136642e-01, 9.685937085371902899e-01, 9.682658657263515378e-01,
9.679424295025619296e-01, 9.676234118132908124e-01, 9.673088246384006217e-01,
9.669986799902344776e-01, 9.666929899137042259e-01, 9.663917664863788115e-01,
9.660950218185727634e-01, 9.658027680534350123e-01, 9.655150173670379310e-01,
9.652317819684667066e-01, 9.649530740999082701e-01, 9.646789060367420010e-01,
9.644092900876289898e-01, 9.641442385946024096e-01, 9.638837639331581109e-01,
9.636278785123455481e-01, 9.633765947748582636e-01, 9.631299251971253694e-01,
9.628878822894031408e-01, 9.626504785958666099e-01, 9.624177266947013809e-01,
9.621896391981960006e-01, 9.619662287528346623e-01, 9.617475080393891318e-01,
9.615334897730127839e-01, 9.613241867033328614e-01, 9.611196116145447332e-01,
9.609197773255048203e-01, 9.607246966898252971e-01, 9.605343825959679060e-01,
9.603488479673386591e-01, 9.601681057623822069e-01, 9.599921689746773179e-01,
9.598210506330320246e-01, 9.596547638015787696e-01, 9.594933215798706616e-01,
9.593367371029771773e-01, 9.591850235415807502e-01, 9.590381941020729162e-01,
9.588962620266514580e-01, 9.587592405934176609e-01, 9.586271431164729018e-01,
9.584999829460172371e-01, 9.583777734684463256e-01, 9.582605281064505709e-01,
9.581482603191123770e-01, 9.580409836020059577e-01, 9.579387114872952580e-01,
9.578414575438342071e-01, 9.577492353772650846e-01, 9.576620586301189952e-01,
9.575799409819160113e-01, 9.575028961492645374e-01, 9.574309378859631181e-01,
9.573640799831001358e-01, 9.573023362691556182e-01, 9.572457206101023797e-01,
9.571942469095077177e-01, 9.571479291086353314e-01, 9.571067811865475727e-01,
9.570708171602075875e-01, 9.570400510845827879e-01, 9.570144970527471040e-01,
9.569941691959850116e-01, 9.569790816838944503e-01, 9.569692487244911838e-01,
9.569646845643128286e-01, 9.569654034885233251e-01, 9.569714198210175216e-01,
9.569827479245263113e-01, 9.569994022007218826e-01, 9.570213970903235223e-01,
9.570487470732028656e-01, 9.570814666684909211e-01, 9.571195704346837640e-01,
9.571630729697496731e-01, 9.572119889112359337e-01, 9.572663329363761964e-01,
9.573261197621985019e-01, 9.573913641456324175e-01, 9.574620808836177277e-01,
9.575382848132127922e-01, 9.576199908117032367e-01, 9.577072137967114207e-01,
9.577999687263049067e-01, 9.578982705991073709e-01, 9.580021344544072948e-01,
9.581115753722692086e-01, 9.582266084736434930e-01, 9.583472489204779565e-01,
9.584735119158284133e-01, 9.586054127039703721e-01, 9.587429665705107240e-01,
9.588861888424999869e-01, 9.590350948885443261e-01, 9.591897001189184646e-01,
9.593500199856788146e-01, 9.595160699827764983e-01, 9.596878656461707013e-01,
9.598654225539432483e-01, 9.600487563264122892e-01, 9.602378826262468747e-01,
9.604328171585819751e-01, 9.606335756711334994e-01, 9.608401739543135367e-01,
9.610526278413467072e-01, 9.612709532083855146e-01, 9.614951659746271417e-01,
9.617252821024303566e-01, 9.619613175974318642e-01, 9.622032885086644338e-01,
9.624512109286739170e-01, 9.627051009936374859e-01, 9.629649748834822054e-01,
9.632308488220031606e-01, 9.635027390769824729e-01, 9.637806619603088709e-01,
9.640646338280971506e-01, 9.643546710808080791e-01, 9.646507901633681881e-01,
9.649530075652912320e-01, 9.652613398207983142e-01, 9.655758035089392344e-01,
9.658964152537145020e-01, 9.662231917241966839e-01, 9.665561496346526393e-01,
9.668953057446663113e-01, 9.672406768592617388e-01, 9.675922798290256255e-01,
9.679501315502314629e-01, 9.683142489649629869e-01, 9.686846490612389671e-01,
9.690613488731369962e-01, 9.694443654809188349e-01, 9.698337160111555333e-01,
9.702294176368531087e-01, 9.706314875775782225e-01, 9.710399430995845238e-01,
9.714548015159391037e-01, 9.718760801866497268e-01, 9.723037965187919518e-01,
9.727379679666363632e-01, 9.731786120317773570e-01, 9.736257462632605941e-01,
9.740793882577122309e-01, 9.745395556594674824e-01, 9.750062661607005188e-01,
9.754795375015535841e-01, 9.759593874702675587e-01, 9.764458339033119660e-01,
9.769388946855159794e-01, 9.774385877501994280e-01, 9.779449310793042471e-01,
9.784579427035267063e-01, 9.789776407024486371e-01, 9.795040432046712153e-01,
9.800371683879468554e-01, 9.805770344793129922e-01, 9.811236597552254191e-01,
9.816770625416927354e-01, 9.822372612144102400e-01, 9.828042741988944897e-01,
9.833781199706193021e-01, 9.839588170551499813e-01, 9.845463840282800971e-01,
9.851408395161672660e-01, 9.857422021954695968e-01, 9.863504907934828037e-01,
9.869657240882776517e-01, 9.875879209088370692e-01, 9.882171001351949258e-01,
9.888532806985737000e-01, 9.894964815815237014e-01, 9.901467218180625141e-01,
9.908040204938135531e-01, 9.914683967461471736e-01, 9.921398697643202258e-01,
9.928184587896166091e-01, 9.935041831154891590e-01, 9.941970620877000897e-01,
9.948971151044636585e-01, 9.956043616165879406e-01, 9.963188211276171602e-01,
9.970405131939754639e-01, 9.977694574251096959e-01, 9.985056734836331715e-01,
9.992491810854701173e-01
};

static const double
	K256ONLN2 = 369.3299304675746271,
	KA2 = 3.66556671660783833261e-06,
	KA1 = 2.70760782821392980564e-03,
	KA0 = 1.0;

static const float extreme[2] = { 1.0e30f, 1.0e-30f };

#define PROCESS(N)						\
	x##N *= K256ONLN2;					\
	k##N = (int) x##N;					\
	x##N -= (double) k##N;					\
	x##N = (KA2 * x##N + KA1) * x##N + KA0;			\
	lres##N = ((long long *)__TBL_exp2f)[k##N & 0xff];	\
	lres##N += (long long)k##N << 44;			\
	*y = (float) (x##N * *(double *)&lres##N);		\
	y += stridey

#ifdef __sparc

#define PREPROCESS(N , index, label)				\
	xi = *(int *)x;						\
	ax = xi & ~0x80000000;					\
	fx = *x;						\
	x += stridex;						\
	if ( ax >= 0x42aeac50 )	/* log(2^126) = 87.3365... */	\
	{							\
		sign = (unsigned)xi >> 31;			\
		if ( ax >= 0x7f800000 )	/* |x| = inf or nan */	\
		{						\
			if ( ax > 0x7f800000 )	/* nan */	\
			{					\
				y[index] = fx * fx;		\
				goto label;			\
			}					\
			y[index] = (sign) ? 0.0f : fx;		\
			goto label;				\
		}						\
		if ( sign || ax > 0x42b17218 ) {		\
			fx = extreme[sign];			\
			y[index] = fx * fx;			\
			goto label;				\
		}						\
	}							\
	x##N = fx

#else

#define PREPROCESS(N , index, label)				\
	xi = *(int *)x;						\
	ax = xi & ~0x80000000;					\
	fx = *x;						\
	x += stridex;						\
	if ( ax > 0x42cff1b4 )	/* 103.972076f */		\
	{							\
		sign = (unsigned)xi >> 31;			\
		if ( ax >= 0x7f800000 )	/* |x| = inf or nan */	\
		{						\
			if ( ax > 0x7f800000 )	/* nan */	\
			{					\
				y[index] = fx * fx;		\
				goto label;			\
			}					\
			y[index] = (sign) ? 0.0f : fx;		\
			goto label;				\
		}						\
		fx = extreme[sign];				\
		y[index] = fx * fx;				\
		goto label;					\
	}							\
	x##N = fx

#endif

void
__vexpf( int n, float * restrict x, int stridex, float * restrict y,
	int stridey )
{
	double		x0, x1, x2, x3, x4;
	double		res0, res1, res2, res3, res4;
	float		fx;
	long long	lres0, lres1, lres2, lres3, lres4;
	int		k0, k1, k2, k3, k4;
	int		xi, ax, sign;

	y -= stridey;

	for ( ; ; )
	{
begin:
		if ( --n < 0 )
			break;
		y += stridey;

		PREPROCESS(0, 0, begin);

		if ( --n < 0 )
			goto process1;

		PREPROCESS(1, stridey, process1);

		if ( --n < 0 )
			goto process2;

		PREPROCESS(2, stridey << 1, process2);

		if ( --n < 0 )
			goto process3;

		PREPROCESS(3, (stridey << 1) + stridey, process3);

		if ( --n < 0 )
			goto process4;

		PREPROCESS(4, (stridey << 2), process4);

		x0 *= K256ONLN2;
		x1 *= K256ONLN2;
		x2 *= K256ONLN2;
		x3 *= K256ONLN2;
		x4 *= K256ONLN2;

		k0 = (int)x0;
		k1 = (int)x1;
		k2 = (int)x2;
		k3 = (int)x3;
		k4 = (int)x4;

		x0 -= (double)k0;
		x1 -= (double)k1;
		x2 -= (double)k2;
		x3 -= (double)k3;
		x4 -= (double)k4;

		x0 = (KA2 * x0 + KA1) * x0 + KA0;
		x1 = (KA2 * x1 + KA1) * x1 + KA0;
		x2 = (KA2 * x2 + KA1) * x2 + KA0;
		x3 = (KA2 * x3 + KA1) * x3 + KA0;
		x4 = (KA2 * x4 + KA1) * x4 + KA0;

		lres0 = ((long long *)__TBL_exp2f)[k0 & 255];
		lres1 = ((long long *)__TBL_exp2f)[k1 & 255];
		lres2 = ((long long *)__TBL_exp2f)[k2 & 255];
		lres3 = ((long long *)__TBL_exp2f)[k3 & 255];
		lres4 = ((long long *)__TBL_exp2f)[k4 & 255];

		lres0 += (long long)k0 << 44;
		res0 = *(double *)&lres0;
		lres1 += (long long)k1 << 44;
		res1 = *(double *)&lres1;
		lres2 += (long long)k2 << 44;
		res2 = *(double *)&lres2;
		lres3 += (long long)k3 << 44;
		res3 = *(double *)&lres3;
		lres4 += (long long)k4 << 44;
		res4 = *(double *)&lres4;

		*y = (float)(res0 * x0);
		y += stridey;
		*y = (float)(res1 * x1);
		y += stridey;
		*y = (float)(res2 * x2);
		y += stridey;
		*y = (float)(res3 * x3);
		y += stridey;
		*y = (float)(res4 * x4);
		continue;

process1:
		PROCESS(0);
		continue;

process2:
		PROCESS(0);
		PROCESS(1);
		continue;

process3:
		PROCESS(0);
		PROCESS(1);
		PROCESS(2);
		continue;

process4:
		PROCESS(0);
		PROCESS(1);
		PROCESS(2);
		PROCESS(3);
	}
}
