﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisvideo/KinesisVideoRequest.h>
#include <aws/kinesisvideo/KinesisVideo_EXPORTS.h>

#include <utility>

namespace Aws {
namespace KinesisVideo {
namespace Model {

/**
 */
class DeleteSignalingChannelRequest : public KinesisVideoRequest {
 public:
  AWS_KINESISVIDEO_API DeleteSignalingChannelRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteSignalingChannel"; }

  AWS_KINESISVIDEO_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the signaling channel that you want to
   * delete.</p>
   */
  inline const Aws::String& GetChannelARN() const { return m_channelARN; }
  inline bool ChannelARNHasBeenSet() const { return m_channelARNHasBeenSet; }
  template <typename ChannelARNT = Aws::String>
  void SetChannelARN(ChannelARNT&& value) {
    m_channelARNHasBeenSet = true;
    m_channelARN = std::forward<ChannelARNT>(value);
  }
  template <typename ChannelARNT = Aws::String>
  DeleteSignalingChannelRequest& WithChannelARN(ChannelARNT&& value) {
    SetChannelARN(std::forward<ChannelARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current version of the signaling channel that you want to delete. You can
   * obtain the current version by invoking the <code>DescribeSignalingChannel</code>
   * or <code>ListSignalingChannels</code> API operations.</p>
   */
  inline const Aws::String& GetCurrentVersion() const { return m_currentVersion; }
  inline bool CurrentVersionHasBeenSet() const { return m_currentVersionHasBeenSet; }
  template <typename CurrentVersionT = Aws::String>
  void SetCurrentVersion(CurrentVersionT&& value) {
    m_currentVersionHasBeenSet = true;
    m_currentVersion = std::forward<CurrentVersionT>(value);
  }
  template <typename CurrentVersionT = Aws::String>
  DeleteSignalingChannelRequest& WithCurrentVersion(CurrentVersionT&& value) {
    SetCurrentVersion(std::forward<CurrentVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_channelARN;

  Aws::String m_currentVersion;
  bool m_channelARNHasBeenSet = false;
  bool m_currentVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisVideo
}  // namespace Aws
