﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ecr-public/ECRPublic_EXPORTS.h>
#include <aws/ecr-public/model/ImageFailureCode.h>
#include <aws/ecr-public/model/ImageIdentifier.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ECRPublic {
namespace Model {

/**
 * <p>An object that represents an Amazon ECR image failure.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/ImageFailure">AWS
 * API Reference</a></p>
 */
class ImageFailure {
 public:
  AWS_ECRPUBLIC_API ImageFailure() = default;
  AWS_ECRPUBLIC_API ImageFailure(Aws::Utils::Json::JsonView jsonValue);
  AWS_ECRPUBLIC_API ImageFailure& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_ECRPUBLIC_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The image ID that's associated with the failure.</p>
   */
  inline const ImageIdentifier& GetImageId() const { return m_imageId; }
  inline bool ImageIdHasBeenSet() const { return m_imageIdHasBeenSet; }
  template <typename ImageIdT = ImageIdentifier>
  void SetImageId(ImageIdT&& value) {
    m_imageIdHasBeenSet = true;
    m_imageId = std::forward<ImageIdT>(value);
  }
  template <typename ImageIdT = ImageIdentifier>
  ImageFailure& WithImageId(ImageIdT&& value) {
    SetImageId(std::forward<ImageIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The code that's associated with the failure.</p>
   */
  inline ImageFailureCode GetFailureCode() const { return m_failureCode; }
  inline bool FailureCodeHasBeenSet() const { return m_failureCodeHasBeenSet; }
  inline void SetFailureCode(ImageFailureCode value) {
    m_failureCodeHasBeenSet = true;
    m_failureCode = value;
  }
  inline ImageFailure& WithFailureCode(ImageFailureCode value) {
    SetFailureCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The reason for the failure.</p>
   */
  inline const Aws::String& GetFailureReason() const { return m_failureReason; }
  inline bool FailureReasonHasBeenSet() const { return m_failureReasonHasBeenSet; }
  template <typename FailureReasonT = Aws::String>
  void SetFailureReason(FailureReasonT&& value) {
    m_failureReasonHasBeenSet = true;
    m_failureReason = std::forward<FailureReasonT>(value);
  }
  template <typename FailureReasonT = Aws::String>
  ImageFailure& WithFailureReason(FailureReasonT&& value) {
    SetFailureReason(std::forward<FailureReasonT>(value));
    return *this;
  }
  ///@}
 private:
  ImageIdentifier m_imageId;

  ImageFailureCode m_failureCode{ImageFailureCode::NOT_SET};

  Aws::String m_failureReason;
  bool m_imageIdHasBeenSet = false;
  bool m_failureCodeHasBeenSet = false;
  bool m_failureReasonHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECRPublic
}  // namespace Aws
