﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/autoscaling/AutoScaling_EXPORTS.h>
#include <aws/autoscaling/model/Metric.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace AutoScaling {
namespace Model {

/**
 * <p>This structure defines the CloudWatch metric to return, along with the
 * statistic and unit.</p> <p>For more information about the CloudWatch terminology
 * below, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html">Amazon
 * CloudWatch concepts</a> in the <i>Amazon CloudWatch User
 * Guide</i>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/TargetTrackingMetricStat">AWS
 * API Reference</a></p>
 */
class TargetTrackingMetricStat {
 public:
  AWS_AUTOSCALING_API TargetTrackingMetricStat() = default;
  AWS_AUTOSCALING_API TargetTrackingMetricStat(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_AUTOSCALING_API TargetTrackingMetricStat& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_AUTOSCALING_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_AUTOSCALING_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The metric to use.</p>
   */
  inline const Metric& GetMetric() const { return m_metric; }
  inline bool MetricHasBeenSet() const { return m_metricHasBeenSet; }
  template <typename MetricT = Metric>
  void SetMetric(MetricT&& value) {
    m_metricHasBeenSet = true;
    m_metric = std::forward<MetricT>(value);
  }
  template <typename MetricT = Metric>
  TargetTrackingMetricStat& WithMetric(MetricT&& value) {
    SetMetric(std::forward<MetricT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The statistic to return. It can include any CloudWatch statistic or extended
   * statistic. For a list of valid values, see the table in <a
   * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Statistic">Statistics</a>
   * in the <i>Amazon CloudWatch User Guide</i>.</p> <p>The most commonly used metric
   * for scaling is <code>Average</code>.</p>
   */
  inline const Aws::String& GetStat() const { return m_stat; }
  inline bool StatHasBeenSet() const { return m_statHasBeenSet; }
  template <typename StatT = Aws::String>
  void SetStat(StatT&& value) {
    m_statHasBeenSet = true;
    m_stat = std::forward<StatT>(value);
  }
  template <typename StatT = Aws::String>
  TargetTrackingMetricStat& WithStat(StatT&& value) {
    SetStat(std::forward<StatT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unit to use for the returned data points. For a complete list of the
   * units that CloudWatch supports, see the <a
   * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html">MetricDatum</a>
   * data type in the <i>Amazon CloudWatch API Reference</i>.</p>
   */
  inline const Aws::String& GetUnit() const { return m_unit; }
  inline bool UnitHasBeenSet() const { return m_unitHasBeenSet; }
  template <typename UnitT = Aws::String>
  void SetUnit(UnitT&& value) {
    m_unitHasBeenSet = true;
    m_unit = std::forward<UnitT>(value);
  }
  template <typename UnitT = Aws::String>
  TargetTrackingMetricStat& WithUnit(UnitT&& value) {
    SetUnit(std::forward<UnitT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The period of the metric in seconds. The default value is 60. Accepted
   * values are 10, 30, and 60. For high resolution metric, set the value to less
   * than 60. For more information, see <a
   * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/policy-creating-high-resolution-metrics.html">Create
   * a target tracking policy using high-resolution metrics for faster response</a>.
   * </p>
   */
  inline int GetPeriod() const { return m_period; }
  inline bool PeriodHasBeenSet() const { return m_periodHasBeenSet; }
  inline void SetPeriod(int value) {
    m_periodHasBeenSet = true;
    m_period = value;
  }
  inline TargetTrackingMetricStat& WithPeriod(int value) {
    SetPeriod(value);
    return *this;
  }
  ///@}
 private:
  Metric m_metric;
  bool m_metricHasBeenSet = false;

  Aws::String m_stat;
  bool m_statHasBeenSet = false;

  Aws::String m_unit;
  bool m_unitHasBeenSet = false;

  int m_period{0};
  bool m_periodHasBeenSet = false;
};

}  // namespace Model
}  // namespace AutoScaling
}  // namespace Aws
