﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workspaces-web/WorkSpacesWebRequest.h>
#include <aws/workspaces-web/WorkSpacesWeb_EXPORTS.h>
#include <aws/workspaces-web/model/AuthenticationType.h>
#include <aws/workspaces-web/model/InstanceType.h>

#include <utility>

namespace Aws {
namespace WorkSpacesWeb {
namespace Model {

/**
 */
class UpdatePortalRequest : public WorkSpacesWebRequest {
 public:
  AWS_WORKSPACESWEB_API UpdatePortalRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdatePortal"; }

  AWS_WORKSPACESWEB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ARN of the web portal.</p>
   */
  inline const Aws::String& GetPortalArn() const { return m_portalArn; }
  inline bool PortalArnHasBeenSet() const { return m_portalArnHasBeenSet; }
  template <typename PortalArnT = Aws::String>
  void SetPortalArn(PortalArnT&& value) {
    m_portalArnHasBeenSet = true;
    m_portalArn = std::forward<PortalArnT>(value);
  }
  template <typename PortalArnT = Aws::String>
  UpdatePortalRequest& WithPortalArn(PortalArnT&& value) {
    SetPortalArn(std::forward<PortalArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the web portal. This is not visible to users who log into the web
   * portal.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  UpdatePortalRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of authentication integration points used when signing into the web
   * portal. Defaults to <code>Standard</code>.</p> <p> <code>Standard</code> web
   * portals are authenticated directly through your identity provider. You need to
   * call <code>CreateIdentityProvider</code> to integrate your identity provider
   * with your web portal. User and group access to your web portal is controlled
   * through your identity provider.</p> <p> <code>IAM Identity Center</code> web
   * portals are authenticated through IAM Identity Center. Identity sources
   * (including external identity provider integration), plus user and group access
   * to your web portal, can be configured in the IAM Identity Center.</p>
   */
  inline AuthenticationType GetAuthenticationType() const { return m_authenticationType; }
  inline bool AuthenticationTypeHasBeenSet() const { return m_authenticationTypeHasBeenSet; }
  inline void SetAuthenticationType(AuthenticationType value) {
    m_authenticationTypeHasBeenSet = true;
    m_authenticationType = value;
  }
  inline UpdatePortalRequest& WithAuthenticationType(AuthenticationType value) {
    SetAuthenticationType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type and resources of the underlying instance.</p>
   */
  inline InstanceType GetInstanceType() const { return m_instanceType; }
  inline bool InstanceTypeHasBeenSet() const { return m_instanceTypeHasBeenSet; }
  inline void SetInstanceType(InstanceType value) {
    m_instanceTypeHasBeenSet = true;
    m_instanceType = value;
  }
  inline UpdatePortalRequest& WithInstanceType(InstanceType value) {
    SetInstanceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of concurrent sessions for the portal.</p>
   */
  inline int GetMaxConcurrentSessions() const { return m_maxConcurrentSessions; }
  inline bool MaxConcurrentSessionsHasBeenSet() const { return m_maxConcurrentSessionsHasBeenSet; }
  inline void SetMaxConcurrentSessions(int value) {
    m_maxConcurrentSessionsHasBeenSet = true;
    m_maxConcurrentSessions = value;
  }
  inline UpdatePortalRequest& WithMaxConcurrentSessions(int value) {
    SetMaxConcurrentSessions(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The custom domain of the web portal that users access in order to start
   * streaming sessions. </p>
   */
  inline const Aws::String& GetPortalCustomDomain() const { return m_portalCustomDomain; }
  inline bool PortalCustomDomainHasBeenSet() const { return m_portalCustomDomainHasBeenSet; }
  template <typename PortalCustomDomainT = Aws::String>
  void SetPortalCustomDomain(PortalCustomDomainT&& value) {
    m_portalCustomDomainHasBeenSet = true;
    m_portalCustomDomain = std::forward<PortalCustomDomainT>(value);
  }
  template <typename PortalCustomDomainT = Aws::String>
  UpdatePortalRequest& WithPortalCustomDomain(PortalCustomDomainT&& value) {
    SetPortalCustomDomain(std::forward<PortalCustomDomainT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_portalArn;

  Aws::String m_displayName;

  AuthenticationType m_authenticationType{AuthenticationType::NOT_SET};

  InstanceType m_instanceType{InstanceType::NOT_SET};

  int m_maxConcurrentSessions{0};

  Aws::String m_portalCustomDomain;
  bool m_portalArnHasBeenSet = false;
  bool m_displayNameHasBeenSet = false;
  bool m_authenticationTypeHasBeenSet = false;
  bool m_instanceTypeHasBeenSet = false;
  bool m_maxConcurrentSessionsHasBeenSet = false;
  bool m_portalCustomDomainHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkSpacesWeb
}  // namespace Aws
