﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudformation/CloudFormation_EXPORTS.h>
#include <aws/cloudformation/model/RegistryType.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace CloudFormation {
namespace Model {

/**
 * <p>Contains summary information about a specific version of a CloudFormation
 * extension.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cloudformation-2010-05-15/TypeVersionSummary">AWS
 * API Reference</a></p>
 */
class TypeVersionSummary {
 public:
  AWS_CLOUDFORMATION_API TypeVersionSummary() = default;
  AWS_CLOUDFORMATION_API TypeVersionSummary(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_CLOUDFORMATION_API TypeVersionSummary& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_CLOUDFORMATION_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_CLOUDFORMATION_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The kind of extension.</p>
   */
  inline RegistryType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(RegistryType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline TypeVersionSummary& WithType(RegistryType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the extension.</p>
   */
  inline const Aws::String& GetTypeName() const { return m_typeName; }
  inline bool TypeNameHasBeenSet() const { return m_typeNameHasBeenSet; }
  template <typename TypeNameT = Aws::String>
  void SetTypeName(TypeNameT&& value) {
    m_typeNameHasBeenSet = true;
    m_typeName = std::forward<TypeNameT>(value);
  }
  template <typename TypeNameT = Aws::String>
  TypeVersionSummary& WithTypeName(TypeNameT&& value) {
    SetTypeName(std::forward<TypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a specific version of the extension. The version ID is the value at
   * the end of the ARN assigned to the extension version when it's registered.</p>
   */
  inline const Aws::String& GetVersionId() const { return m_versionId; }
  inline bool VersionIdHasBeenSet() const { return m_versionIdHasBeenSet; }
  template <typename VersionIdT = Aws::String>
  void SetVersionId(VersionIdT&& value) {
    m_versionIdHasBeenSet = true;
    m_versionId = std::forward<VersionIdT>(value);
  }
  template <typename VersionIdT = Aws::String>
  TypeVersionSummary& WithVersionId(VersionIdT&& value) {
    SetVersionId(std::forward<VersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether the specified extension version is set as the default version.</p>
   * <p>This applies only to private extensions you have registered in your account,
   * and extensions published by Amazon. For public third-party extensions,
   * CloudFormation returns <code>null</code>.</p>
   */
  inline bool GetIsDefaultVersion() const { return m_isDefaultVersion; }
  inline bool IsDefaultVersionHasBeenSet() const { return m_isDefaultVersionHasBeenSet; }
  inline void SetIsDefaultVersion(bool value) {
    m_isDefaultVersionHasBeenSet = true;
    m_isDefaultVersion = value;
  }
  inline TypeVersionSummary& WithIsDefaultVersion(bool value) {
    SetIsDefaultVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the extension version.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  TypeVersionSummary& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the version was registered.</p>
   */
  inline const Aws::Utils::DateTime& GetTimeCreated() const { return m_timeCreated; }
  inline bool TimeCreatedHasBeenSet() const { return m_timeCreatedHasBeenSet; }
  template <typename TimeCreatedT = Aws::Utils::DateTime>
  void SetTimeCreated(TimeCreatedT&& value) {
    m_timeCreatedHasBeenSet = true;
    m_timeCreated = std::forward<TimeCreatedT>(value);
  }
  template <typename TimeCreatedT = Aws::Utils::DateTime>
  TypeVersionSummary& WithTimeCreated(TimeCreatedT&& value) {
    SetTimeCreated(std::forward<TimeCreatedT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the extension version.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  TypeVersionSummary& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For public extensions that have been activated for this account and Region,
   * the version of the public extension to be used for CloudFormation operations in
   * this account and Region. For any extensions other than activated third-party
   * extensions, CloudFormation returns <code>null</code>.</p> <p>How you specified
   * <code>AutoUpdate</code> when enabling the extension affects whether
   * CloudFormation automatically updates the extension in this account and Region
   * when a new version is released. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/registry-public.html#registry-public-enable-auto">Automatically
   * use new versions of extensions</a> in the <i>CloudFormation User Guide</i>.</p>
   */
  inline const Aws::String& GetPublicVersionNumber() const { return m_publicVersionNumber; }
  inline bool PublicVersionNumberHasBeenSet() const { return m_publicVersionNumberHasBeenSet; }
  template <typename PublicVersionNumberT = Aws::String>
  void SetPublicVersionNumber(PublicVersionNumberT&& value) {
    m_publicVersionNumberHasBeenSet = true;
    m_publicVersionNumber = std::forward<PublicVersionNumberT>(value);
  }
  template <typename PublicVersionNumberT = Aws::String>
  TypeVersionSummary& WithPublicVersionNumber(PublicVersionNumberT&& value) {
    SetPublicVersionNumber(std::forward<PublicVersionNumberT>(value));
    return *this;
  }
  ///@}
 private:
  RegistryType m_type{RegistryType::NOT_SET};

  Aws::String m_typeName;

  Aws::String m_versionId;

  bool m_isDefaultVersion{false};

  Aws::String m_arn;

  Aws::Utils::DateTime m_timeCreated{};

  Aws::String m_description;

  Aws::String m_publicVersionNumber;
  bool m_typeHasBeenSet = false;
  bool m_typeNameHasBeenSet = false;
  bool m_versionIdHasBeenSet = false;
  bool m_isDefaultVersionHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_timeCreatedHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_publicVersionNumberHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
