/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.runtime;

import java.io.File;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.Set;
import org.apache.nifi.NiFiServer;
import org.apache.nifi.bundle.Bundle;
import org.apache.nifi.nar.ExtensionMapping;
import org.apache.nifi.nar.NarClassLoaders;
import org.apache.nifi.nar.NarClassLoadersHolder;
import org.apache.nifi.nar.NarUnpackMode;
import org.apache.nifi.nar.NarUnpacker;
import org.apache.nifi.nar.SystemBundle;
import org.apache.nifi.runtime.ManagementServer;
import org.apache.nifi.runtime.ManagementServerProvider;
import org.apache.nifi.runtime.PropertiesProvider;
import org.apache.nifi.runtime.StandardUncaughtExceptionHandler;
import org.apache.nifi.runtime.command.DiagnosticsCommand;
import org.apache.nifi.runtime.command.ShutdownCommand;
import org.apache.nifi.util.NiFiProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Application
implements Runnable {
    private static final String SECURITY_KRB5_CONF_PROPERTY = "java.security.krb5.conf";
    private static final Logger logger = LoggerFactory.getLogger(Application.class);
    private static final String VERSION_UNKNOWN = "UNKNOWN";
    private static final String APPLICATION_VERSION = Objects.requireNonNullElse(Application.class.getPackage().getImplementationVersion(), "UNKNOWN");

    @Override
    public void run() {
        logger.info("Starting NiFi {} using Java {} with PID {}", new Object[]{APPLICATION_VERSION, Runtime.version(), ProcessHandle.current().pid()});
        Instant started = Instant.now();
        try {
            this.run(started);
        }
        catch (Throwable e) {
            logger.error("Starting NiFi failed", e);
        }
    }

    private void run(Instant started) {
        NiFiProperties properties = PropertiesProvider.readProperties();
        Bundle systemBundle = this.createSystemBundle(properties);
        NarUnpackMode unpackMode = properties.isUnpackNarsToUberJar() ? NarUnpackMode.UNPACK_TO_UBER_JAR : NarUnpackMode.UNPACK_INDIVIDUAL_JARS;
        ExtensionMapping extensionMapping = NarUnpacker.unpackNars((NiFiProperties)properties, (Bundle)systemBundle, (NarUnpackMode)unpackMode);
        NarClassLoaders narClassLoaders = this.initializeClassLoaders(properties, systemBundle.getClassLoader());
        NiFiServer applicationServer = narClassLoaders.getServer();
        if (applicationServer == null) {
            logger.error("Server implementation of [{}] not found", (Object)NiFiServer.class.getName());
        } else {
            try {
                this.startServer(systemBundle, extensionMapping, narClassLoaders, applicationServer, started);
            }
            catch (Throwable e) {
                logger.error("Start Server failed", e);
                applicationServer.stop();
            }
        }
    }

    private Bundle createSystemBundle(NiFiProperties properties) {
        ClassLoader systemClassLoader = ClassLoader.getSystemClassLoader();
        String narLibraryDirectory = properties.getProperty("nifi.nar.library.directory");
        return SystemBundle.create((String)narLibraryDirectory, (ClassLoader)systemClassLoader);
    }

    private NarClassLoaders initializeClassLoaders(NiFiProperties properties, ClassLoader systemClassLoader) {
        NarClassLoaders narClassLoaders = NarClassLoadersHolder.getInstance();
        File frameworkWorkingDirectory = properties.getFrameworkWorkingDirectory();
        File extensionsWorkingDirectory = properties.getExtensionsWorkingDirectory();
        try {
            narClassLoaders.init(systemClassLoader, frameworkWorkingDirectory, extensionsWorkingDirectory, true);
        }
        catch (Exception e) {
            logger.error("NAR Class Loaders initialization failed", (Throwable)e);
        }
        return narClassLoaders;
    }

    private void startServer(Bundle systemBundle, ExtensionMapping extensionMapping, NarClassLoaders narClassLoaders, NiFiServer applicationServer, Instant started) {
        ClassLoader applicationServerClassLoader = narClassLoaders.getServer().getClass().getClassLoader();
        Thread.currentThread().setContextClassLoader(applicationServerClassLoader);
        Bundle frameworkBundle = narClassLoaders.getFrameworkBundle();
        ClassLoader frameworkClassLoader = frameworkBundle.getClassLoader();
        NiFiProperties properties = PropertiesProvider.readProperties(frameworkClassLoader);
        this.setKerberosConfiguration(properties);
        Set narBundles = narClassLoaders.getBundles();
        applicationServer.initialize(properties, systemBundle, narBundles, extensionMapping);
        ManagementServer managementServer = ManagementServerProvider.getManagementServer(applicationServer);
        applicationServer.start();
        managementServer.start();
        DiagnosticsCommand diagnosticsCommand = new DiagnosticsCommand(properties, applicationServer);
        ShutdownCommand shutdownCommand = new ShutdownCommand(applicationServer, managementServer, diagnosticsCommand);
        this.addShutdownHook(shutdownCommand);
        Instant completed = Instant.now();
        Duration duration = Duration.between(started, completed);
        double durationSeconds = (double)duration.toMillis() / 1000.0;
        logger.info("Started Application in {} seconds ({} ns)", (Object)durationSeconds, (Object)duration.toNanos());
    }

    private void addShutdownHook(Runnable shutdownCommand) {
        Thread shutdownHook = Thread.ofPlatform().name(shutdownCommand.getClass().getSimpleName()).uncaughtExceptionHandler(new StandardUncaughtExceptionHandler()).unstarted(shutdownCommand);
        Runtime.getRuntime().addShutdownHook(shutdownHook);
    }

    private void setKerberosConfiguration(NiFiProperties properties) {
        File kerberosConfigFile = properties.getKerberosConfigurationFile();
        if (kerberosConfigFile == null) {
            logger.debug("Application Kerberos Configuration not specified");
        } else {
            String kerberosConfigFilePath = kerberosConfigFile.getAbsolutePath();
            System.setProperty(SECURITY_KRB5_CONF_PROPERTY, kerberosConfigFilePath);
        }
    }
}

