#!/bin/sh

# Copyright (C) 2024-2025 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr"

# Include utilities based on location of this script. Check for sources first,
# so that the unexpected situations with weird paths fall on the default
# case of installed.
script_path=$(cd "$(dirname "${0}")" && pwd)
if test "${script_path}" = "/startdir/src/build/src/share/database/scripts/pgsql"; then
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "/startdir/src/build/src/bin/admin/admin-utils.sh"
else
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "${prefix}/share/kea/scripts/admin-utils.sh"
fi

# Check only major version to allow for intermediary backported schema changes.
version=$(pgsql_version "${@}" | cut -d '.' -f 1)
if test "${version}" != '25'; then
    printf 'This script upgrades 25.* to 26.0. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${version}"
    exit 0
fi

psql "$@" >/dev/null <<EOF
START TRANSACTION;

-- This line starts the schema upgrade to version 26.0.

-- Add client_classes column to options tables for option class taggging.
ALTER TABLE dhcp4_options
    ADD COLUMN client_classes TEXT DEFAULT NULL;

ALTER TABLE dhcp6_options
    ADD COLUMN client_classes TEXT DEFAULT NULL;

UPDATE option_def_data_type SET name='int8' WHERE id = 3;

-- Rename require_client_classes and only_if_required.
ALTER TABLE dhcp4_shared_network
    RENAME COLUMN require_client_classes TO evaluate_additional_classes;

ALTER TABLE dhcp4_subnet
    RENAME COLUMN require_client_classes TO evaluate_additional_classes;

ALTER TABLE dhcp4_pool
    RENAME COLUMN require_client_classes TO evaluate_additional_classes;

ALTER TABLE dhcp6_shared_network
    RENAME COLUMN require_client_classes TO evaluate_additional_classes;

ALTER TABLE dhcp6_subnet
    RENAME COLUMN require_client_classes TO evaluate_additional_classes;

ALTER TABLE dhcp6_pool
    RENAME COLUMN require_client_classes TO evaluate_additional_classes;

ALTER TABLE dhcp6_pd_pool
    RENAME COLUMN require_client_classes TO evaluate_additional_classes;

ALTER TABLE dhcp4_client_class
    RENAME COLUMN only_if_required TO only_in_additional_list;

ALTER TABLE dhcp6_client_class
    RENAME COLUMN only_if_required TO only_in_additional_list;

-- Update the schema version number.
UPDATE schema_version
    SET version = '26', minor = '0';

-- This line concludes the schema upgrade to version 26.0.

-- Commit the script transaction.
COMMIT;

EOF
