﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lookoutequipment/LookoutEquipmentRequest.h>
#include <aws/lookoutequipment/LookoutEquipment_EXPORTS.h>
#include <aws/lookoutequipment/model/IngestionInputConfiguration.h>

#include <utility>

namespace Aws {
namespace LookoutEquipment {
namespace Model {

/**
 */
class StartDataIngestionJobRequest : public LookoutEquipmentRequest {
 public:
  AWS_LOOKOUTEQUIPMENT_API StartDataIngestionJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartDataIngestionJob"; }

  AWS_LOOKOUTEQUIPMENT_API Aws::String SerializePayload() const override;

  AWS_LOOKOUTEQUIPMENT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the dataset being used by the data ingestion job. </p>
   */
  inline const Aws::String& GetDatasetName() const { return m_datasetName; }
  inline bool DatasetNameHasBeenSet() const { return m_datasetNameHasBeenSet; }
  template <typename DatasetNameT = Aws::String>
  void SetDatasetName(DatasetNameT&& value) {
    m_datasetNameHasBeenSet = true;
    m_datasetName = std::forward<DatasetNameT>(value);
  }
  template <typename DatasetNameT = Aws::String>
  StartDataIngestionJobRequest& WithDatasetName(DatasetNameT&& value) {
    SetDatasetName(std::forward<DatasetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Specifies information for the input data for the data ingestion job,
   * including dataset S3 location. </p>
   */
  inline const IngestionInputConfiguration& GetIngestionInputConfiguration() const { return m_ingestionInputConfiguration; }
  inline bool IngestionInputConfigurationHasBeenSet() const { return m_ingestionInputConfigurationHasBeenSet; }
  template <typename IngestionInputConfigurationT = IngestionInputConfiguration>
  void SetIngestionInputConfiguration(IngestionInputConfigurationT&& value) {
    m_ingestionInputConfigurationHasBeenSet = true;
    m_ingestionInputConfiguration = std::forward<IngestionInputConfigurationT>(value);
  }
  template <typename IngestionInputConfigurationT = IngestionInputConfiguration>
  StartDataIngestionJobRequest& WithIngestionInputConfiguration(IngestionInputConfigurationT&& value) {
    SetIngestionInputConfiguration(std::forward<IngestionInputConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The Amazon Resource Name (ARN) of a role with permission to access the data
   * source for the data ingestion job. </p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  StartDataIngestionJobRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A unique identifier for the request. If you do not set the client request
   * token, Amazon Lookout for Equipment generates one. </p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartDataIngestionJobRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_datasetName;
  bool m_datasetNameHasBeenSet = false;

  IngestionInputConfiguration m_ingestionInputConfiguration;
  bool m_ingestionInputConfigurationHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace LookoutEquipment
}  // namespace Aws
