﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/datasync/DataSyncRequest.h>
#include <aws/datasync/DataSync_EXPORTS.h>
#include <aws/datasync/model/FilterRule.h>
#include <aws/datasync/model/ManifestConfig.h>
#include <aws/datasync/model/Options.h>
#include <aws/datasync/model/TaskReportConfig.h>
#include <aws/datasync/model/TaskSchedule.h>

#include <utility>

namespace Aws {
namespace DataSync {
namespace Model {

/**
 * <p>UpdateTaskResponse</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UpdateTaskRequest">AWS
 * API Reference</a></p>
 */
class UpdateTaskRequest : public DataSyncRequest {
 public:
  AWS_DATASYNC_API UpdateTaskRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateTask"; }

  AWS_DATASYNC_API Aws::String SerializePayload() const override;

  AWS_DATASYNC_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the ARN of the task that you want to update.</p>
   */
  inline const Aws::String& GetTaskArn() const { return m_taskArn; }
  inline bool TaskArnHasBeenSet() const { return m_taskArnHasBeenSet; }
  template <typename TaskArnT = Aws::String>
  void SetTaskArn(TaskArnT&& value) {
    m_taskArnHasBeenSet = true;
    m_taskArn = std::forward<TaskArnT>(value);
  }
  template <typename TaskArnT = Aws::String>
  UpdateTaskRequest& WithTaskArn(TaskArnT&& value) {
    SetTaskArn(std::forward<TaskArnT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Options& GetOptions() const { return m_options; }
  inline bool OptionsHasBeenSet() const { return m_optionsHasBeenSet; }
  template <typename OptionsT = Options>
  void SetOptions(OptionsT&& value) {
    m_optionsHasBeenSet = true;
    m_options = std::forward<OptionsT>(value);
  }
  template <typename OptionsT = Options>
  UpdateTaskRequest& WithOptions(OptionsT&& value) {
    SetOptions(std::forward<OptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies exclude filters that define the files, objects, and folders in your
   * source location that you don't want DataSync to transfer. For more information
   * and examples, see <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/filtering.html">Specifying
   * what DataSync transfers by using filters</a>.</p>
   */
  inline const Aws::Vector<FilterRule>& GetExcludes() const { return m_excludes; }
  inline bool ExcludesHasBeenSet() const { return m_excludesHasBeenSet; }
  template <typename ExcludesT = Aws::Vector<FilterRule>>
  void SetExcludes(ExcludesT&& value) {
    m_excludesHasBeenSet = true;
    m_excludes = std::forward<ExcludesT>(value);
  }
  template <typename ExcludesT = Aws::Vector<FilterRule>>
  UpdateTaskRequest& WithExcludes(ExcludesT&& value) {
    SetExcludes(std::forward<ExcludesT>(value));
    return *this;
  }
  template <typename ExcludesT = FilterRule>
  UpdateTaskRequest& AddExcludes(ExcludesT&& value) {
    m_excludesHasBeenSet = true;
    m_excludes.emplace_back(std::forward<ExcludesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a schedule for when you want your task to run. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/task-scheduling.html">Scheduling
   * your task</a>.</p>
   */
  inline const TaskSchedule& GetSchedule() const { return m_schedule; }
  inline bool ScheduleHasBeenSet() const { return m_scheduleHasBeenSet; }
  template <typename ScheduleT = TaskSchedule>
  void SetSchedule(ScheduleT&& value) {
    m_scheduleHasBeenSet = true;
    m_schedule = std::forward<ScheduleT>(value);
  }
  template <typename ScheduleT = TaskSchedule>
  UpdateTaskRequest& WithSchedule(ScheduleT&& value) {
    SetSchedule(std::forward<ScheduleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the name of your task.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateTaskRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the Amazon Resource Name (ARN) of an Amazon CloudWatch log group
   * for monitoring your task.</p> <p>For Enhanced mode tasks, you must use
   * <code>/aws/datasync</code> as your log group name. For example:</p> <p>
   * <code>arn:aws:logs:us-east-1:111222333444:log-group:/aws/datasync:*</code> </p>
   * <p>For more information, see <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/configure-logging.html">Monitoring
   * data transfers with CloudWatch Logs</a>.</p>
   */
  inline const Aws::String& GetCloudWatchLogGroupArn() const { return m_cloudWatchLogGroupArn; }
  inline bool CloudWatchLogGroupArnHasBeenSet() const { return m_cloudWatchLogGroupArnHasBeenSet; }
  template <typename CloudWatchLogGroupArnT = Aws::String>
  void SetCloudWatchLogGroupArn(CloudWatchLogGroupArnT&& value) {
    m_cloudWatchLogGroupArnHasBeenSet = true;
    m_cloudWatchLogGroupArn = std::forward<CloudWatchLogGroupArnT>(value);
  }
  template <typename CloudWatchLogGroupArnT = Aws::String>
  UpdateTaskRequest& WithCloudWatchLogGroupArn(CloudWatchLogGroupArnT&& value) {
    SetCloudWatchLogGroupArn(std::forward<CloudWatchLogGroupArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies include filters define the files, objects, and folders in your
   * source location that you want DataSync to transfer. For more information and
   * examples, see <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/filtering.html">Specifying
   * what DataSync transfers by using filters</a>.</p>
   */
  inline const Aws::Vector<FilterRule>& GetIncludes() const { return m_includes; }
  inline bool IncludesHasBeenSet() const { return m_includesHasBeenSet; }
  template <typename IncludesT = Aws::Vector<FilterRule>>
  void SetIncludes(IncludesT&& value) {
    m_includesHasBeenSet = true;
    m_includes = std::forward<IncludesT>(value);
  }
  template <typename IncludesT = Aws::Vector<FilterRule>>
  UpdateTaskRequest& WithIncludes(IncludesT&& value) {
    SetIncludes(std::forward<IncludesT>(value));
    return *this;
  }
  template <typename IncludesT = FilterRule>
  UpdateTaskRequest& AddIncludes(IncludesT&& value) {
    m_includesHasBeenSet = true;
    m_includes.emplace_back(std::forward<IncludesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configures a manifest, which is a list of files or objects that you want
   * DataSync to transfer. For more information and configuration examples, see <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/transferring-with-manifest.html">Specifying
   * what DataSync transfers by using a manifest</a>.</p> <p>When using this
   * parameter, your caller identity (the IAM role that you're using DataSync with)
   * must have the <code>iam:PassRole</code> permission. The <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/security-iam-awsmanpol.html#security-iam-awsmanpol-awsdatasyncfullaccess">AWSDataSyncFullAccess</a>
   * policy includes this permission.</p> <p>To remove a manifest configuration,
   * specify this parameter as empty.</p>
   */
  inline const ManifestConfig& GetManifestConfig() const { return m_manifestConfig; }
  inline bool ManifestConfigHasBeenSet() const { return m_manifestConfigHasBeenSet; }
  template <typename ManifestConfigT = ManifestConfig>
  void SetManifestConfig(ManifestConfigT&& value) {
    m_manifestConfigHasBeenSet = true;
    m_manifestConfig = std::forward<ManifestConfigT>(value);
  }
  template <typename ManifestConfigT = ManifestConfig>
  UpdateTaskRequest& WithManifestConfig(ManifestConfigT&& value) {
    SetManifestConfig(std::forward<ManifestConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies how you want to configure a task report, which provides detailed
   * information about your DataSync transfer. For more information, see <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/task-reports.html">Monitoring
   * your DataSync transfers with task reports</a>.</p> <p>When using this parameter,
   * your caller identity (the IAM role that you're using DataSync with) must have
   * the <code>iam:PassRole</code> permission. The <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/security-iam-awsmanpol.html#security-iam-awsmanpol-awsdatasyncfullaccess">AWSDataSyncFullAccess</a>
   * policy includes this permission.</p> <p>To remove a task report configuration,
   * specify this parameter as empty.</p>
   */
  inline const TaskReportConfig& GetTaskReportConfig() const { return m_taskReportConfig; }
  inline bool TaskReportConfigHasBeenSet() const { return m_taskReportConfigHasBeenSet; }
  template <typename TaskReportConfigT = TaskReportConfig>
  void SetTaskReportConfig(TaskReportConfigT&& value) {
    m_taskReportConfigHasBeenSet = true;
    m_taskReportConfig = std::forward<TaskReportConfigT>(value);
  }
  template <typename TaskReportConfigT = TaskReportConfig>
  UpdateTaskRequest& WithTaskReportConfig(TaskReportConfigT&& value) {
    SetTaskReportConfig(std::forward<TaskReportConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_taskArn;
  bool m_taskArnHasBeenSet = false;

  Options m_options;
  bool m_optionsHasBeenSet = false;

  Aws::Vector<FilterRule> m_excludes;
  bool m_excludesHasBeenSet = false;

  TaskSchedule m_schedule;
  bool m_scheduleHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_cloudWatchLogGroupArn;
  bool m_cloudWatchLogGroupArnHasBeenSet = false;

  Aws::Vector<FilterRule> m_includes;
  bool m_includesHasBeenSet = false;

  ManifestConfig m_manifestConfig;
  bool m_manifestConfigHasBeenSet = false;

  TaskReportConfig m_taskReportConfig;
  bool m_taskReportConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataSync
}  // namespace Aws
