﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codedeploy/CodeDeployRequest.h>
#include <aws/codedeploy/CodeDeploy_EXPORTS.h>
#include <aws/codedeploy/model/ComputePlatform.h>
#include <aws/codedeploy/model/Tag.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CodeDeploy {
namespace Model {

/**
 * <p>Represents the input of a <code>CreateApplication</code>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/codedeploy-2014-10-06/CreateApplicationInput">AWS
 * API Reference</a></p>
 */
class CreateApplicationRequest : public CodeDeployRequest {
 public:
  AWS_CODEDEPLOY_API CreateApplicationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateApplication"; }

  AWS_CODEDEPLOY_API Aws::String SerializePayload() const override;

  AWS_CODEDEPLOY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the application. This name must be unique with the applicable
   * user or Amazon Web Services account.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  CreateApplicationRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The destination platform type for the deployment (<code>Lambda</code>,
   * <code>Server</code>, or <code>ECS</code>).</p>
   */
  inline ComputePlatform GetComputePlatform() const { return m_computePlatform; }
  inline bool ComputePlatformHasBeenSet() const { return m_computePlatformHasBeenSet; }
  inline void SetComputePlatform(ComputePlatform value) {
    m_computePlatformHasBeenSet = true;
    m_computePlatform = value;
  }
  inline CreateApplicationRequest& WithComputePlatform(ComputePlatform value) {
    SetComputePlatform(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The metadata that you apply to CodeDeploy applications to help you organize
   * and categorize them. Each tag consists of a key and an optional value, both of
   * which you define. </p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateApplicationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateApplicationRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;

  ComputePlatform m_computePlatform{ComputePlatform::NOT_SET};

  Aws::Vector<Tag> m_tags;
  bool m_applicationNameHasBeenSet = false;
  bool m_computePlatformHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeDeploy
}  // namespace Aws
