<?php

/*
 * PHP.updateDNS (OPNsense version)
 *
 * +====================================================+
 *  Services Supported:
 *    - DynDns (dyndns.org) [dynamic, static, custom]
 *    - DHSDns (dhs.org)
 *    - No-IP (no-ip.com)
 *    - EasyDNS (easydns.com)
 *    - DHS (www.dhs.org)
 *    - HN (hn.org) -- incomplete checking!
 *    - DynS (dyns.org)
 *    - ZoneEdit (zoneedit.com)
 *    - FreeDNS (freedns.afraid.org)
 *    - Loopia (loopia.se)
 *    - StaticCling (staticcling.org)
 *    - DNSexit (dnsexit.com)
 *    - Namecheap (namecheap.com)
 *    - HE.net (dns.he.net)
 *    - HE.net IPv6 (dns.he.net)
 *    - HE.net Tunnelbroker IP update (ipv4.tunnelbroker.net)
 *    - SelfHost (selfhost.de)
 *    - Amazon Route53 (aws.amazon.com)
 *    - DNS-O-Matic (dnsomatic.com)
 *    - Custom dynamic DNS (any URL)
 *    - Custom dynamic DNS IPv6 (any URL)
 *    - Cloudflare (www.cloudflare.com)
 *    - Cloudflare IPv6 (www.cloudflare.com)
 *    - Cloudflare API token (www.cloudflare.com)
 *    - Cloudflare API token IPv6 (www.cloudflare.com)
 *    - Eurodns (eurodns.com)
 *    - GratisDNS (gratisdns.dk)
 *    - City Network (citynetwork.se)
 *    - Duck DNS (duckdns.org)
 *    - Google Domains (domains.google.com)
 *    - STRATO (strato.com)
 *    - 3322 (3322.net)
 *    - Oray (oray.com)
 *    - regfish (regfish.de)
 *    - regfish IPv6 (regfish.de)
 *    - dynv6 IPv6 (dynv6.com)
 *    - DigitalOcean (digitalocean.com)
 *    - Gandi LiveDNS (gandi.net)
 *    - Azure DNS (azure.microsoft.com)
 *    - Linode (linode.com)
 *    - Linode IPv6 (linode.com)
 *    - GoDaddy (godaddy.com)
 *    - GoDaddy IPv6 (godaddy.com)
 * +----------------------------------------------------+
 *  Requirements:
 *    - PHP version 4.0.2 or higher with the CURL Library and the PCRE Library
 * +----------------------------------------------------+
 *  Public Functions
 *    - updatedns()
 *
 *  Private Functions
 *    - _update()
 *    - _checkStatus()
 *    - _error()
 *    - _detectChange()
 *    - _debug()
 *    - _checkIP()
 * +----------------------------------------------------+
 *  DynDNS Dynamic              - Last Tested: 12 July 2005
 *  DynDNS Static               - Last Tested: NEVER
 *  DynDNS Custom               - Last Tested: NEVER
 *  No-IP                       - Last Tested: 20 July 2008
 *  HN.org                      - Last Tested: 12 July 2005
 *  EasyDNS                     - Last Tested: 20 July 2008
 *  DHS                         - Last Tested: 12 July 2005
 *  ZoneEdit                    - Last Tested: NEVER
 *  Dyns                        - Last Tested: NEVER
 *  ODS                         - Last Tested: 02 August 2005
 *  FreeDNS                     - Last Tested: 23 Feb 2011
 *  Loopia                      - Last Tested: NEVER
 *  StaticCling                 - Last Tested: 27 April 2006
 *  DNSexit                     - Last Tested: 20 July 2008
 *  Namecheap                   - Last Tested: 31 August 2010
 *  HE.net                      - Last Tested: 7 July 2013
 *  HE.net IPv6                 - Last Tested: 7 July 2013
 *  HE.net Tunnel               - Last Tested: 28 June 2011
 *  SelfHost                    - Last Tested: 26 December 2011
 *  Amazon Route53              - Last Tested: 01 April 2012
 *  DNS-O-Matic                 - Last Tested: 9 September 2010
 *  Cloudflare                  - Last Tested: 16 April 2019
 *  Cloudflare IPv6             - Last Tested: 16 April 2019
 *  Cloudflare w/API token      - Last Tested: 13 June 2020
 *  Cloudflare w/API token v6   - Last Tested: NEVER
 *  Eurodns                     - Last Tested: 25 July 2018
 *  GratisDNS                   - Last Tested: 26 January 2020
 *  OVH DynHOST                 - Last Tested: NEVER
 *  City Network                - Last Tested: 13 November 2013
 *  Duck DNS                    - Last Tested: 04 March 2015
 *  Google Domains              - Last Tested: 20 February 2017
 *  STRATO                      - Last Tested: 09 May 2017
 *  3322                        - Last Tested: 26 May 2017
 *  Oray                        - Last Tested: 26 May 2017
 *  regfish                     - Last Tested: 15 August 2017
 *  regfish v6                  - Last Tested: 15 August 2017
 *  Amazon Route53 v6           - Last Tested: 19 November 2017
 *  dynv6                       - Last Tested: 25 June 2019
 *  dynv6 v6                    - Last Tested: 25 June 2019
 *  DigitalOcean                - Last Tested: 25 June 2019
 *  Azure DNS                   - Last Tested: 16 October 2019
 *  Linode                      - Last Tested: 25 February 2020
 *  Linode v6                   - Last Tested: 25 February 2020
 *  GoDaddy                     - Last Tested: 10 July 2020
 *  GoDaddy v6                  - Last Tested: 10 July 2020
 *  Gandi LiveDNS               - Last Tested: 24 August 2020
 * +====================================================+
 *
 * @author    E.Kristensen
 * @link    http://www.idylldesigns.com/projects/phpdns/
 * @version    0.8
 * @updated    13 October 05 at 21:02:42 GMT
 *
 * DNSexit support and multiwan extension for pfSense by Ermal Luçi
 * Custom DNS support by Matt Corallo
 */

class updatedns
{
    var $_cacheFile;
    var $_cacheFile_v6;
    var $_debugFile;
    var $_UserAgent = 'User-Agent: phpDynDNS/0.7';
    var $_errorVerbosity = 0;
    var $_dnsService;
    var $_dnsUser;
    var $_dnsPass;
    var $_dnsHost;
    var $_dnsIP;
    var $_dnsWildcard;
    var $_dnsMX;
    var $_dnsBackMX;
    var $_dnsServer;
    var $_dnsPort;
    var $_dnsUpdateURL;
    var $_dnsZoneID;
    var $_dnsResourceID;
    var $_dnsTTL;
    var $status;
    var $_debugID;
    var $_if;
    var $_dnsResultMatch;
    var $_dnsRequestIf;
    var $_dnsRequestIfIP;
    var $_dnsVerboseLog;
    var $_curlIpresolveV4;
    var $_curlSslVerifypeer;
    var $_dnsMaxCacheAgeDays;
    var $_dnsDummyUpdateDone;
    var $_forceUpdateNeeded;
    var $_useIPv6;

    /*
     * Public Constructor Function (added 12 July 05) [beta]
     *   - Gets the dice rolling for the update.
     *   - $dnsResultMatch should only be used with $dnsService = 'custom'
     *   -  $dnsResultMatch is parsed for '%IP%', which is the IP the provider was updated to,
     *   -  it is otherwise expected to be exactly identical to what is returned by the Provider.
     *   - $dnsUser, and $dnsPass indicate HTTP Auth for custom DNS, if they are needed in the URL (GET Variables), include them in $dnsUpdateURL.
     *   - $For custom requests, $dnsUpdateURL is parsed for '%IP%', which is replaced with the new IP.
     */
    public function __construct(
        $dnsService = '',
        $dnsHost = '',
        $dnsUser = '',
        $dnsPass = '',
        $dnsWildcard = 'OFF',
        $dnsMX = '',
        $dnsIf = '',
        $dnsBackMX = '',
        $dnsServer = '',
        $dnsPort = '',
        $dnsUpdateURL = '',
        $forceUpdate = false,
        $dnsZoneID = '',
        $dnsResourceID = '',
        $dnsTTL = '',
        $dnsResultMatch = '',
        $dnsRequestIf = '',
        $dnsID = '',
        $dnsVerboseLog = false,
        $curlIpresolveV4 = false,
        $curlSslVerifypeer = true
    ) {

        /* XXX because the call stack is upside down we need to reassemble config parts here... */
        $conf = array('host' => $dnsHost, 'id' => $dnsID, 'interface' => $dnsIf);
        $this->_cacheFile = dyndns_cache_file($conf, 4);
        $this->_cacheFile_v6 = dyndns_cache_file($conf, 6);
        $this->_debugFile = dyndns_cache_file($conf, 4) . '.debug';
        $this->_dnsServiceList = dyndns_list();

        $this->_curlIpresolveV4 = $curlIpresolveV4;
        $this->_curlSslVerifypeer = $curlSslVerifypeer;
        $this->_dnsVerboseLog = $dnsVerboseLog;
        if ($this->_dnsVerboseLog) {
            log_error("Dynamic DNS: updatedns() starting");
        }

        $dyndnslck = lock("DDNS" . $dnsID, LOCK_EX);

        if (!$dnsService) {
            $this->_error(2);
        }
        switch ($dnsService) {
            case 'freedns':
                if (!$dnsHost) {
                    $this->_error(5);
                }
                break;
            case 'linode':
            case 'linode-v6':
            case 'namecheap':
                if (!$dnsPass) {
                    $this->_error(4);
                } elseif (!$dnsHost) {
                    $this->_error(5);
                }
                break;
            case 'route53':
            case 'route53-v6':
                if (!$dnsZoneID) {
                    $this->_error(8);
                } elseif (!$dnsTTL) {
                    $this->_error(9);
                }
                break;
            case 'custom':
            case 'custom-v6':
                if (!$dnsUpdateURL) {
                    $this->_error(7);
                }
                break;
            case 'duckdns':
            case 'dynv6':
            case 'dynv6-v6':
            case 'regfish':
            case 'regfish-v6':
                if (!$dnsUser) {
                    $this->_error(3);
                } elseif (!$dnsHost) {
                    $this->_error(5);
                }
                break;
            case 'azure':
            case 'azurev6':
                if (!$dnsUser) {
                    $this->_error(3);
                } elseif (!$dnsPass) {
                    $this->_error(4);
                } elseif (!$dnsHost) {
                    $this->_error(5);
                } elseif (!$dnsResourceID) {
                    $this->_error(8);
                } elseif (!$dnsTTL) {
                    $this->_error(9);
                }
                break;
            case 'cloudflare-token':
            case 'cloudflare-token-v6':
                if (!$dnsPass) {
                    $this->_error(4);
                } elseif (!$dnsHost) {
                    $this->_error(5);
                } elseif (!$dnsTTL) {
                    $this->_error(9);
                }
                break;
            default:
                if (!$dnsUser) {
                    $this->_error(3);
                } elseif (!$dnsPass) {
                    $this->_error(4);
                } elseif (!$dnsHost) {
                    $this->_error(5);
                }
                break;
        }

        switch ($dnsService) {
            case 'azurev6':
            case 'cloudflare-v6':
            case 'custom-v6':
            case 'dynv6-v6':
            case 'he-net-v6':
            case 'godaddy-v6':
            case 'linode-v6':
            case 'regfish-v6':
            case 'route53-v6':
            case 'cloudflare-token-v6':
                $this->_useIPv6 = true;
                break;
            default:
                $this->_useIPv6 = false;
        }
        $this->_dnsService = strtolower($dnsService);
        $this->_dnsUser = $dnsUser;
        $this->_dnsPass = $dnsPass;
        $this->_dnsHost = $dnsHost;
        $this->_dnsServer = $dnsServer;
        $this->_dnsPort = $dnsPort;
        $this->_dnsWildcard = $dnsWildcard;
        $this->_dnsMX = $dnsMX;
        $this->_dnsZoneID = $dnsZoneID;
        $this->_dnsResourceID = $dnsResourceID;
        $this->_dnsTTL = $dnsTTL;
        $this->_if = dyndns_failover_interface($dnsIf, $this->_useIPv6 ? 'inet6' : 'all');
        $this->_checkIP();
        $this->_dnsUpdateURL = $dnsUpdateURL;
        $this->_dnsResultMatch = $dnsResultMatch;
        $this->_dnsRequestIf = dyndns_failover_interface($dnsRequestIf, $this->_useIPv6 ? 'inet6' : 'all');
        if ($this->_dnsVerboseLog) {
            log_error("Dynamic DNS ({$this->_dnsHost}): running dyndns_failover_interface for {$dnsRequestIf}. found {$this->_dnsRequestIf}");
        }
        $this->_dnsRequestIfIP = $this->_useIPv6 ? get_interface_ipv6($this->_dnsRequestIf) : get_interface_ip($this->_dnsRequestIf);
        $this->_dnsMaxCacheAgeDays = 25;
        $this->_dnsDummyUpdateDone = false;
        $this->_forceUpdateNeeded = $forceUpdate;

        // Ensure that we were able to lookup the IP
        if (!is_ipaddr($this->_dnsIP)) {
            log_error("Dynamic DNS ({$this->_dnsHost}) There was an error trying to determine the public IP for interface - {$dnsIf}({$this->_if}). Probably interface is not a WAN interface.");
            unlock($dyndnslck);
            return;
        }

        $this->_debugID = rand(1000000, 9999999);

        if ($forceUpdate == false && $this->_detectChange() == false) {
            $this->_error(10);
        } else {
            switch ($this->_dnsService) {
                case '3322':
                case 'azure':
                case 'azurev6':
                case 'citynetwork':
                case 'cloudflare':
                case 'cloudflare-v6':
                case 'cloudflare-token':
                case 'cloudflare-token-v6':
                case 'custom':
                case 'custom-v6':
                case 'dhs':
                case 'digitalocean':
                case 'gandi-livedns':
                case 'dnsexit':
                case 'dnsomatic':
                case 'duckdns':
                case 'dyndns':
                case 'dyndns-custom':
                case 'dyndns-static':
                case 'dyns':
                case 'dynv6':
                case 'dynv6-v6':
                case 'easydns':
                case 'eurodns':
                case 'freedns':
                case 'godaddy':
                case 'godaddy-v6':
                case 'googledomains':
                case 'gratisdns':
                case 'he-net':
                case 'he-net-tunnelbroker':
                case 'he-net-v6':
                case 'hn':
                case 'linode':
                case 'linode-v6':
                case 'loopia':
                case 'namecheap':
                case 'noip':
                case 'noip-free':
                case 'ods':
                case 'oray':
                case 'ovh-dynhost':
                case 'regfish':
                case 'regfish-v6':
                case 'route53':
                case 'route53-v6':
                case 'selfhost':
                case 'staticcling':
                case 'strato':
                case 'zoneedit':
                    $this->_update();
                    if ($this->_dnsDummyUpdateDone == true) {
                        // If a dummy update was needed, then sleep a while and do the update again to put the proper address back.
                        // Some providers (e.g. No-IP free accounts) need to have at least 1 address change every month.
                        // If the address has not changed recently, or the user did "Force Update", then the code does
                        // a dummy address change for providers like this.
                        sleep(10);
                        $this->_update();
                    }
                    break;
                default:
                    $this->_error(6);
                    break;
            }
        }

        unlock($dyndnslck);
    }

    /*
     * Private Function (added 12 July 05) [beta]
     *   Send Update To Selected Service.
     */
    function _update()
    {
        if ($this->_dnsVerboseLog) {
            log_error("Dynamic DNS ({$this->_dnsHost} via {$this->_dnsServiceList[$this->_dnsService]}): _update() starting.");
        }

        if ($this->_dnsService != 'ods' and $this->_dnsService != 'route53' and $this->_dnsService != 'route53-v6') {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_HEADER, 0);
            curl_setopt($ch, CURLOPT_USERAGENT, $this->_UserAgent);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_INTERFACE, $this->_dnsRequestIfIP);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        }

        switch ($this->_dnsService) {
            case 'dyndns':
            case 'dyndns-static':
            case 'dyndns-custom':
                if (isset($this->_dnsWildcard) && $this->_dnsWildcard != "OFF") {
                    $this->_dnsWildcard = "ON";
                }
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                $server = "https://members.dyndns.org/nic/update";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?system=dyndns&hostname=' . $this->_dnsHost . '&myip=' . $this->_dnsIP . '&wildcard=' . $this->_dnsWildcard . '&mx=' . $this->_dnsMX . '&backmx=NO');
                break;
            case 'dhs':
                $post_data['hostscmd'] = 'edit';
                $post_data['hostscmdstage'] = '2';
                $post_data['type'] = '4';
                $post_data['updatetype'] = 'Online';
                $post_data['mx'] = $this->_dnsMX;
                $post_data['mx2'] = '';
                $post_data['txt'] = '';
                $post_data['offline_url'] = '';
                $post_data['cloak'] = 'Y';
                $post_data['cloak_title'] = '';
                $post_data['ip'] = $this->_dnsIP;
                $post_data['domain'] = 'dyn.dhs.org';
                $post_data['hostname'] = $this->_dnsHost;
                $post_data['submit'] = 'Update';
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                $server = "https://members.dhs.org/nic/hosts";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, '{$server}{$port}');
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $post_data);
                break;
            case 'noip':
            case 'noip-free':
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                $server = "https://dynupdate.no-ip.com/ducupdate.php";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                if (
                    ($this->_dnsService == "noip-free") &&
                    ($this->_forceUpdateNeeded == true) &&
                    ($this->_dnsDummyUpdateDone == false)
                ) {
                    // Update the IP to a dummy value to force No-IP free accounts to see a change.
                    $iptoset = "192.168.1.1";
                    $this->_dnsDummyUpdateDone = true;
                    log_error("Dynamic DNS ({$this->_dnsHost}): Processing dummy update on No-IP free account. IP temporarily set to " . $iptoset);
                } else {
                    $iptoset = $this->_dnsIP;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?username=' . urlencode($this->_dnsUser) . '&pass=' . urlencode($this->_dnsPass) . '&hostname=' . $this->_dnsHost . '&ip=' . $iptoset);
                break;
            case 'easydns':
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                $server = "https://members.easydns.com/dyn/dyndns.php";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?hostname=' . $this->_dnsHost . '&myip=' . $this->_dnsIP . '&wildcard=' . $this->_dnsWildcard . '&mx=' . $this->_dnsMX . '&backmx=' . $this->_dnsBackMX);
                break;
            case 'hn':
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                $server = "http://dup.hn.org/vanity/update";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?ver=1&IP=' . $this->_dnsIP);
                break;
            case 'zoneedit':
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);

                $server = "https://dynamic.zoneedit.com/auth/dynamic.html";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, "{$server}{$port}?host=" . $this->_dnsHost);
                break;
            case 'dyns':
                /* XXX HTTPS is currently broken for them */
                $server = 'http://www.dyns.cx/postscript011.php';
                $port = '';
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?username=' . urlencode($this->_dnsUser) . '&password=' . $this->_dnsPass . '&host=' . $this->_dnsHost);
                break;
            case 'ods':
                $misc_errno = 0;
                $misc_error = "";
                $server = "ods.org";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                $this->con['socket'] = fsockopen("{$server}{$port}", "7070", $misc_errno, $misc_error, 30);
                /* Check that we have connected */
                if (!$this->con['socket']) {
                    print "error! could not connect.";
                    break;
                }
                /* Here is the loop. Read the incoming data (from the socket connection) */
                while (!feof($this->con['socket'])) {
                    $this->con['buffer']['all'] = trim(fgets($this->con['socket'], 4096));
                    $code = substr($this->con['buffer']['all'], 0, 3);
                    sleep(1);
                    switch ($code) {
                        case 100:
                            fputs($this->con['socket'], "LOGIN " . $this->_dnsUser . " " . $this->_dnsPass . "\n");
                            break;
                        case 225:
                            fputs($this->con['socket'], "DELRR " . $this->_dnsHost . " A\n");
                            break;
                        case 901:
                            fputs($this->con['socket'], "ADDRR " . $this->_dnsHost . " A " . $this->_dnsIP . "\n");
                            break;
                        case 795:
                            fputs($this->con['socket'], "QUIT\n");
                            break;
                    }
                }
                $this->_checkStatus(0, $code);
                break;
            case 'freedns':
                curl_setopt($ch, CURLOPT_URL, 'https://freedns.afraid.org/dynamic/update.php?' . $this->_dnsPass);
                break;
            case 'dnsexit':
                curl_setopt($ch, CURLOPT_URL, 'https://update.dnsexit.com/RemoteUpdate.sv?login=' . urlencode($this->_dnsUser) . '&password=' . $this->_dnsPass . '&host=' . $this->_dnsHost . '&myip=' . $this->_dnsIP);
                break;
            case 'loopia':
                $this->_dnsWildcard = (isset($this->_dnsWildcard) && $this->_dnsWildcard == true) ? 'ON' : 'OFF';
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                curl_setopt($ch, CURLOPT_URL, 'https://dns.loopia.se/XDynDNSServer/XDynDNS.php?hostname=' . $this->_dnsHost . '&myip=' . $this->_dnsIP . '&wildcard=' . $this->_dnsWildcard);
                break;
            case 'staticcling':
                curl_setopt($ch, CURLOPT_URL, 'https://www.staticcling.org/update.html?login=' . urlencode($this->_dnsUser) . '&pass=' . $this->_dnsPass);
                break;
            case 'dnsomatic':
                /* Example syntax
                    https://username:password@updates.dnsomatic.com/nic/update?hostname=yourhostname&myip=ipaddress&wildcard=NOCHG&mx=NOCHG&backmx=NOCHG
                */
                if (isset($this->_dnsWildcard) && $this->_dnsWildcard != "OFF") {
                    $this->_dnsWildcard = "ON";
                }
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                /*
                Reference: https://www.dnsomatic.com/wiki/api
                    DNS-O-Matic usernames are 3-25 characters.
                    DNS-O-Matic passwords are 6-20 characters.
                    All ASCII letters and numbers accepted.
                    Dots, dashes, and underscores allowed, but not at the beginning or end of the string.
                Required: "rawurlencode" http://www.php.net/manual/en/function.rawurlencode.php
                    Encodes the given string according to RFC 3986.
                */
                $server = "https://" . rawurlencode($this->_dnsUser) . ":" . rawurlencode($this->_dnsPass) . "@updates.dnsomatic.com/nic/update?hostname=";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $this->_dnsHost . '&myip=' . $this->_dnsIP . '&wildcard=' . $this->_dnsWildcard . '&mx=' . $this->_dnsMX . '&backmx=NOCHG');
                break;
            case 'namecheap':
                /* Example:
                    https://dynamicdns.park-your-domain.com/update?host=[host_name]&domain=[domain.com]&password=[domain_password]&ip=[your_ip]
                */
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                $dparts = explode(".", trim($this->_dnsHost));
                $domain_part_count = ($dparts[count($dparts) - 1] == "uk") ? 3 : 2;
                $domain_offset = count($dparts) - $domain_part_count;
                $hostname = implode(".", array_slice($dparts, 0, $domain_offset));
                $domain = implode(".", array_slice($dparts, $domain_offset));
                $dnspass = trim($this->_dnsPass);
                $server = "https://dynamicdns.park-your-domain.com/update?host={$hostname}&domain={$domain}&password={$dnspass}&ip={$this->_dnsIP}";
                curl_setopt($ch, CURLOPT_URL, $server);
                break;
            case 'he-net':
            case 'he-net-v6':
                $server = "https://dyn.dns.he.net/nic/update?";
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_URL, $server . 'hostname=' . $this->_dnsHost . '&password=' . $this->_dnsPass . '&myip=' . $this->_dnsIP);
                break;
            case 'he-net-tunnelbroker':
                $server = "https://ipv4.tunnelbroker.net/nic/update?";
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                curl_setopt($ch, CURLOPT_URL, $server . 'hostname=' . $this->_dnsHost);
                break;
            case 'digitalocean':
                /*
                * dnsHost should be the root domain
                * dnsUser should be the record ID
                * dnsPass should be the API key
                */
                $server = "https://api.digitalocean.com/v2/domains/" . $this->_dnsHost . "/records/" . $this->_dnsUser;
                $hostData = array("data" => "{$this->_dnsIP}");

                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                    "Authorization: Bearer {$this->_dnsPass}",
                    'Content-Type: application/json'
                ));
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($hostData));
                curl_setopt($ch, CURLOPT_URL, $server);
                break;
            case 'gandi-livedns':
                /*
                * https://github.com/vizion8-dan
                * Tested on OPNsense 20.1.8_1-amd64 - IPv4 (A)
                * dnsHost ("Hostname" field in OPNsense) should be the 2nd-level domain ("example.org")
                * dnsUser ("Username" field in OPNsense) should be the subdomain / A-record ("myrecord" in 2nd-level domain)
                * dnsPass should be the Gandi-API key
                */
                $server = "https://dns.api.gandi.net/api/v5/domains/" . $this->_dnsHost . "/records/" . $this->_dnsUser . "/A";

                $body = '{"rrset_ttl":"' . "300" . '", "rrset_values":["' . $this->_dnsIP . '"]}';

                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                    "X-Api-Key: {$this->_dnsPass}",
                    'Content-Type: application/json'
                ));
                curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
                curl_setopt($ch, CURLOPT_URL, $server);
                break;
            case 'selfhost':
                if (isset($this->_dnsWildcard) && $this->_dnsWildcard != "OFF") {
                    $this->_dnsWildcard = "ON";
                }
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                $server = "https://carol.selfhost.de/nic/update";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?system=dyndns&hostname=' . $this->_dnsHost . '&myip=' . $this->_dnsIP . '&wildcard=' . $this->_dnsWildcard . '&mx=' . $this->_dnsMX . '&backmx=NO');
                break;
            case 'route53':
            case 'route53-v6':
                /* Setting Variables */
                $hostname = "{$this->_dnsHost}.";
                $ZoneID = $this->_dnsZoneID;
                $AccessKeyId = $this->_dnsUser;
                $SecretAccessKey = $this->_dnsPass;
                $NewIP = $this->_dnsIP;
                $NewTTL = $this->_dnsTTL;
                $RecordType = ($this->_useIPv6) ? "AAAA" : "A";

                /* Set Amazon AWS Credentials for this record */
                $r53 = new Route53($AccessKeyId, $SecretAccessKey);

                /* Function to find old values of records in Route 53 */
                if (!function_exists('Searchrecords')) {
                    function SearchRecords($records, $name)
                    {
                        if (!is_array($records)) {
                            return false;
                        }
                        $result = array();
                        foreach ($records as $record) {
                            if (strtolower($record['Name']) == strtolower($name)) {
                                $result [] = $record;
                            }
                        }
                        return ($result) ? $result : false;
                    }
                }

                $records = $r53->listResourceRecordSets("/hostedzone/$ZoneID");

                /* Get IP for your hostname in Route 53 */
                if (false !== ($a_result = SearchRecords($records['ResourceRecordSets'], "$hostname"))) {
                    /**
                     * if hostname for ipv4 and ipv6 is the same, a_result contains more than 1 item
                     * we need to get the item that corresponds to the record type
                     */
                    $oldTTLResult = null;
                    $oldIPResult = null;
                    foreach ($a_result as $resultItem) {
                        if ($RecordType === $resultItem['Type']) {
                            $oldTTLResult = $resultItem["TTL"];
                            $oldIPResult = $resultItem["ResourceRecords"][0];
                        }
                    }

                    $OldTTL = $oldTTLResult;
                    $OldIP = $oldIPResult;
                } else {
                    $OldIP = "";
                }

                /* Check if we need to update DNS Record */
                if ($OldIP !== $NewIP) {
                    if (!empty($OldIP)) {
                        /* Your Hostname already exists, deleting and creating it again */
                        $changes = array();
                        $changes[] = $r53->prepareChange("DELETE", $hostname, $RecordType, $OldTTL, $OldIP);
                        $changes[] = $r53->prepareChange("CREATE", $hostname, $RecordType, $NewTTL, $NewIP);
                        $result = $r53->changeResourceRecordSets("/hostedzone/$ZoneID", $changes);
                    } else {
                        /* Your Hostname does not exist yet, creating it */
                        $changes = $r53->prepareChange("CREATE", $hostname, $RecordType, $NewTTL, $NewIP);
                        $result = $r53->changeResourceRecordSets("/hostedzone/$ZoneID", $changes);
                    }
                }
                $this->_checkStatus(0, $result);
                break;
            case 'custom':
            case 'custom-v6':
                if ($this->_curlIpresolveV4) {
                    curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
                }
                if ($this->_curlSslVerifypeer) {
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                } else {
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                }
                if ($this->_dnsUser != '') {
                    curl_setopt($ch, CURLOPT_USERPWD, "{$this->_dnsUser}:{$this->_dnsPass}");
                }
                $server = str_replace("%IP%", $this->_dnsIP, $this->_dnsUpdateURL);
                curl_setopt($ch, CURLOPT_URL, $server);
                break;
            case 'cloudflare':
            case 'cloudflare-v6':
            case 'cloudflare-token':
            case 'cloudflare-token-v6':
                $baseUrl = 'https://api.cloudflare.com/client/v4';
                $fqdn = str_replace(' ', '', $this->_dnsHost);
                $recordType = ($this->_useIPv6) ? 'AAAA' : 'A';
                $ttlData = intval($this->_dnsTTL) < 1 ? 1 : intval($this->_dnsTTL);
                $hostData = array(
                    "content" => "{$this->_dnsIP}",
                    "type" => $recordType,
                    "name" => $fqdn,
                    "ttl" => $ttlData
                );

                // Determine if service is token based or user/password based and define appropriate header
                if (strpos($this->_dnsService, 'token') !== false) {
                    $headerAuth = array(
                        "Authorization: Bearer {$this->_dnsPass}",
                        'Content-Type: application/json'
                    );
                } else {
                    $headerAuth = array(
                        "X-Auth-Email: {$this->_dnsUser}",
                        "X-Auth-Key: {$this->_dnsPass}",
                        'Content-Type: application/json'
                    );
                }

                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headerAuth);

                // Get all zone info
                $zonesUrl = "$baseUrl/zones";
                curl_setopt($ch, CURLOPT_URL, $zonesUrl);
                $output = json_decode(curl_exec($ch));
                $zoneId = null; // Set default value

                // Iterate zone objects, check if $fqdn is equal to or ends with zone name
                foreach ($output->result as $key => $zoneObj) {
                    if (preg_match("/^{$zoneObj->name}$|\.{$zoneObj->name}$/", $fqdn)) {
                        // Found matching zone
                        $zoneId = $zoneObj->id;
                        // Get $hostName from $fqdn, set $domainName
                        // These are only really used for log messages.
                        $hostName = preg_replace("/\.?{$zoneObj->name}$/", '', $fqdn);
                        $domainName = $zoneObj->name;
                        break;
                    }
                }

                if ($zoneId) { // If zone ID was found get host ID
                    $dnsRecordsUrl = "$zonesUrl/$zoneId/dns_records";
                    $getHostId = "$dnsRecordsUrl?name=$fqdn&type=$recordType";
                    curl_setopt($ch, CURLOPT_URL, $getHostId);
                    $output = json_decode(curl_exec($ch));
                    $recordCount = !empty($output->result) ? count($output->result) : 0;
                    if ($recordCount === 0) {
                        // create record
                        curl_setopt($ch, CURLOPT_URL, $dnsRecordsUrl);
                        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
                    } elseif ($recordCount >= 1) {
                        // update record
                        $recordId = $output->result[0]->id;
                        $hostData["proxied"] = $output->result[0]->proxied;
                        curl_setopt($ch, CURLOPT_URL, "$dnsRecordsUrl/$recordId");
                        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                    }
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($hostData));
                    if ($recordCount > 1) {
                        log_error("Dynamic DNS ($fqdn): Warning: multiple records for $hostName found");
                    }
                }
                break;
            case 'eurodns':
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                $server = "https://update.eurodyndns.org/update/";
                $port = "";
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?hostname=' . $this->_dnsHost . '&myip=' . $this->_dnsIP);
                break;
            case 'gratisdns':
                $server = "https://admin.gratisdns.com/ddns.php";
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                list($hostname, $domain) = explode(".", $this->_dnsHost, 2);
                curl_setopt($ch, CURLOPT_URL, $server . '?u=' . urlencode($this->_dnsUser) . '&p=' . $this->_dnsPass . '&h=' . $this->_dnsHost . '&d=' . $domain . '&i=' . $this->_dnsIP);
                break;
            case 'ovh-dynhost':
                if (isset($this->_dnsWildcard) && $this->_dnsWildcard != "OFF") {
                    $this->_dnsWildcard = "ON";
                }
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                $server = "https://www.ovh.com/nic/update";
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?system=dyndns&hostname=' . $this->_dnsHost . '&myip=' . $this->_dnsIP . '&wildcard=' . $this->_dnsWildcard . '&mx=' . $this->_dnsMX . '&backmx=NO');
                break;
            case 'citynetwork':
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                $server = 'https://dyndns.citynetwork.se/nic/update';
                $port = "";
                if ($this->_dnsServer) {
                    $server = $this->_dnsServer;
                }
                if ($this->_dnsPort) {
                    $port = ":" . $this->_dnsPort;
                }
                curl_setopt($ch, CURLOPT_URL, $server . $port . '?hostname=' . $this->_dnsHost . '&myip=' . $this->_dnsIP);
                break;
            case 'duckdns':
                $server = "https://www.duckdns.org/update";
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_URL, $server . '?domains=' . str_replace('.duckdns.org', '', $this->_dnsHost) . '&token=' . urlencode($this->_dnsUser));
                break;
            case 'dynv6':
                $server = "https://ipv4.dynv6.com/api/update";
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_INTERFACE, $this->_dnsRequestIf);
                curl_setopt($ch, CURLOPT_DNS_LOCAL_IP4, $this->_dnsIP);
                curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
                curl_setopt($ch, CURLOPT_URL, $server . '?hostname=' . $this->_dnsHost . '&ipv4=' . $this->_dnsIP . '&token=' . $this->_dnsUser);
                break;
            case 'dynv6-v6':
                $server = "https://ipv6.dynv6.com/api/update";
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_INTERFACE, $this->_dnsRequestIf);
                curl_setopt($ch, CURLOPT_DNS_LOCAL_IP6, $this->_dnsIP);
                curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V6);
                curl_setopt($ch, CURLOPT_URL, $server . '?hostname=' . $this->_dnsHost . '&ipv6=' . $this->_dnsIP . '&token=' . $this->_dnsUser);
                break;
            case 'googledomains':
                $server = "https://domains.google.com/nic/update";
                $post_data['hostname'] = $this->_dnsHost;
                $post_data['myip'] = $this->_dnsIP;
                $post_data['offline'] = 'no';
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $post_data);
                curl_setopt($ch, CURLOPT_URL, $server);
                curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
                break;
            case 'strato':
                $server = "https://dyndns.strato.com/nic/update?hostname={$this->_dnsHost}&myip={$this->_dnsIP}";
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                curl_setopt($ch, CURLOPT_URL, $server);
                curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
                break;
            case '3322':
                $server = "http://members.3322.net/dyndns/update?hostname={$this->_dnsHost}&myip={$this->_dnsIP}";
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                curl_setopt($ch, CURLOPT_URL, $server);
                curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
                break;
            case 'oray':
                $server = "http://ddns.oray.com/ph/update?hostname={$this->_dnsHost}&myip={$this->_dnsIP}";
                curl_setopt($ch, CURLOPT_USERPWD, $this->_dnsUser . ':' . $this->_dnsPass);
                curl_setopt($ch, CURLOPT_URL, $server);
                curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
                break;
            case 'regfish':
            case 'regfish-v6':
                $family = $this->_useIPv6 ? 'ipv6' : 'ipv4';
                $server = "https://dyndns.regfish.de/?fqdn={$this->_dnsHost}&{$family}={$this->_dnsIP}&forcehost=1&token=" . urlencode($this->_dnsUser);
                curl_setopt($ch, CURLOPT_URL, $server);
                break;
            case 'linode':
            case 'linode-v6':
                $baseUrl = "https://api.linode.com/v4";
                $fqdn = trim($this->_dnsHost);
                $recordType = ($this->_useIPv6) ? 'AAAA' : 'A';

                if ($this->_dnsWildcard == 'ON') {
                    $fqdn = "*.$fqdn";
                }

                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                    'Accept: application/json',
                    'Authorization: Bearer ' . $this->_dnsPass,
                    'Content-Type: application/json'
                ));

                $domainsUrl = "$baseUrl/domains";
                curl_setopt($ch, CURLOPT_URL, $domainsUrl);
                $output = json_decode(curl_exec($ch));
                $domainId = null;

                // Find matching domain and split the hostname part from it
                foreach ($output->data as $key => $domainObj) {
                    if (preg_match("/^{$domainObj->domain}$|\.{$domainObj->domain}$/", $fqdn)) {
                        $domainId = $domainObj->id;
                        $hostName = preg_replace("/\.?{$domainObj->domain}$/", '', $fqdn);
                        $domainName = $domainObj->domain;
                        break;
                    }
                }

                if ($domainId) {
                    if ($this->_dnsVerboseLog) {
                        log_error("Dynamic DNS ($fqdn): Found domain name: $domainName, ID: $domainId");
                    }

                    $dnsRecordsUrl = "$domainsUrl/$domainId/records";
                    curl_setopt($ch, CURLOPT_URL, $dnsRecordsUrl);
                    $output = json_decode(curl_exec($ch));
                    $recordId = null;

                    // Find matching record
                    foreach ($output->data as $key => $recordObj) {
                        if ($recordObj->type == $recordType && $recordObj->name == $hostName) {
                            $recordId = $recordObj->id;
                            break;
                        }
                    }

                    $hostData = array(
                        "target" => "{$this->_dnsIP}",
                    );

                    if ($recordId) {
                        // Update record
                        if ($this->_dnsVerboseLog) {
                            log_error("Dynamic DNS ($fqdn): Updating existing record ID: $recordId");
                        }

                        curl_setopt($ch, CURLOPT_URL, "$dnsRecordsUrl/$recordId");
                        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                    } else {
                        // Create record
                        if ($this->_dnsVerboseLog) {
                            log_error("Dynamic DNS ($fqdn): Creating new record");
                        }

                        $hostData['type'] = $recordType;
                        $hostData['name'] = $hostName;
                        // Linode will round up to the nearest valid TTL
                        $hostData['ttl_sec'] = 0;

                        curl_setopt($ch, CURLOPT_URL, $dnsRecordsUrl);
                        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
                    }

                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($hostData));
                } else {
                    log_error("Dynamic DNS($fqdn): No zone found for domain");
                }
                break;
            case 'azurev6':
            case 'azure':
                $hostname = "{$this->_dnsHost}";
                $resourceid = trim($this->_dnsResourceID);
                $app_id = $this->_dnsUser;
                $client_secret = $this->_dnsPass;
                $newip = $this->_dnsIP;
                $newttl = $this->_dnsTTL;
                // ensure resourceid starts with / and has no trailing /
                $resourceid = '/' . trim($resourceid, '/');
                // extract subscription id from resource id
                preg_match('/\\/subscriptions\\/(?<sid>[^\\/]*)/', $resourceid, $result);
                $subscriptionid = isset($result['sid']) ? $result['sid'] : '';
                if (isset($result['sid'])) {
                    $subscriptionid = $result['sid'];
                } else {
                    log_error("Azure subscription id not found in resource id ({$resourceid})");
                    return false;
                }
                // find tenant id from subscription id
                curl_setopt($ch, CURLOPT_URL, "https://management.azure.com/subscriptions/" . $subscriptionid . "?api-version=2016-09-01");
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($ch, CURLOPT_HEADER, 1);
                curl_setopt($ch, CURLOPT_NOBODY, 1);
                $output = curl_exec($ch);
                $pattern = '/Bearer authorization_uri="https:\\/\\/login.windows.net\\/(?<tid>[^"]*)/i';
                preg_match($pattern, $output, $result);
                if (isset($result['tid'])) {
                    $tenantid = $result['tid'];
                } else {
                    log_error("Tenant ID not found");
                    return false;
                }
                 // get an bearer token
                curl_setopt($ch, CURLOPT_URL, "https://login.microsoftonline.com/" . $tenantid . "/oauth2/token");
                curl_setopt($ch, CURLOPT_POST, 1);
                $body = "resource=" . urlencode("https://management.core.windows.net/") . "&grant_type=client_credentials&client_id=" . $app_id . "&client_secret=" . urlencode($client_secret);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                $server_output = curl_exec($ch);
                $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                preg_match("/\"access_token\":\"(?<tok>[^\"]*)\"/", $server_output, $result);
                if (isset($result['tok'])) {
                    $bearertoken = $result['tok'];
                } else {
                    log_error("no valid bearer token");
                    return false;
                }
                // Update the DNS record
                if ($this->_useIPv6) {
                    $url = "https://management.azure.com" . $resourceid . "/AAAA/" . $hostname . "?api-version=2017-09-01";
                    $body = '{"properties":{"TTL":"' . $newttl . '", "AAAARecords":[{"ipv6Address":"' . $newip . '"}]}}';
                } else {
                    $url = "https://management.azure.com" . $resourceid . "/A/" . $hostname . "?api-version=2017-09-01";
                    $body = '{"properties":{"TTL":"' . $newttl . '", "ARecords":[{"ipv4Address":"' . $newip . '"}]}}';
                }
                $request_headers = array();
                $request_headers[] = 'Accept: application/json';
                $request_headers[] = 'Authorization: Bearer ' . $bearertoken;
                $request_headers[] = 'Content-Type: application/json';
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_USERAGENT, $this->_UserAgent);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $request_headers);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_HEADER, 1);
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                break;
            case 'godaddy':
            case 'godaddy-v6':
                /* Read https://developer.godaddy.com/ for API documentation */
                $baseApiUrl = 'https://api.godaddy.com/v1/domains/';
                $recordType = $this->_useIPv6 ? "AAAA" : "A";
                $splitHost = explode('.', trim($this->_dnsHost));
                $dnsDomain = '*';
                if ($this->_dnsWildcard != 'ON') {
                    $dnsDomain = array_shift($splitHost);
                }
                $dnsHost = implode('.', $splitHost);

                $url = $baseApiUrl  . $dnsHost . '/records/' . $recordType . '/' . $dnsDomain;

                /* body can contain multiple options (data, port, priority, service, ttl, weight) */
                $data = array();
                $data[] = array('data' => $this->_dnsIP);
                if ($this->_dnsTTL) {
                    $data[0]['ttl'] = $this->_dnsTTL;
                } else {
                    // minimum allowed by GoDaddy
                    $data[0]['ttl'] = 600;
                }
                $jsonBody = json_encode($data);
                if ($this->_dnsVerboseLog) {
                    log_error("Dynamic DNS: calling $url with body: $jsonBody");
                }

                /* PUT JSON /v1/domains/{domain}/records/{type}/{name} */
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "PUT");
                curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                        'Accept: application/json',
                        'Content-Type: application/json',
                        'Authorization: sso-key ' . $this->_dnsUser . ':' . $this->_dnsPass
                ));
                curl_setopt($ch, CURLOPT_URL, $url);

                curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonBody);
                break;
            default:
                break;
        }
        if ($this->_dnsService != 'ods' and $this->_dnsService != 'route53' and $this->_dnsService != 'route53-v6') {
            $data = curl_exec($ch);
            $this->_checkStatus($ch, $data);
            @curl_close($ch);
        }
    }

    /*
     * Private Function (added 12 July 2005) [beta]
     *   Retrieve Update Status
     */
    function _checkStatus($ch, $data)
    {
        if ($this->_dnsVerboseLog) {
            log_error("Dynamic DNS ({$this->_dnsHost}): _checkStatus() starting.");
            log_error("Dynamic DNS ({$this->_dnsHost}): Current Service: {$this->_dnsService}");
        }
        $successful_update = false;
        if ($this->_dnsService != 'ods' and $this->_dnsService != 'route53' and $this->_dnsService != 'route53-v6' && @curl_error($ch)) {
            $status = "Curl error occurred: " . curl_error($ch);
            log_error($status);
            $this->status = $status;
            return;
        }
        switch ($this->_dnsService) {
            case 'dhs':
                break;
            case 'noip':
            case 'noip-free':
                list($ip,$code) = explode(":", $data);
                switch ($code) {
                    case 0:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP address is current, no update performed.";
                        $successful_update = true;
                        break;
                    case 1:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Success) DNS hostname update successful.";
                        $successful_update = true;
                        break;
                    case 2:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Hostname supplied does not exist.";
                        break;
                    case 3:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Invalid Username.";
                        break;
                    case 4:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Invalid Password.";
                        break;
                    case 5:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) To many updates sent.";
                        break;
                    case 6:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Account disabled due to violation of No-IP terms of service.";
                        break;
                    case 7:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Invalid IP. IP Address submitted is improperly formatted or is a private IP address or is on a blacklist.";
                        break;
                    case 8:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Disabled / Locked Hostname.";
                        break;
                    case 9:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Host updated is configured as a web redirect and no update was performed.";
                        break;
                    case 10:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Group supplied does not exist.";
                        break;
                    case 11:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Success) DNS group update is successful.";
                        $successful_update = true;
                        break;
                    case 12:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Success) DNS group is current, no update performed.";
                        $successful_update = true;
                        break;
                    case 13:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Update client support not available for supplied hostname or group.";
                        break;
                    case 14:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Hostname supplied does not have offline settings configured.";
                        break;
                    case 99:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Client disabled. Client should exit and not perform any more updates without user intervention.";
                        break;
                    case 100:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Client disabled. Client should exit and not perform any more updates without user intervention.";
                        break;
                    default:
                        $status = "Dynamic DNS ({$this->_dnsHost}): (Unknown Response)";
                        $this->_debug("Unknown Response: " . $data);
                        break;
                }
                break;
            case 'easydns':
                if (preg_match('/NOACCESS/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Authentication Failed: Username and/or Password was Incorrect.";
                } elseif (preg_match('/NOSERVICE/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) No Service: Dynamic DNS Service has been disabled for this domain.";
                } elseif (preg_match('/ILLEGAL INPUT/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Illegal Input: Self-Explanatory";
                } elseif (preg_match('/TOOSOON/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Too Soon: Not Enough Time Has Elapsed Since Last Update";
                } elseif (preg_match('/NOERROR/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP Updated Successfully!";
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Unknown Response)";
                    log_error("Dynamic DNS ({$this->_dnsHost}): PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'hn':
                /* FIXME: add checks */
                break;
            case 'zoneedit':
                if (preg_match('/799/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error 799) Update Failed!";
                } elseif (preg_match('/700/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error 700) Update Failed!";
                } elseif (preg_match('/200/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP Address Updated Successfully!";
                    $successful_update = true;
                } elseif (preg_match('/201/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP Address Updated Successfully!";
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Unknown Response)";
                    log_error("Dynamic DNS ({$this->_dnsHost}): PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'dyns':
                if (preg_match("/400/i", $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Bad Request - The URL was malformed. Required parameters were not provided.";
                } elseif (preg_match('/402/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Update Too Soon - You have tried updating to quickly since last change.";
                } elseif (preg_match('/403/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Database Error - There was a server-sided database error.";
                } elseif (preg_match('/405/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Hostname Error - The hostname (" . $this->_dnsHost . ") doesn't belong to you.";
                } elseif (preg_match('/200/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP Address Updated Successfully!";
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Unknown Response)";
                    log_error("Dynamic DNS ({$this->_dnsHost}): PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'ods':
                if (preg_match("/299/i", $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP Address Updated Successfully!";
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Unknown Response)";
                    log_error("Dynamic DNS ({$this->_dnsHost}): PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'freedns':
                if (preg_match("/has not changed./i", $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) No Change In IP Address";
                    $successful_update = true;
                } elseif (preg_match("/Updated/i", $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP Address Changed Successfully!";
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Unknown Response)";
                    log_error("Dynamic DNS ({$this->_dnsHost}): PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'dnsexit':
                if (preg_match("/IP not changed/i", $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) No Change In IP Address";
                    $successful_update = true;
                } elseif (preg_match("/Success/i", $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP Address Changed Successfully!";
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Unknown Response)";
                    log_error("Dynamic DNS ({$this->_dnsHost}): PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'staticcling':
                if (preg_match("/invalid ip/i", $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Bad Request - The IP provided was invalid.";
                } elseif (preg_match('/required info missing/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Bad Request - Required parameters were not provided.";
                } elseif (preg_match('/invalid characters/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Bad Request - Illegal characters in either the username or the password.";
                } elseif (preg_match('/bad password/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Invalid password.";
                } elseif (preg_match('/account locked/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) This account has been administratively locked.";
                } elseif (preg_match('/update too frequent/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Updating too frequently.";
                } elseif (preg_match('/DB error/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Error) Server side error.";
                } elseif (preg_match('/success/i', $data)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Success) IP Address Updated Successfully!";
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS ({$this->_dnsHost}): (Unknown Response)";
                    log_error("Dynamic DNS ({$this->_dnsHost}): PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'namecheap':
                $tmp = str_replace("^M", "", $data);
                $ncresponse = simplexml_load_string($tmp);
                if (preg_match("/internal server error/i", $data)) {
                    $status = "Dynamic DNS: (Error) Server side error.";
                } elseif (preg_match("/request is badly formed/i", $data)) {
                    $status = "Dynamic DNS: (Error) Badly Formed Request (check your settings).";
                } elseif ((string)$ncresponse->ErrCount === "0") {
                    $status = "Dynamic DNS: (Success) IP Address Updated Successfully!";
                    $successful_update = true;
                } elseif (isset($ncresponse->ErrCount) && is_numeric((string)$ncresponse->ErrCount) && (string)$ncresponse->ErrCount > 0) {
                    $status = "Dynamic DNS: (Error) ";
                    if (isset($ncresponse->errors)) {
                        foreach ($ncresponse->errors->children() as $err) {
                            $status .= (string)$err . " ";
                        }
                    }
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS: (Unknown Response)";
                    log_error("Dynamic DNS: PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'route53':
            case 'route53-v6':
                $successful_update = true;
                break;
            case 'custom':
            case 'custom-v6':
                $successful_update = false;
                if ($this->_dnsResultMatch == "") {
                    $successful_update = true;
                } else {
                    $this->_dnsResultMatch = str_replace("%IP%", $this->_dnsIP, $this->_dnsResultMatch);
                    $matches = preg_split("/(?<!\\\\)\\|/", $this->_dnsResultMatch);
                    foreach ($matches as $match) {
                        $match = str_replace("\\|", "|", $match);
                        if (strcmp($match, trim($data, "\t\n\r")) == 0) {
                            $successful_update = true;
                        }
                    }
                    unset($matches);
                }
                if ($successful_update == true) {
                    $status = "Dynamic DNS: (Success) IP Address Updated Successfully!";
                } else {
                    $status = "Dynamic DNS: (Error) Result did not match.";
                }
                break;
            case 'cloudflare':
            case 'cloudflare-v6':
            case 'cloudflare-token':
            case 'cloudflare-token-v6':
                $output = json_decode($data);
                if ($output->result->content === $this->_dnsIP) {
                    $status = "Dynamic DNS: (Success) {$this->_dnsHost} updated to {$this->_dnsIP}";
                    $successful_update = true;
                } elseif ($output->errors[0]->code === 9103) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): ERROR - Invalid Credentials! Don't forget to use API Key for password field with Cloudflare.";
                } elseif (($output->success) && (!$output->result[0]->id)) {
                    $status = "Dynamic DNS ({$this->_dnsHost}): ERROR - Zone ID was not found.";
                } else {
                    $status = "Dynamic DNS ({$this->_dnsHost}): UNKNOWN ERROR - {$output->errors[0]->message}";
                    log_error("Dynamic DNS ({$this->_dnsHost}): PAYLOAD: {$data}");
                }
                break;
            case 'digitalocean':
                $output = json_decode($data);
                if ($output->domain_record->data === $this->_dnsIP) {
                    $status = "Dynamic DNS: (Success) Record ID {$this->_dnsUser} updated to {$this->_dnsIP}";
                    $successful_update = true;
                } else {
                    $status = "Dynamic DNS Record ID ({$this->_dnsUser}): UNKNOWN ERROR";
                    log_error("Dynamic DNS Record ID ({$this->_dnsUser}): PAYLOAD: {$data}");
                }
                break;
            case 'gandi-livedns':
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                if ($http_code == 401) {
                    $status = 'Dynamic DNS: (Error) Bad authentication attempt because of a wrong API Key.';
                } elseif ($http_code == 403) {
                    $status = 'Dynamic DNS: (Error) Access to the resource is denied. Mainly due to a lack of permissions to access it!';
                } elseif ($http_code == 201) {
                    $status = 'Dynamic DNS: (Success) Record was created!';
                    $successful_update = true;
                } elseif ($http_code == 200) {
                    $status = 'Dynamic DNS: (Success) Same record already exists. Nothing was changed!';
                    $successful_update = true;
                } else {
                    $status = 'Dynamic DNS: (Error) "Unknown Response"';
                    log_error("Dynamic DNS: HTTP Status: {$http_code}  PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'gratisdns':
                if (preg_match('/Forkerte værdier/i', $data)) {
                        $status = "Dynamic DNS: (Error) Wrong values - Update could not be completed.";
                } elseif (preg_match('/Bruger login: Bruger eksistere ikke/i', $data)) {
                        $status = "Dynamic DNS: (Error) Unknown username - User does not exist.";
                } elseif (preg_match('/Bruger login: 1Fejl i kodeord/i', $data)) {
                        $status = "Dynamic DNS: (Error) Wrong password - Remember password is case sensitive.";
                } elseif (preg_match('/Domæne kan IKKE administreres af bruger/i', $data)) {
                        $status = "Dynamic DNS: (Error) User unable to administer the selected domain.";
                } elseif (preg_match('/OK/i', $data)) {
                        $status = "Dynamic DNS: (Success) IP Address Updated Successfully!";
                        $successful_update = true;
                } else {
                        $status = "Dynamic DNS: (Unknown Response)";
                        log_error("Dynamic DNS: PAYLOAD: {$data}");
                        $this->_debug($data);
                }
                break;
            case 'duckdns':
                if (preg_match('/OK/i', $data)) {
                        $status = "Dynamic DNS: (Success) IP Address Updated Successfully!";
                        $successful_update = true;
                } else {
                        $status = "Dynamic DNS: (Unknown Response)";
                        log_error("Dynamic DNS: PAYLOAD: {$data}");
                        $this->_debug($data);
                }
                break;
            case 'dynv6':
            case 'dynv6-v6':
                /* API-Documentation: https://dynv6.com/docs/apis */
                if (preg_match('/addresses updated/i', $data)) {
                        $status = "Dynamic DNS: (Success) IP Address Updated Successfully!";
                        $successful_update = true;
                } elseif (preg_match('/addresses unchanged/i', $data)) {
                        $status = "Dynamic DNS: (Success) IP Address Unchanged!";
                        $successful_update = true;
                } else {
                        $status = "Dynamic DNS: (Unknown Response)";
                        log_error("Dynamic DNS: PAYLOAD: {$data}");
                        $this->_debug($data);
                }
                break;
            case '3322':
            case 'citynetwork':
            case 'dnsomatic':
            case 'dyndns':
            case 'dyndns-custom':
            case 'dyndns-static':
            case 'eurodns':
            case 'googledomains':
            case 'he-net':
            case 'he-net-tunnelbroker':
            case 'he-net-v6':
            case 'loopia':
            case 'oray':
            case 'ovh-dynhost':
            case 'selfhost':
            case 'strato':
                if (preg_match('/notfqdn/i', $data)) {
                    $status = "Dynamic DNS: (Error) Not a FQDN";
                } elseif (preg_match('/nochg/i', $data)) {
                    $status = "Dynamic DNS: (Success) No change in IP address";
                    $successful_update = true;
                } elseif (preg_match('/good/i', $data)) {
                    $status = "Dynamic DNS: (Success) IP address updated successfully ({$this->_dnsIP})";
                    $successful_update = true;
                } elseif (preg_match('/badauth/i', $data)) {
                    $status = "Dynamic DNS: (Error) Authentication failed";
                } elseif (preg_match("/badip/i", $data)) {
                    $status = "Dynamic DNS: (Error) IP address provided is invalid";
                } elseif (preg_match('/nohost/i', $data)) {
                    $status = "Dynamic DNS: (Error) Hostname does not exist or does not have dynamic DNS enabled";
                } elseif (preg_match('/numhost/i', $data)) {
                    $status = "Dynamic DNS: (Error) You may update up to 20 hosts only";
                } elseif (preg_match('/dnserr/i', $data)) {
                    $status = "Dynamic DNS: (Error) DNS error, stop updating for 30 minutes.";
                } elseif (preg_match('/badagent/i', $data)) {
                    $status = "Dynamic DNS: (Error) Bad request";
                } elseif (preg_match('/abuse/i', $data)) {
                    $status = "Dynamic DNS: (Error) Access has been blocked for abuse";
                } elseif (preg_match('/911/i', $data)) {
                    $status = "Dynamic DNS: (Error) Server-side error or maintenance";
                } elseif (preg_match('/yours/i', $data)) {
                    $status = "Dynamic DNS: (Error) Specified hostname does not exist under this username";
                } else {
                    $status = "Dynamic DNS: (Unknown Response)";
                    log_error("Dynamic DNS: PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'regfish':
            case 'regfish-v6':
                if (preg_match('/\|100\|/', $data)) {
                    $status = 'Dynamic DNS: (Success) Update successful';
                    $successful_update = true;
                } elseif (preg_match('/\|101\|/', $data)) {
                    $status = 'Dynamic DNS: (Success) Still up-to-date';
                    $successful_update = true;
                } elseif (preg_match('/\|401\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Standard authentication failed';
                } elseif (preg_match('/\|402\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Authentication failed';
                } elseif (preg_match('/\|406\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Invalid resource record';
                } elseif (preg_match('/\|407\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Invalid TTL range';
                } elseif (preg_match('/\|408\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Invalid IPv4';
                } elseif (preg_match('/\|409\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Invalid IPv6';
                } elseif (preg_match('/\|410\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Unknown authentication type';
                } elseif (preg_match('/\|412\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Domain format is wrong, missing trailing dot?';
                } elseif (preg_match('/\|414\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Unexpected error';
                } elseif (preg_match('/\|415\|/', $data)) {
                    $status = 'Dynamic DNS: (Error) Cannot update load balancer';
                } else {
                    $status = "Dynamic DNS: (Unknown Response)";
                    log_error("Dynamic DNS: PAYLOAD: {$data}");
                    $this->_debug($data);
                }
            case 'linode':
            case 'linode-v6':
                $fqdn = trim($this->_dnsHost);
                if ($this->_dnsWildcard == 'ON') {
                    $fqdn = "*.$fqdn";
                }

                $output = json_decode($data);
                if ($output->target === $this->_dnsIP) {
                    $status = "Dynamic DNS: (Success) $fqdn updated to {$this->_dnsIP}";
                    $successful_update = true;
                } elseif (!empty($output->errors)) {
                    $status = "Dynamic DNS ($fqdn): ERROR - Reason: {$output->errors[0]->reason}";
                } else {
                    $status = "Dynamic DNS ($fqdn): UNKNOWN ERROR";
                    log_error("Dynamic DNS ($fqdn): PAYLOAD: {$data}");
                }
                break;
            case 'azure':
            case 'azurev6':
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                if ($http_code == 401) {
                    $status = 'Dynamic DNS: (Error) User Authorization Failed';
                } elseif ($http_code == 201) {
                    $status = 'Dynamic DNS: (Success) IP Address Changed Successfully!';
                    $successful_update = true;
                } elseif ($http_code == 200) {
                    $status = 'Dynamic DNS: (Success) IP Address Changed Successfully!';
                    $successful_update = true;
                } else {
                    $status = 'Dynamic DNS: (Error) "Unknown Response"';
                    log_error("Dynamic DNS: HTTP Status: {$http_code}  PAYLOAD: {$data}");
                    $this->_debug($data);
                }
                break;
            case 'godaddy':
            case 'godaddy-v6':
                /* See https://developer.godaddy.com/ for API documentation, not all codes are handled. */
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $successful_update = false;
                if ($http_code == 200) {
                    $status = 'Dynamic DNS: (Success) IP Address Updated Successfully!';
                    $successful_update = true;
                } elseif ($http_code == 401) {
                    $status = 'Dynamic DNS: (Error) Authentication info not sent or invalid';
                } elseif ($http_code == 404) {
                    $status = 'Dynamic DNS: (Error) Resource not found';
                } else {
                    $status = "Dynamic DNS: (Error) Repsonse not handled check the following: {$data}";
                    log_error("Dynamic DNS: (Error) HTTPS Status: {$http_code} PAYLOAD: {$data}");
                }
                break;
            default:
                break;
        }

        if ($successful_update == true) {
            /* Write WAN IP to cache file */
            $wan_ip = $this->_checkIP();
            if ($this->_useIPv6 == false && $wan_ip > 0) {
                $currentTime = time();
                log_error("Dynamic DNS: updating cache file {$this->_cacheFile}: {$wan_ip}");
                @file_put_contents($this->_cacheFile, "{$wan_ip}|{$currentTime}");
            } else {
                @unlink($this->_cacheFile);
            }
            if ($this->_useIPv6 == true && $wan_ip > 0) {
                $currentTime = time();
                log_error("Dynamic DNS: updating cache file {$this->_cacheFile_v6}: {$wan_ip}");
                @file_put_contents($this->_cacheFile_v6, "{$wan_ip}|{$currentTime}");
            } else {
                @unlink($this->_cacheFile_v6);
            }
        }
        $this->status = $status;
        log_error($status);
    }

    /*
     * Private Function (added 12 July 05) [beta]
     *   Return Error, Set Last Error, and Die.
     */
    function _error($errorNumber = '1')
    {
        switch ($errorNumber) {
            case 0:
                break;
            case 2:
                $error = 'Dynamic DNS: (ERROR!) No Dynamic DNS Service provider was selected.';
                break;
            case 3:
                $error = 'Dynamic DNS: (ERROR!) No Username Provided.';
                break;
            case 4:
                $error = 'Dynamic DNS: (ERROR!) No Password Provided.';
                break;
            case 5:
                $error = 'Dynamic DNS: (ERROR!) No Hostname Provided.';
                break;
            case 6:
                $error = 'Dynamic DNS: (ERROR!) The Dynamic DNS Service provided is not yet supported.';
                break;
            case 7:
                $error = 'Dynamic DNS: (ERROR!) No Update URL Provided.';
                break;
            case 8:
                $status = "Route 53: (Error) Invalid ZoneID";
                break;
            case 9:
                $status = "Route 53: (Error) Invalid TTL";
                break;
            case 10:
                $error = "Dynamic DNS ({$this->_dnsHost}): No change in my IP address and/or " . $this->_dnsMaxCacheAgeDays . " days has not passed. Not updating dynamic DNS entry.";
                break;
            default:
                $error = "Dynamic DNS: (ERROR!) Unknown Response.";
                /* FIXME: $data isn't in scope here */
                /* $this->_debug($data); */
                break;
        }
        $this->lastError = $error;
        log_error($error);
    }

    /*
     * Private Function (added 12 July 05) [beta]
     *   - Detect whether or not IP needs to be updated.
     *      | Written Specifically for pfSense (https://www.pfsense.org) may
     *      | work with other systems. pfSense base is FreeBSD.
     */
    function _detectChange()
    {
        $currentTime = time();

        $wan_ip = $this->_checkIP();
        if ($wan_ip == 0) {
            log_error("Dynamic DNS ({$this->_dnsHost}): Current WAN IP could not be determined, skipping update process.");
            return false;
        }
        $log_error = "Dynamic DNS ({$this->_dnsHost}): Current WAN IP: {$wan_ip} ";

        if ($this->_useIPv6 == true) {
            if (file_exists($this->_cacheFile_v6)) {
                $contents = file_get_contents($this->_cacheFile_v6);
                list($cacheIP,$cacheTime) = explode('|', $contents);
                $this->_debug($cacheIP . '/' . $cacheTime);
                $initial = false;
                $log_error .= "Cached IPv6: {$cacheIP} ";
            } else {
                $cacheIP = '::';
                @file_put_contents($this->_cacheFile, "::|{$currentTime}");
                $cacheTime = $currentTime;
                $initial = true;
                $log_error .= "No Cached IPv6 found.";
            }
        } else {
            if (file_exists($this->_cacheFile)) {
                $contents = file_get_contents($this->_cacheFile);
                list($cacheIP,$cacheTime) = explode('|', $contents);
                $this->_debug($cacheIP . '/' . $cacheTime);
                $initial = false;
                $log_error .= "Cached IP: {$cacheIP} ";
            } else {
                $cacheIP = '0.0.0.0';
                @file_put_contents($this->_cacheFile, "0.0.0.0|{$currentTime}");
                $cacheTime = $currentTime;
                $initial = true;
                $log_error .= "No Cached IP found.";
            }
        }
        if ($this->_dnsVerboseLog) {
            log_error($log_error);
        }

        // Convert seconds = days * hr/day * min/hr * sec/min
        $maxCacheAgeSecs = $this->_dnsMaxCacheAgeDays * 24 * 60 * 60;

        $needs_updating = false;
        /* lets determine if the item needs updating */
        if ($cacheIP != $wan_ip) {
            $needs_updating = true;
            $update_reason = "Dynamic DNS: cacheIP != wan_ip.  Updating. ";
            $update_reason .= "Cached IP: {$cacheIP} WAN IP: {$wan_ip} ";
        }
        if (($currentTime - $cacheTime) > $maxCacheAgeSecs) {
            $needs_updating = true;
            $this->_forceUpdateNeeded = true;
            $update_reason = "Dynamic DNS: More than " . $this->_dnsMaxCacheAgeDays . " days.  Updating. ";
            $update_reason .= "{$currentTime} - {$cacheTime} > {$maxCacheAgeSecs} ";
        }
        if ($initial == true) {
            $needs_updating = true;
            $update_reason .= "Initial update. ";
        }

        /*   finally if we need updating then store the
         *   new cache value and return true
         */
        if ($needs_updating == true) {
            if ($this->_dnsVerboseLog) {
                log_error("Dynamic DNS ({$this->_dnsHost}): {$update_reason}");
            }
            return true;
        }

        return false;
    }

    /*
     * Private Function (added 16 July 05) [beta]
     *   - Writes debug information to a file.
     *   - This function is only called when a unknown response
     *   - status is returned from a dynamic DNS service provider.
     */
    function _debug($data)
    {
        $string = date('m-d-y h:i:s') . ' - (' . $this->_debugID . ') - [' . $this->_dnsService . '] - ' . $data . "\n";
        $file = fopen($this->_debugFile, 'a');
        fwrite($file, $string);
        fclose($file);
    }

    function _checkIP()
    {
        $ip_address = get_dyndns_ip($this->_if, $this->_useIPv6 ? 6 : 4);
        if (!is_ipaddr($ip_address)) {
            if ($this->_dnsVerboseLog) {
                log_error("Dynamic DNS ({$this->_dnsHost}): IP address could not be extracted");
            }

            $ip_address = 0;
        } else {
            if ($this->_dnsVerboseLog) {
                log_error("Dynamic DNS ({$this->_dnsHost}): {$ip_address} extracted");
            }

            $this->_dnsIP = $ip_address;
        }

        return $ip_address;
    }
}
