/**
 * (C) Copyright IBM Corp. 2007
 *
 * THIS FILE IS PROVIDED UNDER THE TERMS OF THE COMMON PUBLIC LICENSE 
 * ("AGREEMENT"). ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS FILE 
 * CONSTITUTES RECIPIENTS ACCEPTANCE OF THE AGREEMENT.
 *
 * You can obtain a current copy of the Common Public License from
 * http://www.opensource.org/licenses/cpl1.0.php
 *
 * @author: Endre Bak, IBM, ebak@de.ibm.com
 * 
 * Change History
 * Flag       Date        Prog         Description
 *------------------------------------------------------------------------------- 
 * 1804402    2007-09-28  ebak      IPv6 ready SLP
 */


package org.sblim.slp.internal.sa;

import java.io.IOException;

import org.sblim.slp.ServiceLocationException;
import org.sblim.slp.internal.TRC;

/**
 * RecieverThread
 *
 */
public abstract class RecieverThread implements Runnable {
	
	private Thread iThread;
	private volatile boolean iStop;
	private boolean iInited;
	private Object iInitLock = new Object();
	
	protected ServiceAgent iSrvAgent;
	
	/**
	 * Ctor.
	 * @param pName
	 * @param pSrvAgent
	 */
	public RecieverThread(String pName, ServiceAgent pSrvAgent) {
		iThread = new Thread(this, pName);
		iSrvAgent = pSrvAgent;
	}
	
	/**
	 * start
	 */
	public void start() {
		iThread.start();
	}
	
	/**
	 * wait4init
	 */
	public void wait4init() {
		synchronized (iInitLock) {
			try {
				if (iInited) return;
				iInitLock.wait();
				return;
			} catch (InterruptedException e) {
				TRC.error(e);
			}
		}
	}
	
	/**
	 * stop
	 */
	public void stop() { stop(true); }
	
	/**
	 * stop
	 * @param pWait
	 */
	public void stop(boolean pWait) {
		iStop = true;
		if (pWait) join();
	}
	
	public void run() {
		// 1st init
		synchronized (iInitLock) {
			TRC.debug("initing");
			initialize();
			iInited = true;
			TRC.debug("inited");
			try {
				iInitLock.notifyAll();
			} catch (IllegalMonitorStateException e) {
				TRC.error(e);
			}
		}
		while (!iStop) {
			try {
				mainLoop();
			} catch (Exception e) {
				TRC.error(e);
				sleep(100);
				initialize();
			}
		}
		close();
		iStop = false;
		TRC.debug("STOPPED");
	}
	
	// exception of init shoud be reachable for ServiceTable
	
	protected abstract void init() throws ServiceLocationException, IOException;
	
	protected abstract void mainLoop() throws IOException;
	
	protected abstract void close();
	
	private void join() {
		try {
			iThread.join();
		} catch (InterruptedException e) {
			TRC.error(e);
		}
	}
	
	private void initialize() {
		try {
			init();
		} catch (ServiceLocationException e) {
			TRC.error(e);
		} catch (IOException e) {
			TRC.error(e);
		}
	}
	
	private static void sleep(int pMillis) {
		try {
			Thread.sleep(pMillis);
		} catch (InterruptedException e) {
			TRC.error(e);
		}
	}
	
}