/*
Copyright 1990-2008 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/
#include <stdlib.h>
#include <strings.h>
#include <libxml/parser.h>
#include <glib.h>

#include "iiimcf.h"
#include "iiim-properties.h"

typedef struct {
  gchar *name;
  gchar *s1;
  gchar *s2;
  gchar *s3;
  gchar *s4;
  gint code;
} KeyInfo;

static gchar *MEM_ERROR_MSG = "iiimx-settings-init: memory allocation error.\n";

IIIMCF_context cf_get_context ();
gchar *setget_nol10nlang_to_id (gchar *);

static gint p_strcmp (gconstpointer a, gconstpointer b)
{
  return strcmp (a, b);
}

static void free_key_info (KeyInfo *keyInfo)
{
  g_free (keyInfo->name);
  g_free (keyInfo->s1);
  g_free (keyInfo->s2);
  g_free (keyInfo->s3);
  g_free (keyInfo->s4);
  g_free (keyInfo);
}

static void free_layout (CustomLayout *layout)
{
  GSList *p;

  g_free (layout->name);
  g_free (layout->source);
  for (p = layout->key_list; p; p = g_slist_next (p)) {
    free_key_info (p->data);
  }
  g_slist_free (layout->key_list);
  g_free (layout);
}

static GSList *
parseLayoutFile (xmlDocPtr doc) {
  xmlNodePtr cur;
  CustomLayout *ret;
  GSList *layout_list = NULL;

  if (doc == NULL) {
    return NULL;
  }

  cur = xmlDocGetRootElement (doc);
  if (cur == NULL) {
    return NULL;
  }

  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    /* cur is <lo> element */
    xmlNodePtr keyNode;
    xmlChar *name;
    xmlChar *idstr;
    CustomLayout *layout;
    KeyInfo *keyInfo;

    if (!xmlStrcmp (cur->name, (const xmlChar *)"lo")) {
      name = xmlGetProp (cur, (xmlChar *)"type");
      idstr = xmlGetProp (cur, (xmlChar *)"id");
    } else {
      cur = cur->next;
      continue;
    }

    layout = g_new0 (CustomLayout, 1);
    layout->name = (gchar *)name;
    layout->id = (gchar *)idstr;
    layout->source = (gchar *)xmlGetProp (cur, (xmlChar *)"source");
    layout->source_id = setget_nol10nlang_to_id (layout->source);
    
    /* get_id_from_name */
    /* store it in layout */

    keyNode = cur->xmlChildrenNode;

    while (keyNode != NULL) {
      if (!xmlStrcmp (keyNode->name, (const xmlChar *)"key")) {
	xmlNodePtr labelNode;
	labelNode = keyNode->xmlChildrenNode;

	keyInfo = g_new0 (KeyInfo, 1);
	keyInfo->name = (gchar *)xmlGetProp (keyNode, (xmlChar *)"sym");

	while (labelNode != NULL) {
	  if (!xmlStrcmp (labelNode->name, (const xmlChar *)"s1")) {
	    keyInfo->s1 = (gchar *)xmlNodeListGetString (doc, labelNode->xmlChildrenNode, 1);
	  } else if (!xmlStrcmp (labelNode->name, (const xmlChar *)"s2")) {
	    keyInfo->s2 = (gchar *)xmlNodeListGetString (doc, labelNode->xmlChildrenNode, 1);
	  } else if (!xmlStrcmp (labelNode->name, (const xmlChar *)"s3")) {
	    keyInfo->s3 = (gchar *)xmlNodeListGetString (doc, labelNode->xmlChildrenNode, 1);
	  } else if (!xmlStrcmp (labelNode->name, (const xmlChar *)"s4")) {
	    keyInfo->s4 = (gchar *)xmlNodeListGetString (doc, labelNode->xmlChildrenNode, 1);
	  } else if (!xmlStrcmp (labelNode->name, (const xmlChar *)"code")) {
	    xmlChar *codeStr = xmlNodeListGetString (doc, labelNode->xmlChildrenNode, 1);
	    keyInfo->code = g_ascii_strtoull ((gchar *)codeStr, NULL, 16);
	  }
	  labelNode = labelNode->next;
	}
	layout->key_list = g_slist_append (layout->key_list, keyInfo);
      }
      keyNode = keyNode->next;
    }
    layout_list = g_slist_append (layout_list, layout);
    cur = cur->next;
  }
      
  return layout_list;
}


/* for DEBUG */
static int sent_num = 0;
static void print_data (CustomLayout *layout) {
  GSList *p;
  
  printf (" ----------- %d ------\n", sent_num++);
  printf ("   name = %s\n", layout->name);
  printf ("     id = %s\n", layout->id);
  printf (" source = %s\n", layout->source);
  printf ("   s_id = %s\n", layout->source_id);
  printf (" num of keys - %d\n", g_slist_length (layout->key_list));
  for (p = layout->key_list; p; p = g_slist_next (p)) {
    KeyInfo *k = (KeyInfo *)(p->data);
    printf ("     k name [%s] - %s/%s/%s/%s (%d)\n", k->name,
	    k->s1 == NULL ? "" : k->s1,
	    k->s2 == NULL ? "" : k->s2,
	    k->s3 == NULL ? "" : k->s3,
	    k->s4 == NULL ? "" : k->s4,
	    k->code);
  }
  printf ("\n");
}

/*
 * convert UTF8 to UTF16 and store it to given address
 */
static gboolean
setup_text (IIIMP_card16 **text_array, gchar *str)
{
  if (str == NULL)
    str = "";
  *text_array = g_utf8_to_utf16 (str, -1, NULL, NULL, NULL);
  if (*text_array == NULL) {
    fprintf (stderr, "UTF8 to UTF16 conversion failed : %s\n", str);
    return FALSE;
  }

  return TRUE;
}

static IIIMP_card16 AUX_NAME[] = {'c', 'o', 'm', '.', 's', 'u', 'n', '.', 'v', 'k', 'b', '.', 'K', 'B', 'L', 'T', 'r', 'a', 'n', 's', 'E', 'x', 't'};

static void
send_to_server (CustomLayout *layout)
{
  /* This function can assume that cf initialization has done. */
  IIIMCF_event event;
  IIIMF_status st;

  IIIMP_card16 **text_array;
  IIIMP_card32 *int_array;
  gint num_of_keys;
  GSList *p;
  gint i, len, t_idx, i_idx;
  KeyInfo *kinfo;

  /* DEBUG */
  /* print_data (layout); */

  num_of_keys = g_slist_length (layout->key_list);

  if (num_of_keys == 0) {
    /* not register no modified key data */
    return;
  }

  text_array = g_new0 (IIIMP_card16 *, num_of_keys * 5 + 4);
  int_array = g_new0 (IIIMP_card32, num_of_keys);
  if (text_array == NULL || int_array == NULL) {
    fprintf (stderr, MEM_ERROR_MSG);
    return;
  }

  /* add layout name to the text_array as first element */
  if (!setup_text (&(text_array[0]), layout->name)) return;
  /* add source name to the text_array as second element */
  if (!setup_text (&(text_array[1]), layout->source)) return;;
  /* add id name to the text_array as third element */
  if (!setup_text (&(text_array[2]), layout->id)) return;
  /* add key label & code info */
  if (!setup_text (&(text_array[3]), layout->source_id)) return;
  t_idx = 4;
  i_idx = 0;
  for (p = layout->key_list; p; p = g_slist_next (p)) {
    kinfo = (KeyInfo *)(p->data);
    if (!setup_text (&(text_array[t_idx++]), kinfo->name)) return;
    if (!setup_text (&(text_array[t_idx++]), kinfo->s1)) return;
    if (!setup_text (&(text_array[t_idx++]), kinfo->s2)) return;
    if (!setup_text (&(text_array[t_idx++]), kinfo->s3)) return;
    if (!setup_text (&(text_array[t_idx++]), kinfo->s4)) return;

    int_array[i_idx++] = kinfo->code;
  }

  st = iiimcf_create_aux_setvalues_event (AUX_NAME, 0,
					  i_idx, (const IIIMP_card32 *)int_array,
					  t_idx, (const IIIMP_card16 **)text_array, &event);
  if (st != IIIMF_STATUS_SUCCESS) {
    fprintf (stderr, " something wrong at create aux event for layout data. (%d)\n", st);
    return;
  }
  st = iiimcf_forward_event_with_operations (cf_get_context (), event, IIIMCF_SEND_KBLTRANS_LAYOUT);

  // free text_array & int_array
  for (i = 0; i < t_idx; i++) {
    g_free (text_array[i]);
  }
  g_free (text_array);
  g_free (int_array);
}

static GSList *user_defined_layout_list = NULL;

static void
assign_unique_id (GSList *list)
{
  GHashTable *idmap = g_hash_table_new (g_str_hash, g_str_equal);
  GSList *lp;

  for (lp = list; lp; lp = g_slist_next (lp))
    {
      CustomLayout *cl = (CustomLayout *)lp->data;
      
      gpointer i = g_hash_table_lookup (idmap, cl->source_id);
      g_hash_table_insert (idmap, cl->source_id, (gpointer)((gint)i + 1));
      cl->id = g_strdup_printf ("%s-%d-%s", cl->source_id, (gint)i + 1, USER_DEFINED_ID_SUFFIX);
    }
}

/*
 * retrun value should not be freeed.
 */
GSList *
sendlayout_get_use_layout_list()
{
  GSList *unuse_list = NULL;
  xmlDocPtr doc;
  xmlNodePtr cur, node;
  GDir *vkxdir;
  const gchar *vkxname;
  GError *error = NULL;
  GSList *use_list = NULL;
  GSList *p;

  if (user_defined_layout_list != NULL)
    {
      return user_defined_layout_list;
    }

  const gchar *home_dir = getenv ("HOME");
  const gchar *dir = g_strconcat (home_dir, "/.iiim/vkb", NULL);
  const gchar *config = g_strconcat (dir, "/config.xml", NULL);

  vkxdir = g_dir_open (dir, 0, &error);
  if (error != NULL) {
    /* no need to warning */
    g_error_free (error);
    return NULL;
  }

  /* read config for getting unused layout list */
  if (access (config, R_OK) == 0) {
    doc = xmlParseFile (config);
    if (doc != NULL) {
      cur = xmlDocGetRootElement (doc);
      cur = cur->xmlChildrenNode;
      while (cur != NULL) {
	if (!xmlStrcmp (cur->name, (const xmlChar *)"nonuse")) {
	  node = cur->xmlChildrenNode;
	  while (node != NULL) {
	    if (!xmlStrcmp (node->name, (const xmlChar *)"type")) {
	      xmlChar *val = xmlNodeListGetString (doc, node->xmlChildrenNode, 1);
	      unuse_list = g_slist_append (unuse_list, val);
	    }
	    node = node->next;
	  }
	}
	cur = cur->next;
      }
      xmlFreeDoc (doc);
    }
  }

  /* read custom layout files in $HOME/.iiim/vkb dir */
  while ((vkxname = g_dir_read_name (vkxdir)) != NULL) {
    GSList *layout_list = NULL;
    gchar *custom_name;
    gchar *full_path = g_strconcat (dir, "/", vkxname, NULL);
    if (!g_str_has_suffix (full_path, ".vkx")) {
      continue;
    }
    layout_list = parseLayoutFile (xmlParseFile (full_path));

    for (p = layout_list; p; p = g_slist_next (p)) {
      CustomLayout *l = (CustomLayout *)p->data;
      custom_name = l->name;
      if (g_slist_find_custom (unuse_list, custom_name, p_strcmp) == NULL) {
	use_list = g_slist_append (use_list, l);
      } else {
	/* free CustomLayout element */
	free_layout (l);
      }
    }
  }
  g_slist_free (unuse_list);

  /* make unique id from source_id */
  /* unique id is assigned by iiim-keyboard */
  /* assign_unique_id (use_list);*/
  user_defined_layout_list = use_list;

  {
    GSList *lp;
    for (lp = user_defined_layout_list; lp; lp = g_slist_next (lp)) {
      CustomLayout *cl = lp->data;
    }
  }

  return user_defined_layout_list;
}

void
send_user_layout()
{
  GSList *use_list = sendlayout_get_use_layout_list ();
  GSList *p;

  /* now use_list has list of layout which should be sent to server */
  for (p = use_list; p; p = g_slist_next (p)) {
    send_to_server (p->data);
  }
}

gchar *
sendlayout_get_ui_from_id (gchar *id)
{
  GSList *list = sendlayout_get_use_layout_list ();
  GSList *lp;

  for (lp = list; lp; lp = g_slist_next (lp))
    {
      CustomLayout *cl = (CustomLayout *)lp->data;
      if (!strcmp (id, cl->id))
	{
	  return cl->name;
	}
    }
  return id;
}

gchar *
sendlayout_get_id_from_ui (gchar *ui)
{
  GSList *list = sendlayout_get_use_layout_list ();
  GSList *lp;

  for (lp = list; lp; lp = g_slist_next (lp))
    {
      CustomLayout *cl = (CustomLayout *)lp->data;
      if (!strcmp (ui, cl->name))
	{
	  return cl->id;
	}
    }
  return ui;
}
