#include <config.h>
#include <gnome.h>

#include "gimlet.h"
#include "panel.h"
#include "eggtrayicon.h" 

static GdkPixbuf *on_pixbuf;
static GdkPixbuf *off_pixbuf;
static GdkPixbuf *disabled_pixbuf;
GdkPixbuf *kbd_pixbuf;

/*----------------------------------------------------------------------------
 |        panel callback functions                                            |
  ----------------------------------------------------------------------------*/
static void
update_icon_text (GimletWindow *gimlet)
{
  GdkPixbuf *pixbuf;
  if (gimlet->im_enabled)
    if (gimlet->conversion_mode)
      pixbuf = on_pixbuf;
    else
      pixbuf = off_pixbuf;
  else
    pixbuf = disabled_pixbuf;

  int icon_height = gdk_pixbuf_get_height(pixbuf);
  int icon_width = gdk_pixbuf_get_width(pixbuf);

  gint off_x, off_y;
  GtkWidget *drawing = gimlet->drawing_area;
  GtkWidget *parent = gtk_widget_get_parent (drawing);
  gtk_widget_translate_coordinates (drawing, parent, 0, 0, &off_x, &off_y);

  GdkGC *gc = gdk_gc_new(GDK_WINDOW(drawing->window));
  GdkColor fgColor;
  fgColor.red = fgColor.green = fgColor.blue = 0;
  gdk_gc_set_rgb_fg_color (gc, &fgColor);

  /* we actually draw on the event_box's window */
  gdk_draw_pixbuf(GDK_WINDOW(drawing->window),
                  gc,
                  pixbuf,0,0,
                  off_x,off_y,
                  -1,-1,
                  GDK_RGB_DITHER_NONE, 0,0);

  if (gimlet->im_enabled)
    {
      gchar *text = get_language_short_name (gimlet->current_iiim_lang);
      PangoLayout *layout = gtk_widget_create_pango_layout (GTK_WIDGET(drawing), text);
      int pango_width, pango_height, x_pos, y_pos;

      pango_layout_get_pixel_size (layout, &pango_width, &pango_height);

      x_pos = icon_width>pango_width?
              (icon_width-pango_width)/2:
              0;
      y_pos = (icon_height-pango_height)/2;
      gdk_draw_layout (GDK_WINDOW(drawing->window),
                       gc,
                       x_pos+off_x,
                       y_pos+off_y,
                       layout);
      g_object_unref (layout);
      g_free (text);
    }
  gdk_gc_unref (gc);
}

static void
update_tooltips (GimletWindow *gimlet)
{
  gchar *lang;
  gchar *name;
  gchar *descr;
  gchar *text;

  lang = gimlet->current_iiim_lang;

  if (lang)
    {
      name = lang;
      descr = get_language_full_name_by_short (lang);

      if (!descr || (descr && strcmp (name, descr) == 0))
        text = g_strdup (name);
      else
        text = g_strconcat (name, " - ", descr, NULL);

      gtk_tooltips_set_tip (gimlet->tooltips, GTK_WIDGET(gimlet->applet), text, NULL);
      g_free(descr); g_free(text);
    }
}

static gboolean
drawing_area_expose_cb (GtkWidget *widget,
                        GdkEventExpose *event,
                        GimletWindow *gimlet)
{
  update_icon_text (gimlet);
  update_tooltips (gimlet);
  return TRUE;
}

static gboolean
button_release_cb (GtkWidget *widget,
                       GdkEventButton *event,
                       gpointer user_data)
{
  gboolean ret = TRUE;

  switch (event->button) {
  case 1:
    gimlet_show_lang_menu ();
    break;
  case 3:
    gimlet_show_help_menu ();
    break;
  default:
    ret = FALSE;
    break;
  }

  return ret;
}


/*----------------------------------------------------------------------------
 |        create panel ui                                                     |
  ----------------------------------------------------------------------------*/
#define TIMEOUT_INTERVAL 1000
#define TIMEOUT_LIMIT 30
#define TIMEOUT_INTERVAL_2ND 3000
#define TIMEOUT_LIMIT_2ND 50

static guint timeout_id = 0;
static guint timeout_times = 0;

typedef enum {
  SYS_DEFAULT, TRAYICON, WINDOW,
}UIType;

static void panel_create_ui_by_type (GimletWindow *, UIType);
static void panel_create_ui (GimletWindow  *);

static gboolean 
try_to_create_tray_icon_when_timeout (gpointer timeout_nums)
{
  if (is_systray_supported ()) {
    panel_create_ui_by_type (gimlet, TRAYICON);
    timeout_id = 0;
    return FALSE;
  }

  if (++timeout_times < (guint)timeout_nums)
    return TRUE;

  if (!gimlet->applet) {
    /* if timeouts and no container was created yet, create the 
     * standalone window and continue trying */
    panel_create_ui_by_type (gimlet, WINDOW);
    timeout_times = 0;
    timeout_id = g_timeout_add (TIMEOUT_INTERVAL_2ND, 
                                try_to_create_tray_icon_when_timeout, 
                                (gpointer)TIMEOUT_LIMIT_2ND);
  }

  return FALSE;
}

static gint
on_langwin_delete_event(GtkWidget *widget,
                     GdkEvent  *event,
                     gpointer   data)
{
  if (is_systray_supported ()) {
    panel_create_ui_by_type (gimlet, TRAYICON);

    if (timeout_id)
      g_source_remove (timeout_id);

    return FALSE; /* return FALSE to delete this window */
  }
  return TRUE;
}

static gboolean
on_langwin_configure_event(GtkWidget       *widget,
                           GdkEventConfigure *event,
                           gpointer         user_data)
{
  static int evx = 0, evy = 0;

  int x, y;
  if (is_on_CDE()) 
    get_mwm_decoration_size (widget, &x, &y, NULL, NULL);
  else
    gdk_window_get_geometry (widget->window, &x, &y, NULL, NULL, NULL);

  if (evx != event->x || evy != event->y) {
    evx = event->x; evy = event->y;
    panel_save_position (event->x-x, event->y-y);
  }

  return FALSE;
}


/* when gnome-panel is crashed, try to re-create the UI components.*/
static void
tray_icon_destroyed (GtkWidget *widget, GimletWindow *gimlet)
{
  gtk_widget_destroy (widget);
  panel_create_ui_by_type (gimlet, WINDOW);
  timeout_times = 0;
  timeout_id = g_timeout_add (TIMEOUT_INTERVAL, 
                              try_to_create_tray_icon_when_timeout, 
                              (gpointer)TIMEOUT_LIMIT);
}

static GtkWidget *
create_container_trayicon (GimletWindow *gimlet)
{
  GtkWidget *container;

  container = egg_tray_icon_new ("iiim panel");
  g_signal_connect(G_OBJECT(container), "destroy", G_CALLBACK(tray_icon_destroyed), gimlet);

  if(gimlet->lang_menu != NULL) {
    gtk_menu_popdown (gimlet->lang_menu);
  }
  if(gimlet->help_menu != NULL) {
    gtk_menu_popdown (gimlet->help_menu);
  }

  return container;
}

static void 
move_container_window_to_bottom_right (GtkWindow *wnd)
{
  Display *disp  = gdk_x11_get_default_xdisplay ();
  gint    screen = gdk_x11_get_default_screen ();

  int H = DisplayHeight (disp, screen);
  int W = DisplayWidth (disp, screen);

  int h, w;
  gtk_window_get_size (wnd, &w, &h);
  gtk_window_move (wnd, W-w, H-h);
}

static GtkWidget *
create_container_window (GimletWindow *gimlet)
{
  GtkWidget *container;

  container = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  gtk_window_set_type_hint (GTK_WINDOW (container), GDK_WINDOW_TYPE_HINT_DIALOG);
  gtk_window_set_skip_taskbar_hint (GTK_WINDOW (container), TRUE);
  gtk_window_set_skip_pager_hint (GTK_WINDOW (container), TRUE);

  g_signal_connect (G_OBJECT (container), "delete_event",
                    G_CALLBACK (on_langwin_delete_event), NULL);
  g_signal_connect (G_OBJECT (container), "configure_event",
                    G_CALLBACK (on_langwin_configure_event), NULL);

  gtk_window_set_default_size (container, 60, 26);

  if (is_on_CDE()) {
    int x, y;
    panel_get_saved_position (&x, &y);
    gtk_window_move (container, x, y);
    gtk_widget_show (container);
    occupy_all_workspaces_on_CDE (container);
    sm_disable_on_CDE (container);
  } else
    move_container_window_to_bottom_right (container);

  return container;
}

static GtkWidget *
create_container (GimletWindow *gimlet)
{
  if (is_systray_supported ())
    return create_container_trayicon (gimlet);

  if (is_on_CDE())
    return create_container_window (gimlet);

  /* notification area maybe temporarily un-available, try several times */
  timeout_times = 0;
  timeout_id = g_timeout_add (TIMEOUT_INTERVAL, 
                              try_to_create_tray_icon_when_timeout, 
                              (gpointer)TIMEOUT_LIMIT);

  return NULL;
}

static void
append_widgets (GimletWindow *gimlet)
{
  GtkWidget *hbox;
  GtkWidget *status_button;
  GtkWidget *event_box;

  event_box = gtk_event_box_new ();
  gtk_event_box_set_visible_window (GTK_EVENT_BOX (event_box), FALSE);
  gtk_container_add (GTK_CONTAINER (gimlet->applet), event_box);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (event_box), hbox);

  /* status label */
  gimlet->status_label = gtk_label_new (NULL);
  gtk_box_pack_start_defaults (GTK_BOX (hbox),
                               GTK_WIDGET(gimlet->status_label));
  gimlet_status_set_text (gimlet, "");

  /* two-letters indidator */
  on_pixbuf = init_pixbuf ("im-on.png");
  off_pixbuf = init_pixbuf ("im-off.png");
  disabled_pixbuf = init_pixbuf ("im-disable.png");

  /* kbd language indicator */
  kbd_pixbuf = init_pixbuf ("iiim-properties-kbd.png");

  gimlet->drawing_area = gtk_label_new (NULL); /* just a place holder */
  int width = gdk_pixbuf_get_width(off_pixbuf);
  int height = gdk_pixbuf_get_height(off_pixbuf);
  gtk_widget_set_size_request (GTK_WIDGET(gimlet->drawing_area), width, height);
  g_signal_connect (G_OBJECT (gimlet->drawing_area), "expose_event",  
                    G_CALLBACK (drawing_area_expose_cb), gimlet);
  gtk_box_pack_end_defaults (GTK_BOX(hbox), GTK_WIDGET(gimlet->drawing_area));

  /* tooltip */
  gimlet->tooltips = gtk_tooltips_new();

  g_signal_connect_swapped (G_OBJECT(event_box), "button_release_event",
                            G_CALLBACK (button_release_cb), gimlet);
}

static void
panel_create_ui_by_type (GimletWindow *gimlet, UIType uitype)
{
  GtkWidget *container = NULL;

  switch (uitype) {
  case SYS_DEFAULT:
    container = create_container (gimlet);
    break;

  case TRAYICON:
    container = create_container_trayicon (gimlet);
    break;

  case WINDOW:
  default:
    container = create_container_window (gimlet);
    break;
  }

  if (!container)
    return;

  if (gimlet->applet)
    gtk_widget_destroy (gimlet->applet);

  gimlet->applet = container;
  append_widgets (gimlet);
  gtk_window_set_accept_focus (gimlet->applet, FALSE);
  gtk_widget_show_all (gimlet->applet);

  /* setup X properties for the newly created window */
  gimlet_xprop_init (gimlet->applet);
}

static void
panel_create_ui (GimletWindow *gimlet)
{
  panel_create_ui_by_type (gimlet, SYS_DEFAULT);
}


/*----------------------------------------------------------------------------
 |        service routines called by others                                   |
  ----------------------------------------------------------------------------*/
void
gimlet_create_panel (GimletWindow *gimlet)
{
  gimlet->current_iiim_lang = get_initial_language ();
  panel_create_ui (gimlet);
}

void
gimlet_update_lang (GimletWindow *gimlet)
{
  gtk_widget_queue_draw (gimlet->drawing_area);
}

void
gimlet_status_show_or_hide (GimletWindow *gimlet)
{
  if (gimlet->conversion_mode && gimlet->im_enabled &&
      gimlet->status_enabled &&
      gimlet->status_placement == ON_DESKTOP_PANEL)
    gtk_widget_show (GTK_WIDGET (gimlet->status_label));
  else
    gtk_widget_hide (GTK_WIDGET (gimlet->status_label));
}

void
gimlet_status_set_text (GimletWindow *gimlet, gchar *text)
{
  gtk_label_set_text (GTK_LABEL(gimlet->status_label), text);
  gimlet_status_show_or_hide (gimlet);
}
