#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#include <iiimp-data.h>
#include <iiimp-opcode.h>

#include "iiimp-dataP.h"


IIIMP_message *
iiimp_register_hotkeys_new(
    IIIMP_data_s *	data_s,
    IIIMP_card16	im_id,
    IIIMP_card8		scope,
    IIIMP_card16	profile_id,
    IIIMP_hotkey *	hotkey)
{
    IIIMP_message *	m;
    m = (IIIMP_message *)malloc(sizeof (IIIMP_message));
    if (NULL == m) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    m->opcode = IM_REGISTER_HOTKEYS;
    m->im_id = im_id;
    m->ic_id = -1;
    m->v.register_hotkeys.scope = scope;
    m->v.register_hotkeys.profile_id = profile_id;
    m->v.register_hotkeys.hotkey = hotkey;

    return m;
}


void
iiimp_register_hotkeys_delete(IIIMP_data_s * data_s, IIIMP_message * m)
{
    if (NULL == m) return;
    iiimp_hotkey_list_delete(data_s, m->v.register_hotkeys.hotkey);
    free(m);
}


uchar_t *
iiimp_register_hotkeys_pack(
    IIIMP_data_s *	data_s,
    IIIMP_card16	im_id,
    IIIMP_card8		scope,
    IIIMP_card16	profile_id,
    IIIMP_hotkey *	hotkey,
    size_t *		buf_size)
{
    size_t		nbyte;
    size_t		hk_nbyte;
    int			length;
    uchar_t *		buf;
    size_t		rest;
    uchar_t *		p;
    IIIMP_card16	scope_profile_id;
    IIIMP_hotkey *	hk;

    nbyte = 0;
    nbyte += 2;	/* input method id */
    nbyte += 2;	/* PAD(2) */
    nbyte += 2;	/* SCOPE and PROFILE_ID */
    nbyte += 2;	/* byte length of list of HOTKEYS */

    hk_nbyte = 0;
    for (hk = hotkey; NULL != hk; hk = hk->next) {
	hk_nbyte += hk->nbyte;
    }
    nbyte += hk_nbyte;

    length = (nbyte >> 2);
    *buf_size = (1 + 3 + nbyte);

    buf = (uchar_t *)malloc(1 + 3 + nbyte);
    if (NULL == buf) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    PUT_PACKET_HEADER(buf, IM_REGISTER_HOTKEYS, length);

    rest = nbyte;
    p = (buf + 4);

    PUTU16(im_id, rest, p, data_s->byte_swap);
    PUTU16(0, rest, p, data_s->byte_swap);
    scope_profile_id = ((0x8000 & (scope << 15)) | (0x7fff & profile_id));
    PUTU16(scope_profile_id, rest, p, data_s->byte_swap);
    if (NULL == hotkey) {
	PUTU16(0, rest, p, data_s->byte_swap);
    } else {
	PUTU16(hk_nbyte, rest, p, data_s->byte_swap);
	iiimp_hotkey_list_pack(data_s, hotkey, &rest, &p);
    }

    return buf;
}


IIIMP_message *
iiimp_register_hotkeys_unpack(
    IIIMP_data_s *	data_s,
    IIIMP_card7		opcode,
    size_t *		nbyte,
    const uchar_t **	ptr)
{
    IIIMP_message *			m;
    IIIMP_register_hotkeys_v *	        v;
    size_t				rest;
    const uchar_t *			p;
    int					len;
    IIIMP_card16			scope_profile_id;

    rest = *nbyte;
    p = *ptr;

    if (rest < (2 + 2 + 2 + 2 + 0)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    m = (IIIMP_message *)malloc(sizeof (IIIMP_message));
    if (NULL == m) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    v = &(m->v.register_hotkeys);

    m->opcode = opcode;
    v->hotkey = NULL;

    GETU16(m->im_id, rest, p, data_s->byte_swap);
    m->ic_id = -1;
    SKIP16(rest, p);
    GETU16(scope_profile_id, rest, p, data_s->byte_swap);
    v->scope = ((0x8000 & scope_profile_id) >> 15);
    v->profile_id = (0x7fff & scope_profile_id);
    GETU16(len, rest, p, data_s->byte_swap);
    if ((len < 0) || (rest < len)) {
	iiimp_register_hotkeys_delete(data_s, m);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    v->hotkey = iiimp_hotkey_list_unpack(data_s, &rest, &p, len);
    if (NULL == v->hotkey) {
	iiimp_register_hotkeys_delete(data_s, m);
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return (IIIMP_message *)m;
}


void
iiimp_register_hotkeys_print(
    IIIMP_data_s *	data_s,
    IIIMP_message *	m)
{
    iiimp_message_header_print(data_s, m->opcode, m->im_id, -1);
    (void)fprintf(data_s->print_fp, "\tscope = %s (%d)\n",
		  (0 == m->v.register_hotkeys.scope) ?
		  "global" : "session specific",
		  m->v.register_hotkeys.scope);
    (void)fprintf(data_s->print_fp, "\thotkey profile ID = %d\n", 
		  m->v.register_hotkeys.profile_id);
    (void)fprintf(data_s->print_fp, "\tHotkey List:\n");
    iiimp_hotkey_list_print(data_s, m->v.register_hotkeys.hotkey);
}


/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
