/* 
 * Copyright (C) 2008 Sun Microsystems
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Aijin Kim <Aijin.Kim@Sun.COM>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "debug.h"
#include "ime.h"
#include "hangul_im.h"
#include "hangul_properties.h"
#include "hangul_composer.h"

#ifndef LIBHANGUL_PREFIX
#define LIBHANGUL_PREFIX       "/usr"
#endif

#define HANGUL_LIB             LIBHANGUL_PREFIX"/lib/libhangul.so"
#define HANGUL_DATADIR         LIBHANGUL_PREFIX"/lib/libhangul"
#define HANGUL_NAME_UTF8       "한글"
#define HANGUL_UUID            "hangul-8f57ebee-bd88-46d2-8097-38ae1dac420c"
#define HANGUL_VERSION         10
#define AUTHOR \
    "Aijin Kim\n" \
    "Jeffrey Chen"
#define COPYRIGHT        "Copyright (c) 2007-2010 Oracle and/or its affiliates"
#define HINTING                "Hangul Input Method"
#define HANGUL_ICONPATH        "hangul.xpm"

typedef struct _HangulIMConfig {
    char keyboard[32];
} HangulIMConfig;

static HangulIMConfig config;
HanjaTable* hanja_table = NULL;

ImeResult hangul_Initialize(ImeInfo ime_info);
ImeResult hangul_Destroy(ImeInfo ime_info);
ImeResult hangul_Process_Key_Event(ImeInputContext ic, ImeKey key_event);
ImeResult hangul_Create_Session(ImeInputContext ic);
ImeResult hangul_Destroy_Session(ImeInputContext ic);
ImeResult hangul_FocusOut(ImeInputContext ic);
ImeResult hangul_FocusIn(ImeInputContext ic);

ImmServices imm_services;

ImeMethodsRec hangul_methods = {
    1,                /* version */
    hangul_Initialize,        /* ImeInitialize */
    hangul_Destroy,        /* ImeDestroy  */
    hangul_Process_Key_Event,    /* ImeProcessKeyEvent */
    NULL,            /* ImeProcessAuxEvent  */
    hangul_Create_Session,    /* ImeAttachSession */
    hangul_Destroy_Session,    /* ImeDetachSession */
    hangul_FocusIn,        /* ImeFocusIn  */
    hangul_FocusOut,        /* ImeFocusOut */
    NULL,            /* ImeAttachUser */
    NULL,            /* ImeDetachUser */
    NULL,            /* ImeAttachDesktop */
    NULL,            /* ImeDetachDesktop */
    NULL,            /* ImeGetErrorMessage */
#if 0
    NULL,            /* ImeDoConfig */
#endif
};

#ifdef    WIN32
#define EXPORT extern __declspec(dllexport)
EXPORT
#endif
ImeResult RegisterIME(ImmServices srvs, ImeInfo * ppinfo,
              ImeMethods * pmthds, int argc, char **argv)
{
    ImeInfoRec *hangul_info = NULL;

    DEBUG_printf("Register Hangul IM: argc: %d\n", argc);
    
    if (access(HANGUL_LIB, R_OK) != 0) {
        fprintf(stderr, "Error: Hangul: %s not found !!\n", HANGUL_LIB);
        return (IME_FAIL);
    }

    if (access(HANGUL_DATADIR"/hanja", R_OK) != 0) {
        fprintf(stderr, "Error: Hangul: hanja directory: %s not found !!\n", HANGUL_DATADIR);
        return (IME_FAIL);
    }

    hangul_info = (ImeInfoRec *) calloc(1, sizeof(ImeInfoRec));
    DEBUG_printf("hangul_info: %p\n", hangul_info);
    if (hangul_info == NULL) {
        return (IME_FAIL);
    }

    hangul_info->version = HANGUL_VERSION;
    hangul_info->mt_safe = 0;
    hangul_info->encoding = ENCODE_UTF8;
    hangul_info->uuid = HANGUL_UUID;
    hangul_info->name = HANGUL_NAME_UTF8;
    hangul_info->author = AUTHOR;
    hangul_info->hinting = HINTING;
    hangul_info->copyright = COPYRIGHT;
    hangul_info->icon_file = HANGUL_ICONPATH;
    hangul_info->support_locales = "ko_KR.UTF-8";
    hangul_info->pl = NULL;
    hangul_info->specific_data = NULL;

    hangul_Init_Ime_Properties(hangul_info);

    *ppinfo = hangul_info;
    *pmthds = &hangul_methods;

    imm_services = srvs;

    DEBUG_printf("leave Register IME\n");
    return (IME_OK);
}

ImeResult hangul_Initialize(ImeInfo hangul_info)
{
    char *prefix = HANGUL_DATADIR;

    DEBUG_printf("hangul_Initialize\n");

    /* Load hanja dictionary */
    hanja_table = hanja_table_load(NULL);

    DEBUG_printf("Hangul initialize done\n");

    return (IME_OK);
}

ImeResult hangul_Destroy(ImeInfo hangul_info)
{
    DEBUG_printf("hangul_Destroy\n");

    if (hanja_table != NULL) {
        hanja_table_delete(hanja_table);    
    }    

    if (hangul_info != NULL) {
        hangul_Destroy_Ime_Properties(hangul_info);
        free((char *) hangul_info);
    }

    return (IME_OK);
}

ImeResult hangul_Create_Session(ImeInputContext ic)
{
    int i;
    ImmResult imm_result;
    ImeInfoRec *hangul_info = NULL;
    hangul_session_t *hangul_session;

    hangul_session = (hangul_session_t *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    DEBUG_printf ("hangul_Create_Session ======= begin get ime_session_data: %p\n",
                  hangul_session);

    if (hangul_session == NULL) {
        DEBUG_printf("hangul_Create_Session ======= begin calloc for hangul_session\n");
        hangul_session = (hangul_session_t *) calloc(1, sizeof(hangul_session_t));
        if (hangul_session == NULL) {
            return (IME_FAIL);
        }

        /* init hangul core */
        hangul_session->preedit[0] = '\0';
        hangul_session->state = HANGUL_STATE_ENGLISH;
        hangul_session->hic = hangul_ic_new(config.keyboard);
        hangul_session->candidate_key = NULL;
        hangul_session->candidates = NULL;
        hangul_session->candidate_n_per_page = 9;

        imm_result = imm_services->ImmSetData(ic, IME_SCOPE_SESSION, hangul_session);
        if (imm_result == IMM_FAIL) {
            free((char *) hangul_session);
            return (IME_FAIL);
        }
    }
    return (IME_OK);
}

ImeResult hangul_Destroy_Session(ImeInputContext ic)
{
    hangul_session_t *hangul_session;

    hangul_session = (hangul_session_t *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    DEBUG_printf (" ====>hangul_Destroy_Session ======= begin get ime_session_data: %p\n",
                hangul_session);

    if (hangul_session != NULL) {
        hangul_ic_delete(hangul_session->hic);
        free((char *) hangul_session);
    }

    imm_services->ImmSetData(ic, IME_SCOPE_SESSION, NULL);

    return (IME_OK);
}

ImeResult hangul_FocusIn(ImeInputContext ic)
{
    DEBUG_printf("codetable: call hangul_FocusIn()\n");

    hangul_session_t *hangul_session = NULL;
    hangul_session = (hangul_session_t *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    if (hangul_session == NULL) {
        return (IME_FAIL);
    }

    hangul_update_preedit(ic, hangul_session);

    return (IME_OK);
}

ImeResult hangul_FocusOut(ImeInputContext ic)
{
    DEBUG_printf("codetable: call hangul_FocusOut()\n");

    hangul_session_t *hangul_session = NULL;
    hangul_session = (hangul_session_t *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    if (hangul_session == NULL) {
        return (IME_FAIL);
    }

    if (hangul_session->candidates == NULL) {
        hangul_flush(ic, hangul_session);
    }

    return (IME_OK);
}

ImmResult hangul_beep(ImeInputContext ic, hangul_session_t * hangul_session)
{
    return (imm_services->ImmBeep(ic, ImeBeepWarning));
}


/* process key input event */
/* return value:  
 * IME_UNUSED_KEY:if IME don't use this key,return it to systerm directly 
 * IME_OK: if IME has used this key 
 */
ImeResult hangul_Process_Key_Event(ImeInputContext ic, ImeKey key_event)
{
    hangul_session_t *hangul_session = NULL;

    long kcode, kstate;
    unsigned short kchar;
        
    DEBUG_printf("hangul_Process_Key_Event: ic: %p\n", ic);
    hangul_session = (hangul_session_t *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION); 
    if (hangul_session == NULL) {
        return (IME_UNUSED_KEY);
    }

    kcode = key_event->keycode;
    kchar = key_event->keychar;
    kstate = key_event->modifier;

    DEBUG_printf ("  ====> Hangul processing key (0X%X - 0X%X - 0X%X) ...\n",
                    kcode, kchar, kstate);

    hangul_Set_Ime_Properties(ic, hangul_session);

    if (hangul_session->candidates != NULL) {
        switch (kcode) {
        case IM_VK_0:
            hangul_lookup_commit_nth(ic, hangul_session, 9);
            break;
        case IM_VK_1:
        case IM_VK_2:
        case IM_VK_3:
        case IM_VK_4:
        case IM_VK_5:
        case IM_VK_6:
        case IM_VK_7:
        case IM_VK_8:
        case IM_VK_9:
            hangul_lookup_commit_nth(ic, hangul_session, kcode - IM_VK_1);
            break;
        case IM_VK_ENTER:
            hangul_lookup_commit(ic, hangul_session);
            break;
        case IM_VK_ESCAPE:
            hangul_lookup_done(ic, hangul_session);
            break;
        case IM_VK_UP:
        case IM_VK_K:
            hangul_lookup_prev(ic, hangul_session);
            break;
        case IM_VK_DOWN:
        case IM_VK_J:
            hangul_lookup_next(ic, hangul_session);
            break;
        case IM_VK_PAGE_UP:
        case IM_VK_LEFT:
        case IM_VK_H:
            hangul_lookup_prev_page(ic, hangul_session);
            break;
        case IM_VK_PAGE_DOWN:
        case IM_VK_RIGHT:
        case IM_VK_L:
        case IM_VK_SPACE:
            hangul_lookup_next_page(ic, hangul_session);
            break;
        default:
            break;
        }

        hangul_update_preedit(ic, hangul_session);

        return (IME_OK);
    }

    if (hangul_is_candidate_key(key_event)) {
        hangul_lookup_start(ic, hangul_session, key_event);
        return (IME_OK);
    }

    if (kcode == IM_VK_SHIFT) {
        return (IME_UNUSED_KEY);
    }

    if (kstate & (IM_CTRL_MASK | IM_ALT_MASK | IM_META_MASK)) {
        return (IME_UNUSED_KEY);
    }

    int preedit_len = strlen(hangul_session->preedit);
    if (hangul_is_backspace(key_event)) {
        bool ret = hangul_ic_backspace(hangul_session->hic);
        if (ret) {
            hangul_update_preedit(ic, hangul_session);
            return (IME_OK);
        } else if (preedit_len > 0) {
            hangul_session->preedit[preedit_len-sizeof(UTFCHAR)-1] = '\0';
            hangul_update_preedit(ic, hangul_session);
            return (IME_OK);
        }
    } else if (preedit_len > 0) {
        if (!hangul_ic_get_commit_string (hangul_session->hic)) {
            hangul_session->preedit[preedit_len-sizeof(UTFCHAR)-1] = '\0';
        }    
    }

    if (kchar >= ' ' && kchar <= '~') {
        bool ret;

        if (kstate & IM_SHIFT_MASK) {
            if (islower(kchar))
            kchar = toupper(kchar);
        } else {
            if (isupper(kchar))
            kchar = tolower(kchar);
        }

        ret = hangul_ic_process(hangul_session->hic, kchar);

        ucschar* str = hangul_ic_get_commit_string (hangul_session->hic);
        int ucslen = ucs4_strlen(str);

        char* utfstr = utf8_from_ucs4 (hangul_session, str, ucslen);
        if (kchar == ' ') {
            strcat(utfstr, " ");
        }
        int len = strlen(utfstr);

        if (len >= 0) {
            if (hangul_session->commit_by_word) {

                // if preedit length exceeds MAX_STR_LEN,
                // flush the previous preedit strings.
                DEBUG_printf("   preedit_len+len=%d\n", preedit_len+len);
                if (preedit_len+len >= MAX_STR_LEN-sizeof(UTFCHAR)-1) {
                    DEBUG_printf("    preedit length exceeds the maximun value.\n");
                    hangul_commit(ic, hangul_session);
                }
                strcat(hangul_session->preedit,utfstr);
            } else {
                if (kchar == ' ') {
                    strcat(hangul_session->preedit,utfstr);
                } else {
                    hangul_commit(ic, hangul_session);
                }
            }
        }

            if (hangul_ic_is_empty(hangul_session->hic)) {
                hangul_flush(ic, hangul_session);
            }
			/*aijin
        if (hangul_session->commit_by_word) {
            if (hangul_ic_is_empty(hangul_session->hic)) {
                hangul_flush(ic, hangul_session);
            }
        }
		*/

        hangul_update_preedit(ic, hangul_session);

        if (ret || kchar == ' ') {
            return (IME_OK);
        } else {
            return (IME_UNUSED_KEY);
        }
    }

    hangul_flush(ic, hangul_session);

    return (IME_UNUSED_KEY);
}
