/* 
 * Copyright (C) 2008 Sun Microsystems
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Aijin Kim <Aijin.Kim@Sun.COM>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <X11/Xmd.h>

#include "debug.h"
#include "hangul_im.h"
#include "hangul_composer.h"

extern HanjaTable* hanja_table;
extern ImmServices imm_services;

size_t ucs4_strlen(const ucschar* str)
{
    const ucschar* start = str;
    while (*str != 0) {
        str++;
    }
    return str - start;
}

int utf8_from_ucscode(char* buf, ucschar c)
{
    if (c < 0x80) {
        if (buf != NULL) {
            buf[0] = c;
        }
        return 1;
    } else if (c < 0x800) {
        if (buf != NULL) {
            buf[1] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[0] =  c | 0xc0;
        }
        return 2;
    } else if (c < 0x10000) {
        if (buf != NULL) {
            buf[2] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[1] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[0] =  c | 0xe0;
        }
        return 3;
    } else if (c < 0x200000) {
        if (buf != NULL) {
            buf[3] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[2] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[1] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[0] =  c | 0xf0;
        }
        return 4;
    } else if (c < 0x4000000) {
        if (buf != NULL) {
            buf[4] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[3] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[2] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[1] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[0] =  c | 0xf8;
        }
        return 5;
    } else {
        if (buf != NULL) {
            buf[5] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[4] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[3] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[2] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[1] = (c & 0x3f) | 0x80;
            c >>= 6;
            buf[0] =  c | 0xfc;
        }
        return 6;
    }
}

char* utf8_from_ucs4(hangul_session_t* hs, const ucschar* str, ssize_t len)
{
    int i, utf8len = 0;
    char* ret;

    if (len < 0)
        len = ucs4_strlen(str);

    for (i = 0, utf8len = 0; i < len; ++i) {
        utf8len += utf8_from_ucscode(NULL, str[i]);
    }

    ret = calloc(utf8len+1, sizeof(char));
    for (i = 0, utf8len = 0; i < len; ++i) {
        utf8len += utf8_from_ucscode(ret + utf8len, str[i]);
    }

    ret[utf8len] = '\0';

    return ret;
}

ImmResult hangul_update_preedit(ImeInputContext ic, hangul_session_t * hs)
{
    DEBUG_printf("hangul_update_preedit\n");

    ImePreeditRec ime_preedit;

    int preedit_len = 0;
    const ucschar* str;
    const char* utfstr;
    char preedit_str[MAX_STR_LEN];

    str = hangul_ic_get_preedit_string(hs->hic);
    utfstr = utf8_from_ucs4 (hs, str, ucs4_strlen(str));
    
    strcpy(preedit_str,hs->preedit);
    preedit_len = strlen(hs->preedit);

    strcat(preedit_str, utfstr);
    preedit_len += strlen(utfstr);

    if (preedit_len <= 0) {
        return (imm_services->ImmHidePreedit(ic));
    }

    DEBUG_printf("    preedit_str: %s\n",preedit_str);
    DEBUG_printf("    preedit_len: %d\n",preedit_len);
        
    imm_services->ImmShowPreedit(ic);

    memset(&ime_preedit, 0, sizeof(ImePreeditRec));
    ime_preedit.caret = preedit_len/(sizeof(UTFCHAR)+1);
    ime_preedit.preedit.text = (unsigned char*)preedit_str;

    return (imm_services->ImmUpdatePreedit(ic, &ime_preedit));
}

void hangul_reset(hangul_session_t *hs)
{
    DEBUG_printf("hangul_reset\n");

    hangul_ic_reset(hs->hic);

    //reset preedit buffer
    hs->preedit[0] = '\0';
}

int  hangul_flush(ImeInputContext ic, hangul_session_t *hs)
{
    DEBUG_printf("hangul_flush\n");

    int len = 0;
    const ucschar* str;
    const char* utfstr;
    char flush_str[MAX_STR_LEN];

    imm_services->ImmHidePreedit(ic);
    strcpy(flush_str,hs->preedit);

    str = hangul_ic_flush(hs->hic);
    utfstr = utf8_from_ucs4(hs, str, ucs4_strlen(str));

    len = strlen(utfstr);
    if (len > 0) {
        strcat(flush_str,utfstr);
    }

    DEBUG_printf("    ic_flush: %s preedit_str: %s flush_str: %s\n",utfstr,hs->preedit,flush_str);

        imm_services->ImmCommit(ic, (unsigned char*)flush_str);

    // reset preedit buffer
    hs->preedit[0] = '\0';

    return len;
}

ImmResult hangul_commit(ImeInputContext ic, hangul_session_t * hs)
{
    DEBUG_printf("hangul_commit\n");

    int len;
    const ucschar* str;
    const char* utfstr;
    char commit_str[MAX_STR_LEN];

    strcpy(commit_str,hs->preedit);

    str = hangul_ic_get_commit_string(hs->hic);
    utfstr = utf8_from_ucs4 (hs, str, ucs4_strlen(str));
    len = strlen(utfstr);
    if (len > 0) {
        strcat(commit_str,utfstr);
    }

    DEBUG_printf("    ic_commit: %s preedit_str: %s commit_str: %s\n",utfstr,hs->preedit,commit_str);

    imm_services->ImmCommit(ic, (unsigned char*)commit_str);

    // reset preedit buffer
    hs->preedit[0] = '\0';

    return (IMM_OK);
}

Bool hangul_is_candidate_key(ImeKey key_event)
{
    Bool ret = False;        
    long kcode;

    kcode = key_event->keycode;

    if ( kcode == IM_VK_F9 || kcode == IM_VK_HANJA) {
        ret = True;
    }

    return ret;
}

Bool hangul_is_backspace(ImeKey key_event)
{
    Bool ret = False;
    long kcode;

    kcode = key_event->keycode;

    if (kcode == IM_VK_BACK_SPACE) {
        ret = True;
    }

    return ret;
}

ImmResult hangul_lookup_prev(ImeInputContext ic, hangul_session_t *hs)
{
    ImmResult imm_result;

    if (hs->candidate_cur - 1 >= 0) {
        hs->candidate_cur--;
    } else {
        if (hs->candidate_page - hs->candidate_n_per_page >= 0) {
            hs->candidate_page -= hs->candidate_n_per_page;
            hs->candidate_cur = hs->candidate_n_per_page - 1;
        }
    }

    imm_result = hangul_update_candidates(ic, hs);

    return (imm_result);
}

ImmResult hangul_lookup_next(ImeInputContext ic, hangul_session_t *hs)
{
    ImmResult imm_result;
    int total_candidates;

    total_candidates = hanja_list_get_size(hs->candidates);

    if (hs->candidate_cur + 1 <  hs->candidate_n_per_page) {
        if (hs->candidate_page + hs->candidate_cur + 1 < total_candidates) {
            hs->candidate_cur++;
        }
    } else {
        if (hs->candidate_page + hs->candidate_n_per_page < total_candidates) {
            hs->candidate_page += hs->candidate_n_per_page;
            hs->candidate_cur = 0;
        }
    }

    imm_result = hangul_update_candidates(ic, hs);

    return (imm_result);
}

ImmResult hangul_lookup_prev_page(ImeInputContext ic, hangul_session_t *hs)
{
    ImmResult imm_result;

    if (hs->candidate_page - hs->candidate_n_per_page >= 0) {
        hs->candidate_page -= hs->candidate_n_per_page;
    }

    imm_result = hangul_update_candidates(ic, hs);

    return (imm_result);
}

ImmResult hangul_lookup_next_page(ImeInputContext ic, hangul_session_t *hs)
{
    ImmResult imm_result;
    int total_candidates;

    total_candidates = hanja_list_get_size(hs->candidates);

    if (hs->candidate_page + hs->candidate_n_per_page < total_candidates) {
        hs->candidate_page += hs->candidate_n_per_page;

        if (hs->candidate_page + hs->candidate_cur > total_candidates) {
            hs->candidate_cur = total_candidates - hs->candidate_page - 1;
        }
    }

    imm_result = hangul_update_candidates(ic, hs);

    return (imm_result);
}

ImmResult hangul_update_candidates(ImeInputContext ic, hangul_session_t *hs)
{
    ImmResult imm_result;
    ImeCandidatesRec ime_candidates;
    int i, total_candidates, num_candidates;
    int page_state = 0;

    total_candidates = hanja_list_get_size(hs->candidates);
    num_candidates = hs->candidate_n_per_page;

    if (hs->candidate_page == 0) {
        page_state |= ImeCandidatesFirstPage;
    }

    if (hs->candidate_page + num_candidates > total_candidates) {
        num_candidates = total_candidates - hs->candidate_page;
        page_state |= ImeCandidatesLastPage; 
    }

    memset(&ime_candidates, 0, sizeof(ImeCandidatesRec));

    ime_candidates.title = (unsigned char*)hs->candidate_key;
    ime_candidates.numbers = NULL;
    ime_candidates.focus = 0;
    ime_candidates.page_state = page_state;
    ime_candidates.count = num_candidates;
    ime_candidates.candidates = (ImeTextRec *) calloc(num_candidates, sizeof(ImeTextRec));

    if (ime_candidates.candidates == NULL)
        return (IMM_FAIL);

    for (i = 0; i < num_candidates; i++) {
        char str[512];
        const char* value;
        const char* comment;

        value = hanja_list_get_nth_value(hs->candidates,
                                        hs->candidate_page + i);
        comment = hanja_list_get_nth_comment(hs->candidates,
                                            hs->candidate_page + i);
        snprintf(str, sizeof(str), "%s %s", value, comment);

		/*
        char* candidate_text = strdup(str);
        ime_candidates.candidates[i].text = (unsigned char*)candidate_text;
        free(candidate_text);
		*/
        ime_candidates.candidates[i].text = (unsigned char*)strdup(str);
    }

    imm_services->ImmShowCandidates(ic);

    imm_result = imm_services->ImmUpdateCandidates(ic, &ime_candidates);

    free ((char *)ime_candidates.candidates);

    for (i = 0; i < num_candidates; i++) {
        free (ime_candidates.candidates[i].text);
    }

    return imm_result;
}

ImmResult hangul_lookup_start(ImeInputContext ic, hangul_session_t *hs, ImeKey key)
{
    ImmResult imm_result;

    const char* utfstr;
    const ucschar* str;
    int i, num_candidates;
    char preedit_str[MAX_STR_LEN];

    /* find candidate */
    str = hangul_ic_get_preedit_string(hs->hic);
    utfstr = utf8_from_ucs4(hs, str, -1);

    strcpy(preedit_str,hs->preedit);
    strcat(preedit_str,utfstr);

    hs->candidates = hanja_table_match_prefix(hanja_table, preedit_str);
    if (hs->candidates == NULL) {
        /* fail to find candidate */
        return (imm_services->ImmHideCandidates(ic));    
    }
    num_candidates = hanja_list_get_size(hs->candidates);

    hs->state = HANGUL_STATE_HANJA;
    hs->candidate_page = 0;
    hs->candidate_cur = 0;

    char* keystr = strdup(utf8_from_ucs4(hs, str, -1));
    hs->candidate_key =  keystr;

    imm_result = hangul_update_candidates(ic, hs);

    free(keystr);

    return (imm_result);
}

ImmResult hangul_lookup_done(ImeInputContext ic, hangul_session_t *hs)
{
    if (hs->candidates != NULL) {
        hanja_list_delete(hs->candidates);
        hs->candidates = NULL;
    }

    if (hs->candidate_key != NULL) {
        free(hs->candidate_key);
        hs->candidate_key = 0;
    }

    hs->candidate_page = 0;
    hs->candidate_cur = 0;
    hs->state = HANGUL_STATE_HANGUL;

    return (imm_services->ImmHideCandidates(ic));
}

const char* hangul_lookup_get_nth(hangul_session_t* hs, int n)
{
    return hanja_list_get_nth_value(hs->candidates, hs->candidate_page + n);
}

const char* hangul_lookup_get_current(hangul_session_t* hs)
{
    return hangul_lookup_get_nth(hs, hs->candidate_cur);
}

const char* hangul_compose_commit_string(hangul_session_t* hs, const char* hanja_str)
{
    char* commit_str;
    char preedit_str[MAX_STR_LEN];
    char tmp_str[MAX_STR_LEN];
    const char* utfstr;
    const ucschar* str;

    str = hangul_ic_get_preedit_string(hs->hic);
    utfstr = utf8_from_ucs4(hs, str, -1);

    strcpy(preedit_str,hs->preedit);
    strcat(preedit_str,utfstr);
	
    if (hs->commit_by_word) {
        if (strlen(hanja_str) < strlen(preedit_str)) {
            char* tmp = &hs->preedit[strlen(hanja_str)];
            strcpy(hs->preedit,tmp);
        } else {
            hangul_reset(hs);
        }
    } else { 
        hangul_reset(hs);
	}

    switch (hs->hanja_conversion) {
        case HANJA_CONVERSION_HANJA:
            commit_str = strdup(hanja_str);
            break;
        case HANJA_CONVERSION_HANGUL_HANJA:
            preedit_str[strlen(hanja_str)] = '\0';
            snprintf(tmp_str,sizeof(tmp_str),"%s(%s)",preedit_str,hanja_str);
            commit_str = strdup(tmp_str);
            break;
        case HANJA_CONVERSION_HANJA_HANGUL:
            preedit_str[strlen(hanja_str)] = '\0';
            snprintf(tmp_str,sizeof(tmp_str),"%s(%s)",hanja_str,preedit_str);
            commit_str = strdup(tmp_str);
            break;
        default:
            commit_str = strdup(hanja_str);
            break; 
    }

    return (const char*)commit_str;
}

ImmResult hangul_lookup_commit(ImeInputContext ic, hangul_session_t *hs)
{
    const char* hanja_str;
    const char* str;

    hanja_str = hangul_lookup_get_current(hs);
    str = hangul_compose_commit_string(hs, hanja_str);
    DEBUG_printf("commit_str : %s\n",str);

    hangul_lookup_done(ic, hs);
    imm_services->ImmCommit(ic, (unsigned char*)str);

    hs->state = HANGUL_STATE_HANGUL;

    free(str);

    return (IMM_OK);
}

ImmResult hangul_lookup_commit_nth(ImeInputContext ic, hangul_session_t *hs, int n)
{
    const char* hanja_str;
    const char* str;

    hanja_str = hangul_lookup_get_nth(hs, n);
    str = hangul_compose_commit_string(hs, hanja_str);
    DEBUG_printf("commit_str : %s\n",str);

    hangul_lookup_done(ic, hs);
    imm_services->ImmCommit(ic, (unsigned char*)str);
    hs->state = HANGUL_STATE_HANGUL;

    free(str);

    return (IMM_OK);
}

void get_striped(char *src, char *dest)
{
    char *p;

    /* ignore white space from start */
    while (isspace(*src)) {
        src++;
    }

    /* ignore white space at the end */
    for (p = src; !isspace(*p); p++) {
        continue;
    }

    *p = 0;

    strcpy(dest, src);
}

void get_key_value(char *buf, char *key, char *value)
{
    char *p;
    char *end = NULL;

    /* we silently ignore that has no '=' mark */
    p = strchr(buf, '=');
    if (p == NULL) {
        return;
    }

    p = strchr(buf, '#');
    if (p != NULL) {
        *p = 0;
    }

    p = strtok_r(buf, "=", &end);
    if (p != NULL) {
        get_striped(p, key);
    }

    p = strtok_r(NULL, "=", &end);
    if (p != NULL) {
        get_striped(p, value);
    }
}

