/*

Copyright (c) 2006 Yuuhei TERAMOTO <tera@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdarg.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include "syscfg.h"
#include "transintrn.h"

static void assert_( int val, char * fmt,... );

pgc_trans_rec *
pgc_trans_create()
{
    pgc_trans_rec * tr = (pgc_trans_rec*)malloc( sizeof(*tr) );
    if( tr == NULL ){
	return NULL;
    }
    memset( tr, 0, sizeof(*tr) );
    return tr;
}

pgc_trans_rec *
pgc_trans_dup( 
    pgc_trans_rec * trsl
)
{
    pgc_trans_rec * tr;
    if( trsl == NULL ){
	return NULL;
    }
    tr = (pgc_trans_rec*)malloc( sizeof(*tr) );
    if( tr == NULL ){
	return NULL;
    }
    memcpy( tr, trsl, sizeof(*tr) );
    if( trsl->data != NULL ){
	tr->data = (pgc_trans_unit_rec*)malloc( sizeof(*trsl->data) * trsl->bufsize );
	if( tr->data == NULL ){
	    free( tr );
	    return NULL;
	}
	memcpy( tr->data, trsl->data, sizeof(*trsl->data) * trsl->bufsize);
    }
    tr->next = NULL;
    return tr;
}

void
pgc_trans_destroy(
    pgc_trans_rec * trsl
)
{
    if( trsl != NULL ){
	free( trsl->data );
	free( trsl );
    }
}

pgc_trans_rec *
pgc_trans_add(
    pgc_trans_rec * trsl,
    pgc_trans_unit_rec *unit
)
{
    if( trsl == NULL ){
	return NULL;
    }
    if( trsl->len == trsl->bufsize ){
	pgc_trans_unit_rec *p = (pgc_trans_unit_rec*)realloc( trsl->data, 
		  sizeof(*trsl->data) * (trsl->bufsize+PGC_TRANS_BUFUNIT) );
	if( p == NULL ){
	    return NULL;
	}
	trsl->data = p;
	trsl->bufsize += PGC_TRANS_BUFUNIT;
    }
    *(trsl->data + trsl->len++) = *unit;
    return trsl;
}

int
set_translation_unit(
    pgc_trans_unit_rec *unit,
    unsigned char * p,
    size_t size,
    size_t src_s,
    size_t src_e
)
{
    if( unit == NULL ){
	return -1;
    }
    if( size > sizeof(unit->str) ){
	assert_( 0, "pgc_trans>>set_translation_unit() size too big" );
	return -1;
    }
    memcpy( unit->str, p, size );
    unit->bytes = size;
    unit->src_s = src_s;
    unit->src_e = src_e;
    return 0;
}

int
pgc_trans_reencode(
    iconv_t icd,
    pgc_trans_rec *trans
)
{
    size_t ii;

    if( trans == NULL ){
	return 0;
    }
    if( trans->data == NULL ){
	return 0;
    }
    if( trans->len == 0 ){
	return 0;
    }
    for( ii = 0 ; ii < trans->len ; ii++ ){
	char *inbuf, *outbuf;
	size_t inbytesleft, outbytesleft;
	pgc_trans_unit_rec * unit = trans->data + ii;
	if( unit != NULL ){
	    int outsize;
	    unsigned char buf[PGC_TRANS_LENMAX*PGC_TRANS_CHARMAX+1];
	    inbuf = (char*)unit->str;
	    outbuf = (char*)buf;
	    inbytesleft = unit->bytes;
	    outbytesleft = sizeof(buf);
	    if( iconv( icd, &inbuf, &inbytesleft, &outbuf, &outbytesleft ) == -1 ){
		return -1;
	    }
	    if( outbytesleft == 0 ){
		return -1;
	    }
	    outsize = sizeof(buf) - outbytesleft;
	    memcpy( unit->str, buf, outsize );
	    unit->bytes = outsize;
	}
    }
    return 0;
}

int
pgc_trans_unit_count(
    pgc_trans_rec * trans,
    size_t *count
)
{
    if( trans == NULL ){
	*count = 0;
    }else{
	*count = trans->len;
    }
    return 0;
}

int
pgc_trans_get_string(
    pgc_trans_rec * trans,
    size_t su,
    size_t eu,
    size_t *sp,
    size_t *ep,
    unsigned char *buf,
    size_t bufsize
)
{
    size_t ii, bytes=0, left= bufsize;
    *sp = *ep = 0;

    if( trans == NULL || trans->len == 0 ){
	return 0;
    }
    if( eu == PGC_TRANS_LAST ){
	eu = trans->len - 1;
    }
    assert_( su >= 0 && su <= eu,
	     "pgc_trans>>pgc_trans_get_string() su > eu" );
    for( ii = 0 ; ii <= eu ; ii++ ){
	pgc_trans_unit_rec *unit = trans->data + ii;
	if( ii == su ){
	    *sp = bytes;
	}
	if( buf != NULL ){
	    if( su <= ii && ii <= eu ){
		if( left < unit->bytes ){
		    return -1;
		}
		memcpy( buf + bytes - *sp, unit->str, unit->bytes );
		left -= unit->bytes;
	    }
	}
	bytes += unit->bytes;
	if( ii == eu ){
	    *ep = bytes;
	}
    }
    return 0;
}

int
pgc_trans_strcmp(
    pgc_trans_rec * t1,
    pgc_trans_rec * t2
)
{
    int ret=0;
    size_t l1=0,l2=0,b1=0,b2=0;
    while( 1 ){
	unsigned char c1=0,c2=0;
	if( l1 < t1->len ){
	    pgc_trans_unit_rec *u = t1->data + l1;
	    if( u->bytes == b1 ){
		while( l1 < t1->len ){
		    u = t1->data + (l1++);
		    if( u->bytes > 0 ){
			c1 = u->str[0];
			b1 = 1;
			break;
		    }
		}
	    }else{
		c1 = u->str[b1++];
	    }
	}
	if( l2 < t2->len ){
	    pgc_trans_unit_rec *u = t2->data + l2;
	    if( u->bytes == b2 ){
		while( l2 < t2->len ){
		    u = t2->data + (l2++);
		    if( u->bytes > 0 ){
			c2 = u->str[0];
			b2 = 1;
			break;
		    }
		}
	    }else{
		c2 = u->str[b2++];
	    }
	}
	if( c1 != c2 ){
	    ret = c1 < c2 ? -1 : 1;
	    break;
	}else if( c1 == 0 && c2 == 0 ){
	    ret = 0;
	    break;
	}
    }
    return ret;
}


int
pgc_trans_get_source_pos(
    pgc_trans_rec * trans,
    size_t su,
    size_t eu,
    size_t *sp,
    size_t *ep
)
{
    pgc_trans_unit_rec *unit;
    if( trans == NULL || trans->len == 0 ){
	*sp = *ep = 0;
	return 0;
    }
    if( eu == PGC_TRANS_LAST ){
	eu = trans->len - 1;
    }
    assert_( su < trans->len && eu < trans->len, 
	     "pgc_trans>>pgc_trans_get_source_pos() su:%d,eu:%d invalid", su, eu );
    unit = trans->data + su;
    *sp = unit->src_s;
    unit = trans->data + eu;
    *ep = unit->src_e;
    return 0;
}

void
pgc_trans_print(
    pgc_trans_rec * trans,
    FILE *fp
)
{
    unsigned char *s=NULL;
    size_t size,sp,ep;

    fprintf( fp, "#### Phonogram Conv Translation\n" );
    if( trans == NULL ){
	fprintf( fp,"\ttransilation is null\n");
	goto end;
    }

    if( pgc_trans_get_string( trans, 0, PGC_TRANS_LAST, &sp, &ep, NULL, 0 ) != 0 ){
	goto end;
    }
    size = ep - sp;
    s = (unsigned char*)malloc(size+1);
    if( pgc_trans_get_string( trans, 0, PGC_TRANS_LAST, &sp, &ep , s, size ) != 0 ){
	goto end;
    }
    s[size] = 0;

    fprintf( fp, "\tString:\"%s\"(%d translation units)\n", s, trans->len );
    
end:
    free( s );
}

void
pgc_trans_list_destroy(
    pgc_trans_rec *lst
)
{
    pgc_trans_rec *cur = lst;
    while( cur != NULL ){
	pgc_trans_rec *t = cur;
	cur = cur->next;
	pgc_trans_destroy( t );
    }
}

pgc_trans_rec*
pgc_trans_list_next(
    pgc_trans_rec *lst, pgc_trans_rec * cur
)
{
    if( lst == NULL ){
	return NULL;
    }
    if( cur == NULL ){
	return lst;
    }else{
	return cur->next;
    }
}

int
pgc_trans_list_size(
    pgc_trans_rec *lst,
    size_t *size
)
{
    size_t sz = 0;
    pgc_trans_rec *cur = lst;
    while( cur != NULL ){
	cur = cur->next;
	sz++;
    }
    *size = sz;
    return 0;
}

pgc_trans_rec *
pgc_trans_list_append(
    pgc_trans_rec * lst,
    pgc_trans_rec * trans
)
{
    pgc_trans_rec *tr;
    if( lst == NULL ){
	return trans;
    }
    for( tr = lst; tr->next != NULL ; tr = tr->next ){
    }
    tr->next = trans;
    return lst;
}

pgc_trans_rec *
pgc_trans_list_remove(
    pgc_trans_rec * lst,
    pgc_trans_rec * trans
)
{
    pgc_trans_rec *tr,*prev=NULL;

    for( tr = lst; tr != NULL ; tr = tr->next ){
	if( tr == trans ){
	    pgc_trans_rec *next = tr->next;
	    trans->next = NULL;
	    if( prev == NULL ){
		return next;
	    }else{
		prev->next = next;
		return lst;
	    }
	}
	prev = tr;
	tr = tr->next;
    }
    return lst;
}

static void assert_( 
    int val,
    char * fmt,
    ... 
)
{
  static char *pre = "ASSERT!";
  char *p,*pa = NULL;
  char buf[256];
  va_list va;

  if( val ){
      return;
  }
  va_start( va, fmt );
  if( strlen(fmt) + strlen(pre) > sizeof(buf) - 2 ){
    pa = (char*)malloc( strlen(fmt) + strlen(pre) + 2 );
    p = pa;
  }else{
    p = buf;
  }
  strcpy( p, pre );
  strcat( p, fmt );
  strcat( p, "\n" );
  vfprintf(stderr, p, va );
  va_end( va );
  free( pa );  		      
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
