/*

Copyright (c) 2006 Kenji ABE <abek@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "convdef.h"
#include "imcaret.h"
#include "imcommit.h"
#include "imconv.h"
#include "imeventhandler.h"
#include "iminput.h"
#include "imlookup.h"
#include "impreedit.h"
#include "imstatus.h"
#include "imtext.h"


static Bool convert_letter_type_sub(iml_session_t *session, IMKeyEventStruct *key_event, enum IM_INPUT_LETTER_TYPE type, iml_inst **inst_list);


Bool
handle_preedit_append(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch(psd->im_state) {
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       if (!isascii(key_event->keyChar) ||
	   !isprint(key_event->keyChar)) {
	   return False;
       }

       handle_commit(session, key_event, inst_list);
       ASSERT(psd->im_state == IM_STATE_EDIT_EMPTY);
       /* not break, but fall-through. */
      case IM_STATE_EDIT_EMPTY:
	   if (key_event->modifier & (IM_CTRL_MASK|IM_ALT_MASK)) {
	       return handle_send_back_key(session, key_event, inst_list);
	   }
      case IM_STATE_EDITING:
       if (isascii(key_event->keyChar) &&
	   isprint(key_event->keyChar)) {
	   im_input_add_char(psd->im_input, key_event->keyChar);

	   if (psd->im_state == IM_STATE_EDIT_EMPTY &&
	       !im_input_empty_p(psd->im_input)) {
	       change_im_state(session, IM_STATE_EDITING, inst_list, False);
	   }

	   make_preedit_draw_inst(session, inst_list);
	   return True;
       }

       if (psd->im_state != IM_STATE_EDIT_EMPTY) {
	   break;
       }

       /* not break, but fall-through. */
      case IM_STATE_OFF:
       return handle_send_back_key(session, key_event, inst_list);

      default:
       ASSERT(0); /* cannot happen */
       break;
    }

    return False;
}


Bool
handle_delete_prev(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       if (!change_im_state(session, IM_STATE_EDITING, inst_list, True)) {
	   break;
       }
       /* not break, but fall-through. */
      case IM_STATE_EDITING:
       im_input_delete_prev(psd->im_input);

       if (im_input_empty_p(psd->im_input)) {
	   change_im_state(session, IM_STATE_EDIT_EMPTY, inst_list, True);
       } else {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_input_make_preedit_text(psd->im_input, session, IMUnderline),
	       im_input_cur_pos(psd->im_input),
	       inst_list);
       }

       return True;

      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      default:
       ASSERT(0); /* cannot happen */
       break;
    }
    return False;
}

Bool
handle_delete_next(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       if (!change_im_state(session, IM_STATE_EDITING, inst_list, True)) {
	   break;
       }
       /* not break, but fall-through. */
      case IM_STATE_EDITING:
       im_input_delete_next(psd->im_input);

       if (im_input_empty_p(psd->im_input)) {
	   change_im_state(session, IM_STATE_EDIT_EMPTY, inst_list, True);
       } else {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_input_make_preedit_text(psd->im_input, session, IMUnderline),
	       im_input_cur_pos(psd->im_input),
	       inst_list);
       }

       return True;

      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      default:
       ASSERT(0); /* cannot happen */
       break;
    }
    return False;
}

Bool
handle_input_mode_prev(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->im_state == IM_STATE_EDIT_EMPTY) {
	enum IM_INPUT_LETTER_TYPE type = im_input_get_input_letter_type(psd->im_input);
	if (type <= IM_INPUT_LETTER_TYPE_MIN) {
	    type = IM_INPUT_LETTER_TYPE_MAX;
	} else {
	    --type;
	}
	im_input_set_input_letter_type(psd->im_input, type);

	make_status_draw_inst(session, inst_list);

	return True;
    }
    return False;
}

Bool
handle_input_mode_next(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->im_state == IM_STATE_EDIT_EMPTY) {
	enum IM_INPUT_LETTER_TYPE type = im_input_get_input_letter_type(psd->im_input);
	if (IM_INPUT_LETTER_TYPE_MAX <= type) {
	    type = IM_INPUT_LETTER_TYPE_MIN;
	} else {
	    ++type;
	}
	im_input_set_input_letter_type(psd->im_input, type);

	make_status_draw_inst(session, inst_list);

	return True;
    }
    return False;
}

Bool
handle_convert_to_hiragana(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    return convert_letter_type_sub(session, key_event, IM_INPUT_LETTER_TYPE_HIRAGANA, inst_list);
}

Bool
handle_convert_to_katakana(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    return convert_letter_type_sub(session, key_event, IM_INPUT_LETTER_TYPE_KATAKANA, inst_list);
}

Bool
handle_convert_to_halfwidth_katakana(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    return convert_letter_type_sub(session, key_event, IM_INPUT_LETTER_TYPE_HW_KATAKANA, inst_list);
}

Bool
handle_convert_to_fullwidth_alpha(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    return convert_letter_type_sub(session, key_event, IM_INPUT_LETTER_TYPE_FW_ALPHA, inst_list);
}

Bool
handle_convert_to_alpha(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    return convert_letter_type_sub(session, key_event, IM_INPUT_LETTER_TYPE_ALPHA, inst_list);
}

static
Bool
convert_letter_type_sub(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    enum IM_INPUT_LETTER_TYPE type,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    int s, e;
    UTFCHAR *str;
    int len;

    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
       if (!change_im_state(session, IM_STATE_LETTER_TYPE_CONVERSION, inst_list, False)) {
	   return False;
       }
       /* not break, but fall through. */
      case IM_STATE_LETTER_TYPE_CONVERSION:
       im_input_convert_letter_type(psd->im_input, type);
       break;

      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       if (!change_im_state(session, IM_STATE_CONVERSION, inst_list, False)) {
	   return False;
       }
       /* not break, but fall through. */
      case IM_STATE_CONVERSION:
       end_lookup_choice(session, inst_list);
       im_conv_cur_get_unconverted_range(psd->im_conv, &s, &e);
       str = im_input_get_converted_range_strig(psd->im_input, session, s, e, type, &len);
       im_conv_cur_set_string(psd->im_conv, str, len);
       break;

      default:
       ASSERT(0);
       return False;
    }

    make_preedit_draw_inst(session, inst_list);
    return True;
}

void
make_preedit_done_inst(
    iml_session_t *session,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

    if (psd->preedit_p) {
	iml_methods_t *mtbl = METHODS_TBL(session);
	iml_inst *inst = mtbl->iml_make_preedit_done_inst(session);
	mtbl->iml_link_inst_tail(inst_list, inst);

	psd->preedit_p = False;
    }
}

void
make_preedit_erase_inst(
    iml_session_t *session,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

    if (psd->preedit_p) {
	iml_methods_t *mtbl = METHODS_TBL(session);
	iml_inst *inst = mtbl->iml_make_preedit_erase_inst(session);
	mtbl->iml_link_inst_tail(inst_list, inst);

	psd->preedit_length = 0;
    }
}

void
make_preedit_draw_inst(
    iml_session_t *session,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_OFF:
       break;

      case IM_STATE_EDIT_EMPTY:
       make_preedit_erase_inst(session, inst_list);
       break;

      case IM_STATE_EDITING:
      case IM_STATE_LETTER_TYPE_CONVERSION:
       make_preedit_draw_inst_from_text(
	   session,
	   im_input_make_preedit_text(psd->im_input, session, (psd->im_state == IM_STATE_EDITING) ? IMUnderline : IMReverse),
	   im_input_cur_pos(psd->im_input),
	   inst_list
       );
       break;

      case IM_STATE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       make_preedit_draw_inst_from_text(
	   session,
	   im_conv_make_preedit_text(psd->im_conv, session),
	   im_conv_cur_pos(psd->im_conv),
	   inst_list
       );
       break;

      default:
       ASSERT(0); /* cannot happen */
       break;
    }
}

void
make_preedit_draw_inst_from_text(
    iml_session_t *session,
    IMText *im_txt,
    int cur_pos,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (im_txt && im_txt->char_length > 0) {
	iml_methods_t *mtbl = METHODS_TBL(session);
	iml_inst *inst;

	if (cur_pos < 0) {
	    cur_pos = 0;
	} else if (im_txt->char_length < cur_pos) {
	    cur_pos = im_txt->char_length;
	}

	if (!psd->preedit_p) {
	    psd->preedit_p = True;

	    inst = mtbl->iml_make_preedit_start_inst(session);
	    mtbl->iml_link_inst_tail(inst_list, inst);
	    mtbl->iml_execute(session, inst_list);
	    *inst_list = NULL;

	    inst = mtbl->iml_make_preedit_draw_inst(session, im_txt);
	    mtbl->iml_link_inst_tail(inst_list, inst);

	    if (cur_pos < im_txt->char_length) {
		make_caret_move_inst(session, cur_pos, inst_list);
	    }
	} else {
	    inst = mtbl->iml_make_preedit_draw_with_chgpos_inst(session, im_txt, 0, psd->preedit_length, cur_pos);
	    mtbl->iml_link_inst_tail(inst_list, inst);
	}

	psd->preedit_length = im_txt->char_length;
    } else {
	make_preedit_erase_inst(session, inst_list);
    }
}

void
make_preedit_update_inst_from_text(
    iml_session_t *session,
    IMText *im_txt,
    int chg_first,
    int chg_length,
    int cur_pos,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->preedit_p) {
	iml_methods_t *mtbl = METHODS_TBL(session);
	iml_inst *inst = mtbl->iml_make_preedit_draw_with_chgpos_inst(session, im_txt, chg_first, chg_length, cur_pos);
	mtbl->iml_link_inst_tail(inst_list, inst);

	psd->preedit_length -= chg_length;
	psd->preedit_length += im_txt->char_length;
    }
}

IMText *
make_preedit_text(
    iml_session_t *session
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       break;

      case IM_STATE_EDITING:
      case IM_STATE_LETTER_TYPE_CONVERSION:
       return im_input_make_preedit_text(psd->im_input, session, 0);

      case IM_STATE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       return im_conv_make_preedit_text(psd->im_conv, session);

      default:
       ASSERT(0); /* cannot happen */
       break;
    }

    return make_im_text(session, NULL, 0, 0);
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
